;+
; NAME:
;         p3d_misc_logger
;
;         $Id: p3d_misc_logger.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         Outputs a string to the screen, and also optionally, to a log file.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_logger,logstr,funit,rname=,loglevel=,topwid=,verbose=,error=
;
; INPUTS:
;         logstr          - The string to print. Can be an array of strings.
;
; OPTIONAL INPUTS:
;         funit           - If this two-element argument is set, where the
;                           first element is a file unit, then the output is
;                           also written to the file pointed to by this file
;                           unit. Before this file unit is used it is checked
;                           if it is open and can be written to; it is ignored
;                           if this is not the case. A second element must also
;                           be present specifying the logging level associated
;                           with the file (cf. LOGLEVEL).
;
; KEYWORD PARAMETERS:
;         rname ['p3d_misc_logger'] - The name of the routine calling.
;         loglevel [1]    - Before a string is written to funit[0] (if it is
;                           open) it is checked if LOGLEVEL>=funit[1], only
;                           then is the message written to the file.
;         topwid          - If set, then the error message (iff ERROR) is shown
;                           using a dialog instead of printing the message on
;                           the console.
;         verbose         - If set, then the log message is not only printed on
;                           a file (iff FUNIT), but is also output to the
;                           console.
;         error           - Shows the log message as an error, writes the same
;                           message to the log file (iff FUNIT), and thereafter
;                           closes the log file before returning.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required (~).
;
; EXAMPLE:
;         tmps='The warp drive is offline.'
;         ret=p3d_misc_logger(tmps,funit,rname='startrek_warpcore',/error)
;
; MODIFICATION HISTORY:
;         20.10.2008 - Introduced /CS
;
;-
FUNCTION p3d_misc_logger,logstr,funit,rname=rname,loglevel=loglevel, $
             topwid=topwid,verbose=verbose,error=error
  compile_opt hidden,IDL2

  on_error,1

  if ~n_elements(rname) then rname='p3d_misc_logger: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(loglevel) then loglevel=1
  if ~n_elements(topwid) then topwid=0L
  if ~n_params() then begin
    print,'% P3D_MISC_LOGGER: The number of input parameters must be >=1.'
    return,-1
  endif

  ;; Testing if the file unit is open and writeable:
  if ~n_elements(funit) then funit=[1L,verbose>1L]
  tmp=fstat(funit[0L])
  printtofile=tmp.open and (tmp.write?1L:0L) and funit[1L] ge loglevel

  n=n_elements(logstr)
  if n ne 0L and size(logstr,/type) eq 7L then begin

    if keyword_set(error) then begin

      ;; Printing to a logfile:
      if printtofile then $
         for i=0L,n-1L do printf,funit[0L],'% '+strupcase(rname)+logstr

      ;; Displaying the message:
      if widget_info(topwid,/valid_id) then begin
        ret=dialog_message(logstr,/error,dialog_parent=topwid, $
                title=rname+'Error!',/center)
      endif else begin
        for i=0L,n-1L do message,logstr[i],/informational
      endelse

      return,-1
    endif else begin

      ;; Printing to a logfile:
      if printtofile then for i=0L,n-1L do printf,funit[0L],rname+logstr[i]

      ;; Displaying the message:
      if verbose then for i=0L,n-1L do print,rname+logstr[i]

    endelse ;; keyword_set(error)
  endif ;; n ne 0L and size(logstr,/type) eq 7L

  return,0
END ;;; function: p3d_misc_logger
