;+
; NAME:
;         p3d_cflatf
;
;         $Id: p3d_cflatf.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         A more detailed description is pending.
;
;         This program is called from the p3d GUI, but can also be used as a 
;         separate program.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: main routine
;
; CALLING SEQUENCE:
;         p3d_cflatf,filename,parfile,out,icstr=,ostr=,tostr=,masterbias=, $
;             tracemask=,dispmask=,crmask=,userparfile=,ofilename=,opath=, $
;             detector=,sfx=,/compress,/exmonitor,logfile=,loglevel=,stawid=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         filename        - An array of strings that specifies the names of the
;                           raw data files, which are used when creating the
;                           master flat field.
;         parfile         - A scalar string that specifies the name of a file
;                           with instrument-specific setup parameters.
;
; KEYWORD PARAMETERS:
;         icstr ['_imcmb'] - A scalar string with the image-comb. specific
;                           string that is used to create the output filename.
;         ostr ['_flatf'] - A scalar string with the flat-field specific
;                           string that is used to create the output filename.
;         tostr ['_trace'] - A scalar string with the trace-mask specific
;                           string that is used to create the output filename.
;         masterbias      - A scalar string with the name of a master bias
;                           file. This file is required if optimal extraction
;                           is used.
;         tracemask       - A scalar string with the name of a trace mask file.
;                           If TRACEMASK is not present then a new trace mask
;                           is calculated using the (combined) input raw data
;                           file.
;         dispmask        - A scalar string with the name of a dispersion mask
;                           to use -later- when wavelength calibrating the flat
;                           field data.
;         crmask          - A scalar string with the name of a cosmic ray mask.
;                           Note! This mask is only used if optimal extraction
;                                 is switched on in the user parameter file.
;         userparfile     - A scalar string with the name of a file with user-
;                           defined parameters. The following parameters are
;                           read here from this file:
;                             'methodimcombine':: see p3d_misc_imcombine.pro
;                             'methodextract':: see p3d_misc_extract_method.pro
;                             'detsec':: see p3d_misc_detsec.pro
;         ofilename       - This keyword returns the full name of the created
;                           master flat-field file.
;         opath ['.']     - A scalar string that specifies the path, where the
;                           output data is saved.
;         detector [0]    - A scalar integer that specifies the currently
;                           selected detector; DETECTOR is a zero-based value.
;         sfx [.fits]     - A scalar string specifying the file ending (without
;                           a trailing compression suffix, such as .gz or .Z).
;         compress [0]    - If this keyword is set then the output data file is
;                           compressed (using gzip).
;         exmonitor       - If this keyword is set then a line profiles viewer
;                           is shown after all fitting is done when using
;                           optimal extraction.
;         logfile         - If this routine is launched as a separate program
;                           then this keyword specifies the name of the log
;                           file that will be created.
;         loglevel [1]    - If this routine is launched as a separate program,
;                           and LOGFILE is used, then this keyword specifies
;                           the logging level; [1],2,3 (the higher the value
;                           the more output).
;         stawid          - If set, then various messages are written to the
;                           p3d GUI status line (this must be the widget id of
;                           that label widget).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - Upon a successful execution this variable contains
;                           the flat-field (data) when the routine exits.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; MODIFICATION HISTORY:
;         02.09.2009 - Routine introduced (moved conts. from p3d_gui). /CS
;
;-
PRO p3d_cflatf,filename,parfile,out,icstr=icstr,ostr=ostr,tostr=tostr, $
        masterbias=masterbias,tracemask=tracemask,dispmask=dispmask, $
        crmask=crmask,userparfile=userparfile,ofilename=ofilename, $
        opath=opath,detector=d, $
        sfx=sfx,compress=compress,exmonitor=exmonitor,logfile=logfile, $
        loglevel=loglevel,stawid=stawid,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_cflatf: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_cflatf'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  logfileisopened=0L
  if ~n_elements(logunit) and n_elements(logfile) eq 1L then begin
    if size(logfile,/type) ne 7L then begin
      errmsg='LOGFILE must, if specified, be a scalar string with the name' + $
             ' of a file to write.'
      goto,error_handler
    endif

    openw,logunit,logfile,/get_lun,error=error
    if error ne 0 then begin
      errmsg='Could not open a logfile with the name "'+logfile+'".'
      goto,error_handler
    endif
    logfileisopened=1L

    if ~n_elements(loglevel) then loglevel=1L
    s=size(loglevel)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='LOGLEVEL must be a scalar integer; 1<=LOGLEVEL<=3.'
      goto,error_handler
    endif
    if loglevel lt 1L or loglevel gt 3L then begin
      errmsg='LOGLEVEL must be a scalar integer; 1<=LOGLEVEL<=3.'
      goto,error_handler
    endif

    logunit=[logunit,loglevel]
  endif ;; ~n_elements(logunit) and n_elements(logfile) eq 1L

  s=size(filename) & n=s[s[0L]+2L]
  if ~n or s[s[0L]+1L] ne 7L then begin
    errmsg='FILENAME [1] must be specified, and be of string type.'
    goto,error_handler
  endif
  for i=0L,n-1L do begin
    if ~file_test(filename[i],/read,/regular) then begin
      errmsg='Cannot read or find the file "'+filename[i]+'" ['+ $
          strtrim(i+1L,2L)+'/'+strtrim(n,2L)+']'
      goto,error_handler
    endif
  endfor

  if size(parfile,/type) ne 7L then begin
    errmsg='PARFILE [2] must be specified; as the scalar filename of an in' + $
           'strument-specific parameter file.'
    goto,error_handler
  endif
  if ~file_test(parfile,/read,/regular) then begin
    errmsg='The file PARFILE "'+parfile+'" does not exist.'
    goto,error_handler
  endif

  if ~n_elements(ostr) then ostr='_flatf'
  if size(ostr,/type) ne 7L or n_elements(ostr) ne 1L then begin
    errmsg='OSTR must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  if ~n_elements(tostr) then tostr='_trace'
  if size(tostr,/type) ne 7L or n_elements(tostr) ne 1L then begin
    errmsg='TOSTR must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  usebiasfile=0L
  if n_elements(masterbias) eq 1L then begin
    if size(masterbias,/type) ne 7L then begin
      errmsg='MASTERBIAS must be of string type.'
      goto,error_handler
    endif
    if ~file_test(masterbias,/read,/regular) then begin
      errmsg='The file MASTERBIAS "'+masterbias+'" does not exist.'
      goto,error_handler
    endif
    usebiasfile=1L
  endif

  calctraces=0L
  if ~n_elements(tracemask) then begin
    calctraces=1L
  endif else begin
    if size(tracemask,/type) ne 7L then begin
      errmsg='TRACEMASK must, if set, be of string type.'
      goto,error_handler
    endif
    if ~file_test(tracemask,/read,/regular) then begin
      errmsg='The file TRACEMASK "'+tracemask+'" does not exist.'
      goto,error_handler
    endif
  endelse

  usedispmask=0L
  if n_elements(dispmask) eq 1L then begin
    if size(dispmask,/type) ne 7L then begin
      errmsg='DISPMASK must be of string type.'
      goto,error_handler
    endif
    if ~file_test(dispmask,/read,/regular) then begin
      errmsg='The file DISPMASK "'+dispmask+'" does not exist.'
      goto,error_handler
    endif
    usedispmask=1L
  endif ;; n_elements(dispmask) eq 1L

  usecrmask=0L
  if n_elements(crmask) eq 1L then begin
    if size(crmask,/type) ne 7L then begin
      errmsg='CRMASK must, if set, be of string type.'
      goto,error_handler
    endif
    if ~file_test(crmask,/read,/regular) then begin
      errmsg='The file CRMASK "'+crmask+'" does not exist.'
      goto,error_handler
    endif
    usecrmask=1L
  endif ;; n_elements(crmask) eq 1L

  if n_elements(userparfile) ne 0L then begin
    if size(userparfile,/type) ne 7L then begin
      errmsg='USERPARFILE must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(userparfile,/read,/regular) and userparfile ne '' then begin
      errmsg='The file USERPARFILE "'+userparfile+'" does not exist.'
      goto,error_handler
    endif

    if userparfile ne '' then $
      readcol,userparfile,uparname,uparvalue,format='a,a',delimiter=' ', $
          silent=verbose lt 3,comment=';'
  endif ;; n_elements(userparfile) ne 0L

  if n_elements(opath) ne 0L then begin
    if size(opath,/type) ne 7L then begin
      errmsg='OPATH must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(opath,/directory,/read) then begin
      errmsg='The directory OPATH "'+opath+'" does not exist.'
      goto,error_handler
    endif
  endif else begin
    cd,current=opath
  endelse
  idx=strpos(opath,path_sep(),/reverse_search)
  if idx ne strlen(opath)-1L then opath+=path_sep()

  if ~n_elements(d) then d=0L
  s=size(d)
  if s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L then begin
    errmsg='DETECTOR must be of (any) integer type; DETECTOR>=0.'
    goto,error_handler
  endif
  if d lt 0L then begin
    errmsg='DETECTOR must be of (any) integer type; DETECTOR>=0.'
    goto,error_handler
  endif

  if ~n_elements(sfx) then sfx='.fits'
  if size(sfx,/type) ne 7L or n_elements(sfx) ne 1L then begin
    errmsg='SFX must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  compress=keyword_set(compress)

  ;;========================================------------------------------
  ;; Routine information:

  msg=['','==============================--------------------', $
          '========== Creating a flat field', $
          '==========   '+systime(), $
          '==============================--------------------']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 2)

  ;;========================================------------------------------
  ;; Reading instrument-specific parameters:

  ppath=strmid(parfile,0L,strpos(parfile,path_sep(),/reverse_search)+1L)
  parname='' & parvalue=''
  readcol,parfile,parname,parvalue,format='a,a',delimiter=' ', $
      silent=verbose lt 3,comment=';'

  p3d_misc_read_params,parname,parvalue,'daxis',daxis,/must_exist,/nou, $
      type='integer',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2 (not '+strtrim(daxis,2L)+').'
    goto,error_handler
  endif

  p3d_misc_read_params,parname,parvalue,'keywordsfile',kwrdlist,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  kwrdlist=ppath+kwrdlist
  if ~file_test(kwrdlist,/read,/regular) then begin
    errmsg='The file KWRDLIST "'+kwrdlist+'" does not exist.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Combining the data files, at first the data is combined accounting for if
  ;; the number of blocks>1, and if there are more than 3 (groups of) files:

  title='master flat '
  p3d_misc_imcombine_wrapper,filename,ofilename,ofsfx,ostr=icstr, $
      detsec=detsec,parfile=parfile,userparfile=userparfile,opath=opath, $
      sfx=sfx,xstr=xstr,dsh=dsh,compress=compress,title=title,stawid=stawid, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  filename=ofilename
  fsfx=ofsfx

  ;; Writing information to the status line:
  if usestawid and dsh then begin
     tmpstr='[Flat field] Wrote the master flat-field image file "'+ $
            file_basename(ofilename)+'".'
     widget_control,stawid,set_value=tmpstr
  endif

  ;;========================================------------------------------
  ;; Reading some more parameters:

  if ~n_elements(exmonitor) then begin
    p3d_misc_read_params,parname,parvalue,'exmonitor',exmonitor, $
        uparname=uparname,uparvalue=uparvalue,logunit=logunit,topwid=topwid, $
        verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    exmonitor=~n_elements(exmonitor)?1L:(strlowcase(exmonitor) eq 'yes'?1L:0L)
  endif
  exmonitor=keyword_set(exmonitor)

  ;;=============================--------------------
  ;; Creating output filenames:

  tmpstr=opath+file_basename(filename[0L])
  ofilename =tmpstr+ostr+xstr+fsfx[0L]
  tmp=strpos(ofilename,'.gz')
  if compress then begin
    if tmp eq -1L then ofilename+='.gz'
  endif else begin
    if tmp ne -1L then ofilename=strmid(ofilename,0L,tmp)
  endelse

  ofilenamee=tmpstr+ostr+'_err'+xstr+fsfx[0L]
  tmp=strpos(ofilenamee,'.gz')
  if compress then begin
    if tmp eq -1L then ofilenamee+='.gz'
  endif else begin
    if tmp ne -1L then ofilenamee=strmid(ofilenamee,0L,tmp)
  endelse

  hdr=headfits(filename+fsfx)

  ;; Determining which method to use for spectrum extraction:
  method=p3d_misc_extract_method(userparfile,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug)
  if error ne 0 then goto,logfile_close
  if method eq 'optimal' and usecrmask then crmaskfile=crmask

  ;;==============================--------------------
  ;; Extracting traces, if required:

  if calctraces then begin
    p3d_ctrace,filename+fsfx,parfile,ostr=tostr,masterbias=masterbias, $
	userparfile=userparfile,ofilename=tracemask,opath=opath, $
        detector=d,sfx=sfx,compress=compress,stawid=stawid,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
  endif ;; calctraces

  ;;==============================--------------------
  ;; Extracting the binning parameters:

  p3d_misc_binextract,kwrdlist,hdr,xbin=xbin,ybin=ybin,topwid=topwid, $
      logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  bin=daxis?ybin:xbin

  ;;==============================--------------------
  ;; Loading the flat-field image data, the tracing mask, and the bias image
  ;; (if it is loaded), in order to extract the same region on the ccd, which
  ;; was trimmed of the overscan region:

  if usestawid then begin ;; Writing information to the status line:
    tmpstr='[Flat field] Trimming data and extracting spectra ['+method+'].'
    widget_control,stawid,set_value=tmpstr
  endif

  ecalc=0L
  if usebiasfile then begin
    p3d_misc_getinformation,filename+fsfx,kwrdlist,dbias=dbias,gain=gain, $
        rdnoise=rdnoise,masterbias=masterbias,userparfile=userparfile, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close

    if min(dbias) gt 0d0 then ecalc=1L
  endif ;; usebiasfile

  p3d_extract_prepare_extraction,filename+fsfx,tracemask,kwrdlist,traces, $
      trimmedflatdata,objshift,bias,bsw,xbin=xbin,ybin=ybin, $
      crmaskfile=crmaskfile,crmaskdata=crmasko,userparfile=userparfile, $
      masterbias=masterbias,parfile=parfile,/no_shift,daxis=daxis, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  ;;==============================--------------------
  ;; Extracting all spectra from the raw data image by integrating up the
  ;; contribution of all contributing pixels using the trace mask:

  p3d_misc_retrieve_lprofs,tracemask,lprofs,proffun,method=method, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  p3d_extract,trimmedflatdata,traces,flatextract,dout=dflatextract, $
      bias=bias,bin=bin,dbias=dbias,crmask=crmasko,gain=gain,rdnoise=rdnoise, $
      userparfile=userparfile,detsec=detsec,parfile=parfile,xstr=xstr, $
      method=method,lprofs=lprofs,proffun=proffun,exmonitor=exmonitor, $
      subtitle=file_basename(filename+fsfx),daxis=daxis,stawid=stawid, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  ;;==============================--------------------
  ;; Extracting the flat-field data, without normalizing it:

  p3d_flatfield,flatextract,finalflat,dfinalflat,din=dflatextract, $
      daxis=daxis,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  ;;==============================--------------------
  ;; Writing the flat-field data to a file. Note! The data is not
  ;; normalized:

  ;; Writing information to the status line:
  if usestawid then begin
    tmpstr='[Flat field] Writing the extracted flat-field data file.'
    widget_control,stawid,set_value=tmpstr
  endif

  hdr=headfits(filename+fsfx)
  fxhmake,tmphdr,finalflat
  fxaddpar,hdr,'BITPIX',fxpar(tmphdr,'BITPIX')
  fxaddpar,hdr,'BZERO',0
  fxaddpar,hdr,'NAXIS1',fxpar(tmphdr,'NAXIS1')
  fxaddpar,hdr,'NAXIS2',fxpar(tmphdr,'NAXIS2')
  fxaddpar,hdr,'IMTYPE','p3d: flat'

  ;; Adding information on the input files:
  p3d_misc_fitsparsplit,'IMRAW',filename[0L]+fsfx[0L],hdr, $
      ' Raw data filename',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then goto,logfile_close

  p3d_misc_fitsparsplit,'IMTRC',tracemask[0L],hdr,' Trace mask filename', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  if usebiasfile then begin
    p3d_misc_fitsparsplit,'IMMBI',masterbias[0L],hdr,' Master bias filename', $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
  endif ;; usebiasfile

  if usedispmask then begin
    p3d_misc_fitsparsplit,'IMDMK',dispmask[0L],hdr, $
        ' Dispersion mask filename', $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
  endif ;; usedispmask

  etmp=' ' & eetmp='".'
  if ecalc then begin
    p3d_misc_fitsparsplit,'IMERR',ofilenamee,hdr,' Flat error filename', $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    etmp=', and its error, ' & eetmp='", and'
  endif ;; ecalc

  writefits,ofilename,finalflat,hdr,compress=compress
  if n_elements(dfinalflat) ne 0L then begin
    fxaddpar,hdr,'IMTYPE','p3d: flat - error data'
    writefits,ofilenamee,dfinalflat,hdr,compress=compress
  endif

  msg=['Wrote the extracted (not normalized) flat-field data'+etmp+'to:', $
       '  "'+p3d_misc_pathify(ofilename,/dpath)+eetmp]
  if ecalc then msg=[msg,'  "'+p3d_misc_pathify(ofilenamee,/dpath)+'".']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  out=temporary(finalflat)

  ;; Writing information to the status line:
  if usestawid then begin
    tmpstr='[Flat field] Done.'
    widget_control,stawid,set_value=tmpstr
  endif

  ;; Closing the logfile, if it was opened here:
  if logfileisopened then free_lun,logunit[0L]

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
logfile_close:
  if logfileisopened then free_lun,logunit[0L]
  return
END ;; procedure: p3d_cflatf
