;+
; NAME:
;         p3d_ctrace
;
;         $Id: p3d_ctrace.pro 109 2010-03-11 16:48:11Z christersandin $
;
; PURPOSE:
;         A more detailed description is pending.
;
;         This program is called from the p3d GUI, but can also be used as a 
;         separate program.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: main routine
;
; CALLING SEQUENCE:
;         p3d_ctrace,filename,parfile,out,icstr=,ostr=,masterbias=, $
;             userparfile=,ofilename=,opath=,detector=,sfx=,/compress, $
;             logfile=,loglevel=,stawid=,topwid=,logunit=,verbose=,error=, $
;             /debug,/help
;
; INPUTS:
;         filename        - A scalar string that specifies the name of the raw
;                           data file to be used when tracing spectra.
;         parfile         - A scalar string that specifies the name of a file
;                           with instrument-specific setup parameters.
;
; KEYWORD PARAMETERS:
;         icstr ['_imcmb'] - A scalar string with the image-comb. specific
;                           string that is used to create the output filename.
;         ostr ['_trace'] - A scalar string with the trace-mask specific
;                           string that is used to create the output filename.
;         masterbias      - A scalar string with the name of a master bias
;                           file. This file is required if optimal extraction
;                           is used.
;         userparfile     - A scalar string with the name of a file with user-
;                           defined parameters. The following parameters are
;                           read here from this file:
;                             methodextract: see p3d_misc_extract_method.pro
;         ofilename       - This keyword returns the full name of the created
;                           trace mask file.
;         opath ['.']     - A scalar string that specifies the path, where the
;                           output data is saved.
;         detector [0]    - A scalar integer that specifies the currently
;                           selected detector; DETECTOR is a zero-based value.
;         sfx [.fits]     - A scalar string specifying the file ending (without
;                           a trailing compression suffix, such as .gz or .Z).
;         compress [0]    - If this keyword is set then the output data file is
;                           compressed (using gzip).
;         logfile         - If this routine is launched as a separate program
;                           then this keyword specifies the name of the log
;                           file that will be created.
;         loglevel [1]    - If this routine is launched as a separate program,
;                           and LOGFILE is used, then this keyword specifies
;                           the logging level; [1],2,3 (the higher the value
;                           the more output).
;         stawid          - If set, then various messages are written to the
;                           p3d GUI status line (this must be the widget id of
;                           that label widget).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined. This keyword
;                           must not be used if this routine is called as a
;                           script instead of using the GUI.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - Upon a successful execution this variable contains
;                           the trace mask (data) when the routine exits.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; MODIFICATION HISTORY:
;         01.09.2009 - Routine introduced (moved conts. from p3d_gui). /CS
;
;-
PRO p3d_ctrace,filename,parfile,out,icstr=icstr,ostr=ostr, $
        masterbias=masterbias,userparfile=userparfile,ofilename=ofilename, $
        opath=opath,detector=d,sfx=sfx,compress=compress,logfile=logfile, $
        loglevel=loglevel,stawid=stawid,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_ctrace: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_ctrace'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(stawid) then stawid=0L
  ustawid=widget_info(stawid,/valid_id)

  logfileisopened=0L
  if ~n_elements(logunit) and n_elements(logfile) eq 1L then begin
    if size(logfile,/type) ne 7L then begin
      errmsg='LOGFILE must, if specified, be a scalar string with the name' + $
             ' of a file to write.'
      goto,error_handler
    endif

    openw,logunit,logfile,/get_lun,error=error
    if error ne 0 then begin
      errmsg='Could not open a logfile with the name "'+logfile+'".'
      goto,error_handler
    endif
    logfileisopened=1L

    if ~n_elements(loglevel) then loglevel=1L
    s=size(loglevel)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='LOGLEVEL must be a scalar integer; 1<=LOGLEVEL<=3.'
      goto,error_handler
    endif
    if loglevel lt 1L or loglevel gt 3L then begin
      errmsg='LOGLEVEL must be a scalar integer; 1<=LOGLEVEL<=3.'
      goto,error_handler
    endif

    logunit=[logunit,loglevel]
  endif ;; ~n_elements(logunit) and n_elements(logfile) eq 1L

  s=size(filename) & n=s[s[0L]+2L]
  if n ne 1L and n lt 3L or s[s[0L]+1L] ne 7L then begin
    errmsg='FILENAME [1] must be specified, and be of string type.'
    goto,error_handler
  endif
  for i=0L,n-1L do begin
    if ~file_test(filename[i],/read,/regular) then begin
      errmsg='Cannot read or find the file "'+filename[i]+'" ['+ $
          strtrim(i+1L,2L)+'/'+strtrim(n,2L)+']'
      goto,error_handler
    endif
  endfor

  if ~n_elements(d) then d=0L
  s=size(d)
  if s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L then begin
    errmsg='D must be of (any) integer type; D>=0.'
    goto,error_handler
  endif
  if d lt 0L then begin
    errmsg='D must be of (any) integer type; D>=0.'
    goto,error_handler
  endif

  if size(parfile,/type) ne 7L then begin
    errmsg='PARFILE [2] must be specified; as the scalar filename of an in' + $
           'strument-specific parameter file.'
    goto,error_handler
  endif
  if ~file_test(parfile,/read,/regular) then begin
    errmsg='The file PARFILE "'+parfile+'" does not exist.'
    goto,error_handler
  endif

  if ~n_elements(ostr) then ostr='_trace'
  if size(ostr,/type) ne 7L or n_elements(ostr) ne 1L then begin
    errmsg='OSTR must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  usebiasfile=0L
  if n_elements(masterbias) eq 1L then begin
    if size(masterbias,/type) ne 7L then begin
      errmsg='MASTERBIAS must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(masterbias,/read,/regular) then begin
      errmsg='The file MASTERBIAS "'+masterbias+'" does not exist.'
      goto,error_handler
    endif
    usebiasfile=1L
  endif

  if n_elements(userparfile) ne 0L then begin
    if size(userparfile,/type) ne 7L then begin
      errmsg='USERPARFILE must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(userparfile,/read,/regular) and userparfile ne '' then begin
      errmsg='The file USERPARFILE "'+userparfile+'" does not exist.'
      goto,error_handler
    endif
  endif

  if n_elements(opath) ne 0L then begin
    if size(opath,/type) ne 7L then begin
      errmsg='OPATH must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(opath,/directory,/read) then begin
      errmsg='The directory OPATH "'+opath+'" does not exist.'
      goto,error_handler
    endif
  endif else begin
    cd,current=opath
  endelse
  idx=strpos(opath,path_sep(),/reverse_search)
  if idx ne strlen(opath)-1L then opath+=path_sep()

  if n_elements(userparfile) ne 0L then begin
    s=size(userparfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE, if set, must be a scalar string.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='USERPARFILE, cannot read the file "'+userparfile+'".'
        goto,error_handler
      endif

      readcol,userparfile,uparname,uparvalue,format='a,a',comment=';', $
          silent=verbose lt 3,delimiter=' '
    endif
  endif ;; n_elements(userparfile) ne 0L

  if ~n_elements(sfx) then sfx='.fits'
  if size(sfx,/type) ne 7L or n_elements(sfx) ne 1L then begin
    errmsg='SFX must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  compress=keyword_set(compress)

  ;;========================================------------------------------
  ;; Routine information:

  msg=['','==============================--------------------', $
          '========== Creating a trace mask', $
          '==========   '+systime(), $
          '==============================--------------------']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 2)

  ;;========================================------------------------------
  ;; Reading instrument-specific parameters:

  ppath=strmid(parfile,0L,strpos(parfile,path_sep(),/reverse_search)+1L)
  parname='' & parvalue=''
  readcol,parfile,parname,parvalue,format='a,a',delimiter=' ', $
      silent=verbose lt 3,comment=';'

  p3d_misc_read_params,parname,parvalue,'daxis',daxis,/must_exist,/nou, $
      type='integer',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2 (not '+strtrim(daxis,2L)+').'
    goto,error_handler
  endif

  p3d_misc_read_params,parname,parvalue,'ndetector',ndetector,type='integer', $
      /nou,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  if ~n_elements(ndetector) then ndetector=1L ;; default is one detector

  p3d_misc_read_params,parname,parvalue,'keywordsfile',kwrdlist,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  kwrdlist=ppath+kwrdlist
  if ~file_test(kwrdlist,/read,/regular) then begin
    errmsg='The file KWRDLIST "'+kwrdlist+'" does not exist.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Reading the gaps-array filenames:

  gapfile=strarr(ndetector)
  for i=0L,ndetector-1L do begin
    str=ndetector eq 1L?'':'_'+strtrim(i+1L,2L)
    p3d_misc_read_params,parname,parvalue,'gapfile'+str,gapfilename, $
        /must_exist,uparname=uparname,uparvalue=uparvalue, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    gapfile[i]=gapfilename
  endfor
  tmp=strpos(gapfile,path_sep())
  if tmp[0L] eq -1L then $
     gapfile=ppath+gapfile ;; adding the path of the parameter file

  for i=0L,ndetector-1L do begin
    if ~file_test(gapfile[i],/regular,/read) then begin
      errmsg='The gaps array file "'+gapfile[i]+'" does not exist.'
      goto,error_handler
    endif
  endfor

  ;;========================================------------------------------
  ;; Determining if line profiles are to be calculated:

  method=p3d_misc_extract_method(userparfile,/tracemask,topwid=topwid, $
      logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then goto,logfile_close

  if ~usebiasfile and method eq 'optimal' and ustawid then begin
    msg='[Trace mask] Extraction method "optimal": a masterbias is r' + $
        'equired!.'
    widget_control,stawid,set_value=msg
    return
  endif

  ;;========================================------------------------------
  ;; Combining the data files, at first the data is combined accounting for if
  ;; the number of blocks>1, and if there are more than 3 (groups of) files:

  title='master trace '
  p3d_misc_imcombine_wrapper,filename,ofilename,ofsfx,ostr=icstr, $
      detsec=detsec,parfile=parfile,userparfile=userparfile,opath=opath, $
      sfx=sfx,xstr=xstr,dsh=dsh,compress=compress,title=title,stawid=stawid, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  filename=ofilename
  fsfx=ofsfx

  ;; Writing information to the status line:
  if usestawid and dsh then begin
    tmpstr='[Trace mask] Wrote the master trace file "'+ $
           file_basename(ofilename)+'".'
    widget_control,stawid,set_value=tmpstr
  endif

  ;;=============================--------------------
  ;; Creating output filenames:

  tmpstr=opath+file_basename(filename[0L])
  ofilename =tmpstr+ostr+xstr+fsfx[0L]
  tmp=strpos(ofilename,'.gz')
  if compress then begin
    if tmp eq -1L then ofilename+='.gz'
  endif else begin
    if tmp ne -1L then ofilename=strmid(ofilename,0L,tmp)
  endelse

  ofilenamec=tmpstr+ostr+'_cdlp'+xstr+fsfx[0L]
  tmp=strpos(ofilenamec,'.gz')
  if compress then begin
    if tmp eq -1L then ofilenamec+='.gz'
  endif else begin
    if tmp ne -1L then ofilenamec=strmid(ofilenamec,0L,tmp)
  endelse

  ;;=============================--------------------
  ;; Reading the fits file with the data that will be used for tracing:

  array=readfits(filename+fsfx,hdr,silent=verbose lt 3,/no_unsigned)

  msg='[create tracemask] Read the file "'+filename+fsfx+'".'
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  ;;=============================--------------------
  ;; Reading the cross-dispersion binning parameter:

  p3d_misc_binextract,kwrdlist,hdr,xbin=xbin,ybin=ybin,topwid=topwid, $
      logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  bin=daxis?ybin:xbin

  ;;=============================--------------------
  ;; Reading the gaps file:

  readcol,gapfile[d],gaps,format='l',silent=verbose lt 3,comment=';'

  ;;=============================--------------------
  ;; Launching the fiber tracing routine:

  title='[Trace mask] "'+filename+fsfx+'"'
  p3d_tracing_trace,parfile,array,traces,gaps=gaps,bin=bin,xstr=xstr, $
      userparfile=userparfile,daxis=daxis,stawid=stawid,topwid=topwid, $
      logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  ;;=============================--------------------
  ;; Writing the outcome of the tracing procedure to a file:

  s=size(traces)
  fxaddpar,hdr,'NAXIS1',s[1L]
  fxaddpar,hdr,'NAXIS2',s[2L]
  fxhmake,tmphdr,traces
  fxaddpar,hdr,'BITPIX',fxpar(tmphdr,'BITPIX')
  fxaddpar,hdr,'BZERO',0

  p3d_misc_fitsparsplit,'IMRAW',filename+fsfx,hdr, $
      ' Raw data filename',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug

  if method eq 'optimal' then begin
    p3d_misc_fitsparsplit,'IMPRF',ofilenamec,hdr, $
        ' Line profiles filename',topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug
  endif ;; method eq 'optimal'

  writefits,ofilename,traces,hdr,compress=compress
  tracefile=file_basename(ofilename)

  msg='[create tracemask] Wrote the file "'+tracefile+'".'
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  ;;=============================--------------------
  ;; Writing information to the status line:

  if ustawid then begin
    msg='[Trace mask] Wrote the trace mask file "'+tracefile+'".'
    widget_control,stawid,set_value=msg
  endif

  ;;=============================--------------------
  ;; Optionally calculating cross-dispersion line profiles for every
  ;; spectrum and wavelength bin:

  if method eq 'optimal' then begin
    p3d_misc_getinformation,filename+fsfx,kwrdlist,dbias=dbias,gain=gain, $
        rdnoise=rdnoise,masterbias=masterbias,userparfile=userparfile, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close

    p3d_tracing_calculate_lprofs,filename+fsfx,masterbias,traces,lprofs, $
        dmbias=dbias,gaps=gaps,xstr=xstr,xbin=xbin,ybin=ybin,gain=gain, $
        rdnoise=rdnoise,detector=d,kwrdlist=kwrdlist,parfile=parfile, $
        userparfile=userparfile,oprof=oprof,daxis=daxis,stawid=stawid, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close

    s=size(lprofs)
    fxaddpar,hdr,'NAXIS1',s[1L]
    fxaddpar,hdr,'NAXIS2',s[2L]
    fxaddpar,hdr,'NAXIS3',s[3L]
    fxhmake,tmphdr,lprofs
    fxaddpar,hdr,'BITPIX',fxpar(tmphdr,'BITPIX')
    fxaddpar,hdr,'BZERO',0

    p3d_misc_fitsparsplit,'IMTRC',ofilename,hdr, $
        ' Trace filename',topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    p3d_misc_fitsparsplit,'IMMBI',masterbias[0L],hdr, $
        ' Master bias filename',topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close

    fxaddpar,hdr,'P3DFFUNC',oprof

    ;; Writing the file with all line profile fit parameters:
    writefits,ofilenamec,lprofs,hdr,compress=compress

    msg='[create tracemask] Wrote the line profiles file "'+ $
        p3d_misc_pathify(ofilenamec,/dpath)+'".'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  endif ;; method eq 'optimal'

  out=temporary(traces)

  ;; Writing information to the status line:
  if usestawid then begin
    tmpstr='[Trace mask] Done.'
    widget_control,stawid,set_value=tmpstr
  endif

  ;; Closing the logfile, if it was opened here:
  if logfileisopened then free_lun,logunit[0L]

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
logfile_close:
  if logfileisopened then free_lun,logunit[0L]
  return
END ;; procedure: p3d_ctrace
