;+
; NAME:
;         p3d_wavecal_grating_eq
;
;         $Id: p3d_wavecal_grating_eq.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         Computes various parameters using the grating equation.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: wavelength calibration
;
; CALLING SEQUENCE:
;         p3d_wavecal_grating_eq,lpmm,blaze,alpha,npixdisp,m, $
;             pix,lambda,lam_min,lam_cen,lam_max,rec_disp,pix_disp,spcres,r, $
;             eta=,view=,pixsize=,psf=,flip=,grotpos=,flen=,grotoffset=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         lpmm            - Groove density [lines per mm].
;         blaze           - Blaze angle [degrees].
;         alpha           - Grating angle normal to the incident beam.
;         npixdisp        - Number of pixels on CCD in the dispersion
;                           direction.
;         m               - Spectrum order. Note! -1 is FORWARD diffraction.
;
; KEYWORD PARAMETERS:
;         eta [42]       - Collimator/camera angle [degrees].
;         view            - Plots the lambda and grating geometry.
;         pixsize [0.015mm] - The pixel size.
;         psf [4.0]       - The width of the instrumental profile [pixels].
;         flip            - Reverses the order of pixels.
;         grotpos         - The grating rotator encoder position. A default
;                           value of -58.9 is used if this keyword is set to
;                           either GROTPOS==1b or GROTPOS==1 (i.e. byte or
;                           integer type).
;         flen [270mm]    - The focal length [mm].
;         grotoffset [-214.8||0.0] - The grating rotator of PMAS was broke in
;                           2008. The new encoder values are offset compared to
;                           the tabulated values. This offset value can be set
;                           in order to calculate the actual grating rotator
;                           position. The default value of -214.8 is used iff
;                           GROTPOS>0.0, otherwise it is set to 0.0. This value
;                           is added to GROTPOS.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         pix             - An array with NPIXDISP elements: pixel number.
;         lambda          - An array with NPIXDISP elements; corresponding
;                           Lambda values [].
;         lam_min         - lambda_min, the minimum wavelength [].
;         lam_cen         - lambda_cen, the center  wavelength [].
;         lam_max         - lambda_max, the maximum wavelength [].
;         rdisp           - The reciprocal linear dispersion [/mm].
;         pdisp           - The reciprocal linear dispersion [/pix].
;         dlam            - The spectrum resolution [].
;         R               - The resolving power; Lambda/d_Lambda.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; EXAMPLES:
;         p3d_wavecal_grating_eq, 1200,17.5,2,2048,1,pix,lam, $
;             lmin,lcen,lmax,ldisp,rdisp,dlam,r,/v,/i,/flip 
;
;         p3d_wavecal_grating_eq, 1200,17.5,-48,2048,1,pix,lam, $
;             lmin,lcen,lmax,ld,rd,dl,r,/v,/i,/flip,/gr
;
; MODIFICATION HISTORY:
;         13.10.2008 - Converted from the original routine grating of
;                      Thomas Becker. /CS
;-
PRO p3d_wavecal_grating_eq,lpmm,blaze,alpha,npixdisp,m,pix,lambda, $
        lam_min,lam_cen,lam_max,rec_disp,pix_disp,spcres,r, $
        eta=vareta,view=varview,pixsize=varpixsize,psf=varpsf,flip=varflip, $
        grotpos=grotpos,flen=varfl,grotoffset=grotoffset,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_wavecal_grating_eq: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_wavecal_grating_eq'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(lpmm)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='LPMM must be set; to a scalar floating point value.'
    goto,error_handler
  endif

  s=size(blaze)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='BLAZE must be set; to a scalar floating point value.'
    goto,error_handler
  endif

  s=size(alpha)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='ALPHA must be set; to a scalar floating point value.'
    goto,error_handler
  endif

  s=size(npixdisp)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='NPIXDISP must be set; to a scalar integer.'
    goto,error_handler
  endif

  s=size(m)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='M must be set; to a scalar integer.'
    goto,error_handler
  endif
  varforward=m lt 0L?0L:1L

  ;; The usual offset:
  vargrotpos=0.0
  sl=size(grotpos) & vargrotpos_specified=0L
  if sl[sl[0L]+2L] ne 0L then begin
    vargrotpos=grotpos
    if sl[sl[0L]+1L] le 2L then $
       if vargrotpos eq 1L then vargrotpos=-58.9 
    sl=size(vargrotpos)
    if sl[sl[0L]+2L] ne 1L or $
      (sl[sl[0L]+1L] ge 6L and sl[sl[0L]+1L] le 11L) then begin
      errmsg='GROTPOS must be a scalar of decimal type.'
      goto,error_handler
    endif
    vargrotpos_specified=1L
    vargrotpos=float(vargrotpos)
  endif ;; sl[sl[0L]+2L] ne 0L

  ;; An extra offset:
  if vargrotpos_specified then begin
    grotoffset_def=alpha gt 0.0?214.8:0.0
    grotoffset=n_elements(grotoffset) eq 1L?grotoffset:grotoffset_def
    s=size(grotoffset)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
      errmsg='GROTOFFSET must be set; to a decimal scalar.'
      goto,error_handler
    endif
  endif else grotoffset=0.0

  if ~n_elements(vareta) then vareta=42.0
  s=size(vareta)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='ETA must be set; to a decimal scalar.'
    goto,error_handler
  endif

  if ~keyword_set(varview) then varview=0

  if ~n_elements(varpixsize) then varpixsize=0.015
  s=size(varpixsize)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='PIXSIZE must be set; to a scalar decimal.'
    goto,error_handler
  endif

  if ~n_elements(varpsf) then varpsf=4.0
  s=size(varpsf)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='PSF must be set; to a scalar decimal.'
    goto,error_handler
  endif

  if ~keyword_set(varflip) then varflip=0
  s=size(varflip)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='FLIP must be a scalar integer; 0||1.'
    goto,error_handler
  endif

  if ~n_elements(varfl) then varfl=270.0
  s=size(varfl)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='FLEN must be set; to a decimal scalar.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Calculating relevant properties:

  pix=lindgen(npixdisp)
  y=varpixsize*(double(pix)-round(npixdisp/2d0)) ; detector dispersion axis

  ;; Is the following correct?
  if vargrotpos_specified then alpha=grotoffset-alpha+vargrotpos

  ;; alpha=angle (grating normal,collimator axis)
  ;; Angle (grating normal,camera axis):
  beta=alpha-vareta

  ;; Half angle subtended by the CCD, as seen from the camera:
  cam_angle=atan(y/varfl)

  lambda =abs((1d0/(m*lpmm))*(sin(!dtor*alpha)+sin(!dtor*beta+cam_angle)))*1d7
  lam_cen=abs((1d0/(m*lpmm))*(sin(!dtor*alpha)+sin(!dtor*beta          )))*1d7
  lam_min=min([lambda[0L],lambda[npixdisp-1L]])
  lam_max=max([lambda[0L],lambda[npixdisp-1L]])
  lamrange=lam_max-lam_min
  if varflip then lambda=rotate(lambda,2L) ;; Array reversal

  ;; Angular dispersion [radians/mm]:
  ang_disp=m*lpmm/cos(!dtor*beta)

  ;; Linear dispersion [mm/mm]:
  lin_disp=ang_disp*varfl

  ;; Reciprocal linear dispersion [/mm]:
  rec_disp=(1d0/lin_disp)*1d7

  ;; Reciprocal linear dispersion per pixel [/pix]:
  pix_disp=rec_disp*varpixsize

  ;; Spectral resolution []:
  spcres=abs(pix_disp*varpsf)

  ;; Resolving power:
  R=round((lam_cen)/spcres)


  ;;========================================------------------------------
  ;; Plot a dispersion curve and grating geometry:

  if varview then begin
    window,/free,retain=2

    plot,pix,lambda,/ystyle,xstyle=9, $
         xtitle=' Camera Field:      top scale: [mm]   /   bottom scale:' + $
         ' [pixel]',ytitle='Lambda [A]' , $
         xrange=[0,(floor(npixdisp/100.0)+1)*100]
    axis,xaxis=1,xrange=!x.crange*varpixsize-0.5*npixdisp*varpixsize,xstyle=1

    p3d_display_grat_plot,vareta,alpha,blaze,forward=varforward, $
        ur=strmid(string(lpmm),4,4)+' l/mm', $
        ul='GROTPOS = -'+strmid(string((alpha-GROT_OFFSET)),6,6) , $ 
        ll='Alpha ='+strmid(string(alpha),5,6)+' !EO', $ 
        lr='Blaze ='+strmid(string(blaze),5,5)+' !EO', $
        topwid=topwid,verbose=verbose,error=error,debug=debug
    if error ne 0 then return
  endif ;; varview


  ;;========================================------------------------------
  ;; Print the outcome on the screen:

  msg=[string('lpmm:'    ,lpmm    ,format='(a10,i10,"  grooves per mm")'), $
       string('blaze:'   ,blaze   ,format='(a10,f10.1,"  blaze angle []")'), $
       string('grotpos:' ,-(alpha-vargrotpos),format='(a10,f10.1,"  grat' + $
              'ing position [encoder units]")'), $
       string('alpha:'   ,alpha   ,format='(a10,f10.2,"  collimator on-a' + $
              'xis beam angle to grating normal")'), $
       string('beta:'    ,float(beta)  ,format='(a10,f10.2,"  diffracted' + $
              ' on-axis beam angle to grating normal")'), $
       '----------------------------------------', $
       string('eta:'     ,float(vareta),format='(a10,f10.2,"  collimator' + $
              '-camera angle")'), $
       string('gamma_min:',!radeg*cam_angle[0L],format='(a10,f10.2,"  mi' + $
              'n. camera field angle")'), $
       string('gamma_max:',!radeg*cam_angle[npixdisp-1L], $
              format='(a10,f10.2,"  max. camera field angle")'), $
       '----------------------------------------', $
       string('lam_cen:' ,lam_cen ,format='(a10,f10.2,"  central wavelen' + $
              'gth []")'), $
       string('lam_min:' ,lam_min ,format='(a10,f10.2,"  min. wavelength' + $
              ' []")'), + $
       string('lam_max:' ,lam_max ,format='(a10,d10.2,"  max. wavelength' + $
              ' []")'), $
       string('lamrange:',lamrange,format='(a10,d10.2,"  wavelength cove' + $
              'rage []")'),'----------------------------------------', $
       string('ang_disp:',ang_disp,format='(a10,f10.2,"  angular dispers' + $
              'ion")'), $
       string('lin_disp:',lin_disp,format='(a10,f10.2,"  linear dispersi' + $
              'on")'), $
       string('rec_disp:',rec_disp,format='(a10,f10.3,"  recipr. dispers' + $
              'ion [/mm]")'), $
       string('pix_disp:',pix_disp,format='(a10,f10.3,"  pixel dispersio' + $
              'n [/pix]")'), $
       string('spcres:'  ,spcres  ,format='(a10,f10.3,"  spectral resolu' + $
              'tion []")'), $
       string('R:'       ,R       ,format='(a10,i10,"  resolving power [' + $
              'Lambda/d_Lambda]")')]

  error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose ge 2)

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
            verbose=verbose,/error)
  return
END ;;; procedure: p3d_wavecal_grating_eq
