;+
; NAME:
;         p3d_wavecal_delete_spec
;
;         $Id: p3d_wavecal_delete_spec.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         Deletes numbered spectra in an input image. 
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: wavelength calibration
;
; CALLING SEQUENCE:
;         p3d_wavecal_delete_spec,inspec,rownum,outspec, $
;             daxis=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         inspec          - A two-dimensional array of decimal type.
;         rownum          - A one-dimensional array of integer indices
;                           indicating which rows of INSPEC are to be removed.
;
; KEYWORD PARAMETERS:
;         daxis [1]       - Specifies the dispersion axis of INSPEC.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         outspec         - A two-dimensional array with the spectra indicated
;                           by ROWNUM removed from INSPEC.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         13.10.2008 - Inspired by the original routine delete_spec of
;                      Thomas Becker. /CS
;-

PRO p3d_wavecal_delete_spec,inspec,rownum,outspec,daxis=daxis, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
        debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_wavecal_delete_spec: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_wavecal_delete_spec'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(daxis) then daxis=1L
  s=size(daxis)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  sid=daxis?2L:1L

  s=size(inspec)
  if s[0L] ne 2L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='INSPEC [1] must be set; to a two-dimensional array of decimal ' + $
           'type.'
    goto,error_handler
  endif

  sr=size(rownum)
  if sr[sr[0L]+2L] lt 1L or $
    (sr[sr[0L]+1L] ge 4L and sr[sr[0L]+1L] le 11L) then begin
    errmsg='ROWNUM [2] must be an integer array.'
    goto,error_handler
  endif
  if sr[sr[0L]+2L] eq 1L and rownum[0L] eq -1L then begin
    msg='No spectra were removed.'
    error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
              verbose=verbose ge 2)
    outspec=inspec
    return
  endif

  ;;========================================------------------------------
  ;; Creating an output array where the rows ROWNUM have been removed:

  tmp=strtrim(rownum+1L,2L)
  msg='Removing the following lines in the image: '+strjoin(tmp,', ')
  error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
            verbose=verbose ge 2)

  specid=lonarr(s[sid])+1L

  k=-1L
  while ++k lt n_elements(rownum) do $
     if rownum[k] ge 0L and rownum[k] le s[sid]-1L then specid[rownum[k]]=0L

  specpos=where(specid,count)
  if count ne 0L then outspec=daxis?inspec[*,specpos]:inspec[specpos,*]

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
                        verbose=verbose,/error)
  return
END ;;; procedure: p3d_wavecal_delete_spec
