;+
; NAME:
;         p3d_misc_get_hdr_kwrd
;
;         $Id: p3d_misc_get_hdr_kwrd.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine reads the information from a file, which contains a list
;         of what fits file header keywords are to be used with various
;         properties in the raw data file. If the file header keyword contains
;         minus signs ('-') then these are replaced with spaces (' ') before
;         returning.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_get_hdr_kwrd,table,p3d_kwrd, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         table           - Name of the keyword translation table.
;         p3d_kwrd        - Name of the keyword p3d uses.
;
; KEYWORD PARAMETERS:
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         returned value  - Name of the fitsheader keyword. Is set to
;                           -1 if the keyword was not found.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         06.10.2008 - Converted from the original routine p3d_get_hdr_kwrd of
;                      Thomas Becker. /CS
;-
FUNCTION p3d_misc_get_hdr_kwrd,table,p3d_kwrd,topwid=topwid, $
             logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  error=0 & rname='p3d_misc_get_hdr_kwrd: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_get_hdr_kwrd'
    return,-1L
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return,error
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking if the table file exists:

  s=size(table)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='TABLE [1] must be se to a scalar string pointing to an existi' + $
           'ng file.'
    goto,error_handler
  endif
  if ~file_test(table,/regular,/read) then begin
    errmsg='The file "'+table+'" cannot be found.'
    goto,error_handler
  endif
  if file_search(table) eq '' then return,-1L

  ;;========================================------------------------------
  ;; Reading data:

  readcol,table,p3d_kwrds,hdr_kwrds,format='a,a',silent=verbose lt 3, $
    delimiter=' ',comment=';'
  pos=where(strupcase(p3d_kwrds) eq strupcase(p3d_kwrd),count)

  out=~count?-1L:hdr_kwrds[pos[0L]]

  ;; Replacing '-' with ' ', if there are any '-':
  if out ne '-1' then $
    out=strjoin(strsplit(out,'-',/extract),' ')

  return,out

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
                        verbose=verbose,/error)
  return,error
END ;;; function: p3d_misc_get_hdr_kwrd
