;+
; NAME:
;         p3d_misc_fitsstrex
;
;         $Id: p3d_misc_fitsstrex.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine extracts the value with the keyword name STR from the
;         fits header array HDR. If possible the output value is converted to a
;         double or a long. If the value is the same when evaluated as a double
;         and a long then the long value is returned. If any error is
;         encountered or if the keyword does not exist in HDR then -1 is
;         returned.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         ret=p3d_misc_fitsstrex(hdr,str, $
;                 topwid=,logunit=,verbose=,error=,/debug,/help)
;
; INPUTS:
;         hdr             - A string array with the fits file data keyword
;                           header.
;         str             - A scalar string specifying the keyword which value
;                           will be read from HDR.
;
; KEYWORD PARAMETERS:
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         30.05.2009 - Inluded error handling and this file header. /CS
;-
FUNCTION p3d_misc_fitsstrex,hdr,str,topwid=topwid,logunit=logunit, $
             verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  error=0 & rname='p3d_misc_fitsstrex: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_fitsstrex'
    return,-1L
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return,-1L
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(hdr)
  if s[s[0L]+1L] ne 7L or ~s[s[0L]+2L] then begin
    errmsg='HDR must be set; to a string array.'
    goto,error_handler
  endif

  s=size(str)
  if s[s[0L]+1L] ne 7L or s[s[0L]+2L] ne 1L then begin
    errmsg='STR must be set; to a scalar string.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Extracting the value:

  tmp=strpos(hdr,str)
  idx=where(tmp eq 0L,count)
  if ~count or count gt 1L then return,-1L

  tmp=hdr[idx[0L]]
  p1=strpos(tmp,'=')
  if p1 eq -1L then return,-1L
  tmp=strmid(tmp,p1+1L)
  p2=strpos(tmp,'/')
  if p2 eq -1L then return,-1L
  tmp=strmid(tmp,0L,p2)
  tmp=strtrim(tmp,2L)
  out=tmp

  ;;========================================------------------------------
  ;; Removing leading and trailing apostrophes, if such exist:

  tmp__=strpos(out,'''') & if tmp__ eq 0L then out=strmid(out,1L)
  tmp__=strpos(out,'''',/reverse_search)
  if tmp__ eq strlen(out)-1L then out=strmid(out,0L,strlen(out)-1L)
  out=strtrim(out,2L)

  ;;========================================------------------------------
  ;; Converting to a long or double, if possible:

  double_ok=0L
  on_ioerror,done
  tmp2=double(tmp)
  tmp3=long(tmp)
  out=tmp2 eq tmp3?tmp3:tmp2
  goto,done

done:
  return,out

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return,-1L
END ;;; function: p3d_misc_fitsstrex
