;+
; NAME:
;         p3d_misc_fileallbl
;
;         $Id: p3d_misc_fileallbl.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         Routine description pending.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_fileallbl,filename,fsfx,nblocks,bsfx,nfilename,nfsfx, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         filename        - .
;         fsfx            - .
;         nblocks         - .
;         bsfx            - .
;
; KEYWORD PARAMETERS:
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         nfilename       - .
;         nfsfx           - .
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; MODIFICATION HISTORY:
;         02.10.2009 - Routine introduced. /CS
;
;-
PRO p3d_misc_fileallbl,filename,fsfx,nblocks,bsfx,nfilename,nfsfx, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
        debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_fileallbl: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_fileallbl'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(nblocks)
  if s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L then begin
    errmsg='NBLOCKS [3] must be specified, and be of (any) integer type;' + $
           ' NBLOCKS>=1.'
    goto,error_handler
  endif
  if nblocks le 0L then begin
    errmsg='NBLOCKS [3] must be specified, and be of (any) integer type;' + $
           ' NBLOCKS>=1.'
    goto,error_handler
  endif
  if nblocks eq 1L then begin
    nfilename=filename
    nfsfx=fsfx
    return
  endif

  s=size(filename) & n=s[s[0L]+2L]
  if ~n or s[s[0L]+1L] ne 7L then begin
    errmsg='FILENAME [1] must be specified, and be of string type.'
    goto,error_handler
  endif

  sf=size(fsfx) & nf=sf[sf[0L]+2L]
  if nf ne n or sf[sf[0L]+1L] ne 7L then begin
    errmsg='FSFX [2] must be of string type and have as many elements as' + $
           ' FILENAME [1], i.e. '+strtrim(n,2L)+' elements, not '+ $
           strtrim(nf,2L)+'.'
    goto,error_handler
  endif

  sf=size(bsfx) & nf=sf[sf[0L]+2L]
  if nf ne nblocks or sf[sf[0L]+1L] ne 7L then begin
    errmsg='BSFX [4] must be of string type and have NBLOCKS='+ $
           strtrim(nblocks,2L)+' elements.'
    goto,error_handler
  endif
  blen=strlen(bsfx)

  ;;========================================------------------------------
  ;; Constructing an array of filenames excluding the block suffix:

  tfilename=strarr(n)
  i=-1L
  while ++i lt n do begin
    tfile=filename[i] & slen=strlen(tfile)
    for j=0L,nblocks-1L do begin
      pos=strpos(tfile,bsfx[j],/reverse_search)
      if pos+blen[j] eq slen then begin
        tfilename[i]=strmid(tfile,0L,pos)
        continue
      endif
    endfor ;; j=0L,nblocks-1L
    if tfilename[i] eq '' then begin
      errmsg='  [file '+strtrim(i+1L,2L)+'/'+strtrim(n,2L)+'] :: Could not' + $
             ' extract a filename without any of the block suffixes BSFX='+ $
             strjoin('"'+bsfx+'"',', ')+' for FILENAME="'+tfile+'".'
      goto,error_handler
    endif
  endwhile ;; ++i lt n

  ;;========================================------------------------------
  ;; Creating a unique array of filenames (in case filenames belonging to the
  ;; same data set were given as input these duplicates are now removed):

  tmp=uniq(tfilename,sort(tfilename))
  tfilename=tfilename[tmp]
  tfsfx=fsfx[tmp]
  n=n_elements(tfilename)
  
  ;;========================================------------------------------
  ;; Creating a two-dimensional array of filenames where block-specific
  ;; filenames are provided in the second dimensions (i.e. along rows):

  nfilename=strarr(n,nblocks)
  nfsfx=strarr(n,nblocks)
  for j=0L,nblocks-1L do begin
    nfilename[*,j]=tfilename+bsfx[j]
    nfsfx[*,j]=tfsfx
  endfor

  ;;========================================------------------------------
  ;; Checking if all files exist:

  for i=0L,n-1L do begin
    exist=lonarr(nblocks)
    for j=0L,nblocks-1L do $
       exist[j]=file_test(nfilename[i,j]+fsfx[i],/regular,/read)
    if long(total(exist)) ne nblocks then begin
      errmsg='  [file '+strtrim(i+1L,2L)+'/'+strtrim(n,2L)+'] :: Cannot fi' + $
             'nd the files for all blocks:'
      for j=0L,nblocks-1L do begin
        errmsg=[errmsg,'    [block '+strtrim(j+1L,2L)+'/'+ $
                strtrim(nblocks,2L)+'] :: FILE="'+nfilename[i,j]+fsfx[i]+ $
                '" :: file '+(exist[j]?'':'not ')+'found.']
      endfor
      goto,error_handler
    endif ;; long(total(exist)) ne nblocks
  endfor ;; i=0L,n-1L

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_fileallbl
