;+
; NAME:        
;         p3d_misc_e3dio_write
;
;         $Id: p3d_misc_e3dio_write.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:     
;         Create a e3d format data file from a RSS array and its
;         associated position table
;
; AUTHOR:
;         Joris Gerssen
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:    
;         p3d_io_e3write,rss=,xpos=,ypos=,e3dname=,shape=,size=, $
;             error=,verbose=,/debug,/help 
;
; KEYWORDS: 
;         rss           - (float) 2D array
;         xpos          - (float) 1D array with spaxel x-positions
;         ypos          - (float) 1D array with spaxel y-positions
;         e3dname       - (string) name of the output e3d file
;         size          - (float) size of a spaxel
;         shape         - (int)   spaxel shape. 0: square,  1: circular 
;         topwid        - If set, then error messages are displayed using
;                         DIALOG_MESSAGE, using this widget id as
;                         DIALOG_PARENT, instead of MESSAGE.
;         logunit       - Messages are saved to the file pointed to by this
;                         logical file unit, if it is defined.
;         error         - Returns an error code if set
;         verbose       - Show more information on what is being done
;         debug         - The error handler is not setup if debug is set.
;         help          - Show this routine documentation, and exit
;
; INPUT FILES:   
;         e3d_template_data_table
;         e3d_template_group_table
;
; OUTPUT:        
;         A fits file is e3d format.
;
; EXAMPLE:   
;         p3d_misc_e3dio_write,rss,xpos,ypos,'test.e3d'
;         p3d_misc_e3dio_write,rss,xpos,ypos,'test.e3d', error=error, /verbose
;
; MODIFICATION HISTORY:
;         02.09.2009 - Adapted to p3d from an earlier version by JAG
;
;-
PRO p3d_misc_e3dio_write_primary_header,header
  compile_opt hidden,IDL2

  ;; Create and return the primary header for the E3D file.

  mkhdr,header,0
  sxdelpar,header,'DATE'
  sxdelpar,header,'COMMENT'
  sxaddpar,header,'SIMPLE','T','file does conform to FITS standard'
  sxaddpar,header,'BITPIX',8

  sxaddpar,header,'EURO3D','T',' All mandatory EuroE3D extensions present'
  sxaddpar,header,'E3D_ADC','F',' data corrected from atm dispersion'
  sxaddpar,header,'E3D_VERS',1.0,format='(f3.1)', $
           ' version number of the EuroE3D format'

  return
END ;; procedure: p3d_misc_e3dio_write_primary_header


PRO p3d_misc_e3dio_write_make_data_table,header,nrow,nbyte,spec_len
  compile_opt hidden,IDL2

  ;; Create a data table for the E3D file, requires 'e3d_template_data_table':

  template_dir=!p3d_path+'data'+path_sep()+'tables'+path_sep()

  tfields= 11
  ctypes='Units x-axis      Units y-axis'
  crvals=0.0E
  cdelts=1.0E 
  crpixs=1L

  nkeyword=50L ;; define the number of keywords
  header=strarr(nkeyword)

  ;; Use a template file to define the header keywords
  line=''   ;; define string variable
  openr,lun,template_dir+'e3d_template_data_table',/get_lun
  for k=0L,nkeyword-1L do begin
    readf,lun,line,format='(a80)'
    header[k]=line
  endfor
  close,lun

  ;; Update and/or add keywords:
  sxaddpar,header,'NAXIS1',nbyte
  sxaddpar,header,'NAXIS2',nrow
  sxaddpar,header,'TFIELDS',tfields

  sxaddpar,header,'CTYPES',ctypes
  sxaddpar,header,'CRVALS',crvals,' reference wavelength'
  sxaddpar,header,'CDELTS',cdelts,' wavelength increment per pixel'
  sxaddpar,header,'CRPIXS',crpixs,' reference pixel'

  sxaddpar,header,'TFORM10',string(spec_len,format='(I0)')+'E'
  sxaddpar,header,'TFORM11',string(spec_len,format='(I0)')+'J'

  if (tfields eq 12) then begin
    sxaddpar,header,'TTYPE12','STAT_SPE',' label for field'
    sxaddpar,header,'TFORM12',string(spec_len,format='(I0)')+'E'
  endif

  return
end ;; procedure: p3d_misc_e3di_owrite_make_data_table


;; Auxiliary procedure for p3d_misc_e3diowrite.
PRO p3d_misc_e3dio_write_make_bin_table,lun,nrow,spec_len,xpos,ypos,rss
  compile_opt hidden,IDL2

  spec_id=0L
  selected=0L     
  nspax=1L
  spec_sta=0L
  group_n=0L
  spax_id='S'

  qual_spe=lonarr(spec_len)
  if keyword_set(stat) then stat_spe=fltarr(spec_len)

  for row=1L,nrow do begin
    data_spe=rss[*,row-1L]
    fxbwrite,lun,spec_id,1,row
    fxbwrite,lun,selected,2,row
    fxbwrite,lun,nspax,3,row
    fxbwrite,lun,spec_len,4,row
    fxbwrite,lun,spec_sta,5,row
    fxbwrite,lun,float(xpos[row-1]),6,row
    fxbwrite,lun,float(ypos[row-1]),7,row
    fxbwrite,lun,group_n,8,row
    fxbwrite,lun,spax_id,9,row
    fxbwrite,lun,data_spe,10,row
    fxbwrite,lun,qual_spe,11,row
  endfor ;; row=1L,nrow

  return
END ;; procedure: p3d_misc_e3dio_write_make_bin_table

;; Auxiliary procedure for p3d_misc_e3dio_write.
;; Requires the file 'e3d_template_group_table'
PRO p3d_misc_e3dio_write_make_group_table,header
  compile_opt hidden,IDL2

  template_dir=!p3d_path+'data'+path_sep()+'tables'+path_sep()

  ;; define string variable
  line=''

  ;; define the number of keywords
  nkeyword=54 

  header=strarr(nkeyword)

  ;; use a template file to define the header keywords
  openr,lun,template_dir+'e3d_template_group_table',/get_lun
  for k=0L,nkeyword-1L do begin
    readf,lun,line,format='(a80)'
    header[k]=line
  endfor
  close,lun

  return
end ;; procedure: p3d_misc_e3dio_write_make_group_table


;; Auxiliary procedure for p3d_misc_e3dio_write.
PRO p3d_misc_e3dio_write_make_bin_group,lun,shape,size
  compile_opt hidden,IDL2

  group_n=0L
;shape='S'
;size=1.0
  angle=0.0
  size2=size
  wave=500.0        ;; what is this ?
  airmass=1.0
  parangle=0.0
  pressure=0.0
  temperature=0.0
  humidity=0.0

  row=1

  fxbwrite,lun,group_n,1,row
  fxbwrite,lun,shape,2,row
  fxbwrite,lun,float(size),3,row
  fxbwrite,lun,angle,4,row
  fxbwrite,lun,float(size2),5,row
  fxbwrite,lun,wave,6,row
  fxbwrite,lun,airmass,7,row
  fxbwrite,lun,parangle,8,row
  fxbwrite,lun,pressure,9,row
  fxbwrite,lun,temperature,10,row
  fxbwrite,lun,humidity,11,row

  return
END ;; procedure: p3d_misc_e3dio_write_make_bin_group


PRO p3d_misc_e3dio_write,rss,xpos,ypos,e3dname,shape=shape_,size=size, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
        debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_e3dio_write: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_e3dio_write'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  ;; If this routine is called as a standalone module then the !p3d_path
  ;; variable is  likely not yet defined.
  defsysv,'!p3d_path',exists=exists
  if ~exists then begin
    errmsg=['The system variable !p3d_path is not defined.', $
            '       Start p3d once to set this variable.']
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Check input arguments

  if ~n_elements(size) then size=1.0

  tmp=~n_elements(shape_)?0L:shape_

  ;; Convert shape-number to a character:
  if tmp eq 0L then shape='S'
  if tmp eq 1L then shape='C'

  ;; Set error flag if spaxel shape is neither circular nor square:
  if ~(tmp eq 0L or tmp eq 1L) then begin
    ;; error=1
    errmsg='Shape of spaxels is neither square nor circular.'
    goto,error_handler
  endif

  ;; Filename of the created e3d file:
  e3d_filename=~n_elements(e3dname)?'rub.e3d':e3dname

  compress=0L
  tmp=strpos(e3d_filename,'.gz',/reverse_search)
  if tmp eq strlen(e3d_filename)-3L then begin
    compress=1L
    e3d_filename=strmid(e3d_filename,0L,tmp)
  endif

  if ~n_elements(rss) then begin
   errmsg='Please provide a valid RSS array.'
   goto,error_handler
  endif

  if ~n_elements(xpos) then begin
    errmsg='Please provide a valid xpos vector.'
    goto,error_handler
  endif

  if ~n_elements(ypos) then begin
    errmsg='Please provide a valid ypos vector.'
    goto,error_handler
  endif


  ;; Determine the rss size of the rss image and cross check with
  ;; position table vectors:
  sz=size(rss)
  nrow=sz[2]
  spec_len=sz[1]

  if verbose ge 1 then begin
    msg='Dimension of RSS image [col,row] :: '+ $
        string(format='(i4,",",i4)',spec_len,nrow)
    error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
              verbose=verbose)
  endif

  ;; Are the number of row in the rss and postion table consistent?:
  if nrow ne n_elements(xpos) then begin
    errmsg='Number of rows in RSS and position table arrays is not equal.'
    goto,error_handler
  endif

  ;; Set the number of bytes, i.e. the NAXIS value in the output E3D header:
  nbyte=33+8*spec_len

  ;; Create a primary fits array:
  p3d_misc_e3dio_write_primary_header,header
  writefits,e3d_filename,0,header,compress=0L

  ;; Create the data table:
  p3d_misc_e3dio_write_make_data_table,header,nrow,nbyte,spec_len
  fxbcreate,lun,e3d_filename,header
  p3d_misc_e3dio_write_make_bin_table,lun,nrow,spec_len,xpos,ypos,rss
  fxbfinish,lun

  ;; Add a group table:
  p3d_misc_e3dio_write_make_group_table,header
  fxbcreate,lun,e3d_filename,header
  p3d_misc_e3dio_write_make_bin_group,lun,shape,size
  fxbfinish,lun

  msg='Created the E3D image: "'+string(format='(a)',e3d_filename)+'".'
  error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid,verbose=verbose)

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
end ;; procedure: p3d_misc_e3dio_write
