;+
; NAME:
;         p3d_misc_checkfile
;
;         $Id: p3d_misc_checkfile.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine checks whether the fits header in the supplied filename
;         contains the required detector id. This is useful with instruments
;         which have several detectors (such as VIMOS and IMACS).
;         If the instrument keywords file does not contain the p3d keyword
;         specifier 'DETECTOR' then the routine just exits (for all instrument
;         which only have one detector).
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_checkfile,file,nd,d,kwrdlist,topwid=,logunit=,verbose=, $
;             error=,/debug
;
; INPUTS:
;         file            - A fits file filename.
;         nd              - A scalar integer that specifies the total number of
;                           detectors.
;         d               - An integer that specifies the id of the current
;                           detector.
;         kwrdlist        - A scalar string that specifies the name of a file
;                           that contains a list of instrument-specific
;                           keyword names.
;
; KEYWORD PARAMETERS:
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         24.06.2009 - Included this file header. /CS
;-
PRO p3d_misc_checkfile,file,nd,d,kwrdlist, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_checkfile: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_checkfile'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking if the file exists:

  if ~n_elements(nd) then nd=1L
  if nd eq 1L then return  ;; Nothing is done with only one detector

  if ~file_test(file,/regular,/read) then begin
    errmsg='The file FILE "'+file+'" does not exist.'
    goto,error_handler
  endif

  if ~n_elements(d) then d=0L
  s=size(d)
  if s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L then begin
    errmsg='D must be of (any) integer type; D>=0.'
    goto,error_handler
  endif
  if d lt 0L then begin
    errmsg='D must be of (any) integer type; D>=0.'
    goto,error_handler
  endif

  if size(kwrdlist,/type) ne 7L then begin
    errmsg='KWRDLIST [4] must be specified; as the scalar filename of an i' + $
           'nstrument-specific file with header keywords.'
    goto,error_handler
  endif
  if ~file_test(kwrdlist,/read,/regular) then begin
    errmsg='The file KWRDLIST "'+kwrdlist+'" does not exist.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Checking if the 'DETECTOR' keyword is present for the current
  ;; instrument:

  tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'DETECTOR',logunit=logunit, $
          topwid=topwid,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return

  ;; No check is done if 'DETECTOR' is not found:
  if tmp eq '-1' then return

  omp=p3d_misc_get_hdr_kwrd(kwrdlist,'DETECTOR_OFF',logunit=logunit, $
          topwid=topwid,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return

  offset=omp eq '-1'?0L:omp
  on_ioerror,format_error
  offset=long(offset)
  on_ioerror,NULL

  ;;========================================------------------------------
  ;; Checking if the detector is the same as it should be:

  hdr=headfits(file,silent=verbose lt 3)
  tmp=p3d_misc_fitsstrex(hdr,tmp,error=error)
  if error ne 0 then return

  if tmp ne d+1L+offset then begin
    errmsg=['DETECTOR mismatch: The detector of the file:', $
            '  "'+p3d_misc_pathify(file,/dpath)+'" ['+strtrim(tmp,2L)+']', $
            'is different from the selected detector ['+ $
            strtrim(d+1L+offset,2L)+']. Cannot continue!']
    goto,error_handler
  endif ;; tmp ne d+1L+offset

  return

format_error:
  errmsg='The value of the keyword DETECTOR_OFF ['+omp+'] cannot be conver' + $
         'ted to an integer.'
  on_ioerror,NULL

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure p3d_misc_checkfile
