;+
; NAME:
;         p3d_masterbias_smooth_file
;
;         $Id: p3d_masterbias_smooth_file.pro 115 2010-03-19 12:17:32Z christersandin $
;
; PURPOSE:
;         For a specified raw bias image this routine calls
;         P3D_MASTERBIAS_SMOOTH, which cleans the data from cosmic rays and bad
;         pixels. The result is an image where the original information in
;         every pixel has been smoothed with the value of neighbor pixels.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: master bias creation
;
; CALLING SEQUENCE:
;         p3d_masterbias_smooth_file,filename,parfile,out,ostr=,quadrant=, $
;             block=,opath=,ofilename=,sfx=,/compress,stawid=,topwid=, $
;             logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         filename        - A scalar string specifying the filename of the raw
;                           bias data file.
;         parfile         - The parameter list filename, this argument must be
;                           present.
;
; KEYWORD PARAMETERS:
;         ostr ['_mbias']  - A scalar string with the master-bias specific
;                           string that is used to create the output filename.
;         quadrant ['']   - A scalar integer, or character, that is appended to
;                           some of the tracing parameter names.
;         block ['']      - A scalar integer, or character, that is appended to
;                           some of the tracing parameter names.
;                           Note! QUADRANT must be specified if BLOCK is.
;         opath           - A scalar string that, optionally, specifies the
;                           path where the output data is written.
;         ofilename       - Upon exit this string contains the name of the
;                           written output file.
;         sfx [.fits]     - A scalar string specifying the file ending (without
;                           a trailing compression suffix, such as .gz or .Z).
;         compress [0]    - If this keyword is set then the output data file is
;                           compressed (using gzip).
;         stawid          - If set to a valid ID then a log message is written
;                           using this ID for relevant actions.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - The 'cleaned' bias image.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         07.10.2008 - Converted from the original routine p3d_bias_file of
;                      Thomas Becker and Petra Bhm. /CS
;
;-
PRO p3d_masterbias_smooth_file,filename,parfile,out,ostr=ostr, $
        quadrant=quadrant,block=block,opath=opath_,ofilename=ofilename, $
        sfx=sfx,compress=compress,stawid=stawid,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_masterbias_smooth_file: '
  if ~n_elements(verbose) then verbose=0
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_masterbias_smooth_file'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  n=n_elements(filename)
  if n ne 1L then begin
    errmsg='FILENAME must be set to a scalar string.'
    goto,error_handler
  endif
  if ~file_test(filename,/regular,/read) then begin
    errmsg='The file with FILENAME="'+filename+'" does not exist.'
    goto,error_handler
  endif

  s=size(parfile)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='PARFILE must be set to a string pointing at an existing file w' + $
           'ith an instrument parameter list.'
    goto,error_handler
  endif
  if ~file_test(parfile,/regular,/read) then begin
    errmsg='PARFILE must be set to a string pointing at an existing file w' + $
           'ith an instrument parameter list.'
    goto,error_handler
  endif

  if ~n_elements(quadrant) then quadrant=''
  s=size(quadrant)
  if s[0L] ne 0L or s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ne 7L and s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='QUADRANT must be a scalar integer or character.'
    goto,error_handler
  endif
  if s[s[0L]+1L] eq 7L then begin
    if strlen(quadrant) gt 1L then begin
      errmsg='If QUADRANT is of string type it must be of length 1!'
      goto,error_handler
    endif
    quadrant_=quadrant
  endif else begin
    quadrant_=strtrim(quadrant,2L)
    if strlen(quadrant_) gt 1L then begin
      errmsg='If QUADRANT is of type integer it must only have one digit!'
      goto,error_handler
    endif
  endelse

  if ~n_elements(block) then block=''
  s=size(block)
  if s[0L] ne 0L or s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ne 7L and s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='BLOCK must be a scalar integer or character.'
    goto,error_handler
  endif
  if s[s[0L]+1L] eq 7L then begin
    if strlen(block) gt 1L then begin
      errmsg='If BLOCK is of string type it must be of length 1!'
      goto,error_handler
    endif
    block_=block
  endif else begin
    block_=strtrim(block,2L)
    if strlen(block_) gt 1L then begin
      errmsg='If BLOCK is of type integer it must only have one digit!'
      goto,error_handler
    endif
  endelse

  if strlen(block_) eq 1L and ~strlen(quadrant_) then begin
    errmsg='If BLOCK is set, then so must QUADRANT be, it is not!'
    goto,error_handler
  endif
  xstr=strlen(quadrant_) eq 1L?('_'+quadrant_+block_):''

  ;; Creating an output filename if one was not defined (using the first file):
  if ~n_elements(ostr) then ostr='_mbias'
  if size(ostr,/type) ne 7L or n_elements(ostr) ne 1L then begin
    errmsg='OSTR must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  if ~n_elements(sfx) then sfx='.fits'
  if size(sfx,/type) ne 7L or n_elements(sfx) ne 1L then begin
    errmsg='SFX must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  compress=keyword_set(compress)

  tmppos=strpos(filename[0L],sfx,/reverse_search)
  fsfx=tmppos eq -1L?'':strmid(filename[0L],tmppos)
  tmpfile=tmppos eq -1L?filename[0L]:strmid(filename[0L],0L,tmppos)
  ofilename=tmpfile+ostr+xstr+fsfx

  tmp=strpos(ofilename,'.gz')
  if compress then begin
    if tmp eq -1L then ofilename+='.gz'
  endif else begin
    if tmp ne -1L then ofilename=strmid(ofilename,0L,tmp)
  endelse

  s=size(ofilename)
  if s[s[0L]+1L] ne 7L or s[s[0L]+2L] ne 1L then begin
    errmsg='OFILENAME must be a scalar string.'
    goto,error_handler
  endif

  if n_elements(opath_) ne 0L then begin
    if ~file_test(opath_,/directory) then begin
      errmsg='The output directory OPATH does not exist.'
      goto,error_handler
    endif
    opath=opath_
    tmp=strpos(opath,path_sep(),/reverse_search)
    if tmp ne strlen(opath)-1L then opath+=path_sep()

    ofilename=opath+file_basename(ofilename)
  endif


  ;;========================================------------------------------
  ;; The following values used to be in the instrument parameter file.
  ;; This routine is not a standard routine, which is why I have removed
  ;; this feature /CS:

  xcleanbox=5L & ycleanbox=5L
  xsmobox=5L & ysmobox=5L

  ;;========================================------------------------------
  ;; Logging information:

  if usestawid then begin
    tmpstr='[Master bias] Creating a new -smoothed- masterbias from "'+ $
           file_basename(filename)+'".'
    widget_control,stawid,set_value=tmpstr
  endif ;; usestawid

  ;;========================================------------------------------
  ;; Reading the bias raw data file:

  bias=readfits(filename,hdr,silent=verbose lt 3,/no_unsigned)

  tmp=strpos(filename,path_sep(),/reverse_search)+1L
  tfilename=strmid(filename,tmp)

  msg='Read the file "'+tfilename+'".'
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  ;;========================================------------------------------
  ;; Handing over the control to the smoothing routine:

  p3d_masterbias_smooth,bias,out,xclean=long(xcleanbox), $
      yclean=long(ycleanbox),xsmo=long(xsmobox),ysmo=long(ysmobox), $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  ;;========================================------------------------------
  ;; Writing the bias output file:

  fxhmake,tmphdr,out
  fxaddpar,hdr,'BITPIX',fxpar(tmphdr,'BITPIX')
  fxaddpar,hdr,'BZERO',0
  p3d_misc_fitsparsplit,'IMRAW',file_basename(filename[0L]), $
      hdr,' Raw data filename',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  p3d_misc_fitsparsplit,'IMPATH',file_dirname(filename[0L],/mark_directory), $
      hdr,' Raw data path',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug

  writefits,ofilename,out,hdr,compress=compress

  msg='Wrote the -smoothed- file "'+file_basename(ofilename)+'".'
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  ;;========================================------------------------------
  ;; Writing information to the status line:

  if usestawid then begin
    tmpstr='[Master bias] Wrote the -smoothed- file "'+ $
           file_basename(ofilename)+'".'
    widget_control,stawid,set_value=tmpstr
  endif ;; usestawid

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_masterbias_smooth_file
