function whichgdas,obsdate
;+
; NAME:
;	WHICHGDAS
;
; PURPOSE:
;	This function selects the GDAS sounding data file that lies closest to the observation date.
;
; CALLING SEQUENCE:
;	Result = WHICHGDAS(Obsdate)
;
; INPUTS:
;	Obsdate:      Julian date of the observation (scalar float or double)

; OUTPUTS:
;	Result:       Scalar string with absolute path of the closest GDAS sounding data file
;
; COMMON BLOCKS:
;       TELLREM_INFO:  This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; EXAMPLE:
;	filename = WHICHGDAS(2456590.9276736d)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info

; check the input parameters
if n_params() lt 1 then begin
    message,/info,'Syntax: Result = WHICHGDAS(Obsdate)'
    retall
endif

; check whether folder path has / at the end
if strmid(gdasfolder,0,1,/reverse_offset) ne '/' then gdasf=gdasfolder+'/*' else gdasf=gdasfolder+'*'
; get content of folder
gdasfiles=file_search(gdasf)
; create array to take the available souding data dates
gdasdates=dblarr(n_elements(gdasfiles))
; extract the dates from the filenames
for i=0,n_elements(gdasfiles)-1 do begin
    ; get the filename
    split=strsplit(gdasfiles[i],'/',/extract)
    filename=split[n_elements(split)-1]
    ; check whether it adheres to naming convention GDASYYYY_MM_DD_HH
    if strpos(filename,'.') ne -1 || strmid(filename,0,4) ne 'GDAS' then begin
	print,'The specified folder for the GDAS sounding files contains files that do not adhere to the naming convention!'
	print,'Please check correctness of stated folder and its content!'
	retall
    endif
    ; extract the date
    date=strsplit(filename,'_',/extract)
    date[0]=strmid(date[0],4)
    ; transform date to julian date
    gdasdates[i]=julday(date[1],date[2],date[0],date[3])
endfor
; which gdas is the closest in time to obsdate
thisone=where(abs(gdasdates-obsdate) eq min(abs(gdasdates-obsdate)))
file=gdasfiles[thisone]
return,file[0]
end