;+
; NAME:
;         p3d_wavecal_unfold_dispmask_file
;
;         $Id: p3d_wavecal_unfold_dispmask_file.pro 152 2010-04-16 08:04:10Z christersandin $
;
; PURPOSE:
;         This routine reads a dispersion mask file, that contains the
;         polynomial fitting parameters, and sets up a matrix with a
;         wavelength array for every spectrum.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: wavelength calibration
;
; CALLING SEQUENCE:
;         p3d_wavecal_unfold_dispmask_file,filename,npix,out,bin=bin, $
;             daxis=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         filename        - A string specifying the filename of the file that
;                           holds the dispersion mask data.
;         npix            - A scalar integer specifying the number of
;                           wavelength bins on the dispersion axis.
;
; KEYWORD PARAMETERS:
;         bin             - A scalar integer specifying the detector pixel bin
;                           size.
;         daxis [1]       - Defines the dispersion axis. The default, 1, is in
;                           the x-direction.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - A two-dimensional array holding a wavelength array
;                           of NPIX elements for every spectrum.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         07.11.2008 - Converted from original routine unfold_dispmask_file of
;                      Thomas Becker. /CS
;         07.06.2009 - Added logging information. /CS
;-
PRO p3d_wavecal_unfold_dispmask_file,filename,npix,out,bin=bin, $
        daxis=daxis,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_wavecal_unfold_dispmask_file: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_wavecal_unfold_dispmask_file'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(daxis) then daxis=1L
  sd=size(daxis)
  if sd[sd[0L]+2L] ne 1L or $
    (sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  sid=daxis?2L:1L

  s=size(filename)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='FILENAME [1] must be set; to the name of an existing file.'
    goto,error_handler
  endif
  if ~file_test(filename,/regular,/read) then begin
    errmsg='The file FILENAME "'+filename+'" cannot be read.'
    goto,error_handler
  endif

  s=size(npix)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='NPIX [2] must be a scalar integer; >=1.'
    goto,error_handler
  endif
  if npix lt 1L then begin
    errmsg='NPIX [2] must be a scalar integer; >=1.'
    goto,error_handler
  endif

  s=size(bin)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='BIN must be a scalar integer; >0.'
    goto,error_handler
  endif
  if bin lt 1L then begin
    errmsg='BIN must be a scalar integer; >0.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Reading the dispersion mask file, that contains polynomial fit parameters:

  params=readfits(filename,hdr,silent=verbose lt 3,/no_unsigned)

  msg=['Reading a dispersion mask and calculating the wavelength array', $
       'Read the dispersion mask file: "'+ $
       p3d_misc_pathify(filename,/dpath)+'".']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  s=size(params)
  deg=s[sid]-1L ;; The order of the stored polynomial(s).

  ;;========================================------------------------------
  ;; Setting up a wavelength for every spectrum bin:

  x=dindgen(npix)/bin

  if daxis then begin
    out=dblarr(npix,s[daxis])
    for k=0L,s[daxis]-1L do for L=0L,deg do out[*,k]+=params[k,L]*x^L
  endif else begin
    out=dblarr(s[daxis],npix)
    for k=0L,s[daxis]-1L do for L=0L,deg do out[k,*]+=params[k,L]*x^L
  endelse ;; daxis

  ;; Logging the information:
  msg='Calculated '+strtrim(npix,2L)+'-element wavelength arrays for all'+ $
      ' '+strtrim(s[daxis],2L)+' spectra; polynomial degree='+ $
      strtrim(deg,2L)+' (BIN='+strtrim(bin,2L)+').'
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_wavecal_unfold_dispmask_file
