;+
; NAME:
;         p3d_misc_binextract
;
;         $Id: p3d_misc_binextract.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine extracts the CCD binning parameters XBIN and YBIN which
;         must be present, in some form, in the input fits file header HDR. The
;         name of the keywords in HDR are looked up in the keyword file
;         KWRDLIST. If XBIN and YBIN have the same keyword header then it is
;         assumed that the keyword consists of three characters. XBIN is the
;         first character of the header keyword field and YBIN the third
;         character.
;
;         If either XBIN==0 or YBIN==0, then it is set to 1 (and a warning is
;         issued).
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_binextract,kwrdlist,hdr,xbin=,ybin=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         kwrdlist        - The name of a file, that contains a two-column list
;                           of parameters to use with p3d for the instrument in
;                           question.
;         hdr             - A string array with the fits file data keyword
;                           header.
;
; KEYWORD PARAMETERS:
;         xbin            - Returns XBIN (long).
;         ybin            - Returns YBIN (long).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         30.05.2009 - Inluded this file header. /CS
;-
PRO p3d_misc_binextract,kwrdlist,hdr,xbin=xbin,ybin=ybin,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_binextract: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_binextract'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(hdr)
  if s[s[0L]+1L] ne 7L or ~s[s[0L]+2L] then begin
    errmsg='HDR must be set; to a string array.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Extracting XBIN and YBIN from the keyword header data:

  tmpx=p3d_misc_get_hdr_kwrd(kwrdlist,'XBIN',topwid=topwid, $
           logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return

  if tmpx eq '-1' then begin
    xbin=1L
  endif else begin
    xbin=p3d_misc_fitsstrex(hdr,tmpx,topwid=topwid, $
             logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
  endelse

  tmpy=p3d_misc_get_hdr_kwrd(kwrdlist,'YBIN',topwid=topwid, $
           logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return

  if tmpy eq '-1' then begin
    ybin=1L
  endif else begin
    ybin=p3d_misc_fitsstrex(hdr,tmpy,topwid=topwid, $
             logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
  endelse

  if tmpx eq tmpy and size(tmpx,/type) eq 7L then begin
    bin=strtrim(xbin,2L)
    if strlen(bin) ne 3L then begin
      errmsg='The header keyword '+strupcase(tmpx)+' ['+bin+'] must be 3 c' + $
        'haracters long.'
      goto,error_handler
    endif

    on_ioerror,format_error
    defstr='XBIN' & tmp=strmid(bin,0L,1L) & xbin=long(tmp)
    defstr='YBIN' & tmp=strmid(bin,2L,1L) & ybin=long(tmp)
    on_ioerror,NULL
  endif else begin
    on_ioerror,format_error
    defstr='XBIN' & tmp=xbin & xbin=long(tmp)
    defstr='YBIN' & tmp=ybin & ybin=long(tmp)
    on_ioerror,NULL
  endelse ;; tmpx eq tmpy

  if ~xbin then begin
    msg='WARNING! XBIN==0, reverting to the default value XBIN=1.'
    error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
              verbose=verbose)
    xbin=1L
  endif

  if ~ybin then begin
    msg='WARNING! YBIN==0, reverting to the default value YBIN=1.'
    error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
              verbose=verbose)
    ybin=1L
  endif

  return

format_error:
  errmsg='Error extracting '+defstr+' ['+tmp+' is not an integer].'
  on_ioerror,NULL

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
                        verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_binextract
