;+
; NAME:
;         p3d_extract_optimal
;
;         $Id: p3d_extract_optimal.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         This routine is a wrapper for the two routines that perform the
;         actual task of optimal extraction, viz. p3d_extract_optimal_mox and
;         p3d_extract_optimal_mpd.
;
;         After the spectra have been extracted the line profile viewer of p3d
;         is launched to allow an examination of the result.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: spectrum extraction
;
; CALLING SEQUENCE:
;         p3d_extract_optimal,im,traces,dim,lprofs,out,dout,/ecalc,crmask=, $
;             proffun=,profwidth=,userparfile=,/monitor,stawid=,topwid=, $
;             logunit=,verbose=,error=,debug=,/help,_extra=
;
; INPUTS:
;         im              - A two-dimensional array, that holds the spectra
;                           that will be extracted here.
;         traces          - A two-dimensional array, that for every spectrum
;                           bin provides the y-position of every spectrum line
;                           on the raw data CCD image.
;         dim             - A two-dimensional array of the same dimensions as
;                           IM, with the errors of IM. DIMAGE must be present.
;                           The variance of IM is DIMAGE.
;         lprofs [sp=size(lprofs)]
;                         - A three-dimensional array, that for every spectrum
;                           bin, of every spectrum, holds the fitting
;                           parameters of a cross-dispersion profile.
;
; KEYWORD PARAMETERS:
;         ecalc [1]       - Errors are calculated and returned (DOUT) if this
;                           keyword is set.
;         crmask          - A two-dimensional array of integer type, and of the
;                           same dimensions as IM. If this variable is set then
;                           DIMAGE[masked pixels]=10^6. This way those pixels,
;                           which are hit by cosmic ray, are given a minimal
;                           weight when the spectra are extracted. All elements
;                           of CRMASK must be set to either 0 or 1.
;
;                           Note! If this mask is specified then the
;                           eliminate_crays option of the modified
;                           optimal extraction is always inactive.
;
;         proffun         - A scalar string with the name of the function to
;                           use when (re-)calculating the line profile.
;         profwidth       - 2*PROFWIDTH+1 is the total pixel-width of the
;                           interval where the flux is integrated. This
;                           parameter is only used if OPTEXMETHOD=='modhorne',
;                           but is always used when plotting the profiles if
;                           MONITOR is set.
;         userparfile     - A scalar string specifying the name of an optional
;                           user parameter file, that could contain any of the
;                           keywords that are described in the routine
;                           description.
;         monitor [1]     - If this keyword is set then a line profiles viewer
;                           is shown after the extraction is done.
;         subtitle        - If this keyword is set to a string then the string
;                           is added to the spectrum viewer title.
;         stawid          - If set, then various messages are written to the
;                           p3d GUI status line (this must be the widget id of
;                           that label widget).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;         _extra          - Contains keywords that are passed on to other
;                           routines.
;
; OUTPUTS:
;         out             - A two-dimensional array of the extracted spectra
;                           with the same dimensions as TRACES.
;         dout            - A two-dimensional array of the extracted spectra
;                           with the same dimensions as OUT. This is the error
;                           of OUT.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
;-
PRO p3d_extract_optimal,im,traces,dim_,lprofs,out,dout,ecalc=ecalc, $
        crmask=crmask,proffun=proffun_,profwidth=profwidth, $
        userparfile=userparfile,subtitle=subtitle,monitor=monitor, $
        stawid=stawid,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help,_extra=_extra
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_extract_optimal: '
  if ~n_elements(verbose) then verbose=0
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)
  loglevel=~n_elements(logunit)?0L:logunit[1L]

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_extract_optimal'
    return
  endif ;; keyword_set(help) or ~n_params()

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(im)
  if ~s[s[0L]+2L] or s[0L] ne 2L or $
     (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='IM must be set to a two-dimensional array of decimal type.'
    goto,error_handler
  endif

  st=size(traces)
  if ~st[st[0L]+2L] or st[0L] ne 2L or $
     (st[st[0L]+1L] ge 6L and st[st[0L]+1L] le 11L) then begin
    errmsg='TRACES must be set to a two-dimensional array of decimal type.'
    goto,error_handler
  endif

  sp=size(lprofs)
  if ~sp[sp[0L]+2L] or sp[0L] ne 3L or $
     (sp[sp[0L]+1L] ge 6L and sp[sp[0L]+1L] le 11L) then begin
    errmsg='LPROFS must be set to a three-dimensional array of decimal type.'
    goto,error_handler
  endif

  se=size(dim_)
  if ~se[se[0L]+2L] or se[0L] ne 2L or $
     (se[se[0L]+1L] ge 6L and se[se[0L]+1L] le 11L) then begin
    errmsg='DIMAGE {'+strtrim(se[se[0L]+1L],2L)+',['+strtrim(se[1L],2L)+ $
           ','+strtrim(se[2L],2L)+']} must be of the same ' + $
           'dimensions as IM {'+strtrim(s[s[0L]+1L],2L)+',['+ $
           strtrim(s[1L],2L)+','+strtrim(s[2L],2L)+']}.'
    goto,error_handler
  endif
  if se[0L] ne s[0L] or se[1L] ne s[1L] or $
     se[se[0L]+2L] ne s[s[0L]+2L] then begin
    errmsg='DIMAGE {'+strtrim(se[se[0L]+1L],2L)+',['+strtrim(se[1L],2L)+ $
           ','+strtrim(se[2L],2L)+']} must be of the same ' + $
           'dimensions as IM {'+strtrim(s[s[0L]+1L],2L)+',['+ $
           strtrim(s[1L],2L)+','+strtrim(s[2L],2L)+']}.'
    goto,error_handler
  endif
  dim=dim_

  usecrmask=0L
  sp=size(crmask)
  if sp[sp[0L]+2L] ge 1L then begin
    if sp[0L] ne 2L or $
      (sp[sp[0L]+1L] ge 4L and sp[sp[0L]+1L] le 11L) then begin
      errmsg='CRMASK must be a two-dimensional array of integer type;' + $
             ' CRMASK[*]=0||1.'
      goto,error_handler
    endif
    if min(crmask) lt 0L or max(crmask) gt 1L then begin
      errmsg='CRMASK must be a two-dimensional array of integer type;' + $
             ' CRMASK[*]=0||1.'
      goto,error_handler
    endif ;; min(crmask) lt 0L or max(crmask) gt 1L
    if sp[1L] ne s[1L] or sp[2L] ne s[2L] then begin
      errmsg='CRMASK ['+strtrim(sp[1L],2L)+','+strtrim(s[1L],2L)+'] must h' + $
             'ave the same dimensions as the input image IM [' + $
             strtrim(sp[2L],2L)+','+strtrim(s[2L],2L)+'].'
      goto,error_handler
    endif ;; sp[1L] ne s[1L] or sp[2L] ne s[2L]
    usecrmask=1L
  endif

  sb=size(proffun_)
  if sb[sb[0L]+2L] ne 1L or sb[sb[0L]+1L] ne 7L then begin
    errmsg='PROFFUN must be set to a scalar string with the name of the fu' + $
           'nction to use.'
    goto,error_handler
  endif

  proffun=strlowcase(proffun_)
  case proffun of
    'gaussian':      iidx=2L
    'lorentzian':    iidx=2L
    'gauss/lorentz': iidx=3L
    'doublegauss':   iidx=4L
    else: begin
      errmsg=['PROFFUN must be one of the four options:', $
              '  "gaussian", "lorentzian", "gauss/lorentz", "doublegauss"', $
              ' PROFFUN="'+proffun_+'" is not a valid option.']
      goto,error_handler
    end
  endcase ;; proffun

  if n_elements(userparfile) ne 0L then begin
    sp=size(userparfile)
    if sp[sp[0L]+2L] ne 1L or sp[sp[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE, if set, must be a scalar string.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='USERPARFILE, cannot read the file "'+userparfile+'".'
        goto,error_handler
      endif

      ;; Reading the user parameter file data:
      uparname_='' & uparvalue_=''
      readcol,userparfile,uparname_,uparvalue_,format='a,a',comment=';', $
          silent=verbose lt 3,delimiter=' '
      if n_elements(uparname_) ne 0L then begin
        uparname=uparname_
        uparval=uparvalue_
      endif ;; nelements(uparname_) ne 0L
    endif ;; userparfile ne ''
  endif ;; n_elements(userparfile) ne 0L

  if ~n_elements(monitor) then monitor=1L
  sbi=size(monitor)
  if sbi[sbi[0L]+2L] ne 1L or $
    (sbi[sbi[0L]+1L] ge 4L and sbi[sbi[0L]+1L] le 11L) then begin
    errmsg='MONITOR must be set to a scalar integer; MONITOR=0||1.'
    goto,error_handler
  endif
  if monitor ne 0L and monitor ne 1L then begin
    errmsg='MONITOR must be set to a scalar integer; MONITOR=0||1.'
    goto,error_handler
  endif

  if ~n_elements(ecalc) then ecalc=1L
  sbi=size(ecalc)
  if sbi[sbi[0L]+2L] ne 1L or $
    (sbi[sbi[0L]+1L] ge 4L and sbi[sbi[0L]+1L] le 11L) then begin
    errmsg='ECALC must be set to a scalar integer; ECALC=0||1.'
    goto,error_handler
  endif
  if ecalc ne 0L and ecalc ne 1L then begin
    errmsg='ECALC must be set to a scalar integer; ECALC=0||1.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Setting the error of masked pixels to a large number (10^16) in order to
  ;; give the corresponding pixel values a minimal weight when extracting the
  ;; spectra:

  if usecrmask then begin
    idx=where(crmask,count)
    if count ne 0L then dim[idx]=1d16

    msg='Set the error of '+strtrim(count,2L)+' masked pixels to 1.0e16.'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  endif ;; usecrmask

  ;;========================================------------------------------
  ;; Determining which extraction method to use:

  p3d_misc_read_params,uparname,uparval,'optexmethod',optexmethod,/upo,/a0, $
      topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(optexmethod) then optexmethod='modhorne'
  optexmethod=strlowcase(optexmethod)
  if optexmethod ne 'modhorne' and optexmethod ne 'mp-deconvolution' then begin
    errmsg='OPTEXMETHOD must be set to either "horne" or "mp-deconvolution' + $
           '", not "'+optexmethod+'".'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Calling the MOX and MPD spectrum extraction routines:

  if optexmethod eq 'modhorne' then begin
    p3d_extract_optimal_mox,im,dim,lprofs,out,dout,profwidth=profwidth, $
        proffun=proffun,ecalc=ecalc,usecrmask=usecrmask, $
        userparfile=userparfile,stawid=stawid,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,_extra=_extra
    if error ne 0 then return
  endif else begin
    p3d_extract_optimal_mpd,im,dim,lprofs,out,dout,proffun=proffun, $
        ecalc=ecalc,userparfile=userparfile,stawid=stawid,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,_extra=_extra
    if error ne 0 then return
  endelse

  ;;========================================------------------------------
  ;; Examining the resulting spectra:

  if monitor then begin

    ;;========================================-----------------------------
    ;; Extracting pixel-based cross-dispersion profiles for every spectrum
    ;; and wavelength bin:

    c=reform(lprofs[*,*,0L])
    n=2L*profwidth+1L
    c=round(c) & clo=c-profwidth

    ;; The array that will hold the normalized line profiles:
    prof=dblarr(st[1L],st[2L],n)

    pmultfac=10L
    x0=rebin(dindgen(n)-profwidth,n,pmultfac)
    x1=rebin(dindgen(1L,pmultfac)/pmultfac,n,pmultfac)
    x=x0+x1

    ;; Logging:
    msg=['Extracting pixel-based cross-dispersion line profiles.', $
         '     pmultfac='+strtrim(pmultfac,2L)+'.']
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

    ;; Looping over wavelength bins and spectra:
    wlstr='/'+strtrim(st[1L],2L)
    i=-1L & while ++i lt st[1L] do begin

      if usestawid and ~(i mod 100L) then begin
        tmp=strtrim(i+1L,2L)+'-'+strtrim((i+100L)<st[1L],2L)
        msg='[Optimal spectrum extraction] extracting pixel-based' + $
            ' profiles; w.l.bins '+tmp+wlstr
        widget_control,stawid,set_value=msg
      endif ;; usestawid and ~(i mod 100L)

      j=-1L & while ++j lt st[2L] do begin
        f=p3d_misc_profunction(x+c[i,j],lprofs[i,j,*],proffun=proffun, $
              /nobg,topwid=topwid,error=error,verbose=verbose)
        if error ne 0 then return

        if pmultfac gt 1L then f=total(f,2L)/pmultfac

        ;;==============================--------------------
        ;; Normalizing every pixel-based cross-dispersion profile:

        if lprofs[i,j,iidx] ne 0d0 then prof[i,j,*]=f/total(f)

      endwhile ;; ++j lt st[2L]
    endwhile ;; ++i lt st[1L]

    optex=prof*rebin(out,st[1L],st[2L],n)
    title='p3d: Optimal extraction spectrum viewer'
    title+=n_elements(subtitle) ne 0L?' ['+subtitle+'].':'.'
    p3d_display_lineprofiles,im,traces,lprofs,rawdarray=dim, $
        title=title,optex=optex,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return
  endif ;; monitor

  ;; Converting results to floating point precision:
  out=float(out)
  if ecalc then dout=float(dout)

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_extract_optimal
