;+
; NAME:
;         p3d_tracing_trace
;
;         $Id: p3d_tracing_trace.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         This routine does an automatic tracing of raw data files. The steps
;         that are performed are:
;           1) Locate the spectra
;           2) Follow the spectra in the direction of dispersion (optional)
;         ad 1: Peaks are looked for in several (max. 81) columns
;               and compared to minimize errors because of noise.
;               The procedure works save for at least 10-20 columns,
;               better more than 40 columns (depending on SNR).
;
;         The input parameter file (PARFILE) must contain the following
;         keywords (all values must be specified for data before it is read out
;         - in pixels [b]). Optional additional specifiers are the quadrant (Q)
;         and the block (B) - these are specified with the XSTR keyword:
;           spnum[_QB]          :: the number of spectra.
;           dist_tr[_QB]  [b]   :: the average separation between consecutive
;                                  spectra.
;           dmin_tr[_QB]  [b]   :: the minimum allowed spectrum separation.
;           dmax_tr[_QB]  [b]   :: the maximum allowed spectrum separation.
;           cut_tr              :: specifies the fraction of all pixels
;                                  (2*findwidth_tr+1) that must be found to
;                                  be a maximum in order for their average to
;                                  be defined as a maximum.
;           findwidth_tr        :: specifies the number of pixels (/2) that
;                                  are averaged across the dispersion
;                                  direction when searching for spectra. The
;                                  actually used number of pixels is
;                                  2*findwidth+1. The maximum number allowed
;                                  is half the number of pixels in the
;                                  dispersion direction (although it is not
;                                  recommended to use that many pixels).
;           findcolumn_tr[_QB]  :: specifies the pixel number, in the
;                                  dispersion direction, that is used in the
;                                  first search of spectra in the
;                                  cross-dispersion direction. The central bin
;                                  is used if this value is set to -1, or if
;                                  the specified number is larger than the
;                                  number of bins.
;           centervar_tr[_QB] [b]::specifies a half-width of every spectrum in
;                                  the cross-dispersion direction. The actual
;                                  width is 2*centervar_tr+1.
;           fwhm_tr       [b]   :: specifies the full width at half maximum of
;                                  the Gaussian profile, that is used when
;                                  weighting the cross-dispersion spectrum
;                                  profile.
;           refindwidth_tr [b]  :: specifies the half-width of the region that
;                                  is used when calculating a reduced-size
;                                  positions array.
;           refinddist_tr       :: specifies the dividing factor when
;                                  determining the size of the reduced size
;                                  spectrum-positions array.
;           smowidth_tr         :: the reduced-size data is smoothed across
;                                  the cross-dispersion direction using a box
;                                  car smoothing algorithm and this half-width.
;           dispsmowidth_tr     :: the spectrum positions of the reduced-size
;                                  data are smoothed across the dispersion
;                                  dimension using a box car smoothing
;                                  algorithm and this half-width.
;           niterat_tr          :: specifies how many times the calculation of
;                                  the central position, in the
;                                  cross-dispersion direction, of every
;                                  spectrum is iterated.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: tracing of spectra on the CCD
;
; CALLING SEQUENCE:
;         p3d_tracing_trace,parfile,array,traces,gaps=,/const,bin=,spec0=, $
;             yrange=,var_spec0=,xstr=,/no_display,daxis=,userparfile=, $
;             stawid=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         parfile         - A scalar string specifying the filename of an ASCII
;                           file that contains parameters that are used in the
;                           tracing process.
;         array           - A one- or two-dimensional array of decimal type,
;                           that provides the data which is used to find the
;                           spectrum positions.
;
; KEYWORD PARAMETERS:
;         gaps            - An integer array specifying a pre-defined pattern
;                           of gaps between spectra in the data; across the
;                           cross-dispersion direction. This is a required
;                           input.
;
;                           Assuming equally spaced spectra on the CCD in the
;                           cross-dispersion direction the values in this array
;                           specify the sequential positions where no spectra
;                           are found. For example if GAPS contains
;                           [16,16... then the pre-defined spectrum sequence
;                              0,1,2,...,15,16,17,18,19,...
;                           is changed to:
;                              0,1,2,...,15,18,19,20,21,...
;                           i.e. there must be no spectra in the linearly
;                           sequential positions 16 and 17.
;
;         const [0]       - If set then it is assumed that the spectrum
;                           positions are constant in the dispersion direction
;                           (this is generally not true).
;         bin [1]         - Binning factor for the y-direction (DAXIS==1) or
;                           the x-direction (DAXIS==2).
;         spec0 [-1]      - A scalar decimal value specifying the position of
;                           the first spectrum.
;         yrange          - this variable is forwarded to P3D_TRACING_FINDSPEC.
;         var_spec0       - A scalar decimal value. If SPEC0 is set then
;                           VAR_SPEC0 provides the uncertainty in the position.
;         xstr ['']       - An empty string, or a three character string that
;                           is appended to some of the tracing parameter names.
;         no_display      - If set, then results are not displayed.
;         userparfile     - A scalar string specifying the name of an optional
;                           user parameter file, that could contain any of the
;                           keywords read from the instrument parameter file
;                           (except for 'spnum').
;         daxis [1]       - Defines the dispersion direction dimension of
;                           IMAGE. The default, 1, is in the x-direction.
;         stawid          - If set to a valid ID then a log message is written
;                           using this ID for relevant actions.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         traces          - A one- or two-dimensional array of floating point
;                           type specifying the spectrum position for every
;                           spectrum at every pixel in the dispersion
;                           direction. This is the trace mask.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         06.10.2008 - Converted from the original routine p3de_trace of
;                      Thomas Becker and Petra Bhm. /CS
;-
PRO p3d_tracing_trace,parfile,array_,traces,gaps=gaps,const=const,bin=bin, $
        spec0=spec0,yrange=yrange,var_spec0=var_spec0,xstr=xstr, $
        no_display=no_display,userparfile=userparfile,daxis=daxis, $
        stawid=stawid,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_tracing_trace: '
  if ~n_elements(verbose) then verbose=0
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_tracing_trace'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(daxis) then daxis=1L
  s=size(daxis)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif

  s=size(parfile)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='PARFILE must be a scalar string.'
    goto,error_handler
  endif
  if ~file_test(parfile,/regular,/read) then begin
    errmsg='PARFILE must be the name of an existing file.'
    goto,error_handler
  endif
  parname='' & parvalue=''
  readcol,parfile,parname,parvalue,format='a,a',delimiter=' ', $
          silent=verbose lt 3,comment=';'

  s=size(array_) & s_=s
  if s[0L] ne 1L and s[0L] ne 2L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='ARRAY must be a one- or two-dimensional array of floating poin' + $
           't type.'
    goto,error_handler
  endif
  array=daxis?array_:transpose(array_)
  s=size(array)

  if s[0L] eq 1L then begin
    n_col=s[1L] & n_row=1L
    image=reform(array,n_col,n_row)
  endif else begin
    n_col=s[1L] & n_row=s[2L]
    image=array
  endelse

  s=size(gaps)
  if ~s[s[0L]+2L] or s[0L] ne 1L or $
     (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='GAPS must be a one-dimensional array of integer type.'
    goto,error_handler
  endif

  bin=~n_elements(bin)?1L:bin
  s=size(bin)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='BIN must be a scalar of integer type.'
    goto,error_handler
  endif
  if bin lt 1L then begin
    errmsg='BIN must be an integer >=1.'
    goto,error_handler
  endif

  const=keyword_set(const)
  s=size(const)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='CONST must be a scalar integer; 0||1.'
    goto,error_handler
  endif
  if const lt 0L or const gt 1L then begin
    errmsg='CONST must be either 0 or 1.'
    goto,error_handler
  endif

  if ~n_elements(spec0) then spec0=-1L
  s=size(spec0)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='SPEC0 must be a scalar of floating point type.'
    goto,error_handler
  endif

  var=spec0 eq -1L?-1L:(~keyword_set(var_spec0)?dist/2L:var_spec0)
  s=size(var)
  if s[s[0L]+2L] ne 1L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='VAR must be a scalar of floating point type.'
    goto,error_handler
  endif

  if ~n_elements(xstr) then xstr=''
  s=size(xstr)
  if s[0L] ne 0L or s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='XSTR, if set, must be a two- or three-character scalar string.'
    goto,error_handler
  endif
  if strlen(xstr) ne 0L and $
     strlen(xstr) ne 3L and strlen(xstr) ne 2L then begin
    errmsg='XSTR, if set, must be a two- or three-character scalar string.'
    goto,error_handler
  endif

  if n_elements(userparfile) ne 0L then begin
    s=size(userparfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE, if set, must be a scalar string.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='USERPARFILE, cannot read the file "'+userparfile+'".'
        goto,error_handler
      endif

      ;; Reading the user parameter file data:
      uparname_='' & uparvalue_=''
      readcol,userparfile,uparname_,uparvalue_,format='a,a',comment=';', $
          silent=verbose lt 3,delimiter=' '
      if n_elements(uparname_) ne 0L then begin
        uparname=uparname_
        uparvalue=uparvalue_
      endif ;; nelements(uparname_) ne 0L
    endif ;; userparfile ne ''
  endif ;; n_elements(userparfile) ne 0L

  s=size(array)

  ;;========================================------------------------------
  ;; Reading pre-defined parameter values from PARFILE (file.prm):

  ;; General parameters:

  ;; The number of spectra:
  p3d_misc_read_params,parname,parvalue,'spnum'+xstr,spnum,/must_exist, $
      type='integer',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return

  ;; Tracing-specific parameters:
  p3d_misc_read_params,parname,parvalue,'dist_tr'+xstr,dist,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,type='float', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  dist/=bin

  p3d_misc_read_params,parname,parvalue,'dmin_tr'+xstr,dmin,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,type='float', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  dmin/=bin

  p3d_misc_read_params,parname,parvalue,'dmax_tr'+xstr,dmax,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,type='float', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  dmax/=bin

  p3d_misc_read_params,parname,parvalue,'cut_tr',cut,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,type='float', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  p3d_misc_read_params,parname,parvalue,'findwidth_tr',findwidth,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,type='integer', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  findwidth=findwidth<((n_col-1L)/2L)

  p3d_misc_read_params,parname,parvalue,'findcolumn_tr'+xstr,column, $
      /must_exist,uparname=uparname,uparvalue=uparvalue,type='integer', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  p3d_misc_read_params,parname,parvalue,'centervar_tr'+xstr,centervar, $
      /must_exist,uparname=uparname,uparvalue=uparvalue,type='float', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  centervar/=bin

  p3d_misc_read_params,parname,parvalue,'fwhm_tr',fwhm,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,type='float', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  fwhm/=bin

  p3d_misc_read_params,parname,parvalue,'refindwidth_tr',refindwidth, $
      /must_exist,uparname=uparname,uparvalue=uparvalue,type='integer', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  refindwidth=long(float(refindwidth)/bin)

  p3d_misc_read_params,parname,parvalue,'refinddist_tr',refinddist, $
      /must_exist,uparname=uparname,uparvalue=uparvalue,type='integer', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  p3d_misc_read_params,parname,parvalue,'smowidth_tr',smowidth, $
      /must_exist,uparname=uparname,uparvalue=uparvalue,type='integer', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  p3d_misc_read_params,parname,parvalue,'dispsmowidth_tr',dispsmowidth, $
      /must_exist,uparname=uparname,uparvalue=uparvalue,type='integer', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  p3d_misc_read_params,parname,parvalue,'niterat_tr',niterat, $
      /must_exist,uparname=uparname,uparvalue=uparvalue,type='integer', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  column=column lt 0L or column ge n_col?n_col/2L:long(column)
  specmask=lindgen(spnum)

  for k=0L,n_elements(gaps)-1L do specmask[gaps[k]:spnum-1L]++

  if ~keyword_set(no_display) then begin
    screensize=get_screen_size()
    xsize=(1200L>(0.85*screensize[0L]))<(0.85*screensize[0L])
    ysize=( 800L>(0.85*screensize[1L]))<(0.85*screensize[1L])

    xpos=long(0.5*(screensize[0L]-xsize))
    ypos=long(0.5*(screensize[1L]-ysize))
  endif ;; ~keyword_set(no_display)

  ;;====================----------
  ;; Step 1:
  ;;====================----------

  ;;========================================------------------------------
  ;; Locating the spectra in the raw data image; for a dispersion-direction
  ;; spectrum region of the width 2*FINDWIDTH+1 that is centered on COLUMN. A
  ;; central component of the spectrum locating algorithm are the spectrum
  ;; separation distance DIST, and the spectrum mask SPECMASK:

  p3d_tracing_findspec,image,specmask,dist,dmin,dmax,cut,specpos, $
      yrange=yrange,width=findwidth,column=column,spec0=spec0, $
      var=vardisplay,display=1L-keyword_set(no_display), $
      fwhm=fwhm,centervar=centervar,niterat=niterat, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  ;; Writing information to the status line:
  if usestawid then begin
    tmpstr='[Trace 1/3] -findspec- Located '+strtrim(n_elements(specpos),2L)+ $
           ' spectrum positions.'
    widget_control,stawid,set_value=tmpstr
  endif

  ;;====================----------
  ;; Step 2:
  ;;====================----------

  if const or n_col le 2L*findwidth+1L then begin

    ;; The traces are constant in the dispersion direction, the position array
    ;; is returned as it is:

    if daxis then begin
      traces=specpos
    endif else begin
      traces=transpose(specpos)
    endelse

    msg=['Traces are constant in the dispersion direction']
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  endif else begin

    ;; Storing the just calculated positions for the central wavelength:
    oldpos=specpos

    ;;========================================------------------------------
    ;; Resizing ARRAY to have a smaller size in the dispersion direction; the
    ;; resize involves an averaging:

    nlower=column/refinddist
    ntotal=s[1L]/refinddist+1L
    pos   =lindgen(ntotal)*refinddist + (column mod refinddist)

    p3d_misc_mean_smooth_small,array,pos,refindwidth,smo,pos, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    ;;===================----------
    ;; Logging the performed operations:

    msg=['Rescaling tracing data :: Step 2a [to use fewer wavelength bins]',$
         '  Wavelength bin separation [refinddist_tr]='+ $
         strtrim(refinddist,2L), $
         '  Wavelength smoothing bin width [refindwidth_tr]='+ $
         strtrim(refindwidth,2L)]
    error=p3d_misc_logger(msg,logunit,loglevel=2,rname=rname, $
        verbose=verbose ge 1)

    ;;===================----------
    ;; Writing information to the status line:

    if usestawid then begin
      tmpstr='[Trace 2/3] -rescale- Rescaled data to contain '+ $
             strtrim(ntotal,2L)+', instead of '+strtrim(s[1L],2L)+ $
             ', wavelength bins.'
      widget_control,stawid,set_value=tmpstr
    endif

    ;;========================================------------------------------
    ;; Reference trace positions:

    nsmo=size(smo)
    reftraces=fltarr(nsmo[1L],spnum)
    diffs=fltarr(nsmo[1L]-1L,spnum)

    reftraces[nlower+1L,*]=specpos ;; the central wavelength bin

    ;;========================================------------------------------
    ;; Smoothing the scaled data further:
    ;;   looping over all wavelength bins from the center-column to 0:

    for k=nlower,0L,-1L do begin

      ;; Re-calculating corrected spectrum positions in the scaled data:
      p3d_tracing_correctpos,smo[k,*], $
          oldpos,-1L,dist,fwhm,centervar,niterat,newpos, $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return

      ;; Smoothing the positions across spectra -
      ;;   first with a median and then with a mean:
      p3d_misc_smooth_1d,newpos-oldpos,smowidth,newmed ,/median, $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return
      p3d_misc_smooth_1d,newmed       ,smowidth,newmean, $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return
      newpos=oldpos+newmean

      ;; The positions of the current bin are used as starting values for the
      ;; consecutive bin:
      oldpos=newpos
      
      diffs[k,*]=-newmean & reftraces[k,*]=newpos

    endfor ;; k=nlower,0L,-1L

    ;;========================================------------------------------
    ;; Smoothing the scaled data further:
    ;;   looping over all wave. bins from the center-column to the max bin:
    ;; Note! Starting with data which has already been smoothed:

    startpos=(nlower-10L)>0L
    oldpos=transpose(reftraces[startpos,*])

    for k=startpos+1L,nsmo[1L]-1L do begin

      ;; Re-calculating corrected spectrum positions in the scaled data:
      p3d_tracing_correctpos,smo[k,*], $
          oldpos,-1L,dist,fwhm,centervar,niterat,newpos, $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return

      ;; Smoothing the positions across spectra -
      ;;   first with a median and then with a mean:
      p3d_misc_smooth_1d,newpos-oldpos,smowidth,newmed ,/median, $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return
      p3d_misc_smooth_1d,newmed       ,smowidth,newmean, $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then return
      newpos=oldpos+newmean

      ;; The positions of the current bin are used as starting values for the
      ;; consecutive bin:
      oldpos=newpos

      diffs[k-1L,*]=newmean & reftraces[k,*]=newpos

    endfor ;; k=startpos+1L,nsmo[1L]-1L


    ;;========================================------------------------------
    ;; Smoothing the resized positions once more -
    ;;   this time in the dispersion direction:

    p3d_misc_median_smooth,diffs  ,diffmed ,dispsmowidth,0L, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return
    p3d_misc_mean_smooth  ,diffmed,diffmean,dispsmowidth,0L, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    model_traces=fltarr(nsmo[1L],spnum)
    for k=1L,nsmo[1L]-1L do $
       model_traces[k,*]=model_traces[k-1L,*]+diffmean[k-1L,*]

    ;; Calculating an average value of the difference between smoothed and
    ;; original values across the dispersion dimension. This average value is
    ;; thereafter added to the smoothed offset positions to give the final
    ;; tracing positions in the reduced-size array:

    value=total(reftraces-model_traces,1L)/nsmo[1L]
    reftraces=model_traces+float(rebin(transpose(value),nsmo[1L],spnum))


    ;;===================----------
    ;; Logging the performed operations:

    msg=['Calculated corrected positions :: Step 2b [using the scaled data]', $
         '  Smoothing bin half-width [smowidth_tr; cross-dispersion dire' + $
         'ction]='+strtrim(smowidth,2L), $
         '  Smoothing bin half-width [dispsmowidth_tr; dispersion direct' + $
         'ion]='+strtrim(dispsmowidth,2L)]
    error=p3d_misc_logger(msg,logunit,loglevel=1,rname=rname, $
        verbose=verbose ge 1)

    ;; Writing information to the status line:
    if usestawid then begin
      tmpstr='[Trace 3/3] -smoothing- finished. Now expanding the data to ' + $
             'all wavelength bins'
      widget_control,stawid,set_value=tmpstr
    endif

    ;;========================================------------------------------
    ;; Expanding the cross-dispersion spectrum positions to cover all wave-
    ;; length bins (using linear interpolation):

    p3d_tracing_expand_traces,reftraces,pos,s[1L],traces,$
        logunit=logunit,topwid=topwid,verbose=verbose,error=error,debug=debug

    if daxis eq 2L then traces=transpose(traces)

  endelse ;; const or n_col le 2L*findwidth+1L

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_tracing_trace
