;------------------------------------------------------------------------------;
pro DopTomogPlotPre, sdir = sdir, inf = inf, suf = suf, phr = phr, $
	grd = grd, ctb = ctb, eps = eps, psc = psc, dsp = dsp
;------------------------------------------------------------------------------;
; plot doppler tomography pre output
;------------------------------------------------------------------------------;
; sdir	:	map source directory; default ''
; inf	:	map source file; default 'doptomog.out'
; suf	:	output file suffix; default ''
; phr	:	phase range; default 1
; grd	:	polar-grid=0, xy-grid=1; default 0
; ctb	:	color table; default 15 (stern-special)
; eps	:	output encapsulated postscript; default 1 (yes)
; psc	:	output postscript in color; default 1 (yes)
; dsp	:	display plots on screen; default 1 (yes)
;------------------------------------------------------------------------------;
	common griddata, grds, gs, ds
;------------------------------------------------------------------------------;
	common procdata, pmod, proj
;------------------------------------------------------------------------------;
	common pardata, vm, vf, gam, w0, abs, atm, src, ih, nrm, ns, wd, am, al, ac
;------------------------------------------------------------------------------;
	common specdata, npb, nvp, phase, dphase, vp, spi, sp, spri, spr, spf, sprf
;------------------------------------------------------------------------------;
	common mapdata, np, mapi, map, dmapi, dmap, mmapi, mmap
;------------------------------------------------------------------------------;
	common workdata, vxy, vmax, dv
;------------------------------------------------------------------------------;
	; check keywords, set defaults
	if n_elements(sdir) eq 0 then sdir = ''
	if n_elements(inf) eq 0 then inf = 'doptomog.out'
	if n_elements(suf) eq 0 then suf = ''
	if n_elements(phr) eq 0 then phr = 1
	if n_elements(grd) eq 0 then grd = 0
	if n_elements(ctb) eq 0 then ctb = 15
	if n_elements(eps) eq 0 then eps = 1
	if n_elements(psc) eq 0 then psc = 1
	if n_elements(dsp) eq 0 then dsp = 1
;------------------------------------------------------------------------------;
	; grid data
	grds = grd
	; check if grid parameter file exists
	exist = findfile('pGrid')
	if exist(0) ne '' then begin
		; load parameter values from pGrid file
		@pGrid
	endif else begin
		print, 'Parameter file <pGrid> not found!'
		read, 'Press <Enter> to exit... ', key
		return
	endelse
;------------------------------------------------------------------------------;
	; read input
	DopTomogRead, sdir + inf
;------------------------------------------------------------------------------;
	; set plotting to postscript
	set_plot, 'ps'
	; set up device
	device, font_size = 12, bits_per_pixel = 8
	; set encapsulated postscript if necessary
	if eps eq 1 then device, /encaps else device, encaps = 0
	; set color postscript if necessary
	if psc eq 1 then device, /color else device, color = 0
	; load color table
	loadct, ctb, /silent
	; max no of colors of display
	mcol = !d.table_size - 1
;------------------------------------------------------------------------------;
	; scale maximum velocity with velocity units
	vmax = vm / vf
	; override velocity grid scale
	if vmax le 0.5 then gs = 0.1
;------------------------------------------------------------------------------;
	; if dopmap and spectra indicators is on
	if mapi gt 0 and spi gt 0 then begin
		; set up device for doptomog output
		device, /portrait, xsize = 30, ysize = 30
		file = sdir + 'doptomog' + suf + '.ps'
		device, filename = file
		; string for top of doptomog.ps page
		lbl = src + ' ' + string(format = '(''!7k!5!D0!N = '', f6.1)', w0)
		if abs then lbl = lbl + '  ABSORPTION  '
		lbl = lbl + string(format = '('' !7c!5 = '', e8.2)', gam)
		lbl = lbl + string(format = '('' np = '', i3)', np)
		lbl = lbl + string(format = '('' npb = '', i4)', npb)
		lbl = lbl + string(format = '('' nvp = '', i4)', nvp)
		lbl = lbl + string(format = '('' nsw = '', i1)', ns)
		lbl = lbl + string(format = '('' wid = '', e7.1)', wd)
		lbl = lbl + string(format = '('' amp = '', e7.1)', am)
		lbl = lbl + string(format = '('' !7a!5 = '', f6.3)', al)
		xyouts, 0.05 * !d.x_size, 0.95 * !d.y_size, lbl, /device
		; DOPMAPS
		; character size for dopmap velocity grid labels
		chs = 0.7
		!x.charsize = chs
		!y.charsize = chs
		; offsets and sizes
		del = 0.05
		xof = 0.075
		yof = 0.65
		vsi = 0.25
		msi = 300
		; XY-velocity coordinates of centres of bins
		vxy = (2 * findgen(np) / (np - 1) - 1) * vmax
		dv = vxy(1) - vxy(0)
		vxy = [vxy(0) - dv / 2, vxy(np - 1) + dv / 2]
		; stretch dopmap to fixed dimension
		lmap = congrid(map, msi, msi)
		ma = max(lmap)
		if dsp eq 1 then mapdsp = fltarr(msi, msi, 3)
		; scalings of dopmap
		; 1st position
		pos = [xof, yof, xof + vsi, yof + vsi]
		; 3 intensity scalings
		for i = 1, 3 do begin
			; set specifics for each dopmap
			; 1st scaling: linear
			if i eq 1 then map0 = lmap / ma > 0
			; 2nd and 3rd scaling
			if i eq 2 or i eq 3 then begin
				; 2nd scaling: log
				if i eq 2 then sc = 2
				; 3rd scaling: log over larger range
				if i eq 3 then sc = 4
				map0 = (alog((lmap / ma) > 0.0001) / sc + 1) > 0
			endif
			map0 = (mcol * (1 - map0) > 0) < mcol
			if dsp eq 1 then mapdsp(*, *, i - 1) = map0
			result = DopMapPlot(pos, vxy, map0, 0)
			if result ne 0 then return
			; next position
			pos = pos + [vsi + del, 0, vsi + del, 0]
		endfor
		; SPECTRA
		; character size for spectra velocity grid labels
		chs = 0.9
		!x.charsize = chs
		!y.charsize = chs
		; offsets and xy-sizes
		del = 0.05
		xof = 0.075
		yof = 0.25
		xsi = 0.175
		ysi = 0.30
		; spectra preparations
		result = SpectraPrep(phr)
		if result ne 0 then return
		npp = size(spf)
		if dsp eq 1 then spdsp = fltarr(npp(2), npp(1), 4)
		; scale velocity points with velocity units
		vp0 = vp / vf
		ph0 = phase
		; maximum intensity from input spectra
		spm = max(spf)
		; 1st position
		pos = [xof, yof, xof + xsi, yof + ysi]
		; input (O) and reconstructed (C) spectra, O-C, C-O
		for i = 1, 4 do begin
			; set specifics for each set of spectra
			; input spectra
			if i eq 1 then begin
				sp0 = spf / spm
				ttl = 'Input (O)'
			endif
			; reconstructed spectra
			if i eq 2 then begin
				sp0 = sprf / spm
				ttl = 'Reconstructed (C)'
			endif
			; O-C spectra
			if i eq 3 then begin
				sp0 = 0.5 + (spf - sprf) / spm
				ttl = 'O - C'
			endif
			; C-O spectra
			if i eq 4 then begin
				sp0 = 0.5 + (sprf - spf) / spm
				ttl = 'C - O'
			endif
			sp0 = rotate((mcol * (1 - sp0) > 0) < mcol, 4)
			if dsp eq 1 then spdsp(*, *, i - 1) = sp0
			result = SpectraPlot(pos, phr, vp0, ph0, sp0, ttl)
			if result ne 0 then return
			; next position
			pos = pos + [xsi + del, 0, xsi + del, 0]
		endfor
		device, /close
		if eps eq 1 then spawn, "sed -i '$ i\showpage' " + file
	endif
;------------------------------------------------------------------------------;
	; increase plot definition
	!p.thick = 3
	!p.charthick = 3
	!p.charsize = 1.25
	!x.charsize = 1
	!y.charsize = 1
;------------------------------------------------------------------------------;
	; output spectra stuff to ps files
	if spi gt 0 then begin
		; set up device for phase average line profile
		device, /portrait, xsize = 15, ysize = 15
		; separate ps file for phase average line profile
		file = sdir + 'avgp' + suf + '.ps'
		device, filename = file
		AvgProfPlot, inf = inf, vsf = vf, ovr = 1
		device, /close
		if eps eq 1 then spawn, "sed -i '$ i\showpage' " + file
		; offsets and xy-sizes
		if phr eq 1.0 then begin
			xof = 0.225
			xsi = 0.65
			yof = 0.20
			ysi = 0.75
		endif else begin
			xof = 0.175
			xsi = 0.70
			yof = 0.15
			ysi = 0.80
		endelse
		; position
		pos = [xof, yof, xof + xsi, yof + ysi]
		; set up device for spectra output
		if phr eq 1.0 then device, /portrait, xsize = 7.5, ysize = 10
		if phr eq 2.0 then device, /portrait, xsize = 10, ysize = 15
		; loop for input and reconstructed spectra
		for j = 0, 1 do begin
			; separate ps files for input and reconstructed spectra
			if j eq 0 then file = sdir + 'specin' + suf + '.ps'
			if j eq 1 then file = sdir + 'specre' + suf + '.ps'
			device, filename = file
			sp0 = spdsp(*, *, j)
			result = SpectraPlot(pos, phr, vp0, ph0, sp0)
			if result ne 0 then return
			device, /close
			if eps eq 1 then spawn, "sed -i '$ i\showpage' " + file
		endfor
	endif
;------------------------------------------------------------------------------;
	!p.thick = 1
	!p.charthick = 1
	!p.charsize = 1
;------------------------------------------------------------------------------;
	; display screen output
	if dsp eq 1 then begin
		set_plot, 'x'
		; average line profile
		window, 0, xsize = 600, ysize = 600, xpos = 0, ypos = 0
		AvgProfPlot, inf = inf, vsf = vf
		; scalings of dopmap and spectra
		device, decomposed = 0
		loadct, ctb, /silent
		window, 1, xsize = 3 * msi, ysize = 2 * msi, colors = mcol
		; 3 dopmap intensity scalings
		pos = [0, msi, 2 * msi]
		for j = 0, 2 do begin
			map0 = mapdsp(*, *, j)
			tv, congrid(map0, msi - 1, msi - 1), pos(j), pos(1)
		endfor
		; spectra
		xsi = msi / 2
		pos = [0, xsi, 2 * xsi, 3 * xsi]
		for j = 0, 3 do begin
			sp0 = spdsp(*, *, j)
			tv, congrid(sp0, xsi - 1, msi - 1), pos(j), pos(0)
		endfor
		; delta maps
		for j = 0, 3 do begin
			map0 = dmap(*, *, j)
			map0 = map0 / max(map0)
			map0 = mcol * (0.5 + 2 * map0 < 1 > 0)
			dx = msi / 2 * ((j eq 1) or (j eq 3))
			dy = msi / 2 * ((j eq 0) or (j eq 1))
			tv, congrid(map0, msi / 2 - 1, msi / 2 - 1), 2 * msi + dx, dy
		endfor
		key = ''
		read, 'Press <Enter> to continue... ', key
	endif
;------------------------------------------------------------------------------;
	return
end
;------------------------------------------------------------------------------;
