function uvextinction,tellremparameters,wave,gdasdata
;+
; NAME:
;	UVEXTINCTION
;
; PURPOSE:
;	This function calculates the UV extinction using an LBLRTM model for
;	the stated parameters.
;
; CALLING SEQUENCE:
;	Result = UVEXTINCTION(Tellremparameters,Wave,Gdasdata)
;
; INPUTS:
;	Tellremparameters:  Structure for the individual object from the
;			    array of structures created by TELLREMPARAMETERS.
;	Wave:		    UVB wavelength array of the object.
;	Gdasdata:	    Scalar string containing absolute path to GDAS
;			    file
;
; OUTPUTS:
;	This function returns the extinction in the UVB region appropriate for
;	the input LBLRTM model parameters at the wavelengths given by wave.
;	Since the model only reaches down to 3672 AA the extinction is
;	extrapolated below that value.
;
; EXAMPLE:
;       uvext = UVEXTINCTION(Tellremparameters_object,Wave,'/here/lies/the/Gdasdata')
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 3 then begin
    message,/info,'Syntax: Result = uvextinction(tellremparameters,wave,gdasdata)'
    retall
endif

; check input wavelength array
if min(wave) gt 3500. then begin
    print,'UVEXTINCTION: This seems not to be the UVB wavelength array!'
    print,'Please check!'
    retall
endif

; regions to calculate telluric spectrum in
x=[3672.,3965.,4300.,4700.,4950.,5190.,5790.,6000.]

; do calulations
for i=0,n_elements(x)-2 do begin
    runlblrtm,x[i],x[i+1],gdasdata,tellremparameters.obsaltitude,wp,tp,water=tellremparameters.para[0].p[4],methane=tellremparameters.para[0].p[5],carbondioxide=tellremparameters.para[0].p[6],oxygen=tellremparameters.para[0].p[7],/silent
    if i eq 0 then w=wp else w=[w,wp]
    if i eq 0 then t=tp else t=[t,tp]
endfor
vactoair,w

; smooth out the little lines that are there, that can not be distinguished from noise in the data
tsm=gaussconv(w,t,50)

; fit the part between 3700 AA and 4600 AA to get transmission below 3672 AA
ind=where(w ge 3700. and w le 4600.)
wf=w[ind]
tf=tsm[ind]
c=poly_fit(wf,tf,4)
; observed wavelength array below 3700
indobsbelow=where(wave lt 3700.)
wobsbelow=wave[indobsbelow]
; build transmission from polynom
tsmfit=poly(wobsbelow,c)

; rebin calculated transmission above 3700 AA to observed wavelength
indcalc=where(w ge 3700.)
wcalc=w[indcalc] & tsmcalc=tsm[indcalc]
; observed wavelength array
indobs=where(wave ge 3700.)
wobs=wave[indobs]
trbcalc=interpol(tsmcalc,wcalc,wobs)

; put the two transmissions together
t=[tsmfit,trbcalc]
; check that it has the same number of elements as the observed wavelength array
if n_elements(t) ne n_elements(wave) then begin
    print,'t and wave do not have the same number of elements!'
    retall
endif

return,t
end