pro tellrem_test_run,packagefolder,executablename,mipasmodel
;+
; NAME:
;	TELLREM_TEST_RUN
;
; PURPOSE:
;	This procedure perfoms a test run of the tellrem package using
;	exemplary data provided in the package to check whether the 
;	installation of the related software was sucessful.
;
; CALLING SEQUENCE:
;	TELLREM_TEST_RUN, Packagefolder, Executablename, Mipasmodel
;
; INPUTS:
;	Packagefolder:	String containing the path to where the tellrempackage
;			folder is located.
;	Executablename:	String containing the name of LBLRTM executable.
;	Mipasmodel:	String containing the absolute path to the MIPAS model
;			atmosphere.
;
; OUTPUTS:
;	None. But creates plot file Tellrem_test_run_comparison.ps comparing the
;	freshly determined telluric line removed spectrum to the spectrum
;	provided in the package.
;
; EXAMPLE:
;	TELLREM_TEST_RUN, '/the/tellrem/package/is/here/', 'lblrtm_executable',
;	'/MIPAS/model/is/here/equ.atm'
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, December 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-

; print informational message
print,'This is a test run of the tellrem package.'
; check whether folder path has / at the end
if strmid(packagefolder,0,1,/reverse_offset) ne '/' then packagefolder=packagefolder+'/*'
; create info_for_tellrem file from input to run test
openw,unit,'test_info',/get_lun
; first line GDAS
printf,unit,packagefolder+'tellrempackage/example_files/Test_Run/GDAS/'
; second line MIPAS model
printf,unit,mipasmodel
; third line executable name
printf,unit,executablename
; fourth line folder containing data
printf,unit,packagefolder+'tellrempackage/example_files/Test_Run/Test_data/'
; fifth line obstype
printf,unit,'SCI'
; close info_for_tellrem file
free_lun,unit

; get the current folder
spawn,'pwd',currentfolder

; get the info needed to run tellrem, i.e. create the common block TELLREM_INFO
loadtellreminfo,currentfolder+'/test_info'

; get the objects to work on
objects=getobjects()
; determine parameters needed for the removal of telluric lines
tellremparams=tellremparameters(objects,plotname='tellrem_test_run_',savname='testrun_params')
; do telluric line removal and build cleaned spectra
testspec=tellremspectra(tellremparams,savname='testrun_spectra')

; create plots enabling comparison between provided result and freshly calculated result
; restore provided result (variable 'spec')
restore,packagefolder+'tellrempackage/example_files/Test_Run/tellrem_test_run_spectra.sav',/verbose

; prepare plotting
set_plot,'ps'
filename='Tellrem_test_run_comparison.ps'
device,/color,filename=filename
; plot arms together and their difference
!p.multi=[0,1,2]
; UVB
plot,spec.wclu,spec.clu,title='Comparison UVB - black=provided, red=new calculated',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B)
oplot,testspec.wclu,testspec.clu,col=235
; check shape of difference to determine plot range
diffu=spec.clu-testspec.clu
minmaxu=minmax(diffu)
meanu=mean(diffu)
if meanu eq 0 then begin
    y1=-0.2
    y2=0.2
endif else begin
    y1=minmaxu[0]-0.1*minmaxu[0]
    y2=minmaxu[1]+0.1*minmaxu[1]
endelse
plot,spec.wclu,diffu,title='Difference between provided and newly calculated UVB',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),yra=[y1,y2],/yst

; VIS
plot,spec.wclv,spec.clv,title='Comparison VIS - black=provided, red=new calculated',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),xra=[5400,10300],/xst,xtickformat='(I5)'
oplot,testspec.wclv,testspec.clv,col=235
; check shape of difference to determine plot range
diffv=spec.clv-testspec.clv
minmaxv=minmax(diffv)
meanv=mean(diffv)
if meanv eq 0 then begin
    y1=-0.2
    y2=0.2
endif else begin
    y1=minmaxv[0]-0.1*minmaxv[0]
    y2=minmaxv[1]+0.1*minmaxv[1]
endelse
plot,spec.wclv,diffv,title='Difference between provided and newly calculated VIS',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),yra=[y1,y2],/yst,xra=[5400,10300],/xst,xtickformat='(I5)'

; NIR1
plot,spec.wcln1,spec.cln1,title='Comparison NIR1 - black=provided, red=new calculated',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),xra=[9900,13500],/xst,xtickformat='(I5)'
oplot,testspec.wcln1,testspec.cln1,col=235
; check shape of difference to determine plot range
diffn1=spec.cln1-testspec.cln1
minmaxn1=minmax(diffn1)
meann1=mean(diffn1)
if meann1 eq 0 then begin
    y1=-0.2
    y2=0.2
endif else begin
    y1=minmaxn1[0]-0.1*minmaxn1[0]
    y2=minmaxn1[1]+0.1*minmaxn1[1]
endelse
plot,spec.wcln1,diffn1,title='Difference between provided and newly calculated NIR1',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),yra=[y1,y2],/yst,xra=[9900,13500],/xst,xtickformat='(I5)'

; NIR2
plot,spec.wcln2,spec.cln2,title='Comparison NIR2 - black=provided, red=new calculated',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),xra=[14400,18200],/xst,xtickformat='(I5)'
oplot,testspec.wcln2,testspec.cln2,col=235
; check shape of difference to determine plot range
diffn2=spec.cln2-testspec.cln2
minmaxn2=minmax(diffn2)
meann2=mean(diffn2)
if meann2 eq 0 then begin
    y1=-0.2
    y2=0.2
endif else begin
    y1=minmaxn2[0]-0.1*minmaxn2[0]
    y2=minmaxn2[1]+0.1*minmaxn2[1]
endelse
plot,spec.wcln2,diffn2,title='Difference between provided and newly calculated NIR2',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),yra=[y1,y2],/yst,xra=[14400,18200],/xst,xtickformat='(I5)'

; NIR3
plot,spec.wcln3,spec.cln3,title='Comparison NIR3 - black=provided, red=new calculated',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),xra=[19500,24100],/xst,xtickformat='(I5)'
oplot,testspec.wcln3,testspec.cln3,col=235
; check shape of difference to determine plot range
diffn3=spec.cln3-testspec.cln3
minmaxn3=minmax(diffn3)
meann3=mean(diffn3)
if meann3 eq 0 then begin
    y1=-0.2
    y2=0.2
endif else begin
    y1=minmaxn3[0]-0.1*minmaxn3[0]
    y2=minmaxn3[1]+0.1*minmaxn3[1]
endelse
plot,spec.wcln3,diffn3,title='Difference between provided and newly calculated NIR3',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),yra=[y1,y2],/yst,xra=[19500,24100],/xst,xtickformat='(I5)'
!p.multi=0
device,/close
idlplottitlechanger,filename
set_plot,'x'

print,'Tellrem test run complete. Take a look at the file Tellrem_test_run_comparison.ps created in this folder to compare the telluric line removed spectrum provided in the package and the one just produced from the same data.'
end