function tell,wave,p,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent
;+
; NAME:
;	TELL
;
; PURPOSE:
;	This function calculates a model transmission spectrum with the abundances
;	stated and adjusts it by convolving with a Gaussian, shifting in
;	wavelength and multiplying by a straight line so that it can be compared
;	to an observed spectrum.
;
; CALLING SEQUENCE:
;	model = TELL(Wavelength,Parameters,Gdasdata=Gdasdata,Obsaltitude=Obsaltitude)
;
; INPUTS:
;	Wavelength:   Wavelength array in AA for which model is requested
;	Parameters:   Array of parameters of model ([smoothing_FWHM,velocity_shift,
;		      slope_of_straight,intercept_of_straight,water_abundance,
;		      methane_abundance,carbondioxide_abundance,oxygen_abundance])
;
; REQUIRED KEYWORD PARAMETERS:
;	GDASDATA:     Scalar string containing absolute path to GDAS file
;	OBSALTITUDE:  Altitude angle observation was taken at in deg
;
; KEYWORD PARAMETERS:
;	SILENT:	      Set to 1 if you do not want to get informational messages.
;
; OUTPUTS:
;	This function returns an array of the same size as wavelength containing
;	the transmission spectrum modelled and adjusted according to the parameters.
;
;
; COMMON BLOCKS:
;       MERKEN:       This common block saves the parameters and results to avoid
;		      having to run LBLRTM to often.
;
; EXAMPLE:
;       mod = TELL(wave_array,[0.5,0.1,-5.8,2.,1.0,1.0,1.0,1.0],gdasdata='/here/lies/the/Gdasdata',
;	obsaltitude=82.2)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 2 then begin
    message,/info,'Syntax: model=tell(wave,p,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent)'
    retall
endif
if not keyword_set(gdasdata) then begin
    print,'Keyword GDASDATA not set. This has to be set!'
    retall
endif
if not keyword_set(obsaltitude) then begin
    print,'Keyword OBSALTITUDE not set. This has to be set!'
    retall
endif

; make sure silent keyword has a value
if not keyword_set(silent) then silent=0

;common block to cache lblrtm results
common merken,par_old,range_old,w_old,t_old

; parameter assignment (general spectrum adjustment, needed below)
smooth=p[0]
v=p[1]
slope=p[2]*1.d2    ; parameters should be of equivalent magnitude (see MPFIT FAQ)
b=p[3]*1.d5

; determine range model is needed in
range=minmax(wave)
; convert to vacuum
airtovac,range

; these are the abundances for this run
par_new= p[4:7]

; cache results of lblrtm
; if abundances did not change compared to last call, just return cached results
if ((n_elements(range_old) ne 0) and (n_elements(par_old) ne 0)) && (array_equal(range,range_old) and array_equal(par_new,par_old)) then begin
    w = w_old
    t = t_old
endif else begin
    ; parameter assignment (abundances)
    wat=p[4]
    meth=p[5]
    co2=p[6]
    o2=p[7]
    
    ; run lblrtm for this set up (extend range by 10 AA to avoid edge effects)
    runlblrtm,range[0]-10.,range[1]+10.,gdasdata,obsaltitude,w,t,water=wat,methane=meth,oxygen=o2,carbondioxide=co2,silent=silent
    ; return wavelength array to air
    vactoair,w

    ; keep the result to make further calls faster
    w_old = w
    t_old = t
    range_old = range
    par_old = par_new
endelse

; convolve high res model spectrum with gaussian to get to instrument resolution
tsm=gaussconv(w,t,smooth)
; apply shift in wavelength
c=299792.458d
wm=w*(1.-v/c)
; rebin to wavelength scale of data
tsmrb=interpol(tsm,wm,wave)
; multiply by straight line to adjust to data
straight=slope*wave + b
ymod=tsmrb*straight
; return the model
return,ymod
end