!/*-----------------------------------------------------------------*/
!/*! 
!  \file f77single.f 
!  \brief Example of usage of the ARoME library using the Fortran 77 interface
!         This example computes the Rossiter-McLaughlin effect at one given position of the planet
!         It returns the value computed simulating 3 different techniques:
!           - the CCF technique
!           - the iodine cell technique
!           - the weigthed mean (like Ohta et al. 2005)
!
!  \author  G. Boue
!           EXOEarths, Centro de Astrofisica, Universidade do Porto.
!
!   Copyright (C) 2012, CAUP
!   email of the author : gwenael.boue@astro.up.pt
!
!
!   This work has been supported by the European Research Council/European
!   Community under the FP7 through Starting Grant agreement number 239953, as
!   well as from Fundacao para a Ciencia e a Tecnologia (FCT) through program
!   Ciencia 2007 funded by FCT/MCTES (Portugal) and POPH/FSE (EC), and in the
!   form of grants reference PTDC/CTE-AST/098528/2008, SFRH/BPD/71230/2010, and
!   SFRH/BPD/81084/2011.
!
!
!  History:                                                                
!*/
!/*-----------------------------------------------------------------*/

!/*----------------------------------------------------------------*/
!/* License of the file :
!
!    This program is free software: you can redistribute it and/or modify
!    it under the terms of the GNU General Public License as published by
!    the Free Software Foundation, either version 3 of the License, or
!    (at your option) any later version.
!
!    This program is distributed in the hope that it will be useful,
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!*/
!/*----------------------------------------------------------------*/



       program main
           implicit none
           include 'f90arome.h'
           double precision u1, u2, beta0, Vsini, sigma0, zeta, Rp
           double precision x0, y0, z0, f, vp, betapR, betapT
           double precision own_f, own_betapR
           double precision v_CCF, v_iodine, v_mean
           integer      Kmax
           integer      nerror
           integer*8 :: arome
          
           !/* planet coordinates in stellar radius */
           x0 = 0.2d0
           y0 = 0.0d0
           z0 = 5.0d0  !/* z0 should be positive to have transit */
          
           !/* limb-darkening */
           u1 = 0.69d0
           u2 = 0.00d0
          
           !/* line profile */
           beta0  =  2.0d0  !/* width of the non-rotating star */
           Vsini  = 15.0d0  !/* Vsini */
           sigma0 =  8.0d0  !/* width of the best Gaussian fit */
           zeta   =  2.0d0  !/* macro-turbulence parameter */
           Rp     =  0.1d0  !/* radius of the planet */
           Kmax   =    4    !/* order of expansion for the Iodine cell technique */
          
           !/* you can choose to never exit on error */
           call f90arome_set_continue_on_error()
          
           !/* allocation of the ARoME structure */
           nerror = f90arome_alloc_quad(arome, u1, u2)
          
           !/* or with a nonlinear limb-darkening law */
           !/* nerror = f90arome_alloc_nl(arome, c1, c2, c3, c4) */
          
           !/* set the lineprofile */
           nerror = f90arome_set_lineprofile(beta0, Vsini, 
     .              sigma0, zeta, Kmax, arome)
          
           !/* print error message and exit if nerror .neq. 0 */
           if (nerror .ne. 0) stop
          
           !/* you can choose to always exit on error */
           call f90arome_set_exit_on_error()
          
           !/* set the planet parameters */
           nerror = f90arome_set_planet(Rp, arome)
          
           !/* initialize the techniques you are going to use in the following */
           !/* here we initialize both the CCF and the iodine cell techniques */
           !/* should be done once the planet and line profile parameters are set */
           nerror = f90arome_init_CCF(arome)
           nerror = f90arome_init_iodine(arome)
          
          
           !/* for all planet positions, do the following -----------------------*/
           !/* compute the flux, vp and betap at a given position (x0, y0, z0) of the planet */
           nerror = f90arome_calc_fvpbetap(x0,y0,z0, arome)
          
           !/* you can have access to the computed flux, subplanet vp, betapR, and betapT */
           f = f90arome_get_flux(arome)
           vp = f90arome_get_vp(arome)
           betapR = f90arome_get_betapR(arome)
           betapT = f90arome_get_betapT(arome)
          
           print *, "------------------------------"
           print *, "values obtained by ARoME ....."
           print *, "------------------------------"
           print *, "f = ", f
           print *, "vp = ", vp
           print *, "betapR = ", betapR
           print *, "betapT = ", betapT
           print *, "------------------------------"
           print *, ""

           !/* you can change any of the flux, vp, betap if you have your own formulae */
           !/* let us modify the value of the flux and betapR */
           own_f      = 0.01d0 
           own_betapR = 15.0d0  !/* km/s */

           print *, "------------------------------"
           print *, "reset the values of the flux and radial disp..."
           print *, "------------------------------"
           print *, "own_f = ", own_f
           print *, "own_betapR = ", own_betapR
           print *, "------------------------------"
           print *, ""

           nerror = f90arome_set_flux(own_f, arome)
           nerror = f90arome_set_betapR(own_betapR, arome);
          
           !/* then you can compute the RM signal for the technique(s) already initialized */
           !/* you can use the error-handling function with suffix "_e" */
           nerror = f90arome_get_RM_CCF_e(arome, v_CCF)
          
           !/* or simply (without the suffix "_e") */
           v_iodine = f90arome_get_RM_iodine(arome)
           v_mean = f90arome_get_RM_mean(arome)
          
           print *, "------------------------------"
           print *, "RM signals..."
           print *, "------------------------------"
           print *, "v_CCF = ", v_CCF
           print *, "v_iodine = ", v_iodine
           print *, "v_mean = ", v_mean
           print *, "------------------------------"
          
           !/* end of the loop over the positions of the planet ---------------*/
          
           !/* release memory */
           call f90arome_free(arome)

       end program main
