/*-----------------------------------------------------------------*/
/*! 
  \file corbit.c 
  \brief Example of usage of the ARoME library using the C interface
         This example computes planet coordinates from elliptic elements

  \author  G. Boue
           EXOEarths, Centro de Astrofisica, Universidade do Porto.

   Copyright (C) 2012, CAUP
   email of the author : gwenael.boue@astro.up.pt
   
   This work has been supported by the European Research Council/European
   Community under the FP7 through Starting Grant agreement number 239953, as
   well as from Fundacao para a Ciencia e a Tecnologia (FCT) through program
   Ciencia 2007 funded by FCT/MCTES (Portugal) and POPH/FSE (EC), and in the
   form of grants reference PTDC/CTE-AST/098528/2008, SFRH/BPD/71230/2010, and
   SFRH/BPD/81084/2011.

  History:                                                                

*/
/*-----------------------------------------------------------------*/   

/*----------------------------------------------------------------*/
/* License of the file :

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*----------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "arome.h"


/*************************/
/* main routine          */
/* just a test           */
/*************************/
int main(int argc, char **argv)
{
   const int n = 11;
   double per    = 2.E0;     /* orbital period (day) */
   double sma    = 4.E0;      /* semi-major axis in Rs */
   double ecc    = 0.02E0;    /* eccentricity */
   double inc    = 82.8E0;    /* inclination (deg) */
   double om     = 32.E0;     /* arg. of periastron (deg) */
   double lambda = 21.E0;     /* projected spin-orbit angle (deg) */
   double ttransit;           /* transit time */
   double x, y, z;
   double tab_t[n];
   double tab_x[n], tab_y[n], tab_z[n];
   t_orbit *orbit1;
   int status = 0;
   
   /* never exit on error */
   arome_set_continue_on_error();
   
   /* creation of a new orbit */
   orbit1 = arome_new_orbit();
   
   /* set the elliptic parameters */
   status += arome_set_orbit_eo(per, sma, ecc, om, inc, lambda, orbit1);
   
   /* compute the time of transit */
   ttransit = arome_get_orbit_transit_time(orbit1);
   
   /* compute one planet position at the middle of the transit */
   status += arome_get_orbit_xyz(orbit1, ttransit, &x, &y, &z);
   
   /* array of time (+-2h around transit) */
   for(int k=0; k<n; k++)
     tab_t[k] = (k-n/2)/((n-1)*6.E0)+ttransit;
   
   /* compute multiple planet positions for the same orbit */
   status += arome_mget_orbit_xyz(orbit1, tab_t, n, tab_x, tab_y, tab_z);
   
   
   /* release memory */
   arome_free_orbit(orbit1);
   
   printf("------------------------------\n");
   printf("Coordinates of the planet ....\n");
   printf("------------------------------\n");
   printf("time = %+.15E\n", ttransit);
   printf("x    = %+.15E\n", x);
   printf("y    = %+.15E\n", y);
   printf("z    = %+.15E\n", z);
   printf("------------------------------\n\n");
   
   printf("------------------------------\n");
   printf(" %22s %22s %22s %22s\n", "time", "x coordinate", "y coordinate", "z coordinate");
   for(int k=0; k<n; k++)
     printf(" %+.15E %+.15E %+.15E %+.15E\n", tab_t[k], tab_x[k], tab_y[k], tab_z[k]);
   
   
   return 0;
}
