;+
; NAME:
;         p3d_tracing_findspec
;
;         $Id: p3d_tracing_findspec.pro 119 2010-03-22 09:45:56Z christersandin $
;
; PURPOSE:
;         This routine scans input data in the cross-dispersion direction for
;         the presence and the location of spectrum lines, in the form of local
;         maxima. This is done in a four step algorithm with the following
;         steps:
;
;         i) A stripe of width 2*WIDTH+1, that is centered on pixel COLUMN in
;            the dispersion direction, is searched for local maxima. Rows of
;            pixels, where at least a fraction CUT of the total number of
;            elements in the row are found to be a local maximum, are indicated
;            as locations of spectra. In order to account for slightly
;            tilted spectra the consecutive row of pixels is also taken into
;            account (resulting in fractional positions). The result
;            is stored in the array MASKPOS_0.
;
;         ii) The actual spectrum positions on the CCD should be separated by a
;            roughly constant value - DIST, allowing for possible deviations
;            from a constant deviation using DMIN and DMAX. The output array of
;            step i), MASKPOS_0, is traversed in order to find the largest
;            group of spectra which are separated by about DIST pixels; local
;            maxima due to cosmic rays, for instance, would unlikely appear in
;            regularly separated 'hits'. The result is stored in MASKPOS.
;
;         iii) The list of spectra returned in the result array of step ii)
;            could still contain elements which are not expected for the used
;            instrument. In this third step the calculated spectrum mask of
;            MASKPOS (REGMASK, REMASKPOS) is compared to a pre-defined spectrum
;            mask, REFMASK. REFMASK is moved over REGMASK in order to find the
;            position with the largest number of hits. For positions of the
;            matched elements are stored in the results array SPECPOS.
;
;            Note! SPECPOS only gives the position for the specified pixel in
;                  the cross-dispersion direction.
;
;            The format of the integer-array REFMASK is as follows: It should
;            contain enough DIST-separated elements to cover all spectra along
;            the cross-dispersion axis. The location of a real spectrum is
;            marked with a 1 in this array. Elements should correspondingly be
;            set to 0 for positions where there is a gap in the sequence of
;            spectra.
;
;         iv) In a fourth step spectrum positions for elements of REFMASK,
;            which could not be identified in the matching of step iii), are
;            interpolated (or extrapolated) from the positions identified in
;            the previous steps.
;
;         v) More accurate positions are calculated by weighting the positions
;            array with the cross-dispersion profile of the data.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: tracing of spectra on the CCD
;
; CALLING SEQUENCE:
;         p3d_tracing_findspec,array,refmask,dist,dmin,dmax,cut,specpos, $
;             width=,column=,spec0=,var=,/display,fwhm=,centervar=,niterat=, $
;             yrange=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         array           - A two-dimensional array of floating point type.
;         refmask         - A one-dimensional array of integer type.
;         dist            - A scalar decimal value which defines the expected
;                           separation of spectra in the cross-dispersion
;                           direction of ARRAY.
;         dmin            - A scalar decimal value, which defines that a local
;                           maximum at a position of DIST-DMIN is OK for the
;                           identification of a spectrum.
;         dmax            - A scalar decimal value, which defines that a local
;                           maximum at a position of DIST+DMAX is OK for the
;                           identification of a spectrum.
;         cut             - A scalar decimal value; 0.0<=CUT<=1.0. CUT
;                           determines what fraction of all pixels along the
;                           extracted spectrum (which is centered on COLUMN)
;                           should have a local maximum in order for the
;                           collapsed value (for all spectrum bins) to be a
;                           real spectrum.
;
; KEYWORD PARAMETERS:
;         width [12]      - A scalar integer specifying the half-width of the
;                           spectrum region, which is searched for local maxima
;                           in the first step. The unit is pixels.
;         column          - A scalar integer specifying the pixel position, in
;                           the dispersion-direction, where the spectrum is
;                           extracted.
;         spec0           - A scalar decimal value specifying the position of
;                           the first spectrum (in the cross-dispersion
;                           direction).
;         var             - not sure.
;         display         - Makes a plot of the spectrum data, together with
;                           symbols showing positive identifications of spectra
;                           from the different steps.
;         fwhm            - This value is passed to p3d_tracing_correctpos.pro.
;         centervar       - This value is passed to p3d_tracing_correctpos.pro.
;         niterat         - This value is passed to p3d_tracing_correctpos.pro.
;         yrange          - Defines an y-direction range. If this variable is
;                           not set to a two-element array, then a min-max
;                           value is used instead.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         specpos         - A one-dimensional array which contains the
;                           fractional positions of all spectra, which were
;                           identified in steps i)-iv).
;
;                           Note! The number of elements in SPECPOS is always
;                                 the same as the number of elements in
;                                 REFMASK.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
;-
FUNCTION p3d_tracing_findspec_logarray,array
  compile_opt hidden,IDL2

  stride=12L
  narray=n_elements(array)
  nrw=narray/stride & nmd=narray mod stride
  tmp=strarr(nrw+(~nmd?0L:1L))
  for i=0L,nrw-1L do $
     tmp[i]=strjoin(strtrim(array[(i*stride):(stride*(i+1L)-1L)],2L),', ')
  if nmd ne 0L then $
     tmp[nrw]=strjoin(strtrim(array[(nrw*stride):narray-1L],2L),', ')
  tmp[0L]='  ['+tmp[0L]
  tmp[n_elements(tmp)-1L]+=']'
  if n_elements(tmp) ge 2L then $
     tmp[1L:n_elements(tmp)-1L]='   '+tmp[1L:n_elements(tmp)-1L]

  return,tmp
END ;;; procedure: p3d_tracing_findspec_logarray


PRO p3d_tracing_findspec,array,refmask,dist_,dmin_,dmax_,cut,specpos, $
        width=width,column=column,spec0=spec0,var=var,display=display, $
        fwhm=fwhm,centervar=centervar,niterat=niterat,yrange=yrange, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
        debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_tracing_findspec: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_tracing_findspec'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking input and output arguments:

  s=size(array)
  if s[0L] ne 2L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='ARRAY must be a two-dimensional array of floating point type.'
    goto,error_handler
  endif

  sb=size(refmask)
  if sb[0L] ne 1L or (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='REFMASK must be a one-dimensional array of floating point type.'
    goto,error_handler
  endif

  sb=size(dist_)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='DIST must be a scalar of floating point type.'
    goto,error_handler
  endif
  dist=double(dist_)

  sb=size(dmin_)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='DMIN must be a scalar of floating point type.'
    goto,error_handler
  endif
  dmin=double(dmin_)

  sb=size(dmax_)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='DMAX must be a scalar of floating point type.'
    goto,error_handler
  endif
  dmax=double(dmax_)

  sb=size(cut)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='CUT must be a scalar of floating point type; 0<=cut<=1.'
    goto,error_handler
  endif
  if cut lt 0d0 or cut gt 1d0 then begin
    errmsg='CUT must be a scalar of floating point type; 0<=cut<=1.'
    goto,error_handler
  endif

  if ~n_elements(width) then width=12L
  sb=size(width)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='WIDTH must be a scalar of integer type; >=1.'
    goto,error_handler
  endif
  if width lt 1L then begin
    errmsg='WIDTH must be a scalar of integer type; >=1.'
    goto,error_handler
  endif

  if ~n_elements(column) then column=s[1L]/2L
  sb=size(column)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='COLUMN must be a scalar of integer type; >=1.'
    goto,error_handler
  endif
  if width lt 1L then begin
    errmsg='COLUMN must be a scalar of integer type; >=1.'
    goto,error_handler
  endif

  borders=1L
  if ~n_elements(spec0) then begin
    borders=0L
  endif else if spec0 lt 0L then begin
    borders=0L
  endif
  
  if ~n_elements(var) then $
     borders=0L $
  else if var lt 0L then $
     borders=0L

  if borders eq 1L then begin
    sb=size(spec0)
    if sb[sb[0L]+2L] ne 1L or $
      (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
      errmsg='SPEC0 must be a scalar of floating point type.'
      goto,error_handler
    endif

    sb=size(var)
    if sb[sb[0L]+2L] ne 1L or $
      (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
      errmsg='VAR must be a scalar of floating point type.'
      goto,error_handler
    endif

    lowrow=floor((spec0-var)>0L)
    upprow= ceil((spec0+var)<(s[2L]-1L)) 
  endif else begin
    lowrow=0L
    upprow=s[2L]-1L
  endelse

  ;;========================================------------------------------
  ;;========================================------------------------------
  ;; i) Searching for the pixel position of all spectra in the data; along the
  ;;    cross-dispersion direction, starting at pixel COLUMN and using a
  ;;    wavelength range of 2*WIDTH+1:

  p3d_tracing_findspec0,array,column,width,cut,maskpos, $
      topwid=topwid,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  sb=size(maskpos)
  if sb[sb[0L]+2L] eq 1L or maskpos[0L] eq -1L then begin
    errmsg='[FINDSPEC0] Could not locate a single spectrum in the input data.'
    goto,error_handler
  endif

  maskpos_0=maskpos ;; The spectrum-positions found by findpos0

  ;;========================================------------------------------
  ;; Logging the performed operations:

  nmaskpos=n_elements(maskpos)
  if logunit[1L] ge 2L then begin
    tmp=logunit[1L] ge 2L?[p3d_tracing_findspec_logarray(maskpos),'']:''

    msg=['Finding spectrum positions :: Step 1a [finding maxima in the data]',$
         '  Dispersion bin [find_column_tr]='+strtrim(column,2L), $
         '  Dispersion bin range [findwidth_tr]='+strtrim(width,2L), $
         '  Fraction of bin range required [cut_tr]='+strtrim(cut,2L), $
         '  Found '+strtrim(nmaskpos,2L)+' maxima at the following cross-' + $
         'dispersion positions [px]:',tmp]
  endif else begin
    msg=['Finding spectrum positions :: Step 1a [finding maxima in the data]',$
         '  Dispersion bin [find_column_tr]='+strtrim(column,2L), $
         '  Dispersion bin range [findwidth_tr]='+strtrim(width,2L), $
         '  Fraction of bin range required [cut_tr]='+strtrim(cut,2L), $
         '  Found '+strtrim(nmaskpos,2L)+' maxima.']
  endelse
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)


  ;;========================================------------------------------
  ;;========================================------------------------------
  ;; ii) Matching the found spectrum positions to a regular grid with a pre-
  ;;     defined separation (DIST) between spectra:

  m=n_elements(maskpos)
  chain=dblarr(m+1L,m) & chain[1L,0L]=maskpos[0L] & chain[0L,0L]=1d0
  nebor=dblarr(m+1L,m) &                            nebor[0L,0L]=1d0
  maxcounter=0L

  ;; Looping over all elements in the mask array:
  for k=1L,m-1L do begin
    counter=-1L & found=0L

    ;; Locating the current spectrum position among the already calculated
    ;; list of primary and secondary(, etc.) DIST-separated spectra:
    while chain[0L,counter+1L] ne 0d0 and ~found do begin
      counter++
      tmp=maskpos[k]-chain[chain[0L,counter],counter]
      rest=tmp mod dist

      ;; Use the currently calculated spectrum position, if the remainder <=
      ;;  maximum allowed upper value of the current spectrum:
      if rest le dmax then begin
        next=floor(tmp/dist)    & found=next
      endif

      ;; Use the next calculated spectrum position, if the remainder <=
      ;;  minimum allowed lower value of the consecutive spectrum:
      if dist-rest le dmin and ~found then begin
        next=floor(tmp/dist)+1L & found=next
      endif

    endwhile ;; condition: chain[0L,counter+1L] ne 0d0 and ~found

    if found gt 0L then begin

      ;;========================================------------------------------
      ;; Performing additional checks on array elements which were found to
      ;; contain a real spectrum:

      ;; Indicating in the zero column index of CHAIN and NEBOR that
      ;; another spectrum has been positively identified:
      chain[0L,counter]++ & nebor[0L,counter]++

      ;; Storing the pixel position of the current spectrum:
      chain[chain[0L,counter],counter]=maskpos[k]

      ;; Also storing the integer showing how many spectrum separation
      ;; distances DIST there are to the previous spectrum:
      nebor[nebor[0L,counter],counter]=found

      ;; Finding the primary list of spectra; 0 is the default primary list.
      ;; If the list with the largest number of spectra is >0 then change==1:
      change=1L & counter2=counter
      while counter2 gt 0L and change do begin
        counter2--
        if chain[0L,counter] lt chain[0L,counter2] then change=0L
      endwhile

      if change then counter2=-1L

      ;; Swapping the rows of CHAIN and NEBOR in order to have the list with
      ;; the largest number of identified spectra in row 0:
      if counter gt counter2+1L then begin
        tmp=chain[*,counter]
        chain[*,(counter2+2L):counter]=chain[*,(counter2+1L):(counter-1L)]
        chain[*,(counter2+1L)]=tmp

        tmp=nebor[*,counter]
        nebor[*,(counter2+2L):counter]=nebor[*,(counter2+1L):(counter-1L)]
        nebor[*,(counter2+1L)]=tmp
      endif

    endif else begin ;; found gt 0L

      ;;========================================------------------------------
      ;; The spectrum indicated in MASKPOS[K] was found to not correspond to
      ;; an actual spectrum, storing the result in the consecutive row of
      ;; CHAIN and NEBOR:

      maxcounter++
      chain[0L,maxcounter]=1d0 & chain[1L,maxcounter]=maskpos[k]
      nebor[0L,maxcounter]=1L  & nebor[1L,maxcounter]=found

    endelse ;; found gt 0L

  endfor ;; k=1L,m-1L

  ;; Storing the list of spectra and their (integer-)separation:
    maskpos=chain[1L:chain[0L,0L],0L]
  remaskpos=nebor[1L:nebor[0L,0L],0L]

  ;; Storing the total number of identified spectra:
  nfound=long(nebor[0L,0L])

  ;;========================================------------------------------
  ;; Logging the performed operations:

  if logunit[1L] ge 2L then begin
    tmp =p3d_tracing_findspec_logarray(maskpos)
    tmp2=p3d_tracing_findspec_logarray(long(remaskpos))
    nmaskpos=n_elements(maskpos)

    msg=['Finding spectrum positions :: Step 1b [matching the found positi' + $
         'ons with a regular grid]', $
         '     Spectrum separation [dist_tr]='+strtrim(dist,2L), $
         '  Permitted lower offset [dmin_tr]='+strtrim(dmin,2L), $
         '  Permitted upper offset [dmax_tr]='+strtrim(dmax,2L), $
         '  Matched '+strtrim(nfound,2L)+' spectra at the following cross-' + $
         'dispersion positions [px]:',tmp, $
         '  The regular grid positions are:',tmp2,'']
  endif else begin
    msg=['Finding spectrum positions :: Step 1b [matching the found positi' + $
         'ons with a regular grid]', $
         '     Spectrum separation [dist_tr]='+strtrim(dist,2L), $
         '  Permitted lower offset [dmin_tr]='+strtrim(dmin,2L), $
         '  Permitted upper offset [dmax_tr]='+strtrim(dmax,2L), $
         '  Matched '+strtrim(nfound,2L)+' spectra.']
  endelse

  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)


  ;;========================================------------------------------
  ;;========================================------------------------------
  ;; iii) Matching the identified spectra (MASKPOS) with the pre-defined
  ;;      spectrum mask (REFMASK):

  ;; Creating a mask using the pre-defined array REFMASK:
  mm=long(max(refmask))+1L
  mastermask=bytarr(mm)  & invmastermask=lonarr(mm)
  mastermask[refmask]=1b & invmastermask[refmask]=lindgen(n_elements(refmask))

  ;; Creating a mask of the re-identified spectra:
  mr=long(total(remaskpos))+1L
  regmask=bytarr(mr) & invregmask=lonarr(mr)
  regmask[0L]=1b
  counter=0L
  for k=1L,nfound-1L do begin
    counter+=remaskpos[k]
       regmask[counter]=1b
    invregmask[counter]=k
  endfor

  mfirst=where(mastermask) & mfirst=mfirst[0L]

  klow=(mr-mm)<0L
  khig=(mr-mm)>0L
  nmatch=lonarr(khig-klow+1L)
  for k=klow,khig do begin

    mlow=(-k)>0L ;; mlow=...,3,2,1,0,0,0,0,...
    rlow=  k >0L ;; rlow=...,0,0,0,0,1,2,3,...

    vecsize=(mm-mlow)<(mr-rlow) ;; large->0->large

    match=mastermask[mlow:(mlow+vecsize-1L)] and $
             regmask[rlow:(rlow+vecsize-1L)]

    firstmatch=where(match,count) & firstmatch=firstmatch[0L]

    if count ne 0L then begin
      regfirstmatch=invregmask[rlow+firstmatch]
      firstpos=maskpos[regfirstmatch]-(firstmatch+mlow-mfirst)*dist
      if firstpos ge lowrow and firstpos le upprow or ~borders then $
         nmatch[k-klow]=total(match)
    endif

  endfor ;; k=klow,khig

  ;; OPTPOS holds the index for the fit where the largest number of
  ;; spectra of MASKPOS and REFMASK match:
  dummy=max(nmatch,optpos)

  ;;========================================------------------------------
  ;; Saving the positions of the matched spectra to SPECPOS:

  k=optpos+klow
  mlow=(-k)>0L & rlow=k>0L & vecsize=(mm-mlow)<(mr-rlow)

  tmp=mastermask[mlow:(mlow+vecsize-1L)]*regmask[rlow:(rlow+vecsize-1L)]

  mastermatch=bytarr(mm)
  mastermatch[mlow:(mlow+vecsize-1L)]=tmp
  mastermatchpos=invmastermask[where(mastermatch,nmastermatchpos, $
                                     ncomplement=ncmastermatchpos, $
                                     complement=cmastermatchpos)]

     regmatch=bytarr(mr)
     regmatch[rlow:(rlow+vecsize-1L)]=tmp
     regmatchpos=   invregmask[where(   regmatch)]

  specpos=dblarr(n_elements(refmask))
  specpos[mastermatchpos]=maskpos[regmatchpos]

  ;;========================================------------------------------
  ;; Logging the performed operations:

  if logunit[1L] ge 2L then begin
    tmp =p3d_tracing_findspec_logarray(specpos)
    tmp2=[p3d_tracing_findspec_logarray(refmask),'']
    msg=['Finding spectrum positions :: Step 1c [matching the matched pos' + $
         'itions with a pre-defined spectrum mask]', $
         ' Found '+strtrim(n_elements(specpos),2L)+' spectra at the fol' + $
         'lowing cross-dispersion positions [px]:',tmp, $
         ' The used spectrum mask was:',tmp2]
  endif else begin
    msg=['Finding spectrum positions :: Step 1c [matching the matched pos' + $
         'itions with a pre-defined spectrum mask]', $
         ' Found '+strtrim(n_elements(specpos),2L)+' spectra.']
  endelse

  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)


  ;;========================================------------------------------
  ;; iv) Determining positions for those elements in the pre-defined mask
  ;;     REFMASK, which could not be found in the data:

  if nmastermatchpos lt n_elements(refmask) then begin

    mastermispos=invmastermask[cmastermatchpos]

    for k=0L,ncmastermatchpos-1L do begin

      ;; Finding the closest matched positions:
      lowmatch=where(mastermatchpos lt mastermispos[k],lcount)
      if lcount ne 0L then lowmatch=mastermatchpos[lowmatch[lcount-1L]]

      higmatch=where(mastermatchpos gt mastermispos[k],hcount)
      if hcount ne 0L then higmatch=mastermatchpos[higmatch[0L]]

      if lowmatch ge 0L and higmatch ge 0L then begin
        ;; Calculating an interpolated position:
        lowdist=refmask[mastermispos[k]]-refmask[lowmatch]
        higdist=refmask[higmatch]-refmask[mastermispos[k]]
        specpos[mastermispos[k]]= $
           ((specpos[lowmatch]+lowdist*dist)*higdist + $
            (specpos[higmatch]-higdist*dist)*lowdist)/(lowdist+higdist) 
      endif else if ~lcount then begin
        ;; Calculating an extrapolated position, using the upper value only:
        higdist=refmask[higmatch]       -refmask[mastermispos[k]]
        specpos[mastermispos[k]]=specpos[higmatch]-higdist*dist
      endif else begin
        ;; Calculating an extrapolated position, using the lower value only:
        lowdist=refmask[mastermispos[k]]-refmask[lowmatch]
        specpos[mastermispos[k]]=specpos[lowmatch]+lowdist*dist
      endelse

    endfor ;; k=0L,ncmastermatchpos-1L

    ;;========================================------------------------------
    ;; Logging the performed operations:

    if logunit[1L] ge 2L then begin
      tmp=[p3d_tracing_findspec_logarray(specpos[mastermispos]),'']

      msg=['Finding spectrum positions :: Step 1d [calculating positions o' + $
           'f non-found, but expected spectra]', $
           ' Calculated a position for '+strtrim(ncmastermatchpos,2L)+ $
           ' spectra at the following cross-dispersion positions [px]:',tmp]
    endif else begin
      msg=['Finding spectrum positions :: Step 1d [calculating positions o' + $
           'f non-found, but expected spectra]', $
           ' Calculated a position for '+strtrim(ncmastermatchpos,2L)+ $
           ' spectra.']
    endelse

    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  endif ;; n_elements(mastermatchpos) lt n_elements(refmask)


  ;;========================================------------------------------
  ;; Calculating weighted positions across the cross-dispersion profile. The
  ;; resulting array SPECPOS is the dispersion-direction median of every
  ;; positional value calculated for every spectrum bin:

  tmp=array[(column-width):(column+width),*]
  p3d_tracing_correctpos,tmp,specpos,-1L,dist,fwhm,centervar,niterat,specpos, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  ;;========================================------------------------------
  ;; Plotting the data in the cross-dispersion direction for all pixels and bin
  ;; COLUMN in the dispersion direction. The location of the initially
  ;; identified spectra are shown with a '+'-sign. The identified spectra,
  ;; which were also found to be separated by the right distance are shown with
  ;; an 'X'-sign. Finally, the location of spectra, which match the pre-defined
  ;; mask REFMASK, are shown with a '*'-sign:

  if keyword_set(display) then begin
    n=n_elements(specpos)
    specpos_=reform(transpose(specpos),1L,n)
    title='p3d: {p3d_tracing_findspec} :: finding spectrum positions of th' + $
          'e central wavelength bin, step 1'
    p3d_display_lineprofiles,array[column,*],specpos_,trcp1=maskpos_0, $
        trcp2=maskpos,title=title,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug
  endif ;; keyword_set(display)

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_tracing_findspec
