;+
; NAME:
;         p3d_misc_correlate_arrays
;
;         $Id: p3d_misc_correlate_arrays.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         Find the optimal offset of array VECA relative to array VECB. The
;         optimal offset is calculated by maximizing the correlation function
;         of the two arrays - using the NASA astro-lib functions CORREL_IMAGES
;         and CORREL_OPTIMIZE.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         doffset=p3d_misc_correlate_arrays(veca,vecb,topwid=, $
;             logunit=,verbose=,error=,/debug,/help)
;
; INPUTS:
;         veca            - A one-dimensional array of decimal type.
;         vecb            - A one-dimensional array of decimal type.
;
; KEYWORD PARAMETERS:
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         doffset         - A scalar decimal value containing the optimal
;                           offset of VECB relative to VECA.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         29.06.2009 - Created /CS
;-
FUNCTION p3d_misc_correlate_arrays,veca,vecb,topwid=topwid, $
             logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_correlate_arrays: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_correlate_arrays'
    return,-1
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return,error
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  sa=size(veca)
  if sa[0L] ne 1L or (sa[sa[0L]+1L] ge 6L and sa[sa[0L]+1L] le 11L) then begin
    errmsg='VECA [1] must be set; to a one-dimensional array of decimal type.'
    goto,error_handler
  endif

  sb=size(veca)
  if sb[0L] ne 1L or (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='VECB [2] must be set; to a one-dimensional array of decimal type.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Correlating:

  imagea=[[veca],[veca]] & sima=size(imagea)
  imageb=[[vecb],[vecb]] & simb=size(imageb)

  xoffset=0L
  yoffset=0L
  reducf=min([sima[1L:2L],simb[1L:2L]])/8L ;; Bin average to about 8x8 image.
  xshift=max([sima[1L],simb[1L]])
  yshift=max([sima[2L],simb[2L]])

  while reducf gt 1L do begin
    corrmat=correl_images(imagea,imageb,xoffset_b=xoffset,yoffset_b=yoffset, $
        xshift=xshift,yshift=yshift,reduction=reducf,monitor=verbose ge 3L)
    corrmat_analyze,corrmat,xoffset,yoffset,reduction=reducf, $
        xoff_init=xoffset,yoff_init=yoffset,print=verbose ge 3L

    xshift=2L*reducf & yshift=2L*reducf
    reducf/=2L
  endwhile ;; reducf gt 1L

  xshift/=2L & yshift/=2L
  corrmat=correl_images(imagea,imageb,xoffset_b=xoffset,yoffset_b=yoffset, $
      xshift=xshift,yshift=yshift,monitor=verbose ge 3L)
  corrmat_analyze,corrmat,doffset,oyoffset, $
      xoff_init=xoffset,yoff_init=yoffset,print=verbose ge 3L

  return,float(doffset)

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return,error
END ;;; function: p3d_misc_correlate_arrays
