;+
; NAME:
;         p3d_cmbias
;
;         $Id: p3d_cmbias.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         A more detailed description is pending.
;
;         This program is called from the p3d GUI, but can also be used as a
;         separate program.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: main routine
;
; CALLING SEQUENCE:
;         p3d_cmbias,filename,parfile,out,icstr=,ostr=,userparfile=, $
;             ofilename=,opath=,detector=,sfx=,/compress,logfile=,loglevel=, $
;             stawid=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         filename        - An array of strings that specifies the names of the
;                           raw data files, which are used when creating the
;                           master bias.
;         parfile         - A scalar string that specifies the name of a file
;                           with instrument-specific setup parameters.
;
; KEYWORD PARAMETERS:
;         icstr ['_imcmb'] - A scalar string with the image-comb. specific
;                           string that is used to create the output filename.
;         ostr ['_mbias'] - A scalar string with the master-bias specific
;                           string that is used to create the output filename.
;         userparfile     - A scalar string with the name of a file with user-
;                           defined parameters. The following parameters are
;                           read here from this file:
;                             'methodimcombine': see p3d_misc_imcombine.pro
;                             'detsec': see p3d_misc_detsec.pro
;         ofilename       - This keyword returns the full name of the created
;                           master bias file.
;         opath ['.']     - A scalar string that specifies the path, where the
;                           output data is saved.
;         detector [0]    - A scalar integer that specifies the currently
;                           selected detector; DETECTOR is a zero-based value.
;         sfx [.fits]     - A scalar string specifying the file ending (without
;                           a trailing compression suffix, such as .gz or .Z).
;         compress [0]    - If this keyword is set then the output data file is
;                           compressed (using gzip).
;         logfile         - If this routine is launched as a separate program
;                           then this keyword specifies the name of the log
;                           file that will be created.
;         loglevel [1]    - If this routine is launched as a separate program,
;                           and LOGFILE is used, then this keyword specifies
;                           the logging level; [1],2,3 (the higher the value
;                           the more output).
;         stawid          - If set, then various messages are written to the
;                           p3d GUI status line (this must be the widget id of
;                           that label widget).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - Upon a successful execution this variable contains
;                           the master bias (data) when the routine exits.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; MODIFICATION HISTORY:
;         01.09.2009 - Routine introduced (moved conts. from p3d_gui). /CS
;
;-
PRO p3d_cmbias,filename,parfile,out,icstr=icstr,ostr=ostr, $
        userparfile=userparfile,ofilename=ofilename,opath=opath,detector=d, $
        sfx=sfx,compress=compress,logfile=logfile,loglevel=loglevel, $
        stawid=stawid,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_cmbias: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_cmbias'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  logfileisopened=0L
  if ~n_elements(logunit) and n_elements(logfile) eq 1L then begin
    if size(logfile,/type) ne 7L then begin
      errmsg='LOGFILE must, if specified, be a scalar string with the name' + $
             ' of a file to write.'
      goto,error_handler
    endif

    openw,logunit,logfile,/get_lun,error=error
    if error ne 0 then begin
      errmsg='Could not open a logfile with the name "'+logfile+'".'
      goto,error_handler
    endif
    logfileisopened=1L

    if ~n_elements(loglevel) then loglevel=1L
    s=size(loglevel)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='LOGLEVEL must be a scalar integer; 1<=LOGLEVEL<=3.'
      goto,error_handler
    endif
    if loglevel lt 1L or loglevel gt 3L then begin
      errmsg='LOGLEVEL must be a scalar integer; 1<=LOGLEVEL<=3.'
      goto,error_handler
    endif

    logunit=[logunit,loglevel]
  endif ;; ~n_elements(logunit) and n_elements(logfile) eq 1L

  s=size(filename) & n=s[s[0L]+2L]
  if ~n or s[s[0L]+1L] ne 7L then begin
    errmsg='FILENAME [1] must be specified, and be of string type.'
    goto,error_handler
  endif
  for i=0L,n-1L do begin
    if ~file_test(filename[i],/read,/regular) then begin
      errmsg='Cannot read or find the file "'+filename[i]+'" ['+ $
          strtrim(i+1L,2L)+'/'+strtrim(n,2L)+']'
      goto,error_handler
    endif
  endfor

  if size(parfile,/type) ne 7L then begin
    errmsg='PARFILE [2] must be specified; as the scalar filename of an in' + $
           'strument-specific parameter file.'
    goto,error_handler
  endif
  if ~file_test(parfile,/read,/regular) then begin
    errmsg='The file PARFILE "'+parfile+'" does not exist.'
    goto,error_handler
  endif

  if ~n_elements(ostr) then ostr='_mbias'
  if size(ostr,/type) ne 7L or n_elements(ostr) ne 1L then begin
    errmsg='OSTR must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  if n_elements(userparfile) ne 0L then begin
    if size(userparfile,/type) ne 7L then begin
      errmsg='USERPARFILE must, if specified, be of string type.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/read,/regular) then begin
        errmsg='The file USERPARFILE "'+userparfile+'" does not exist.'
        goto,error_handler
      endif
    endif ;; userparfile ne ''
  endif ;; n_elements(userparfile) ne 0L

  if n_elements(opath) ne 0L then begin
    if size(opath,/type) ne 7L then begin
      errmsg='OPATH must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(opath,/directory,/read) then begin
      errmsg='The directory OPATH "'+opath+'" does not exist.'
      goto,error_handler
    endif
  endif else begin
    cd,current=opath
  endelse
  idx=strpos(opath,path_sep(),/reverse_search)
  if idx ne strlen(opath)-1L then opath+=path_sep()

  if ~n_elements(d) then d=0L
  s=size(d)
  if s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L then begin
    errmsg='DETECTOR must be a scalar integer; DETECTOR>=0.'
    goto,error_handler
  endif
  if d lt 0L then begin
    errmsg='DETECTOR must be a scalar integer; DETECTOR>=0.'
    goto,error_handler
  endif

  if ~n_elements(sfx) then sfx='.fits'
  if size(sfx,/type) ne 7L or n_elements(sfx) ne 1L then begin
    errmsg='SFX must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  compress=keyword_set(compress)

  ;;========================================------------------------------
  ;; Routine information:

  msg=['','==============================--------------------', $
          '========== Creating a master bias', $
          '==========   '+systime(), $
          '==============================--------------------']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 2)

  ;;========================================------------------------------
  ;; Combining the data files, at first the data is combined accounting for
  ;; if the number of blocks>1, and if there are more than 3 (groups of)
  ;; files:

  title='master bias '
  p3d_misc_imcombine_wrapper,filename,ofilename,ofsfx,masterbias, $
      ostr=icstr,mostr=ostr,detsec=detsec,/mbias,parfile=parfile, $
      userparfile=userparfile,opath=opath,sfx=sfx,dsh=dsh,compress=compress, $
      title=title,stawid=stawid,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  ;;========================================------------------------------
  ;; Preparing the master bias:

  n=fxpar(headfits(ofilename+ofsfx),'NCOMBINE',comment=comment)
  xnum=1L
  if n_elements(comment) eq 1L then begin
    tmp=strpos(comment,'out of ')
    if tmp ne -1L then begin
      tmp=strmid(comment,tmp+7L)
      on_ioerror,conv_error
      xnum=long(tmp)
      conv_error:
      on_ioerror,NULL
    endif
  endif ;; n_elements(comment) eq 1L

  onlyone=0L
  if n eq 1L and xnum eq 1L then begin
    msg=['Note! Making a smoothed masterbias image from only one fr' + $
         'ame is not','      the recommended procedure. If at all p' + $
         'ossible use 3 images, or more, instead.', $
         '', $
         'Do you want to proceed and smooth the selected bias image?']

    ret=dialog_message(msg,/question,/default_no,dialog_parent=topwid, $
            title='p3d: create new master bias.',/center)
    if ret eq 'Yes' then begin
      onlyone=1L
      ifilename=ofilename+ofsfx

      ;; Creating a smoothed bias (a less than good method):
      p3d_masterbias_smooth_file,ifilename,parfile,masterbias,ostr=ostr, $
          quadrant=quadrant,block=block,opath=opath,ofilename=ofilename, $
          sfx=sfx,compress=compress,stawid=stawid,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then goto,logfile_close
    endif ;; ret eq 'Yes'
  endif ;; n eq 1L and xnum eq 1L

  if ~onlyone then begin
    ofilename+=ofsfx

    ;; Writing information to the status line:
    if usestawid then begin
      tmpstr='[Master bias] Wrote the master bias file "'+ $
             file_basename(ofilename)+'".'
      widget_control,stawid,set_value=tmpstr
    endif
  endif ;; ~onlyone

  ;;==============================--------------------
  ;; Calculating the median, mean and the standard deviation of the data,
  ;; first trimming the data of prescan and overscan regions (for logging):

  ;; At first for the un-trimmed data (if LOGLEVEL>=2):
  msg='Printing image statistics for the full data:'
  msg=[msg,'The master bias median value is:       '+ $
       string(format='(e10.3)',median(masterbias))+' [ADU]']
  msg=[msg,'The master bias mean value is:         '+ $
       string(format='(e10.3)',mean(masterbias))+' [ADU]']
  msg=[msg,'The master bias standard deviation is: '+ $
       string(format='(e10.3)',stddev(masterbias))+' [ADU]']
  error=p3d_misc_logger(msg,logunit,rname=rname,loglevel=2L, $
                        verbose=verbose ge 2)
  if error ne 0 then goto,logfile_close

  msg='Calculating image statistics for the -trimmed- data:'
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  if error ne 0 then goto,logfile_close

  xmint=min(min(detsec[0L:1L,*],dimension=1L),minjx,max=xmaxt)
  xmaxt=max(max(detsec[0L:1L,*],dimension=1L))
  ymint=min(min(detsec[2L:3L,*],dimension=1L),minjy,max=ymaxt)
  ymaxt=max(max(detsec[2L:3L,*],dimension=1L))

  nblocks=1L
  if size(detsec,/n_dimensions) eq 2L then $
     nblocks=(size(detsec,/dimensions))[1L]

  if nblocks gt 1L then begin
    msg='For the individual blocks the masterbias image statistics are:'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
    for j=0L,nblocks-1L do begin
      xmin=~(min(detsec[0L:1L,j])-xmint)?xmint:(max(detsec[0L:1L,minjx])+1L)
      xmax=xmin+max(detsec[0L:1L,j])-min(detsec[0L:1L,j])

      ymin=~(min(detsec[2L:3L,j])-ymint)?ymint:(max(detsec[2L:3L,minjy])+1L)
      ymax=ymin+max(detsec[2L:3L,j])-min(detsec[2L:3L,j])

      masterbias_=masterbias[xmin:xmax,ymin:ymax]
      tmp=string(format='(i4,":",i4,",",i4,":",i4)', $
                 xmin+1L,xmax+1L,ymin+1L,ymax+1L)
      msg='  block '+strtrim(j+1L,2L)+'/'+strtrim(nblocks,2L)+ $
          ' ['+tmp+'] (pre-scan preserved):'
      msg=[msg,'    The median value is:       '+ $
           string(format='(e10.3)',median(masterbias_))+' [ADU]']
      msg=[msg,'    The mean value is:         '+ $
           string(format='(e10.3)',mean(masterbias_))+' [ADU]']
      msg=[msg,'    The standard deviation is: '+ $
           string(format='(e10.3)',stddev(masterbias_))+' [ADU]']
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
      if error ne 0 then goto,logfile_close
    endfor ;; j=0L,nblocks-1L
  endif    ;; nblocks gt 1L

  masterbias_=masterbias[xmint:xmaxt-xmint,ymint:ymaxt-ymint]
  msg='For the full trimmed masterbias the image statistics are:'
  msg=[msg,'The master bias median value is:       '+ $
       string(format='(e10.3)',median(masterbias_))+' [ADU]']
  msg=[msg,'The master bias mean value is:         '+ $
       string(format='(e10.3)',mean(masterbias_))+' [ADU]']
  msg=[msg,'The master bias standard deviation is: '+ $
       string(format='(e10.3)',stddev(masterbias_))+' [ADU]']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  if error ne 0 then goto,logfile_close

  out=temporary(masterbias)

  ;; Closing the logfile, if it was opened here:
  if logfileisopened then free_lun,logunit[0L]

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
logfile_close:
  if logfileisopened then free_lun,logunit[0L]
  return
END ;; procedure: p3d_cmbias
