;+
; NAME:
;         p3d_misc_pathify
;
;         $Id: p3d_misc_pathify.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine takes a string as input and checks if it contains either
;         !p3d_path or !p3d_data_path (/DPATH). If it does then that part of
;         the string is replaced with either of those two variable names.
;
;         Alternatively, if the string does not contain !p3d_path or
;         !p3d_data_path then teh directory above is checked.
;
;         If !p3d_path (DPATH=0), or !p3d_data_path (DPATH=1), is not set then
;         the input string is returned unchanged.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         ret=p3d_misc_pathify(str,/dpath)
;
; INPUTS:
;         str             - The input string. Should contain a filename with
;                           a path.
;
; KEYWORD PARAMETERS:
;         dpath           - If this keyword is set then the string is checked
;                           for !p3d_data_path, otherwise !p3d_path is used.
;
; OUTPUTS:
;         return value    - This is the input string where the part correspon-
;                           ding to !p3d_path or!p3d_data_path has been
;                           replaced with '!p3d_path' or '!p3d_data_path'.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required (~).
;
; EXAMPLE:
;         str='/home/kif/data/pmas/run9010/run9010_30100902/'
;         ret=p3d_misc_pathify(str,/dpath)
;         print,ret
;         > !p3d_data_path/run9010_30100902/
;
; MODIFICATION HISTORY:
;         01.06.2009 - Introduced /CS
;-
FUNCTION p3d_misc_pathify,str,dpath=dpath
  compile_opt hidden,IDL2

  on_error,3
  error=0 & rname='p3d_misc_pathify: '

  s=size(str)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    message,'STR [1] must be a scalar string.'
  endif

  path=keyword_set(dpath)?'!p3d_data_path':'!p3d_path'
  defsysv,path,exists=exists
  if ~exists then return,str

  tmpstr=keyword_set(dpath)?!p3d_data_path:!p3d_path
  tmp=strpos(str,tmpstr) & slen=strlen(tmpstr)
  if tmp eq -1L then begin
    path+=path_sep()+'..'
    tmpstr=keyword_set(dpath)?!p3d_data_path:!p3d_path
    tmpstr=file_dirname(tmpstr,/mark_directory)
    tmp=strpos(str,tmpstr) & slen=strlen(tmpstr)
    if tmp eq -1L then return,str
  endif

  ostr=(~tmp?path:strmid(str,0L,tmp)+path)+strmid(str,tmp+slen-1L)

  return,ostr
END ;; function: p3d_misc_pathify
