#!/bin/bash
##
## p3d_vm.sh
##
## $Id: p3d_vm.sh 79 2010-03-04 14:24:25Z christersandin $
##
## --
##
## p3d: a general data-reduction tool for fiber-fed IFSs
##
## Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, see <http://www.gnu.org/licenses>.
##
## Additional permission under GNU GPL version 3 section 7
##
## If you modify this Program, or any covered work, by linking or
## combining it with IDL (or a modified version of that library),
## containing parts covered by the terms of the IDL license, the
## licensors of this Program grant you additional permission to convey
## the resulting work.
##
## --
##
## The purpose of this script is to start the p3d data reduction tool
## using the IDL Virtual Machine, i.e. without using an IDL license.
##
## NOTE! It is necessary to have an IDL license in order to create the
##       binary that is used by this script.
##
## Requirements (that are all checked for in advance) are:
##  1) IDL must be present, and the environmental variable $IDL_DIR
##     set (to the idl root directory).
##  2) IDL must be version >=6.2. If a python binary is present the
##     version is checked for automatically.
##  3) The environmental variable 'p3d_path' must be set prior to
##     calling this routine. It can be set in your system startup
##     script. It may also be useful to set the data path variable
##     'p3d_data_path'. If the vm/ directory is also added to $PATH
##     it is then possible to start p3d from any directory by just
##     calling this script.
##  4) The IDL virtual machine binary ${p3d_path}/vm/p3d.sav must have
##     been generated prior to starting this script. It could be
##     created with the script ${p3d_path}/vm/p3d_tool_makevm.sh.
##
##     Example of lines to add to the shell startup scripts:
##      .bashrc)  export p3d_path=/path/to/p3d/
##                export p3d_data_path=/path/to/my/data/
##                export PATH=$PATH:${p3d_path}/vm
##      .cshrc)   setenv p3d_path /path/to/p3d/
##                setenv p3d_data_path /path/to/my/data/
##                setenv PATH $PATH:${p3d_path}/vm
##
##  This script is designed to work with the following platforms:
##    MacOSX, Sun Solaris, and Linux.

screxe=${0##*/}

# Checking if $IDL_DIR is set:
if [ -z "$IDL_DIR" ]; then
    echo "$screxe: The IDL variable \$IDL_DIR is not set, it must be."
    echo "$screxe:"
    echo "$screxe: \$IDL_DIR must be set to point at the IDL directory."
    echo "$screxe:   Example (bash): export IDL_DIR=/opt/itt/idl_6.2"
    echo "$screxe:           (tcsh): setenv IDL_DIR /opt/itt/idl_6.2"
    echo "$screxe: Cannot continue!"
    exit 1
fi

# Checking if the IDL binary exists:
if [ ! -x ${IDL_DIR}/bin/idl ]; then
    echo "$screxe: Cannot find the IDL binary '\$IDL_DIR/bin/idl'."
    echo "$screxe: Cannot continue!"
    exit 1
fi

# Checking if the IDL version is correct:
if [ \( -x `which python` \) -a \( -f ${IDL_DIR}/version.txt \) ]; then
    idlversion=`cat ${IDL_DIR}/version.txt`
    versionok=`echo $idlversion | python -c 'a=input() >= 6.2; print a'`
    if [ $versionok = False ]; then
	echo "$screxe: The used IDL version $idlversion < 6.2."
	echo "$screxe: Cannot continue!"
	exit 1
    fi
fi

# Checking if $p3d_path is set:
if [ -z "$p3d_path" ]; then
    echo "$screxe: The environmental variable \$p3d_path"
    echo "$screxe:   has not been set. Cannot continue!"
    exit 1
fi

path=`pwd`
cd ${p3d_path}

# Checking if p3d.sav is found in the vm/ directory:
if [ ! -f ./vm/p3d.sav ]; then
    echo "$screxe: Cannot find \${p3d_path}/vm/p3d.sav."
    echo "$screxe:   The required file can be created using"
    echo "$screxe:   the script \${p3d_path}/vm/p3d_tool_makevm.sh."
    echo "$screxe:"
    echo "$screxe: Cannot continue!"
    cd $path
    exit 1
fi

# Checking if p3d.sav can be read:
if [ ! -r ./vm/p3d.sav ]; then
    echo "$screxe: \${p3d_path}/vm/p3d.sav exists, but cannot be read."
    echo "$screxe:"
    echo "$screxe: Cannot continue!"
    cd $path
    exit 1
fi

# Checking for xrdb (to set some colors for p3d):
if [ -x `which xrdb` -a -r ./vm/Xdefaults ];then
    xrdb -merge ./vm/Xdefaults
fi

# Now the virtual machine can be started:
${IDL_DIR}/bin/idl -vm=${p3d_path}/vm/p3d.sav

cd $path
