;+
; NAME:
;         p3d
;
;         $Id: p3d.pro 124 2010-03-22 09:52:21Z christersandin $
;
; PURPOSE:
;         This is the main wrapper for the p3d GUI. All (implemented) IFUs can
;         be called from this program. See the keyword parameter list below for
;         a list of implemented instruments.
;
;         Although the programs for the separate data reduction steps can be
;         called separately the easiest way to use (and understand) p3d is to
;         use this wrapper.
;
; AUTHORS:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         DE-14482 Potsdam
;         GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: main routine
;
; CALLING SEQUENCE:
;         p3d,path=,/cwd,detector=,/larr2k,/ppak2k,/larr2k,/larr4k,/imacs, $
;             /virus,/flames,/vimos,/spiral,/oldvirus,/gmos,/mpfs3k, $
;             title=,parfile=,userparfile=,verbose=,/debug,/help
;
; KEYWORD PARAMETERS:
;         path [!p3d_path] - Defines the default directory to look for
;                            data inside (used by all file dialogs).
;         cwd              - If this keyword is set then all output files are
;                            saved in the current directory, instead of in the
;                            raw data directory (that is the default).
;         detector         - Selects which detector to use, if there are
;                            several, as with /VIMOS.
;         flames,imacs,[larr2k],larr4k,ppak2k,ppak4k,spiral,vimos, ...
;           virus,oldvirus,gmos,mpfs3k
;                          - Selects the instrument; LARR2K is the default.
;         userparfile      - Defines a file that is used to read some of the
;                            parameters (so that they do not have to be queried
;                            from the user).
;
;   If an instrument is not selected with the above flags it can be chosen
;   manually with the following variables:
;         title           - Sets the title of the GUI.
;         parfile         - The filename of the file containing instrument
;                           parameters.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - If this keyword is set (as /DEBUG) then no error
;                           handlers are setup. This is useful when debugging
;                           the code.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; EXAMPLE:
;         p3d,/ppak4k
;
;-
PRO p3d,path=path,cwd=cwd,detector=detector,flames=flames,vimos=vimos, $
        larr2k=larr2k,ppak2k=ppak2k,larr4k=larr4k,ppak4k=ppak4k,imacs=imacs, $
        spiral=spiral,virus=virus,oldvirus=oldvirus,gmos=gmos,mpfs3k=mpfs3k, $
        title=title,parfile=parfile,userparfile=userparfile, $
        verbose=verbose,debug=debug,help=help,_extra=_extra
  compile_opt IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d: '
  verbose=~n_elements(verbose)?0:byte(verbose)
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d'
    return
  endif

  ;;========================================------------------------------
  ;; Testing for the existence of widgets:

  if ~(!d.flags and 65536L) then begin
    errmsg=rname+'The currently active DEVICE ['+!d.name+'] does not suppo' + $
           'rt widgets, cannot continue.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; First output:

  print,'p3d  Copyright (C) 2009,2010  Astrophysikalisches Insititut Potsd' + $
        'am (AIP), Germany'
  print,'  This program comes with ABSOLUTELY NO WARRANTY. For details type '
  print,'  p3d,/help (from the command prompt) or see the file header of the'
  print,'  routine p3d.pro in the p3d directory. This is free software, and '
  print,'  you are welcome to redistribute it under certain conditions; see '
  print,'  the file COPYING in the p3d directory (or select Help/Copying in '
  print,'  the GUI) for more information.'
  print

  ;;========================================------------------------------
  ;; Initializing p3d system variables:

  p3d_misc_initialize,mpfitpresent,verbose=verbose,error=error
  if error ne 0 then return

  ;;========================================------------------------------
  ;; Setting default paths:

  psp=path_sep()
  tpath='data'+psp+'instruments'+psp+'pmas'+psp
  dlarr2k={title:'p3d: Calar Alto 3.5m/PMAS [LARR] (2kx4k CCD) ' + $
           'Reduction Panel',parfile:!p3d_path+tpath+'larr2k.prm'}

  tpath='data'+psp+'instruments'+psp+'pmas'+psp
  dlarr4k={title:'p3d: Calar Alto 3.5m/PMAS [LARR] (4kx4k CCD) ' + $
           'Reduction Panel',parfile:!p3d_path+tpath+'larr4k.prm'}

  tpath='data'+psp+'instruments'+psp+'pmas'+psp
  dppak2k={title:'p3d: Calar Alto 3.5m/PMAS [PPAK] (2kx4k CCD) ' + $
           'Reduction Panel',parfile:!p3d_path+tpath+'ppak2k.prm'}

  tpath='data'+psp+'instruments'+psp+'pmas'+psp
  dppak4k={title:'p3d: Calar Alto 3.5m/PMAS [PPAK] (4kx4k CCD) ' + $
           'Reduction Panel',parfile:!p3d_path+tpath+'ppak4k.prm'}

  tpath='data'+psp+'instruments'+psp+'gmos'+psp
  dgmos={title:'p3d: Gemini/GMOS Reduction Panel', $
         parfile:!p3d_path+tpath+'gmos.prm'}

  tpath='data'+psp+'instruments'+psp+'mpfs'+psp
  dmpfs3k={title:'p3d: BTA 6m/MPFS (3kx2k CCD) Reduction Panel', $
           parfile:!p3d_path+tpath+'mpfs3k.prm'}

  tpath='data'+psp+'instruments'+psp+'imacs'+psp
  dimacs={title:'p3d: Magellan I 6.5m/IMACS Reduction Panel', $
         parfile:!p3d_path+tpath+'imacs.prm'}

  tpath='data'+psp+'instruments'+psp+'virus'+psp
   dviru={title:'p3d: McDonald/HJS. 2.7m/VIRUS-P (Bundle2) Reduction Panel', $
         parfile:!p3d_path+tpath+'virus_bundle2.prm'}
  doviru={title:'p3d: McDonald/HJS. 2.7m/VIRUS-P (Bundle1) Reduction Panel', $
         parfile:!p3d_path+tpath+'virus_bundle1.prm'}
  
  tpath='data'+psp+'instruments'+psp+'flames'+psp
  dflam={title:'p3d: VLT/UT2/GIRAFFE/FLAMES [ARGUS] Reduction Panel', $
         parfile:!p3d_path+tpath+'argus.prm'}

  tpath='data'+psp+'instruments'+psp+'spiral'+psp
  dspira={title:'p3d: AAT 3.9m/AAOmega/SPIRAL Reduction Panel', $
          parfile:!p3d_path+tpath+'spiral.prm'}

  tpath='data'+psp+'instruments'+psp+'vimos'+psp
  dvimo={title:'p3d: VLT/UT3/VIMOS Reduction Panel', $
         parfile:!p3d_path+tpath+'vimos.prm'}

  if keyword_set(ppak2k) then begin
      title=dppak2k.title
    parfile=dppak2k.parfile
  endif else if keyword_set(ppak4k) then begin
      title=dppak4k.title
    parfile=dppak4k.parfile
  endif else if keyword_set(imacs)  then begin
      title=dimacs.title
    parfile=dimacs.parfile
  endif else if keyword_set(gmos)  then begin
      title=dgmos.title
    parfile=dgmos.parfile
  endif else if keyword_set(mpfs3k)  then begin
      title=dmpfs3k.title
    parfile=dmpfs3k.parfile
  endif else if keyword_set(virus)  then begin
      title=dviru.title
    parfile=dviru.parfile
  endif else if keyword_set(oldvirus)  then begin
      title=doviru.title
    parfile=doviru.parfile
  endif else if keyword_set(flames) then begin
      title=dflam.title
    parfile=dflam.parfile
  endif else if keyword_set(vimos)  then begin
      title=dvimo.title
    parfile=dvimo.parfile
  endif else if keyword_set(larr2k) then begin
      title=dlarr2k.title
    parfile=dlarr2k.parfile
  endif else if keyword_set(larr4k) then begin
      title=dlarr4k.title
    parfile=dlarr4k.parfile
  endif else if keyword_set(spiral) then begin
      title=dspira.title
    parfile=dspira.parfile
  endif else begin

    defv={larr2k:dlarr2k,larr4k:dlarr4k,ppak2k:dppak2k,ppak4k:dppak4k, $
          imacs:dimacs,vimo:dvimo,viru:dviru,oviru:doviru, $
          flam:dflam,spir:dspira,gmos:dgmos,mpfs3k:dmpfs3k}
    data=p3d_misc_choose(defv,error=error,verbose=verbose,debug=debug)
    if error ne 0 then return

    ;; Not continuing if the window was closed without using the OK button:
    if ~data.okexit then return

      title= data.title
    parfile= data.parfile
  endelse

  ;;========================================------------------------------
  ;; Checking if all claimed files exist (to be sure):

  if ~file_test(parfile) then begin
    errmsg='The PARFILE "'+parfile+'" does not exist.'
    goto,error_handler
  endif

  if ~n_elements(path) then begin
    cd,current=curdir
    tmp=strpos(curdir,path_sep(),/reverse_search)
    if tmp ne strlen(curdir)-1L then curdir+=path_sep()
    defsysv,'!p3d_data_path',exists=exists
    path=exists?!p3d_data_path:curdir
  endif

  ;;========================================------------------------------
  ;; Launching the tool:

  p3d_gui,parfile,cwd=cwd,path=path,title=title,detector=detector, $
      userparfile=userparfile,mpfit=mpfitpresent, $
      verbose=verbose,error=error,debug=debug,_extra=_extra

  return

error_handler:
  message,errmsg,/informational
  return
END ;;; procedure: p3d
