/*------------------------------------------------------------------------------
NAME
	cv2fits.c - convert into standard fits

DESCRIPTION
	Replace in the FITS files the keyswords
	SIMPLE  =  T by   XTENSION= 'IMAGE '
	DATE    = 'YYYY:MM:DD'  by DATE    = 'YYYY-MM-DD'

USAGE
	cv2fits file ...

OPTIONS
	-h, -help		print help for this command
	-r			replace file (default is write stdout)

REVISION
	15-May-2014

------------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>

static char cards[5][80]= {
    "", /* Empty string */
    "PCOUNT  =                    0 / Required by FITS standard", 
    "GCOUNT  =                    1 / Required by FITS standard", 
    "COMMENT PCOUNT ...",
    "COMMENT GCOUNT ..."
};


static int optr = O_RDONLY;
static int optv = 0;
static char usage[] = "\
Usage: cv2fits [-h] [-v] [-r] file [file...]";
static char simple[] = "SIMPLE  =                    T";
static char xtens[]  = "XTENSION= 'IMAGE   '           / replaced at CDS   ";
static char fdate[]  = "DATE    = '";
static char help[] = "\
  -h: print this help\n\
  -v: verbose option\n\
  -r: rewrite the input file (the default is to write to stdout)\n\
 file : one or more file to fix";

int main(int argc, char **argv) {

    char block[2881];
    char *arg, *p;
    int  bytes, fd, headno, blockno;

    /* Initialize additional FITS cards */
    for(fd=0; fd<sizeof(cards)/sizeof(cards[0]); fd++) {
        bytes = strlen(cards[fd]); memset(cards[fd]+bytes, ' ', 80-bytes);
    }

    /* No argument => help */
    if (argc<=1) {
	printf("%s\n", usage);
	exit(0);
    }
    while (--argc>0) {
	arg = *++argv;
	if (arg[0] == '-') switch(arg[1]) {
	    case 'h':
		printf("%s\n%s\b", usage, help);
		exit(0);
	    case 'r':
		optr = O_RDWR;
		continue;
	    case 'v':
		optv = 1;
		continue;
	    default:
		fprintf(stderr, "#***Invalid option: %s\n", arg);
		exit(1);
	}
	/* It's a file to process */
	fflush(stdout);
	if(optv) fprintf(stderr, "#===dealing with file: %s\n", arg);
	fd = open(arg, optr);
	if (fd<0) {
	    fprintf(stderr, "#***Can't open ");
	    perror(arg);
	    exit(2);
	}
	headno = blockno =0;
	/* Modify headers */
	while((bytes=read(fd, block, 2880))>0) {
	    if(bytes!=2880) {
		fprintf(stderr, "#***Read only %d/%d bytes in: %s\n", 
			bytes, 2880, arg);
		exit(2);
	    }
	    blockno++;
	    if(memcmp(block, simple, sizeof(simple)-1) == 0) { /* A header */
		block[2880] = 0;   /* be sure the block is zero-terminated */
		if(memcmp(block+2720, cards[0], 80) != 0) {
		    fprintf(stderr, 
			    "#*** beader#%d in block#%d too long:\n%s\n", 
			    headno, blockno, block);
		    fflush(stdout);
		    exit(2);
		}
			                        
		if(optv) fprintf(stderr, "#...Fix Header#%d at block#%d\n",
			headno, blockno);
		/* Replace SIMPLE by XTENSION, but only from second block */
		if(headno>0) memcpy(block, xtens, sizeof(xtens)-1);

		/* Move data to add the PCOUNT/GCOUNT */
		memmove(block+480, block+320, 2880-480);
		memcpy(block+320, headno>0 ? cards[1] : cards[3], 160);

		block[2880] = 0;   /* be sure the block is zero-terminated */
		headno++;
		if((p = strstr(block, fdate))) {
		    for (p=strchr(p, '\''); isgraph(*p); p++) {
			if (*p == ':') *p = '-';
		    }
		}
		if(optr == O_RDWR) {
		    if(lseek(fd, -2880, SEEK_CUR) == (off_t)(-1)) {
			fprintf(stderr, "#***lseek on ");
			perror(arg);
			exit(2);
		    }
		    if((bytes=write(fd, block, 2880))!=2880) {
			fprintf(stderr, "#***written %d/%d on ", bytes, 2880);
			perror(arg);
			exit(2);
		    }
		    continue;
		}
	    }
	    if (optr == O_RDWR) continue;
	    bytes = write(1, block, 2880);
	    if(bytes!=2880) {
		fprintf(stderr, "#***written %d/%d bytes ", bytes, 2880);
		perror("");
		exit(2);
	    }
	}
        if(optv) fprintf(stderr, "#===end %s %d blocks, %d headers\n",
		arg, blockno, headno);
    }
    exit(0);
}
