;+
; NAME:
;         p3d_misc_smooth_1d
;
;         $Id: p3d_misc_smooth_1d.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine provides an array smoothing algorithm. A box of the
;         width "2*XBOX+1" is moved across the array and the center of the box
;         is replaced by the average or median of the box.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_smooth_1d,image,xbox,out, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         image           - A one-dimensional array of data to smooth.
;         xbox            - The half-width of the smoothing box for the filter.
;
; KEYWORD PARAMETERS:
;         median          - Uses a median instead of the default average.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - The smoothed two-dimensional array.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         07.10.2008 - Converted from the original routines med_smo_1d and
;                      mean_smo_1d of Thomas Becker. /CS
;-
PRO p3d_misc_smooth_1d,image,xbox,out,median=median,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  error=0 & rname='p3d_misc_smooth_1d: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_smooth_1d'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(image)
  if s[0L] ne 1L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='IMAGE must be set; to a one-dimensional array of floating poin' + $
           't type.'
    goto,error_handler
  endif

  sb=size(xbox)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='XBOX must be set to a scalar integer; >=0.'
    goto,error_handler
  endif

  if ~xbox then begin
    out=image
    return
  endif

  ;;========================================------------------------------
  ;; Smoothing the data:

  if keyword_set(median) then begin

    ;; Using the median:
    out=float(median(image,2L*xbox+1L,/double))
    xbox=long(xbox)

    for k=0L,xbox-1L do out[k]=median(image[0L:k+xbox],/double)
    for k=xbox,s[1L]-xbox-1L do $
       out[k]=median(image[(k-xbox):(k+xbox)],/double)
    for k=s[1L]-xbox,s[1L]-1L do $
       out[k]=median(image[(k-xbox):(s[1L]-1L)],/double)

  endif else begin

    ;; Using the average:
    out=smooth(image,2L*xbox+1L)
    for k=0L,xbox-1L do out[k]=mean(image[0L:k+xbox],/double)
    for L=s[1L]-xbox,s[1L]-1L do $
       out[L]=mean(image[(L-xbox):(s[1L]-1L)],/double)

  endelse ;; keyword_set(median)

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_smooth_1d
