;+
; NAME:
;         p3d_extract
;
;         $Id: p3d_extract.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;	  This is the p3d spetral extraction routine. The algorithm works by
;	  integrating along the cross-dispersion direction. Optionally a shift
;	  of the image to the tracemask can be calculated and a flat field
;	  correction can be made.
;
;         This routine is actually a driver for the spectrum extraction
;         routines p3d_extract_tophat and p3d_extract_optimal.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: spectrum extraction
;
; CALLING SEQUENCE:
;         p3d_extract,array,traces,out,out_bsw,dout=,bdout=,parfile=, $
;             flat=,dflat=,bias=,dbias=,crmask=,gain=,rdnoise=,userparfile=, $
;             detsec=,bsw=,trcshift=,bin=,xstr=,method=,lprofs=,proffun=, $
;             /exmonitor,subtitle=,daxis=,stawid=,topwid=,logunit=,verbose=, $
;             error=,/debug,/help
;
; INPUTS:
;         array           - A raw spectrum image, of two dimensions.
;         traces          - A tracemask, that must have the same number of
;                           elements on the dispersion axis as ARRAY.
;
; KEYWORD PARAMETERS:
;         dout            - The resulting error of the extraction image OUT.
;                           This is only calculated if DBIAS is set.
;         bdout           - The resulting error of the extr. image OUT_BSW.
;                           This is only calculated if DBIAS is set.
;         parfile         - The parameter list filename, this keyword must be
;                           present. This file is used to extract the number of
;                           spectra (SPNUM) and the spectrum extraction width
;                           (PROFWIDTH).
;         flat            - If flat fielding is requested then this variable
;                           should be set to the raw flatfield data, or the
;                           final flatmask.
;         dflat           - If present, then this is the error in FLAT. The
;                           dimensions of DFLAT must be the same as those of
;                           FLAT. If BIAS and FLAT are present, then DFLAT must
;                           also be present.
;         bias            - If bias subtraction is requested, then this
;                           variable should give the bias image data.
;         dbias           - If present, then this is the error in BIAS [ADU].
;                           This value must be a decimal value (scalar or
;                           array). DOUT is only calculated if DBIAS is set.
;                           The number of elements of DBIAS must correspond to
;                           the number of rows of DETSEC.
;         crmask          - This two-dimensional array, of integer type, is
;                           used, if set, by the routine p3d_extract_optimal.
;         gain            - The data gain factor [e-/ADU]; a decimal value
;                           (scalar or array). If DBIAS is present then GAIN
;                           must also be present.
;                           The number of elements of GAIN must correspond to
;                           the number of rows of DETSEC.
;         rdnoise         - The readout noise [ADU]; a decimal value (scalar or
;                           array). If DBIAS is present then GAIN must also be
;                           present.
;                           The number of elements of RDNOISE must correspond
;                           to the number of rows of DETSEC.
;         userparfile     - A scalar string specifying the name of an optional
;                           user parameter file, that could contain some of the
;                           keywords below. If it does, then these values are
;                           read and thereafter used in the spectrum
;                           extraction.
;                             'profwidth'         ::
;                             'eliminate_crays'   :: optimal extraction
;                             'correct_crosstalk' :: optimal extraction
;                             'pmultfac'          :: optimal extraction
;                             'cr_sigma'          :: optimal extr., elim. CR
;                             'cr_maxit'          :: optimal extr., elim. CR
;                             'ctalk_eps'         :: optimal extr., cross-talk
;                             'ctalk_maxit'       :: optimal extr., cross-talk
;                             'ctalk_reiterate'   :: optimal extr., cross-talk
;
;         detsec          - A four-element (columns) -by- number of blocks
;                           (rows) integer array that specifies the detector
;                           region to use on the CCD for each block. DBIAS,
;                           GAIN, and RDNOISE must have as many elements as
;                           DETSEC has rows. Since all arrays of this routine
;                           and its subroutines are assumed to have a as the
;                           dispersion axis, the 0:1 and 2:3 columns of DETSEC
;                           are swapped if DAXIS==2 (this swapping does not
;                           affect the input array).
;         bsw [0]         - Specifies beam switching mode.
;         trcshift        - Shift of traces with respect to the object.
;         bin [1]         - The CCD binning factor of the cross-dispersion
;                           axis; either the y-direction (DAXIS==1) or the
;                           x-direction (DAXIS==2).
;         xstr ['']       - An empty string, or a three character string that
;                           is appended to some of the tracing parameter names.
;         method ['tophat']
;                         - This scalar string specifies the spectrum
;                           extraction method that is used:
;                            'tophat'::  cf. p3d_extract_tophat
;                            'optimal':: cf. p3d_extract_optimal. In this case
;                            it is mandatory to also provide LPROFS.
;         lprofs          - A three-dimensional array with the cross-dispersion
;                           line profiles of every spectrum and wavelength bin
;                           (in the third dimension).
;                           LPROFS is used iff METHOD=='optimal'.
;         proffun         - A scalar string with the name of the function to
;                           use when (re-)calculating the profiles
;                           (using LPROFS).
;         exmonitor       - If this keyword is set then a line profiles viewer
;                           is shown after all fitting is done when using
;                           optimal extraction.
;         subtitle        - If this keyword is set to a string then the string
;                           is added to the spectrum viewer title
;                           (in p3d_extract_optimal).
;         daxis [1]       - Defines the dimension of the dispersion direction
;                           in the spectrum image; must be either 1 or 2.
;         stawid          - If set, then various messages are written to the
;                           p3d GUI status line (this must be the widget id of
;                           that label widget).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - The resulting image of the extraction.
;
; OPTIONAL OUTPUTS:
;         out_bsw         - The resulting image of the extraction - when using
;                           the beam switch mode (bsw).
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         10.10.2008 - Converted from the original routine p3d_extract of
;                      Thomas Becker. /CS
;-
PRO p3d_extract,array_,traces_,out,out_bsw,dout=dout,bdout=dout_bsw, $
        parfile=parfile,flat=flat_,dflat=dflat_,bias=bias_,dbias=dbias, $
        crmask=crmask,gain=gain,rdnoise=rdnoise,userparfile=userparfile, $
        detsec=detsec_,bsw=bsw,trcshift=trcshift,bin=bin,xstr=xstr, $
        method=method_,lprofs=lprofs_,proffun=proffun,exmonitor=exmonitor, $
        subtitle=subtitle,daxis=daxis,stawid=stawid,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_extract: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_extract'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(stawid) then stawid=0L
  usestawid=widget_info(stawid,/valid_id)

  if ~n_elements(daxis) then daxis=1L
  sd=size(daxis)
  if sd[sd[0L]+2L] ne 1L or $
    (sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  sid=daxis?2L:1L

  s=size(array_)
  if s[0L] ne 1L and s[0L] ne 2L then begin
    errmsg='ARRAY [1] must be set; to a one- or two-dimensional array.'
    goto,error_handler
  endif
  array=daxis?array_:transpose(array_)
  s_=s
  s=size(array)

  if s[0L] eq 1L then begin
    ;; Creating an image with one spectrum:
    n_col=1L & n_row=s[1L]
    image=rebin(rotate(float(array),1L),1L,n_row)
  endif else begin
    n_col=s[1L] & n_row=s[2L]
    image=array
  endelse

  if ~n_elements(bin) then bin=1L
  sbi=size(bin)
  if sbi[sbi[0L]+2L] ne 1L or $
    (sbi[sbi[0L]+1L] ge 4L and sbi[sbi[0L]+1L] le 11L) then begin
    errmsg='BIN must be set to a scalar integer.'
    goto,error_handler
  endif
  if bin lt 1L then begin
    errmsg='BIN must be >=1.'
    goto,error_handler
  endif
    
  if ~n_elements(method_) then method_='tophat'
  method=strlowcase(method_)
  sb=size(method)
  if sb[sb[0L]+2L] ne 1L or sb[sb[0L]+1L] ne 7L then begin
    errmsg='METHOD must be set to a scalar string; METHOD=''tophat''' + $
           '||''optimal''.'
    goto,error_handler
  endif
  switch method of
    'tophat':
    'optimal': break
    else: begin
      errmsg='METHOD must be either "tophat" or "optimal", not "'+method+'".'
      goto,error_handler
    end
  endswitch

  if ~n_elements(bias_) then bias_=fltarr(s_[1L],s_[2L])
  szb=size(bias_)
  if szb[1L] ne s_[1L] or szb[2L] ne s_[2L] then begin

    tmp1='{[n='+strtrim(szb[daxis],2L)
    if szb[0L] eq 2L then tmp1+=',spnum='+strtrim(szb[sid],2L)
    tmp1+=']}'

    tmp2='{[n='+strtrim(s_[daxis],2L)
    if s[0L] eq 2L then tmp2+=',spnum='+strtrim(s_[sid],2L)
    tmp2+=']}'

    errmsg='BIAS must be of the same size as ARRAY; currently '+ $
           tmp1+' vs. '+tmp2+'.'
    goto,error_handler
  endif
  bias=daxis?bias_:transpose(bias_)

  nblocks=1L
  sdb=size(detsec_)
  if sdb[sdb[0L]+2L] ne 0L then begin
    nblocks=sdb[0L] eq 1L?1L:sdb[2L]
    if sdb[1L] ne 4L or $
      (sdb[sdb[0L]+1L] ge 4L and sdb[sdb[0L]+1L] le 11L) then begin
      errmsg='DETSEC must, if set, have 4 columns and be of integer type.'
      goto,error_handler
    endif

    ;; Swapping the x&y coordinates if DAXIS=2:
    detsec=daxis?detsec_:[detsec_[2L:3L,*],detsec_[0L:1L,*]]
  endif ;; sdb[sdb[0L]+2L] ne 0L

  ecalc=0L
  szb=size(dbias)
  if szb[szb[0L]+2L] ne 0L then begin
    if szb[szb[0L]+2L] ne nblocks or $
      (szb[szb[0L]+1L] ge 6L and szb[szb[0L]+1L] le 11L) then begin
      errmsg='DBIAS must, if set, be of decimal type and have '+ $
             strtrim(nblocks,2L)+' elements, not '+ $
             strtrim(szb[szb[0L]+2L],2L)+'.'
      goto,error_handler
    endif
    ecalc=1L
  endif

  st=size(traces_)
  if st[0L] ne s_[0L] or st[daxis] ne s_[daxis] then begin

    tmp1='{[n='+strtrim(st[daxis],2L)
    if st[0L] eq 2L then tmp1+=',spnum='+strtrim(st[sid],2L)
    tmp1+=']}'

    tmp2='{[n='+strtrim(s_[daxis],2L)
    if s[0L] eq 2L then tmp2+=',spnum='+strtrim(s_[sid],2L)
    tmp2+=']}'

    errmsg='TRACES [2] must be set; to an array with the same number of di' + $
           'mensions and spectrum bins as ARRAY; currently '+ $
           tmp1+' vs. '+tmp2+'.'
    goto,error_handler
  endif
  traces=daxis?traces_:transpose(traces_)

  if ecalc then begin
    szb=size(gain)
    if szb[szb[0L]+2L] ne 1L then begin
      errmsg='GAIN must be set if DBIAS is set, and have 1 element.'
      goto,error_handler
    endif
    if szb[szb[0L]+1L] ge 6L and szb[szb[0L]+1L] le 11L then begin
      errmsg='GAIN must, when set, be of decimal type.'
      goto,error_handler
    endif

    szb=size(rdnoise)
    if szb[szb[0L]+2L] ne nblocks then begin
      errmsg='RDNOISE must be set if DBIAS is set, and have '+ $
             strtrim(nblocks,2L)+' elements.'
      goto,error_handler
    endif
    if szb[szb[0L]+1L] ge 6L and szb[szb[0L]+1L] le 11L then begin
      errmsg='RDNOISE must, when set, be of decimal type.'
      goto,error_handler
    endif
  endif ;; ecalc

  if ~ecalc and method eq 'optimal' then begin
    errmsg='METHOD=="optimal": DBIAS, GAIN, and RDNOISE must all be set!'
    goto,error_handler
  endif

  if method eq 'optimal' then begin
    if size(lprofs_,/n_dimensions) ne 3L then begin
      errmsg='LPROFS must be a three-dimensional array.'
      goto,error_handler
    endif
    lprofs=daxis?lprofs_:transpose(lprofs_,[1L,0L,2L])
    sb=size(lprofs_)
    if ~sb[sb[0L]+2L] or $
       (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
      errmsg='METHOD=="optimal": LPROFS must be set to a three-dimensional' + $
             ' array of decimal type.'
      goto,error_handler
    endif
    if sb[1L] ne st[1L] or sb[2L] ne st[2L] then begin
      errmsg='METHOD=="optimal": The first two dimensions of LPROFS (['+ $
             strtrim(sb[1L],2L)+','+strtrim(sb[2L],2L)+']) must agree with' + $
             ' those of TRACES (['+strtrim(st[1L],2L)+','+ $
             strtrim(st[2L],2L)+']).'
      goto,error_handler
    endif

    sb=size(proffun)
    if sb[sb[0L]+2L] ne 1L or sb[sb[0L]+1L] ne 7L then begin
      errmsg='METHOD=="optimal": PROFFUN must be set to a scalar string w' + $
             'ith the name of the function to use.'
      goto,error_handler
    endif

  endif ;; method eq 'optimal'

  if ~keyword_set(trcshift) then trcshift=0L

  if st[0L] eq 1L then begin
    tracemask=rebin(rotate(traces,1L),n_col,spnum)
  endif else tracemask=traces
  tracemask+=trcshift

  if n_elements(parfile) ne 1L or size(parfile,/type) ne 7L then begin
    errmsg='PARFILE must be set to a string pointing at an existing file' + $
           ' with an instrument parameter list.'
    goto,error_handler
  endif
  if ~file_test(parfile,/regular,/read) then begin
    errmsg='PARFILE must be set to a string pointing at an existing' + $
           ' file with an instrument parameter list.'
    goto,error_handler
  endif

  useuserpar=0L
  if n_elements(userparfile) ne 0L then begin
    s=size(userparfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE, if set, must be a scalar string.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='USERPARFILE, cannot read the file "'+userparfile+'".'
        goto,error_handler
      endif

      useuserpar=1L
      readcol,userparfile,uparname,uparval,format='a,a',comment=';', $
          silent=verbose lt 3,delimiter=' '
    endif
  endif ;; n_elements(userparfile) ne 0L

  parname='' & parvalue=''
  readcol,parfile,parname,parvalue,format='a,a',silent=verbose lt 3, $
      comment=';',delimiter=' '

  p3d_misc_read_params,parname,parvalue,'spnum'+xstr,spnum,/must_exist,/nou, $
      type='integer',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  aspnum=lindgen(spnum)
  specmask=aspnum

  if ~keyword_set(bsw) then bsw=0L
  str=method eq 'optimal'?(bsw?'profwidth_bsw_ctex':'profwidth_ctex'): $
                          (bsw?'profwidth_bsw_ex'  :'profwidth_ex')
  p3d_misc_read_params,parname,parvalue,str,profwidth,/must_exist, $
      uparname=uparname,uparvalue=uparval,type='float', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  profwidth/=bin

  bsw_nrows=float(bsw)/bin

  ;; Logging the read parameters:
  msg='Extraction method is: '+method
  msg=~bsw?msg:[msg,'bsw_nrows='+strtrim(bsw,2L)+'/'+strtrim(bin,2L)+'='+ $
             strtrim(bsw_nrows,2L)+'.']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  if bsw_nrows ne 0L then begin
    tracecopy=tracemask-bsw_nrows
    trace_bsw=fltarr(n_col,2L*spnum)
    if bsw_nrows gt 0L then begin
      trace_bsw[*,2L*aspnum   ]=tracemask
      trace_bsw[*,2L*aspnum+1L]=tracecopy
    endif else begin
      trace_bsw[*,2L*aspnum   ]=tracecopy
      trace_bsw[*,2L*aspnum+1L]=tracemask
    endelse
  endif ;; bsw_nrows ne 0L


  extractflat=0L
  if n_elements(flat_) eq 1L then begin

    flat=daxis?flat_:transpose(flat_)
    sf=size(flat)
    if ecalc then begin
      if ~n_elements(dflat) then begin
        errmsg='DFLAT must be set if DBIAS is set.'
        goto,error_handler
      endif
      dflat=daxis?dflat_:transpose(dflat_)
      sfd=size(dflat)
      if sfd[0L] ne sf[0L] or sfd[sfd[0L]+1L] ne sf[sf[0L]+1L] or $
         sfd[sfd[0L]+2L] ne sf[sf[0L]+2L] then begin
        errmsg='DFLAT must be of the same type and dimensions as FLAT.'
        goto,error_handler
      endif
    endif ;; ecalc

    if sf[0L] eq 1L then begin
      ;; Creating an image with one spectrum:
      sfx=1L & sfy=sf[1L]
      flatmask=rebin(rotate(flat,1L),sfx,sfy)
      if ecalc then dflatmask=rebin(rotate(dflat,1L),sfx,sfy)
    endif else begin
      sfx=sf[1L] & sfy=sf[2L]
      flatmask=flat
      if ecalc then dflatmask=dflat
    endelse

    if sfy eq spnum then flatex=flatmask else extractflat=1L
    if ecalc then if sfy eq spnum then dflatex=dflatmask
  endif ;; n_elements(flat) eq 1L


  ;; Extracting the flat-field data:
  if extractflat then begin
    msg='Extracting spectra in the flatfield data.'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

    ;; Subtracting the bias:
    if nblocks eq 1L then begin
      flatmask-=bias
      if ecalc then dflatmask=sqrt(abs(flatmask)/gain+rdnoise^2+dbias^2)
    endif else begin

      flatmask-=bias

      ;; Calculating the error for each block separately:
      if ecalc then begin
        dflatmask=flatmask ;; as initialization
        xmint=min(detsec[0L:1L,*])
        ymint=min(detsec[2L:3L,*])
        for j=0L,nblocks-1L do begin
          xoff=min(detsec[0L:1L,j]) ne xmint?xmint:0L
          yoff=min(detsec[2L:3L,j]) ne ymint?ymint:0L

          xmin=min(detsec[0L:1L,j])-xmint-xoff
          xmax=max(detsec[0L:1L,j])-xmint-xoff
          ymin=min(detsec[2L:3L,j])-ymint-yoff
          ymax=max(detsec[2L:3L,j])-ymint-yoff

          dflatmask[xmin:xmax,ymin:ymax]= $
            sqrt(abs(flatmask[xmin:xmax,ymin:ymax])/gain+ $
                 rdnoise[j]^2+dbias[j]^2)
        endfor ;; j=0L,nblocks-1L
      endif ;; ecalc
    endelse ;; nblocks eq 1L

    case method of
      'optimal': p3d_extract_optimal,flatmask,tracemask,dflatmask,lprofs, $
          flatex,dflatex,crmask=crmask,proffun=proffun,profwidth=profwidth, $
          detsec=detsec,userparfile=userparfile,gain=gain,rdnoise=rdnoise, $
          subtitle=subtitle,stawid=stawid,topwid=topwid,logunit=logunit, $
          verbose=verbose,error=error,debug=debug
      else: p3d_extract_tophat,flatmask,tracemask,flatex,dflatex, $
          dimage=dflatmask,profwidth=profwidth,stawid=stawid,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug
    endcase ;; method
    if error ne 0 then return
  endif ;; extractflat


  ;;========================================------------------------------
  ;; Subtracting the masterbias from the object data:

  if nblocks eq 1L then begin
    image-=bias
    if ecalc then dimage=sqrt(abs(image)/gain+rdnoise^2+dbias^2)
  endif else begin

    image-=bias

    ;; Calculating the error for each block separately:
    if ecalc then begin
      dimage=image ;; as initialization
      xmint=min(detsec[0L:1L,*])
      ymint=min(detsec[2L:3L,*])
      for j=0L,nblocks-1L do begin
        xoff=min(detsec[0L:1L,j]) ne xmint?xmint:0L
        yoff=min(detsec[2L:3L,j]) ne ymint?ymint:0L

        xmin=min(detsec[0L:1L,j])-xmint-xoff
        xmax=max(detsec[0L:1L,j])-xmint-xoff
        ymin=min(detsec[2L:3L,j])-ymint-yoff
        ymax=max(detsec[2L:3L,j])-ymint-yoff

        dimage[xmin:xmax,ymin:ymax]=sqrt(abs(image[xmin:xmax,ymin:ymax])/ $
                                         gain+rdnoise[j]^2+dbias[j]^2)
      endfor ;; j=0L,nblocks-1L
    endif ;; ecalc
  endelse ;; nblocks eq 1L

  ;;========================================------------------------------
  ;; Extracting the object data using the trace mask:

  if ~bsw_nrows then begin
    msg='Extracting spectra in the object data [no beam switching].'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

    case method of
      'optimal': p3d_extract_optimal,image,tracemask,dimage,lprofs,ex,dex, $
          crmask=crmask,proffun=proffun,profwidth=profwidth,detsec=detsec, $
          userparfile=userparfile,gain=gain,rdnoise=rdnoise, $
          monitor=exmonitor,subtitle=subtitle,stawid=stawid,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug
      else: p3d_extract_tophat,image,tracemask,ex,dex,dimage=dimage, $
            profwidth=profwidth,stawid=stawid,topwid=topwid,logunit=logunit, $
            verbose=verbose,error=error,debug=debug
    endcase ;; method
    if error ne 0 then return

  endif else begin
    msg='Extracting spectra in the object data [beam switching is used].'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

    case method of
      'optimal': p3d_extract_optimal,image,trace_bsw,dimage,lprofs,ex,dex, $
          crmask=crmask,proffun=proffun,profwidth=profwidth,detsec=detsec, $
          userparfile=userparfile,gain=gain,rdnoise=rdnoise, $
          monitor=exmonitor,subtitle=subtitle,stawid=stawid,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug
      else: p3d_extract_tophat,image,trace_bsw,ex,dex,dimage=dimage, $
          profwidth=profwidth,stawid=stawid,topwid=topwid,logunit=logunit, $
          verbose=verbose,error=error,debug=debug
    endcase ;; method
    if error ne 0 then return

    ex1=ex[*,2L*aspnum   ]
    ex2=ex[*,2L*aspnum+1L]
    if ecalc then begin
      dex1=dex[*,2L*aspnum   ]
      dex2=dex[*,2L*aspnum+1L]
    endif

    if bsw_nrows gt 0L then begin
      ex=ex1 & ex_bsw=ex2
      if ecalc then begin
        dex=dex1 & dex_bsw=dex2
      endif
    endif else begin
      ex=ex2 & ex_bsw=ex
      if ecalc then begin
        dex=dex2 & dex_bsw=dex
      endif
    endelse
  endelse ;; ~bsw_nrows

  ;;========================================------------------------------
  ;; Flat-fielding the data:

  if n_elements(flat) ne 0L then begin
    if extractflat then begin

      msg=['[flat field] Normalizing the extracted data.', $
           '   i) Took the median across the dispersion axis.', $
           '  ii) Divided the median with its mean over all spectra.', $
           ' iii) Normalized the extracted data.']

      ;; Taking the median of each spectrum in the flat field image:
      tmp=float(median(flatex,dimension=1L,/double))
      ;;; NOTE!!!! Haven't made the error for this one...

      ;; Dividing the median with the mean value of the median:
      tmp_=float(mean(tmp,/double))
      if ecalc then dflatex/=tmp_
      flatex=tmp/tmp_

      ;; Normalizing the extracted image with the flat field:
      tmp=rebin(rotate(flatex,1L),n_col,n_row)
      if ecalc then begin
        tmp2=rebin(rotate(dflatex,1L),n_col,n_row)
        dex=sqrt(dex^2/tmp^2+tmp2^2*(ex/tmp^2)^2)
      endif
      ex/=tmp

      ;; Normalizing the extracted bsw image with the flat field:
      if bsw_nrows ne 0L then begin
        if ecalc then begin
          tmp2=rebin(rotate(dflatex,1L),n_col,n_row)
          dex_bsw=sqrt(dex_bsw^2/tmp^2+tmp2^2*(ex_bsw/tmp^2)^2)
        endif
        ex_bsw/=rebin(rotate(flatex,1L),n_col,n_row)
        msg=[msg,'  iv) Also normalized the beam switched data.']
      endif ;; bsw_nrows ne 0L

    endif else begin ;; extractflat

      msg=['[flat field] Normalizing the extracted data']

      if sfx ne 1L then begin
        flatex=total(flatex,1L)
        if ecalc then dflatex=sqrt(total(dflatex^2,1L))
      endif
      tmp=float(mean(flatex,/double))
      flatex/=tmp
      if ecalc then dflatex/=tmp

      msg=[msg,'   i) Normalized the flat-field data by its mean across' + $
               '      the dispersion axis.']

      ;; Normalizing every spectrum in the extracted image with the
      ;; flat field:
      tmp=rebin(rotate(flatex,1L),n_col,n_row)
      if ecalc then begin
        tmp2=rebin(rotate(dflatex,1L),n_col,n_row)
        dex=sqrt(dex^2/tmp^2+tmp2^2*(ex/tmp^2)^2)
      endif
      ex/=tmp

      ;; Normalizing the extracted bsw image with the flat field:
      if bsw_nrows ne 0L then begin
        tmp=rebin(rotate(flatex,1L),n_col,n_row)
        if ecalc then begin
          tmp2=rebin(rotate(dflatex,1L),n_col,n_row)
          dex_bsw=sqrt(dex_bsw^2/tmp^2+tmp2^2*(ex_bsw/tmp^2)^2)
        endif
        ex_bsw/=tmp
      endif ;; bsw_nrows ne 0L

      msg=[msg,'  ii) Normalized the extracted data.']
      if bsw_nrows ne 0L then $
         msg=[msg,' iii) Also normalized the beam switched data.']

    endelse ;; extractflat

  endif else begin ;; keyword_set(flat)
     msg=['Data is not flat fielded.']
  endelse ;; keyword_set(flat)

  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  ;;========================================------------------------------
  ;; Preparing the output data:

  if daxis then begin
    out=temporary(ex)
    if bsw_nrows ne 0L then out_bsw=temporary(ex_bsw)

    if ecalc then begin
      dout=temporary(dex)
      if bsw_nrows ne 0L then dout_bsw=temporary(dex_bsw)
    endif
  endif else begin ;; daxis
    out=transpose(ex)
    if bsw_nrows ne 0L then out_bsw=transpose(ex_bsw)

    if ecalc then begin
      dout=transpose(dex)
      if bsw_nrows ne 0L then dout_bsw=transpose(dex_bsw)
    endif
  endelse ;; daxis

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_extract
