;+
; NAME:
;         p3d_wavecal_calculate_mcurv
;
;         $Id: p3d_wavecal_calculate_mcurv.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         Given an input image, a spectrum bin (SPECBIN) and a spectrum number
;         (SPECNUM), this routine first locates the maximum position of an
;         emission line on the dispersion axis. Thereafter the same line is
;         searched for in all other spectra in the image; these positions are
;         compared to that of the first determined position using a cross-
;         correlation method.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: wavelength calibration
;
; CALLING SEQUENCE:
;         p3d_wavecal_calculate_mcurv,image,specbin,specnum,out,deadfibers=, $
;             linewidth=,crosswidth=,topwid=,logunit=,verbose=,error=, $
;             /debug,/help
;
; INPUTS:
;         image           - A two-dimensional emission line image.
;         specbin         - Initial spectrum bin to look for an emission line
;                           in (in the dispersion direction).
;         specnum         - Initial spectrum to look for an emission line in
;                           (in the cross-dispersion direction).
;
; KEYWORD PARAMETERS:
;         deadfibers      - A one-dimensional array of integers specifying
;                           which fiber positions are to be interpolated
;                           instead of fitted.
;         linewidth [5]   - [specbin-linewidth,specbin+linewidth] defines the
;                           interval, where the line will be searched for
;                           initialization.
;         crosswidth [3*linewidth] - The width of an interval around the
;                           emission line that is defined for cross-correlating
;                           the spectra.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - An array containing the line geometry (x-positions
;                           of the emission line) for every spectrum.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         14.10.2008 - Converted from the original routine
;                      p3d_wavecal_calc_linegeo of Thomas Becker. /CS
;-
PRO p3d_wavecal_calculate_mcurv,image,specbin,specnum_,out, $
        deadfibers=dfiber_,linewidth=linewidth,crosswidth=crosswidth, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
        debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_wavecal_calculate_mcurv: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_wavecal_calculate_mcurv'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(image)
  if s[0L] ne 2L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='IMAGE [1] must be set; to a two-dimensional array of decimal type.'
    goto,error_handler
  endif

  spc=size(specbin)
  if spc[spc[0L]+2L] ne 1L or $
    (spc[spc[0L]+1L] ge 6L and spc[spc[0L]+1L] le 11L) then begin
    errmsg='SPECBIN [2] must be set; to a scalar value of integer type.'
    goto,error_handler
  endif
  if specbin lt 0L or specbin ge s[1L] then begin
    errmsg='0<=SPECBIN (='+strtrim(specbin,2L)+')<'+strtrim(s[1L],2L)+ $
           ' does not hold.'
    goto,error_handler
  endif

  spc=size(specnum_)
  if spc[spc[0L]+2L] ne 1L or $
    (spc[spc[0L]+1L] ge 6L and spc[spc[0L]+1L] le 11L) then begin
    errmsg='SPECNUM [3] must be set; to a scalar value of integer type.'
    goto,error_handler
  endif
  if specnum_ lt 0L or specnum_ ge s[2L] then begin
    errmsg='0<=SPECNUM (='+strtrim(specnum_,2L)+')<'+strtrim(s[2L],2L)+ $
           ' does not hold.'
    goto,error_handler
  endif
  specnum=specnum_
  specnst=''

  usedeadfibers=0L
  spc=size(dfiber_)
  if spc[spc[0L]+2L] ge 1L then begin
    if spc[spc[0L]+1L] ge 4L and spc[spc[0L]+1L] le 11L then begin
      errmsg='DEADFIBERS must, if set, be an array of integer type.'
      goto,error_handler
    endif
    if min(dfiber_) lt 0L or max(dfiber_) gt s[2L] then begin
      errmsg='DEADFIBERS must, if set, be an array of integer type.'
      goto,error_handler
    endif
    dfiber=dfiber_-1L

    usedeadfibers=1L
    specnum=p3d_misc_findspec(specnum_,s[2L],dfiber_,logstr=specnst, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
  endif ;; spc[spc[0L]+2L] ge 1L

  if ~n_elements(linewidth) then linewidth=5L
  spc=size(linewidth)
  if spc[spc[0L]+2L] ne 1L or $
    (spc[spc[0L]+1L] ge 4L and spc[spc[0L]+1L] le 11L) then begin
    errmsg='LINEWIDTH must, if set, be a scalar of integer type; >0.'
    goto,error_handler
  endif
  if linewidth lt 1L then begin
    errmsg='LINEWIDTH must, if set, be a scalar of integer type; >0.'
    goto,error_handler
  endif

  if ~n_elements(crosswidth) then crosswidth=3L*linewidth
  spc=size(crosswidth)
  if spc[spc[0L]+2L] ne 1L or $
    (spc[spc[0L]+1L] ge 6L and spc[spc[0L]+1L] le 11L) then begin
    errmsg='CROSSWIDTH must, if set, be a scalar of integer type; >LINEWID' + $
           'TH ['+strtrim(linewidth,2L)+'].'
    goto,error_handler
  endif
  if crosswidth lt 1L then begin
    errmsg='CROSSWIDTH must, if set, be a scalar of integer type; >LINEWID' + $
           'TH ['+strtrim(linewidth,2L)+'].'
    goto,error_handler
  endif

  if crosswidth le linewidth then begin
     errmsg='CROSSWIDTH [='+strtrim(crosswidth,2L)+ $
            '] must be larger than LINEWIDTH ['+strtrim(linewidth,2L)+']!'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Logging the input parameters:

  msg='Calculating the line mask curvature across all spectra.'
  msg=[msg, $
       'Using the following parameters:', $
       '    specbin='+string(   specbin,format='(i9)')+', initial spectrum' + $
       ' bin to search for an emission line in.', $
       '    specnum='+string(   specnum,format='(i9)')+', initial spectrum' + $
       ' number'+specnst+'.', $
       '  linewidth='+string( linewidth,format='(i9)')+', 2*linewidth+1 is' + $
       ' the interval on the dispersion axis searched for a maximum.', $
       ' crosswidth='+string(crosswidth,format='(f9.3)')+', the extent abo' + $
       'ut the emission line that is used for defining the background.']
  if usedeadfibers then begin
    msg=[msg,'dead fibers='+strjoin(strtrim(dfiber_,2L),', ')]
  endif
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1L)

  ;;========================================------------------------------
  ;; Find the position change of the emission line, pointed at with
  ;; SPECBIN,SPECNUM, across the spectra:

  out=fltarr(s[2L])

  nc=ceil(crosswidth)
  nl=linewidth

  x=fltarr(2L*nc+1L)+1.0
  x[(nc-nl):(nc+nl)]=0.0 ;; set to zero ==> in order to sample the background
  x=where(x)

  x0=double(specbin)

  ;; Extracting a part of the spectrum:
  spec0=image[(specbin-nc):(specbin+nc),specnum]

  ;;========================================------------------------------
  ;; Fitting a linear function to the background signal, and subtracting this
  ;; function from the spectrum, leaving the emission line:

  params=poly_fit(x,spec0[x],1L,/double)
  spec0-=params[0L]+params[1L]*dindgen(2L*nc+1L)
  spec0=spec0[(nc-nl):(nc+nl)]>0d0

  if max(spec0) gt 0d0 then $
     x0+=total(spec0*dindgen(2L*nl+1L))/total(spec0)-nl

  if x0 lt 0d0 or x0 ge double(s[1L]) then begin
    errmsg='Could not find any line near the provided spectrum bin!'
    goto,error_handler
  endif

  ;; Storing the location of the initial spectrum:
  out[specnum]=x0

  center0=round(out[specnum])
  width0=long(center0<(s[1L]-1L-center0))<crosswidth

  ;;========================================------------------------------
  ;; Finding the offset of all other spectra:

  lstart=[specnum+1L,specnum-1L]
  lfinal=[s[2L]-1L,0L]
  stride=[1L,-1L]

  for j=0,1 do begin
    poffset=0L & prevfiber=0L
    for k=lstart[j],lfinal[j],stride[j] do begin
      if usedeadfibers then begin
        tmp=where(dfiber eq k,count)
        poffset=prevfiber?(j?poffset-1L:poffset+1L):0L
        if count eq 1L then begin
          prevfiber=1L
          continue ;; skip to the next spectrum number
        endif else begin
          prevfiber=0L ;; count eq 1L
        endelse ;; count eq 1L
      endif ;; usedeadfibers

      center=round(out[k-stride[j]-poffset])

      ;; The extraction width is smaller at the upper boundary:
      width=long(center<(s[1L]-1L-center))<width0

      spec0=image[(center0-width):(center0+width),specnum]
      spec =image[(center -width):(center +width),k]

      offset=p3d_misc_correlate_arrays(spec0,spec,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
      if error ne 0 then return

      tmp=center-offset
      out[k]=(tmp lt 0.0 or tmp ge s[1L])?out[k-stride[j]]:tmp
    endfor ;; k=lstart[j],lfinal[j],stride[j]
  endfor ;; j=0,1

  ;;========================================------------------------------
  ;; Interpolating the dead fiber positions:

  if usedeadfibers then begin
    xarr=lindgen(s[2L]) & mask=lonarr(s[2L])+1L
    count=0L
    for i=0L,n_elements(dfiber)-1L do begin
      tmp=where(xarr eq dfiber[i],count_)
      if count_ eq 1L then begin
        mask[tmp[0L]]=0L
        count++
      endif
    endfor ;; i=0L,n_elements(dfiber)
    if count eq s[2L] then begin
      errmsg='All spectra are masked as dead, cannot continue.'
      goto,error_handler
    endif
    if count gt 0L then xarr=xarr[where(mask)]
    if n_elements(xarr) ge 3L then begin
      par=poly_fit(xarr,out[xarr],2L)
      out[dfiber]=par[0L]+par[1L]*dfiber+par[2L]*dfiber^2
    endif else begin
      errmsg='There are fewer than 3 present spectra, cannot continue.'
      goto,error_handler
    endelse ;; count ge 3L and n_elements(xt) ne 0L
  endif ;; usedeadfibers

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_wavecal_calculate_mcurv
