;+
; NAME:
;         p3d_misc_retrieve_lprofs
;
;         $Id: p3d_misc_retrieve_lprofs.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         Routine description pending.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_retrieve_lprofs,tracefile,lprofs,proffun,method=,opath=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         tracefile       - A scalar string that contains the name of ...
;
; KEYWORD PARAMETERS:
;         method          - A scalar string specifying the spectrum extraction
;                           method.
;         opath           - A scalar string that, if it is specified,
;                           specifies an alternative path where the file is
;                           searched for.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         lprofs          - A three-dimensional array of line profile fitting
;                           parameters.
;         proffun         - Contains a scalar string with the name of the
;                           function used to calculate the line profile
;                           parameters of LPROFS.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         03.09.2009 - Moved from p3d_gui. /CS
;
;-
PRO p3d_misc_retrieve_lprofs,tracefile,lprofs,proffun,method=method, $
        opath=opath_,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_retrieve_lprofs: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_retrieve_lprofs'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if method ne 'optimal' then return

  if n_elements(opath_) ne 0L then begin
    if size(opath_,/type) ne 7L then begin
      errmsg='OPATH must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(opath_,/directory,/read) then begin
      errmsg='The directory OPATH "'+opath_+'" does not exist.'
      goto,error_handler
    endif
    opath=opath_
    idx=strpos(opath,path_sep(),/reverse_search)
    if idx ne strlen(opath)-1L then opath+=path_sep()
  endif ;; n_elements(opath) ne 0L

  ;; Retrieving information about the filenames of the line profiles
  ;; and the profile fractions. Thereafter loading the data:

  thdr=headfits(tracefile)

  lprfile=fxpar(thdr,'IMPRF')
  i=1L & more=1L
  while more do begin
    tmp=fxpar(thdr,'IMPRF'+strtrim(i++,2L))
    if tmp ne '0' then lprfile+=strtrim(tmp,2L) else more=0L
  endwhile

  if strtrim(lprfile,2L) eq '0' then begin
    errmsg=['The fits file header keyword IMPRF is not set in the', $
            'trace mask file. Maybe the trace mask was calculated', $
            'without calculating the additional profiles?', $
            'Cannot continue.']
    goto,error_handler
  endif ;; lprfile eq '0'

  if ~file_test(lprfile,/read,/regular) then begin
    if n_elements(opath) ne 0L then begin
      tlprfile=opath+file_basename(lprfile)
      if ~file_test(tlprfile,/read,/regular) then begin
        errmsg=['Optimal extraction: Cannot read the file with the line pr' + $
                'ofiles (also not in the output directory):', $
                '  '+p3d_misc_pathify(lprfile,/dpath)]
        goto,error_handler
      endif
      lprfile=tlprfile
    endif else begin
      errmsg=['Optimal extraction: Cannot read the file with the line prof' + $
              'iles:','  '+p3d_misc_pathify(lprfile,/dpath)]
      goto,error_handler
    endelse ;; n_elements(opath) ne 0L
  endif ;; ~file_test(lprfile,/read,/regular)

  ;; Loading the data:
  lprofs=readfits(lprfile,tmphdr,silent=verbose lt 3,/no_unsigned)

  ;; Testing the dispersion-axis flipping state of the object data and the
  ;; flat-field mask:
  tmp1=fxpar(  thdr,'P3DDFLIP')
  tmp2=fxpar(tmphdr,'P3DDFLIP')
  if tmp1 ne tmp2 then begin
    errmsg='The dispersion axis flipping of the trace-mask file and the' + $
           ' line-profile file differ {P3DDFLIP}.'
    goto,error_handler
  endif

  ;; Reading the functional name:
  proffun=fxpar(tmphdr,'p3dffunc')
  switch strlowcase(proffun) of
    'gaussian':
    'lorentzian':
    'gauss/lorentz':
    'doublegauss': break
    else: begin
      errmsg=['The profile function parameter PROFFUN must be one of the'+ $
              ' four options:', $
              '  "gaussian", "lorentzian", "gauss/lorentz", "doublegauss"', $
              ' PROFFUN="'+proffun+'" is not a valid option.']
      goto,error_handler
    end
  endswitch ;; strlowcase(proffun)
  proffun=strlowcase(proffun)

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_retrieve_lprofs
