! Code Aspects, version 2.0.
! Michel Fioc, 2014-4-15. 

module mod_Aspects

! Main module of Aspects.

  use mod_types, only : real_type, assoc_struct, coord_struct, prob_struct

  use mod_constants, only : pi, two_pi, n_max_sigma, n_max_ngb, &
       min_xi_ratio, epsilon_f, R_ratio, &
       epsilon_oto_prob, epsilon_ln_oto_Lh, d_f, f_max_shift

  implicit none

  private
  public :: & ! Procedures only.
       general_preliminaries, oto_preliminaries, &
       sto_analysis, ots_analysis, oto_analysis, &
       func_ln_sto_Lh, func_ln_ots_Lh, compute_ln_oto_Lh, func_ln_oto_Lh_alt, &
       compute_sto_prob, compute_ots_prob, compute_oto_prob
!......................................................................

  real(kind=real_type) :: R_ctp, R_ngb

  type ctp_list ! Linked list of counterparts.
     type(ctp_list), pointer :: ptr => null()
     integer :: ctp
     real(kind=real_type) :: xi
  end type ctp_list

  type ngb_list ! Linked list of neighbors.
     type(ngb_list), pointer :: ptr => null()
     integer :: ngb
     real(kind=real_type) :: psi
  end type ngb_list

  type point_list ! Linked list of points (used in `compute_ln_oto_Lh`).
     type(point_list), pointer :: ptr => null()
     real(kind=real_type) :: f, d
     integer :: orig
  end type point_list

contains
!======================================================================

  function arc_cos(x)

! Robust version of `acos` in case $|x| > 1$ for numerical reasons.
    implicit none
    real(kind=real_type) :: arc_cos
    real(kind=real_type), intent(in) :: x
!______________________________________________________________________

    if (x < -1) then
       arc_cos = pi
    else if (x > 1) then
       arc_cos = 0
    else
       arc_cos = acos(x)
    endif

  end function arc_cos

!======================================================================

  subroutine alpha_bounds(alpha_1, delta_1, n_2, alpha_2, R, &
       k1, k2, k3, k4)

! Restrict the domain of right ascension in which counterparts are searched.
    implicit none
    real(kind=real_type), intent(in) :: alpha_1, delta_1
    real(kind=real_type), intent(in) :: R ! Either `R_ctp` or `R_ngb`.
    integer, intent(in) :: n_2
    real(kind=real_type), dimension(:), intent(in) :: alpha_2
    integer, intent(out) :: k1, k2, k3, k4
!......................................................................
    real(kind=real_type) :: diff_alpha, expr
!______________________________________________________________________

    if (R < pi/2) then
       expr = cos(R)**2-sin(delta_1)**2
       if (expr > 0) then ! `delta_1 > -pi/2+R` and `delta_1 < pi/2-R`.
          diff_alpha = arc_cos(sqrt(expr)/cos(delta_1))
          if (alpha_1+diff_alpha > two_pi) then
             k1 = 1
             k2 = k_max(alpha_2, alpha_1+diff_alpha-two_pi)
             k3 = k_min(alpha_2, alpha_1-diff_alpha)
             k4 = n_2
          else if (alpha_1-diff_alpha < 0) then
             k1 = 1
             k2 = k_max(alpha_2, alpha_1+diff_alpha)
             k3 = k_min(alpha_2, alpha_1-diff_alpha+two_pi)
             k4 = n_2
          else
             k1 = k_min(alpha_2, alpha_1-diff_alpha)
             k2 = k_max(alpha_2, alpha_1+diff_alpha)
             k3 = 0
             k4 = -1
          endif
       else
          k1 = 1
          k2 = n_2
          k3 = 0
          k4 = -1
       endif
    else ! `R >= pi/2`: not used in practice!
       k1 = 1
       k2 = n_2
       k3 = 0
       k4 = -1
    endif

  end subroutine alpha_bounds

!======================================================================

  function k_min(x, y)

! Given an array `x(1:n)` ordered by increasing values (possibly not strictly
! increasing ones), return the index `k_min` such that
!
!                `x(k_min-1) < y <= x(k_min)`
!
! (with `x(0)` := -infinity and `x(n+1)` := +infinity).
    integer :: k_min
    real(kind=real_type), dimension(:), intent(in) :: x
    real(kind=real_type), intent(in) :: y
!......................................................................
    integer :: n, i_inf, i_sup, i_med
!______________________________________________________________________

    n = size(x)
    if (y <= x(1)) then
       k_min = 1
    else if (x(n) < y) then
       k_min = n + 1
    else ! `x(1) < y <= x(n)`.
       i_inf = 1
       i_sup = n
       do
          if (i_sup <= i_inf + 1) exit
          i_med = (i_inf + i_sup)/2
          if (y <= x(i_med)) then
             i_sup = i_med
          else
             i_inf = i_med
          endif
       enddo
       k_min = i_sup
    endif

  end function k_min

!======================================================================

  function k_max(x, y)

! Given an array `x(1:n)` ordered by increasing values (possibly not strictly
! increasing ones), return the index `k_max` such that
!
!                `x(k_max) <= y < x(k_max+1)`
!
! (with `x(0)` := -infinity and `x(n+1)` := +infinity).
    integer :: k_max
    real(kind=real_type), dimension(:), intent(in) :: x
    real(kind=real_type), intent(in) :: y
!......................................................................
    integer :: n, i_inf, i_sup, i_med
!______________________________________________________________________

    n = size(x)

    if (y < x(1)) then
       k_max = 0
    else if (x(n) <= y) then
       k_max = n
    else ! `x(1) <= y < x(n)`.
       i_inf = 1
       i_sup = n
       do
          if (i_sup <= i_inf + 1) exit
          i_med = (i_inf + i_sup)/2
          if (y < x(i_med)) then
             i_sup = i_med
          else
             i_inf = i_med
          endif
       enddo
       k_max = i_inf
    endif

  end function k_max

!======================================================================

  subroutine find_ctp(alpha_1, delta_1, alpha_2, delta_2, &
       R_ctp, k_first, k_last, semi_axis_a_1, semi_axis_b_1, beta_1, &
       semi_axis_a_2, semi_axis_b_2, beta_2, &
       min_xi, sum_xi_1, ctp_1, current_ctp, n_ctp)

! Find all possible counterparts with index in [`k_first`, `k_last`] to a 
! given source.
    implicit none
    real(kind=real_type), intent(in) :: alpha_1, delta_1, R_ctp
    real(kind=real_type), dimension(:), intent(in) :: alpha_2, delta_2
    integer, intent(in) :: k_first, k_last
    real(kind=real_type), intent(in) :: semi_axis_a_1, semi_axis_b_1
    real(kind=real_type), dimension(:), intent(in) :: semi_axis_a_2, &
         semi_axis_b_2
    real(kind=real_type), intent(in) :: beta_1
    real(kind=real_type), dimension(:), intent(in) :: beta_2
    real(kind=real_type), intent(in) :: min_xi
    real(kind=real_type), intent(inout) :: sum_xi_1
    type(ctp_list), pointer :: ctp_1
    type(ctp_list), pointer :: current_ctp
    integer, intent(inout) :: n_ctp
!......................................................................
    integer :: k
    real(kind=real_type) :: psi, xi
    real(kind=real_type), dimension(2,2) :: &
         Rot_beta_gamma_1, Diag_1, mat_Gamma_1, &
         Rot_beta_gamma_2, Diag_2, mat_Gamma_2, &
         mat_Gamma_tot
    real(kind=real_type) :: cos_gamma_1, sin_gamma_1, cos_gamma_2, &
         sin_gamma_2
!______________________________________________________________________

    do k = k_first, k_last
       if (delta_2(k) < delta_1 - R_ctp .or. delta_2(k) > delta_1 + R_ctp) &
            cycle

       psi = ang_dist(alpha_1 = alpha_1, delta_1 = delta_1, &
            alpha_2 = alpha_2(k), delta_2 = delta_2(k))
       if (psi > R_ctp) cycle

! Compute angles `gamma_...`.
       if (psi > 0) then
          cos_gamma_2 = cos(delta_1)*(sin(alpha_2(k))*cos(alpha_1) &
               - cos(alpha_2(k))*sin(alpha_1))/sin(psi)
          sin_gamma_2 = (cos(delta_1)*sin(delta_2(k)) &
               *(cos(alpha_2(k))*cos(alpha_1) &
               + sin(alpha_2(k))*sin(alpha_1)) &
               - sin(delta_1)*cos(delta_2(k)))/sin(psi)
          cos_gamma_1 = cos(delta_2(k))*(sin(alpha_2(k))*cos(alpha_1) &
               - cos(alpha_2(k))*sin(alpha_1))/sin(psi)
          sin_gamma_1 = (cos(delta_1)*sin(delta_2(k)) &
               - sin(delta_1)*cos(delta_2(k)) * &
               (cos(alpha_2(k))*cos(alpha_1) &
               + sin(alpha_2(k))*sin(alpha_1)))/sin(psi)
       else ! psi = 0.
          cos_gamma_2 = 1
          sin_gamma_2 = 0
          cos_gamma_1 = 1
          sin_gamma_1 = 0
       endif

! Compute covariance matrix `mat_Gamma_tot`.
       Rot_beta_gamma_2(1, 1) = &
            sin(beta_2(k))*cos_gamma_2 + &
            cos(beta_2(k))*sin_gamma_2
       Rot_beta_gamma_2(1, 2) = &
            cos(beta_2(k))*cos_gamma_2 - &
            sin(beta_2(k))*sin_gamma_2
       Rot_beta_gamma_2(2, 1) = -Rot_beta_gamma_2(1, 2)
       Rot_beta_gamma_2(2, 2) = Rot_beta_gamma_2(1, 1)
       Diag_2(1,1) = semi_axis_a_2(k)**2
       Diag_2(1,2) = 0.
       Diag_2(2,1) = 0.
       Diag_2(2,2) = semi_axis_b_2(k)**2
       mat_Gamma_2 = matmul(transpose(Rot_beta_gamma_2), &
            matmul(Diag_2, Rot_beta_gamma_2))

       Rot_beta_gamma_1(1, 1) = &
            sin(beta_1)*cos_gamma_1 + &
            cos(beta_1)*sin_gamma_1
       Rot_beta_gamma_1(1, 2) = &
            cos(beta_1)*cos_gamma_1 - &
            sin(beta_1)*sin_gamma_1
       Rot_beta_gamma_1(2, 1) = -Rot_beta_gamma_1(1, 2)
       Rot_beta_gamma_1(2, 2) = Rot_beta_gamma_1(1, 1)
       Diag_1(1,1) = semi_axis_a_1**2
       Diag_1(1,2) = 0.
       Diag_1(2,1) = 0.
       Diag_1(2,2) = semi_axis_b_1**2
       mat_Gamma_1 = matmul(transpose(Rot_beta_gamma_1), &
            matmul(Diag_1, Rot_beta_gamma_1))

       mat_Gamma_tot = mat_Gamma_1 + mat_Gamma_2

! Compute spatial probability density `xi`.
       xi = Gaussian_2D(cov = mat_Gamma_tot, vector = (/psi, 0._real_type/))

       if (xi < min_xi) cycle

! Add a new source to the list of counterparts.
       n_ctp = n_ctp + 1
       sum_xi_1 = sum_xi_1 + xi

! Add a new node to the linked list of counterparts (by order of increasing
! right ascension).
       if (.not.associated(ctp_1)) then ! First node.
          allocate(ctp_1)
          current_ctp => ctp_1
       else ! Next nodes.
          allocate(current_ctp%ptr)
          current_ctp => current_ctp%ptr
       endif
       nullify(current_ctp%ptr)
       current_ctp%ctp = k
       current_ctp%xi = xi
    enddo

  end subroutine find_ctp

!======================================================================

  subroutine sto_preliminaries(area_S, &
       n_1, alpha_1, delta_1, semi_axis_a_1, semi_axis_b_1, beta_1, &
       n_2, alpha_2, delta_2, semi_axis_a_2, semi_axis_b_2, beta_2, &
       R_ctp, assoc_data_1)

! Called by subroutine `general_preliminaries`; compute fields in `assoc_data_1`
! required in all cases, in particular the several-to-one case.
    implicit none
    real(kind=real_type), intent(in) :: area_S
    integer, intent(in) :: n_1, n_2
    real(kind=real_type), dimension(:), intent(in) :: alpha_1, delta_1, &
         semi_axis_a_1, semi_axis_b_1, beta_1
    real(kind=real_type), dimension(:), intent(in) :: alpha_2, delta_2, &
         semi_axis_a_2, semi_axis_b_2, beta_2
    real(kind=real_type), intent(in) :: R_ctp
    type(assoc_struct), dimension(:), pointer :: assoc_data_1
!......................................................................
    integer :: i, j, k1, k2, k3, k4
    real(kind=real_type) :: min_xi
    type(ctp_list), pointer :: ctp_1
    type(ctp_list), pointer :: current_ctp, ptr
    real(kind=real_type), dimension(n_1) :: xi0_1
!______________________________________________________________________

    if (associated(assoc_data_1)) then
       do i = 1, n_1
          if (associated(assoc_data_1(i)%ctp_main)) &
               deallocate(assoc_data_1(i)%ctp_main)
          if (associated(assoc_data_1(i)%xi_main)) &
               deallocate(assoc_data_1(i)%xi_main)
          if (associated(assoc_data_1(i)%ngb)) &
               deallocate(assoc_data_1(i)%ngb)
          if (associated(assoc_data_1(i)%ctp_any)) &
               deallocate(assoc_data_1(i)%ctp_any)
          if (associated(assoc_data_1(i)%corresp)) &
               deallocate(assoc_data_1(i)%corresp)
          if (associated(assoc_data_1(i)%xi_any)) &
               deallocate(assoc_data_1(i)%xi_any)
       enddo
       deallocate(assoc_data_1)
    endif
    allocate(assoc_data_1(n_1))

    xi0_1(:) = 1/area_S
! (Above definition is appropriate for $K$- and $K'$-sources uniformly
! (and exclusively) distributed on a common surface of area `area_S`.)

    do i = 1, n_1
       nullify(ctp_1)
       assoc_data_1(i)%n_ctp_main = 0
       assoc_data_1(i)%sum_xi_main = 0
       min_xi = min_xi_ratio*xi0_1(i)
       call alpha_bounds(alpha_1(i), delta_1(i), n_2, alpha_2, R_ctp, &
            k1, k2, k3, k4)
! Find counterparts with index in $[`k1`, `k2`]$.
       call find_ctp(alpha_1(i), delta_1(i), alpha_2, delta_2, &
            R_ctp, k1, k2, &
            semi_axis_a_1(i), semi_axis_b_1(i), beta_1(i), &
            semi_axis_a_2, semi_axis_b_2, beta_2, &
            min_xi, assoc_data_1(i)%sum_xi_main, ctp_1, current_ctp, &
            assoc_data_1(i)%n_ctp_main)
! Find counterparts with index in $[`k3`, `k4`]$.
       call find_ctp(alpha_1(i), delta_1(i), alpha_2, delta_2, &
            R_ctp, k3, k4, &
            semi_axis_a_1(i), semi_axis_b_1(i), beta_1(i), &
            semi_axis_a_2, semi_axis_b_2, beta_2, &
            min_xi, assoc_data_1(i)%sum_xi_main, ctp_1, current_ctp, &
            assoc_data_1(i)%n_ctp_main)
       allocate(assoc_data_1(i)%xi_main(0:assoc_data_1(i)%n_ctp_main))
       allocate(assoc_data_1(i)%ctp_main(0:assoc_data_1(i)%n_ctp_main))
       assoc_data_1(i)%xi_main(0) = xi0_1(i)
       assoc_data_1(i)%ctp_main(0) = 0
! Linked lists are much more efficient here than using an array of size
! `n_2` > 1000 and packing it in an array of size `assoc_data_1(i)%n_ctp_main`
! (tested with "ifort" and "gfortran").

! Deallocate linked list of counterparts.
       current_ctp => ctp_1
       do j = 1, assoc_data_1(i)%n_ctp_main
          assoc_data_1(i)%xi_main(j) = current_ctp%xi
          assoc_data_1(i)%ctp_main(j) = current_ctp%ctp
          ptr => current_ctp%ptr
          deallocate(current_ctp)
          current_ctp => ptr
       enddo
    enddo

  end subroutine sto_preliminaries

!======================================================================

  subroutine ots_preliminaries(area_S, &
       n_1, alpha_1, delta_1, semi_axis_a_1, semi_axis_b_1, beta_1, &
       n_2, alpha_2, delta_2, semi_axis_a_2, semi_axis_b_2, beta_2, &
       R_ctp, assoc_data_2)

! Called by subroutine `general_preliminaries`; compute fields in `assoc_data_2`
! required in all cases, in particular the one-to-several case.
    implicit none
    real(kind=real_type), intent(in) :: area_S
    integer, intent(in) :: n_1, n_2
    real(kind=real_type), dimension(:), intent(in) :: alpha_1, delta_1, &
         semi_axis_a_1, semi_axis_b_1, beta_1
    real(kind=real_type), dimension(:), intent(in) :: alpha_2, delta_2, &
         semi_axis_a_2, semi_axis_b_2, beta_2
    real(kind=real_type), intent(in) :: R_ctp
    type(assoc_struct), dimension(:), pointer :: assoc_data_2
!______________________________________________________________________

! o:s preliminaries = s:o preliminaries with catalogs exchanged.
    call sto_preliminaries(area_S = area_S, &
         n_1 = n_2, alpha_1 = alpha_2, delta_1 = delta_2, &
         semi_axis_a_1 = semi_axis_a_2, semi_axis_b_1 = semi_axis_b_2, &
         beta_1 = beta_2, &
         n_2 = n_1, alpha_2 = alpha_1, delta_2 = delta_1, &
         semi_axis_a_2 = semi_axis_a_1, semi_axis_b_2 = semi_axis_b_1, &
         beta_2 = beta_1, &
         R_ctp = R_ctp, &
         assoc_data_1 = assoc_data_2)

  end subroutine ots_preliminaries

!======================================================================

  subroutine general_preliminaries(area_S, &
       coord_1, coord_2, &
       assoc_data_1, assoc_data_2)

! Compute `assoc_data_1` and `assoc_data_2` (data on counterparts used
! in all cases).
    implicit none
    real(kind=real_type), intent(in) :: area_S
    type(coord_struct), dimension(:), intent(in) :: coord_1, coord_2
    type(assoc_struct), dimension(:), pointer :: assoc_data_1, assoc_data_2
!......................................................................
    integer :: n_1, n_2
!______________________________________________________________________

    n_1 = size(coord_1)
    n_2 = size(coord_2)

! Provide the search radius for counterparts `R_ctp`.
    R_ctp = min(n_max_sigma * sqrt(maxval(coord_1(:)%semi_axis_a)**2 &
         + maxval(coord_1(:)%semi_axis_a)**2), pi) ! `R_ctp` sould be << $\pi$;
!& otherwise, the usual [planar] normal distribution will not be appropriate.
! (`n_max_sigma` is defined in file "mod_constants.f90".)

    call sto_preliminaries(area_S, &
         n_1, coord_1(:)%alpha, coord_1(:)%delta, &
         coord_1(:)%semi_axis_a, coord_1(:)%semi_axis_b, coord_1(:)%beta, &
         n_2, coord_2(:)%alpha, coord_2(:)%delta, &
         coord_2(:)%semi_axis_a, coord_2(:)%semi_axis_b, coord_2(:)%beta, &
         R_ctp, assoc_data_1)

    call ots_preliminaries(area_S, &
         n_1, coord_1(:)%alpha, coord_1(:)%delta, &
         coord_1(:)%semi_axis_a, coord_1(:)%semi_axis_b, coord_1(:)%beta, &
         n_2, coord_2(:)%alpha, coord_2(:)%delta, &
         coord_2(:)%semi_axis_a, coord_2(:)%semi_axis_b, coord_2(:)%beta, &
         R_ctp, assoc_data_2)

  end subroutine general_preliminaries

!======================================================================

  subroutine det_inv(mat, inv_mat, det_mat)

! Compute inverse and determinant of a (2, 2)-matrix.
    implicit none
    real(kind=real_type), dimension(2, 2), intent(in) :: mat
    real(kind=real_type), dimension(2, 2), intent(out) :: inv_mat
    real(kind=real_type), intent(out) ::det_mat
!______________________________________________________________________

    det_mat = mat(1, 1)*mat(2, 2)-mat(1, 2)*mat(2, 1)
    inv_mat(1, 1) = mat(2, 2)
    inv_mat(1, 2) = -mat(1, 2)
    inv_mat(2, 1) = -mat(2, 1)
    inv_mat(2, 2) = mat(1, 1)
    inv_mat(:,:) = inv_mat(:,:)/det_mat

  end subroutine det_inv

!======================================================================

  function Gaussian_2D(cov, vector)

! Compute a 2-dimensional normal law from the covariance matrix.
    implicit none
    real(kind=real_type) :: Gaussian_2D
    real(kind=real_type), dimension(2), intent(in) :: vector
    real(kind=real_type), dimension(2,2), intent(in) :: cov
!......................................................................
    real(kind=real_type), dimension(2,2) :: inv_cov
    real(kind=real_type) :: det_cov
!______________________________________________________________________

    call det_inv(mat = cov, inv_mat = inv_cov, det_mat = det_cov)

    Gaussian_2D = exp(-dot_product(vector, matmul(inv_cov, vector))/2) / &
         (two_pi*sqrt(det_cov))

  end function Gaussian_2D

!======================================================================

  function ang_dist(alpha_1, delta_1, alpha_2, delta_2)

! Angular distance.
    implicit none
    real(kind=real_type) :: ang_dist
    real(kind=real_type), intent(in) :: alpha_1, delta_1, alpha_2, delta_2
!______________________________________________________________________

    ang_dist = arc_cos(cos(alpha_2-alpha_1)*cos(delta_1)*cos(delta_2) &
         + sin(delta_1)*sin(delta_2))

  end function ang_dist

!======================================================================

  subroutine compute_sto_prob_i(assoc_data_1, n_2, f_1, n_ctp, ctp, prob)

! Compute probabilities of association in the several-to-one case.
    implicit none
    type(assoc_struct), intent(in) :: assoc_data_1
    real(kind=real_type), intent(in) :: f_1
    integer, intent(in) :: n_2
    integer, intent(out) :: n_ctp
    integer, dimension(0:), intent(out) :: ctp
    real(kind=real_type), dimension(0:), intent(out) :: prob
!......................................................................
    real(kind=real_type) :: denom
!______________________________________________________________________

    n_ctp = assoc_data_1%n_ctp_main
    ctp(0:n_ctp) = assoc_data_1%ctp_main(0:n_ctp)
    denom = (1-f_1)*assoc_data_1%xi_main(0) + f_1/n_2*assoc_data_1%sum_xi_main
    prob(0) = (1-f_1)*assoc_data_1%xi_main(0)/denom
    prob(1:n_ctp) = (f_1/n_2/denom)*assoc_data_1%xi_main(1:n_ctp)

  end subroutine compute_sto_prob_i

!======================================================================

  subroutine compute_sto_prob(assoc_data_1, n_2, f_1, sto_prob_1, sto_prob_2)

    implicit none
    real(kind=real_type), intent(in) :: f_1
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1
    integer, intent(in) :: n_2
    type(prob_struct), dimension(:), pointer :: sto_prob_1
    type(prob_struct), dimension(:), pointer :: sto_prob_2
!......................................................................
    integer :: i, n_1
    integer :: n_ctp
    integer, dimension(0:n_2) :: ctp
    real(kind=real_type), dimension(0:n_2) :: prob
    integer :: j, k, m
!______________________________________________________________________

    n_1 = size(assoc_data_1)
    if (associated(sto_prob_1)) deallocate(sto_prob_1)
    allocate(sto_prob_1(n_1))

    do i = 1, n_1
       call compute_sto_prob_i(assoc_data_1(i), n_2, f_1, n_ctp, ctp, prob)
       sto_prob_1(i)%n_ctp = n_ctp
       if (associated(sto_prob_1(i)%ctp)) deallocate(sto_prob_1(i)%ctp)
       allocate(sto_prob_1(i)%ctp(0:n_ctp))
       sto_prob_1(i)%ctp(0:n_ctp) = ctp(0:n_ctp)
       if (associated(sto_prob_1(i)%prob)) deallocate(sto_prob_1(i)%prob)
       allocate(sto_prob_1(i)%prob(0:n_ctp))
       sto_prob_1(i)%prob(0:n_ctp) = prob(0:n_ctp)
    enddo

    if (associated(sto_prob_2)) deallocate(sto_prob_2)
    allocate(sto_prob_2(n_2))
    sto_prob_2(1:n_2)%n_ctp = 0
    do i = 1, n_1
       do k = 1, sto_prob_1(i)%n_ctp
          j = sto_prob_1(i)%ctp(k)
          sto_prob_2(j)%n_ctp =  sto_prob_2(j)%n_ctp + 1
       enddo
    enddo
    do j = 1, n_2
       n_ctp = sto_prob_2(j)%n_ctp
       if (associated(sto_prob_2(j)%ctp)) deallocate(sto_prob_2(j)%ctp)
       allocate(sto_prob_2(j)%ctp(0:n_ctp))
       sto_prob_2(j)%ctp(0) = 0
       if (associated(sto_prob_2(j)%prob)) deallocate(sto_prob_2(j)%prob)
       allocate(sto_prob_2(j)%prob(0:n_ctp))
       sto_prob_2(j)%prob(0) = 1 ! Modified below.
    enddo
    sto_prob_2(1:n_2)%n_ctp = 0 ! Will be set back to the write value in 
!& the loop.
    do i = 1, n_1
       do k = 1, sto_prob_1(i)%n_ctp
          j = sto_prob_1(i)%ctp(k)
          sto_prob_2(j)%n_ctp = sto_prob_2(j)%n_ctp + 1
          m = sto_prob_2(j)%n_ctp
          sto_prob_2(j)%ctp(m) = i
          sto_prob_2(j)%prob(m) = sto_prob_1(i)%prob(k)
          sto_prob_2(j)%prob(0) = sto_prob_2(j)%prob(0) * &
               (1 - sto_prob_1(i)%prob(k))
       enddo
    enddo

  end subroutine compute_sto_prob

!======================================================================

  subroutine compute_ots_prob(n_1, assoc_data_2, f_2, ots_prob_1, ots_prob_2)

    implicit none
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_2
    integer, intent(in) :: n_1
    real(kind=real_type) :: f_2
    type(prob_struct), dimension(:), pointer :: ots_prob_1, ots_prob_2
!______________________________________________________________________

! o:s probabilities = s:o probalities with catalogs exchanged.
    call compute_sto_prob(assoc_data_1 = assoc_data_2, n_2 = n_1, f_1 = f_2, &
         sto_prob_1 = ots_prob_2, sto_prob_2 = ots_prob_1)

  end subroutine compute_ots_prob

!======================================================================

  subroutine sto_analysis(assoc_data_1, assoc_data_2, sto_f_1, &
       std_dev_sto_f_1, sto_f_2, start_f_1)

! Compute `sto_f_1` (and, incidentally, `std_dev_sto_f_1` and `sto_f_2`) in the
! several-to-one case.
    implicit none
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1, assoc_data_2
    real(kind=real_type), intent(out) :: sto_f_1, std_dev_sto_f_1, sto_f_2
    real(kind=real_type), intent(in), optional :: start_f_1
!......................................................................
    integer :: n_1, n_2
    real(kind=real_type) :: prev_f_1, xi, drv2_Lh_1
    integer :: i, j, k
    real(kind=real_type), dimension(size(assoc_data_1)) :: prob0_1
    real(kind=real_type), dimension(size(assoc_data_2)) :: prob0_2
!______________________________________________________________________

    n_1 = size(assoc_data_1)
    n_2 = size(assoc_data_2)

    if (present(start_f_1)) then
       sto_f_1 = start_f_1
    else
       sto_f_1 = -1. ! To force next condition to be true.
    endif

    if (sto_f_1 <= 0 .or. sto_f_1 >= 1) sto_f_1 = 0.5

    do
       prev_f_1 = sto_f_1
       do i = 1, n_1
          prob0_1(i) = (1-sto_f_1)*n_2*assoc_data_1(i)%xi_main(0) / &
               ((1-sto_f_1)*n_2*assoc_data_1(i)%xi_main(0) &
               + sto_f_1*assoc_data_1(i)%sum_xi_main)
       enddo
       sto_f_1 = 1-sum(prob0_1(1:n_1))/n_1
       if (abs(sto_f_1-prev_f_1) < epsilon_f) exit
    enddo

    drv2_Lh_1 = (n_1 - sum(prob0_1(1:n_1)**2)/(1-sto_f_1)**2)/sto_f_1**2
    std_dev_sto_f_1 = sqrt(-1/drv2_Lh_1)

! Fraction of $K_2$-sources with a counterpart in $K_1$.

    do j = 1, n_2
       prob0_2(j) = 1
       do k = 1, assoc_data_2(j)%n_ctp_main
          i = assoc_data_2(j)%ctp_main(k)
          xi = assoc_data_2(j)%xi_main(k)
          prob0_2(j) = prob0_2(j) * &
               (1 - sto_f_1*xi/((1-sto_f_1)*n_2*assoc_data_1(i)%xi_main(0) + &
               sto_f_1*assoc_data_1(i)%sum_xi_main))
       enddo
    enddo
    sto_f_2 = 1-sum(prob0_2(1:n_2))/n_2

  end subroutine sto_analysis

!======================================================================

  subroutine ots_analysis(assoc_data_1, assoc_data_2, ots_f_1, ots_f_2, &
       std_dev_ots_f_2, start_f_2)

! Compute `ots_f_2` (and, incidentally, `std_dev_ots_f_2` and `ots_f_1`) in the
! one-to-several case.
    implicit none
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1, assoc_data_2
    real(kind=real_type), intent(out) :: ots_f_1, std_dev_ots_f_2, ots_f_2
    real(kind=real_type), intent(in), optional :: start_f_2
!______________________________________________________________________

! o:s analysis = s:o analysis with catalogs exchanged.
    if (present(start_f_2)) then
       call sto_analysis(assoc_data_1 = assoc_data_2, &
            assoc_data_2 = assoc_data_1, sto_f_1 = ots_f_2, &
            std_dev_sto_f_1 = std_dev_ots_f_2, &
            sto_f_2 = ots_f_1, start_f_1 = start_f_2)
    else
       call sto_analysis(assoc_data_1 = assoc_data_2, &
            assoc_data_2 = assoc_data_1, sto_f_1 = ots_f_2, &
            std_dev_sto_f_1 = std_dev_ots_f_2, &
            sto_f_2 = ots_f_1)
    endif

  end subroutine ots_analysis

!======================================================================

  function func_ln_sto_Lh(assoc_data_1, assoc_data_2, f_1)

! Log-likelihood in the several-to-one case.
    implicit none
    real(kind=real_type) :: func_ln_sto_Lh
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1, assoc_data_2
    real(kind=real_type), intent(in) :: f_1
!......................................................................
    integer :: i, n_1, n_2
!______________________________________________________________________

    n_1 = size(assoc_data_1)
    n_2 = size(assoc_data_2)
    func_ln_sto_Lh = 0
    do i = 1, n_1
       func_ln_sto_Lh = func_ln_sto_Lh &
            + log(f_1*assoc_data_1(i)%sum_xi_main/n_2 &
            + (1-f_1)*assoc_data_1(i)%xi_main(0))
    enddo
    do i = 1, n_2
       func_ln_sto_Lh = func_ln_sto_Lh + log(assoc_data_2(i)%xi_main(0))
    enddo

  end function func_ln_sto_Lh

!======================================================================

  function func_ln_ots_Lh(assoc_data_1, assoc_data_2, f_2)

! Log-likelihood in the one-to-several case.
    implicit none
    real(kind=real_type) :: func_ln_ots_Lh
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1, assoc_data_2
    real(kind=real_type), intent(in) :: f_2
!______________________________________________________________________

! o:s likelihood = s:o likelihood with catalogs exchanged.
    func_ln_ots_Lh = func_ln_sto_Lh( &
         assoc_data_1 = assoc_data_2, assoc_data_2 = assoc_data_1, f_1 = f_2)

  end function func_ln_ots_Lh

!======================================================================

  subroutine find_ngb(src, alpha, delta, R_ngb, k_first, k_last, &
       ngb, n_ngb, ctp_to_any, assoc_data)

! Find neighbors of a given source.
    implicit none
    integer, intent(in) :: src
    real(kind=real_type), intent(in) :: R_ngb
    real(kind=real_type), dimension(:), intent(in) :: alpha, delta
    integer, intent(in) :: k_first, k_last
    type(ngb_list), pointer :: ngb
    integer, intent(inout) :: n_ngb
    logical, dimension(:), intent(inout) :: ctp_to_any
    type(assoc_struct), dimension(:), intent(in) :: assoc_data
!......................................................................
    integer :: k
    real(kind=real_type) :: psi
    type(ngb_list), pointer :: current_ngb
!______________________________________________________________________

    do k = k_first, k_last
       if (delta(k) < delta(src) - R_ngb .or. delta(k) > delta(src) + R_ngb) &
            cycle

       psi = ang_dist(alpha_1 = alpha(src), delta_1 = delta(src), &
            alpha_2 = alpha(k), delta_2 = delta(k))

       if (psi > R_ngb) cycle
       n_ngb = n_ngb + 1

       ctp_to_any(assoc_data(k)%ctp_main(1:assoc_data(k)%n_ctp_main)) = .true.

! Add a new node to the linked list of neigbors (by decreasing right ascension,
! but this does not matter since neighbors are ordered by increasing angular
! distance to the source in subroutine `create_xi_array`).
       allocate(current_ngb)
       current_ngb%ngb = k
       current_ngb%psi = psi
       current_ngb%ptr => ngb
       ngb => current_ngb
    enddo

  end subroutine find_ngb

!======================================================================

  subroutine oto_preliminaries(coord_1, n_2, assoc_data_1)

! Computation of `oto_data_1` (used for one-to-one computations).
    implicit none
    integer, intent(in) :: n_2
    type(coord_struct), dimension(:), intent(in) :: coord_1
    type(assoc_struct), dimension(:), intent(inout) :: assoc_data_1
!......................................................................
    integer :: i, k1, k2, k3, k4, n_1
    logical, dimension(n_2) :: ctp_to_any_1
    type(ngb_list), pointer :: ngb_1
    real(kind=real_type) :: R_ngb
!______________________________________________________________________

! Provide `R_ngb`.
    R_ngb = R_ratio*R_ctp
! (`R_ratio` is defined in file "mod_constants.f90".)
    n_1 = size(assoc_data_1)
    do i = 1, n_1
       ctp_to_any_1(1:n_2) = .false.
       nullify(ngb_1)
       assoc_data_1(i)%n_ngb = 0
       assoc_data_1(i)%n_ctp_any = 0
       call alpha_bounds(coord_1(i)%alpha, coord_1(i)%delta, &
            n_1, coord_1(:)%alpha, R_ngb, &
            k1, k2, k3, k4)
       call find_ngb(i, coord_1(:)%alpha, coord_1(:)%delta, &
            R_ngb, k1, k2, ngb_1, assoc_data_1(i)%n_ngb, &
            ctp_to_any_1, assoc_data_1)
       call find_ngb(i, coord_1(:)%alpha, coord_1(:)%delta, &
            R_ngb, k3, k4, ngb_1, assoc_data_1(i)%n_ngb, &
            ctp_to_any_1, assoc_data_1)
       call create_xi_array(i, ngb_1, ctp_to_any_1, n_2, &
            assoc_data_1)
    enddo

  end subroutine oto_preliminaries

!======================================================================

  subroutine create_xi_array(src, ngb_1, ctp_to_any_1, n_2, &
       assoc_data_1)
! Create array `assoc_data_1%xi_any` containing the value of $\xi_{k, j}$
! for all neighbors ($k$) and counterparts ($j$) to a given source (used in
! one-to-one calculations).
! Caution: `assoc_data_1%n_ctp_any /= assoc_data_1%n_ctp_main`: the first one 
! is the number of counterparts of all nearby sources; the second only that 
! of the central source.
    use mod_heap_index, only : heap_index
    implicit none
    integer, intent(in) :: n_2
    type(ngb_list), pointer :: ngb_1
    logical, dimension(:), intent(in) :: ctp_to_any_1
    type(assoc_struct), dimension(:), intent(inout) :: assoc_data_1
    integer, intent(in) :: src
!......................................................................
    real(kind=real_type), dimension(assoc_data_1(src)%n_ngb) :: psi
    integer, dimension(assoc_data_1(src)%n_ngb) :: ngb, index_psi
    type(ngb_list), pointer :: current_ngb, ptr
    integer :: j, k
    integer, dimension(0:n_2) :: index_ctp
    integer :: n_ctp_main, n_ctp_any, n_ngb
!______________________________________________________________________

    assoc_data_1(src)%n_ctp_any = count(ctp_to_any_1)
    n_ctp_main = assoc_data_1(src)%n_ctp_main
    n_ctp_any = assoc_data_1(src)%n_ctp_any
    n_ngb = assoc_data_1(src)%n_ngb
    allocate(assoc_data_1(src)%ngb(n_ngb))
    allocate(assoc_data_1(src)%ctp_any(0:n_ctp_any))
    allocate(assoc_data_1(src)%corresp(0:n_ctp_main))
    allocate(assoc_data_1(src)%xi_any(n_ngb, 0:n_ctp_any))
    current_ngb => ngb_1
    do k = 1, n_ngb
       psi(k) = current_ngb%psi
       ngb(k) = current_ngb%ngb
       ptr => current_ngb%ptr
       deallocate(current_ngb)
       current_ngb => ptr
    enddo

! Order neighbors by increasing angular distance to the main source.
    call heap_index(psi, index_psi)
    ngb(1:n_ngb) = ngb(index_psi(1:n_ngb))

    index_ctp(0) = 0
    assoc_data_1(src)%ctp_any(0) = 0
    k = 0
    do j = 1, n_2
       if (ctp_to_any_1(j)) then
          k = k+1
          index_ctp(j) = k
          assoc_data_1(src)%ctp_any(k) = j
       endif
    enddo

    assoc_data_1(src)%corresp(0:assoc_data_1(src)%n_ctp_main) = &
         index_ctp(assoc_data_1(src)%ctp_main(0:assoc_data_1(src)%n_ctp_main))

    assoc_data_1(src)%xi_any(:,0:) = 0
    do k = 1, n_ngb
       assoc_data_1(src)%xi_any(k,0) = assoc_data_1(ngb(k))%xi_main(0)
    enddo
    do k = 1, n_ngb
       assoc_data_1(src)%ngb(k) = ngb(k)
       do j = 1, assoc_data_1(ngb(k))%n_ctp_main
          assoc_data_1(src)%xi_any(k, &
               index_ctp(assoc_data_1(ngb(k))%ctp_main(j))) = &
               assoc_data_1(ngb(k))%xi_main(j)
       enddo
    enddo

  end subroutine create_xi_array

!======================================================================

  subroutine compute_oto_prob_i(n_1, n_2, assoc_data_1, f_1, f_2, &
       prob0_prev, sum_prob_all, n_ctp, ctp, prob, i_min)

! This subroutine is called by `compute_oto_prob` iteratively, until 
! convergence, for all sources in the first catalog, and with 
! `f_2 = f_1*n_1/n_2`.
! It should not be called directly.

    implicit none
    integer, intent(in) :: n_1, n_2
    real(kind=real_type), intent(in) :: f_1, f_2, sum_prob_all
    type(assoc_struct), intent(in) :: assoc_data_1
    integer, intent(out) :: n_ctp
    integer, dimension(0:), intent(out) :: ctp
    real(kind=real_type), dimension(0:), intent(out) :: prob
    real(kind=real_type), dimension(:), intent(in) :: prob0_prev
    integer, intent(in), optional :: i_min ! Used only for the alternate way of
!& computing the o:o likelihood.
!......................................................................
    integer :: n_ngb
    integer :: j, n_ctp_any, k, k2
    logical, dimension(0:assoc_data_1%n_ctp_any) :: exclude
    real(kind=real_type) :: rec_sum, sum_prob_local, sum_prob_distant, n_eff
    logical, dimension(:), allocatable :: exclude_ngb
    real(kind=real_type), dimension(:,:), allocatable :: xi
!______________________________________________________________________

    n_ngb = assoc_data_1%n_ngb
    n_ctp_any = assoc_data_1%n_ctp_any
    if (present(i_min)) then
       allocate(exclude_ngb(n_ngb))
       do k = 1, n_ngb
          if (assoc_data_1%ngb(k) < i_min) then ! Neighbors with ordinal number
!&           < `i_min` are excluded from the computation.
             exclude_ngb(k) = .true.
          else
             exclude_ngb(k) = .false.
          endif
       enddo
       allocate(xi(n_ngb-count(exclude_ngb(1:n_ngb)), 0:n_ctp_any))
       k2 = 0
       sum_prob_local = 0
       do k = 1, n_ngb
          if (.not.exclude_ngb(k)) then ! Neighbors with ordinal number
!&           < `i` are excluded from the computation.
             k2 = k2 + 1
             xi(k2,:) = assoc_data_1%xi_any(k,:)
             sum_prob_local = sum_prob_local + &
                  1 - prob0_prev(assoc_data_1%ngb(k))
          endif
       enddo
       n_ngb = n_ngb-count(exclude_ngb(1:n_ngb))
       deallocate(exclude_ngb)
    else
       allocate(xi(n_ngb, 0:n_ctp_any))
       xi = assoc_data_1%xi_any
       sum_prob_local = 0
       do k = 1, n_ngb
          sum_prob_local = sum_prob_local + 1 - prob0_prev(assoc_data_1%ngb(k))
       enddo
    endif
!
    exclude = .false.

! To check the convergence of `prob`, on might compute it for `n_ngb` = 1
! (no neighbor, main source only), `n_ngb` = 2 (main source + first neighbor),
! etc.
! Hopefully, the probabilities will have converged for the following value of
! `n_ngb`.

    n_ngb = min(n_ngb, n_max_ngb)

! Compute the effective number of sources in the other catalog which are 
! available for an association to one of the nearby sources (i.e. all those 
! not associated to a "distant" source).
    sum_prob_distant = sum_prob_all - sum_prob_local
    if (n_1 <= n_2) then
       n_eff = n_2-sum_prob_distant
    else
       n_eff = n_1-sum_prob_distant
    endif

    n_ctp = assoc_data_1%n_ctp_main
    ctp(0:n_ctp) = assoc_data_1%ctp_main(0:n_ctp)
    do k = 0, n_ctp
       j = assoc_data_1%corresp(k)
       if (n_1 <= n_2) then
          prob(k) = rec_sum1_j(j = j, xi = xi, f_1 = f_1, n_ngb = n_ngb, &
               n_ctp = n_ctp_any, exclude = exclude, n_2_eff = n_eff)
       else
          prob(k) = rec_sum2_j(j = j, xi = xi, f_2 = f_2, n_ngb = n_ngb, &
               n_ctp = n_ctp_any, exclude = exclude, n_1_eff = n_eff)
       endif
    enddo
    rec_sum = sum(prob(0:n_ctp))
    prob(0:n_ctp) = prob(0:n_ctp)/rec_sum
    deallocate(xi)

  end subroutine compute_oto_prob_i

!======================================================================

  subroutine compute_oto_prob(assoc_data_1, n_2, f_1, &
       oto_prob_1, oto_prob_2, prob0, i_min)

    implicit none
    real(kind=real_type), intent(in) :: f_1
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1
    integer, intent(in) :: n_2
    integer, intent(in), optional :: i_min ! Used only by `func_ln_oto_Lh_alt`. 
!& Should not be called by the user.
    real(kind=real_type), dimension(:), intent(out), optional :: &
         prob0 ! Probability that a source has no counterpart. 
!& Should not be called by the user.
    type(prob_struct), dimension(:), pointer, optional :: oto_prob_1, oto_prob_2
!......................................................................
    integer :: n_1, i, j, k, m
    real(kind=real_type) :: f_2
    real(kind=real_type) :: sum_prob_all ! Sum, on all sources, of the 
!& probability that it has a counterpart.
    logical :: last
    real(kind=real_type), dimension(size(assoc_data_1)) :: prob0_tmp, prob0_prev
    integer :: n_ctp
    integer, dimension(0:n_2) :: ctp
    real(kind=real_type), dimension(0:n_2) :: prob
!______________________________________________________________________

    if (present(i_min) .and. .not.present(prob0)) then
       write(*,"(a)") "Subroutine `compute_oto_prob`: `prob0` must be present &
            &if `i_min` is present. Stopped."
       stop
    endif

    if (present(prob0) .and. (present(oto_prob_1) .or. present(oto_prob_2))) &
         then
       write(*,"(a)") "Subroutine `compute_oto_prob`: neither `oto_prob_1` &
            &nor `oto_prob_2` should be present if `prob0` is present. &
            &Stopped."
       stop
    endif

    if (.not.present(prob0) .and. .not.(present(oto_prob_1) .and. &
         present(oto_prob_2))) then
       write(*,"(a)") "Subroutine `compute_oto_prob`: both `oto_prob_1` and &
            &`oto_prob_2` should be present if `prob0` is absent. Stopped."
       stop
    endif

    n_1 = size(assoc_data_1)
    if (present(oto_prob_1)) then
       if (associated(oto_prob_1)) deallocate(oto_prob_1)
       allocate(oto_prob_1(n_1))
       if (associated(oto_prob_2)) deallocate(oto_prob_2)
       allocate(oto_prob_2(n_2))
    endif

! s:o probabilities used as a first order approximation to o:o probabilities.

    do i = 1, n_1
       if (present(i_min)) then
          if (i < i_min) then
             prob0_tmp(i) = 1
          else
             call compute_sto_prob_i(assoc_data_1(i), n_2, f_1, n_ctp, ctp, &
                  prob)
             prob0_tmp(i) = prob(0)
          endif
       else
          call compute_sto_prob_i(assoc_data_1(i), n_2, f_1, n_ctp, ctp, prob)
          prob0_tmp(i) = prob(0)
       endif
    enddo
    sum_prob_all = n_1 - sum(prob0_tmp(1:n_1))

! Compute o:o probabilities (and the effective number of distant sources with a
! counterpart) by iteration until they are stable.

    f_2 = (f_1 * n_1)/n_2
    last = .false.
    do
       prob0_prev(:) = prob0_tmp(:)
       do i = 1, n_1
          if (present(i_min)) then
             if (i < i_min) then
                prob0_tmp(i) = 1
             else
                call compute_oto_prob_i(n_1, n_2, assoc_data_1(i), f_1, f_2, &
                     prob0_prev, sum_prob_all, n_ctp, ctp, prob, i_min = i_min)
                prob0_tmp(i) = prob(0)
             endif
          else
             call compute_oto_prob_i(n_1, n_2, assoc_data_1(i), f_1, f_2, &
                  prob0_prev, sum_prob_all, n_ctp, ctp, prob)
             prob0_tmp(i) = prob(0)
          endif

! If requested and if the iteration has converged, store the probabilities in 
! `oto_prob_1` and `oto_prob_2`.
          if (present(oto_prob_1) .and. last) then
             oto_prob_1(i)%n_ctp = n_ctp
             if (associated(oto_prob_1(i)%ctp)) deallocate(oto_prob_1(i)%ctp)
             allocate(oto_prob_1(i)%ctp(0:n_ctp))
             oto_prob_1(i)%ctp(0:n_ctp) = ctp(0:n_ctp)
             if (associated(oto_prob_1(i)%prob)) deallocate(oto_prob_1(i)%prob)
             allocate(oto_prob_1(i)%prob(0:n_ctp))
             oto_prob_1(i)%prob(0:n_ctp) = prob(0:n_ctp)
          endif
       enddo
       if (present(oto_prob_2) .and. last) then 
          oto_prob_2(1:n_2)%n_ctp = 0
          do i = 1, n_1
             do k = 1, oto_prob_1(i)%n_ctp
                j = oto_prob_1(i)%ctp(k)
                oto_prob_2(j)%n_ctp =  oto_prob_2(j)%n_ctp + 1
             enddo
          enddo
          do j = 1, n_2
             n_ctp = oto_prob_2(j)%n_ctp
             if (associated(oto_prob_2(j)%ctp)) deallocate(oto_prob_2(j)%ctp)
             allocate(oto_prob_2(j)%ctp(0:n_ctp))
             oto_prob_2(j)%ctp(0) = 0
             if (associated(oto_prob_2(j)%prob)) deallocate(oto_prob_2(j)%prob)
             allocate(oto_prob_2(j)%prob(0:n_ctp))
             oto_prob_2(j)%prob(0) = 1 ! Modified below.
          enddo
          oto_prob_2(1:n_2)%n_ctp = 0 ! Will be set back to the write value 
!& in the loop.
          do i = 1, n_1
             do k = 1, oto_prob_1(i)%n_ctp
                j = oto_prob_1(i)%ctp(k)
                oto_prob_2(j)%n_ctp = oto_prob_2(j)%n_ctp + 1
                m = oto_prob_2(j)%n_ctp
                oto_prob_2(j)%ctp(m) = i
                oto_prob_2(j)%prob(m) = oto_prob_1(i)%prob(k)
                oto_prob_2(j)%prob(0) = oto_prob_2(j)%prob(0) &
                     - oto_prob_1(i)%prob(k)
             enddo
          enddo
       endif

       sum_prob_all = n_1 - sum(prob0_tmp(1:n_1))
       if (last) exit
       if (sum(abs(prob0_tmp(:) - prob0_prev(:))) < epsilon_oto_prob * n_1) &
            last = .true. ! Converged. Iterate one more time and exit.
    enddo

    if (present(prob0)) then
       prob0(:) = prob0_tmp(:)
    endif

  end subroutine compute_oto_prob

!======================================================================

  subroutine oto_analysis(assoc_data_1, n_2, oto_f_1, std_dev_oto_f_1, &
       oto_f_2, start_f_1)

    implicit none
    integer, intent(in) :: n_2
    real(kind=real_type), intent(out) :: oto_f_1, std_dev_oto_f_1, oto_f_2
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1
    real(kind=real_type), intent(in), optional :: start_f_1
!......................................................................
    integer :: n_1
    real(kind=real_type) :: prev_f_1, drv2_ln_oto_Lh, f_max
    real(kind=real_type), dimension(size(assoc_data_1)) :: prob0
!______________________________________________________________________

    n_1 = size(assoc_data_1)

! Compute `oto_f_1`.

    if (present(start_f_1)) then
       oto_f_1 = start_f_1
    else
       oto_f_1 = -1. ! To force next condition to be true.
    endif

! `f_1` must be in $]0, \min(1, `n_2`/`n_1`)[$.
    f_max = min(1., real(n_2)/n_1)
    if (oto_f_1 <= 0 .or. oto_f_1 >= f_max) &
         then
! Starting iteration with $f_0 = \min(1, n'/n)/2$.
       oto_f_1 = 0.5*f_max
    endif

    do
       prev_f_1 = oto_f_1
       call compute_oto_prob(assoc_data_1, n_2, oto_f_1, &
            prob0 = prob0)
       oto_f_1 = 1 - sum(prob0(1:n_1))/n_1
       if (abs(oto_f_1-prev_f_1) < epsilon_f) exit
    enddo
    oto_f_2 = (n_1*oto_f_1)/n_2

! Compute `std_dev_oto_f_1`.

    if (oto_f_1 < 0.5*f_max) then ! To avoid problems if `oto_f_1` is close 
!& to either `f_max` or 0.
       drv2_ln_oto_Lh = &
            (drv_ln_oto_Lh(assoc_data_1, n_2, oto_f_1 + d_f) - 0)/d_f
    else
       drv2_ln_oto_Lh = &
            (drv_ln_oto_Lh(assoc_data_1, n_2, oto_f_1 - d_f) - 0)/(-d_f)
    endif
! (The "0" above is `drv_ln_oto_Lh` at `oto_f_1`.)
    std_dev_oto_f_1 = 1/sqrt(-drv2_ln_oto_Lh)

  end subroutine oto_analysis

!======================================================================

  recursive function rec_sum1_j(j, xi, f_1, n_ngb, n_ctp, &
       exclude, n_2_eff, depth) result(res)

! Recursive sum used when `n_1 <= n_2`.
    implicit none
    real(kind=real_type) :: res
    integer, intent(in) :: j
    real(kind=real_type), dimension(:,0:), intent(in) :: xi
    real(kind=real_type), intent(in) :: f_1
    integer, intent(in) :: n_ngb, n_ctp
    logical, dimension(0:), intent(in) :: exclude
    real(kind=real_type), intent(in) :: n_2_eff
    integer, intent(in), optional :: depth
!......................................................................
    integer :: k ! Current depth of the recursive sum.
    integer :: j_k ! Sndex of the counterparts considered at depth `k`.
    logical, dimension(0:n_ctp) :: exclude_next ! Sources excluded at the next 
!& (deeper) level of the recursion.
!______________________________________________________________________

    if (present(depth)) then
       k = depth
    else
       k = 1
    endif

    if (k == 1) then
       exclude_next = exclude
       if (j == 0) then
          res = (1-f_1)*xi(k, j)* &
               rec_sum1_j(j = j, xi = xi, f_1 = f_1, &
               n_ngb = n_ngb, n_ctp = n_ctp, &
               exclude = exclude_next, n_2_eff = n_2_eff, depth = k+1)
       else
          exclude_next(j) = .true.
          if (xi(k, j) /= 0) then
             res = f_1*xi(k, j)/n_2_eff* &
                  rec_sum1_j(j = j, xi = xi, f_1 = f_1, &
                  n_ngb = n_ngb, n_ctp = n_ctp, &
                  exclude = exclude_next, n_2_eff = n_2_eff, depth = k+1)
          else
             res = 0
          endif
       endif
    else if (k < n_ngb) then
       res = 0
       do j_k = 0, n_ctp
          if (exclude(j_k)) cycle
          exclude_next = exclude
          if (j_k == 0) then
             res = res + (1-f_1)*xi(k, j_k)* &
                  rec_sum1_j(j = j, xi = xi, f_1 = f_1, &
                  n_ngb = n_ngb, n_ctp = n_ctp, &
                  exclude = exclude_next, n_2_eff = n_2_eff, depth = k+1)
          else
             exclude_next(j_k) = .true.
             if (xi(k, j_k) /= 0) then
                res = res + f_1*xi(k, j_k)/(n_2_eff-count(exclude))* &
                     rec_sum1_j(j = j, xi = xi, f_1 = f_1, &
                     n_ngb = n_ngb, n_ctp = n_ctp, &
                     exclude = exclude_next, n_2_eff = n_2_eff, depth = k+1)
             endif
          endif
       enddo
    else if (k == n_ngb) then
       res = 0
       do j_k = 0, n_ctp
          if (exclude(j_k)) cycle
          if (j_k == 0) then
             res = res + (1-f_1)*xi(k, j_k)
          else
             res = res + f_1*xi(k, j_k)/(n_2_eff-count(exclude))
          endif
       enddo
    else ! Occurs when `n_ngb` = 1.
       res = 1
    endif

  end function rec_sum1_j

!======================================================================

  recursive function rec_sum2_j(j, xi, f_2, n_ngb, n_ctp, &
       exclude, n_1_eff, depth) result(res)

! Recursive sum used when `n_1 > n_2`.
    implicit none
    real(kind=real_type) :: res
    integer, intent(in) :: j
    real(kind=real_type), dimension(:,0:), intent(in) :: xi
    real(kind=real_type), intent(in) :: f_2
    integer, intent(in) :: n_ngb, n_ctp
    logical, dimension(0:), intent(in) :: exclude
    real(kind=real_type), intent(in) :: n_1_eff
    integer, intent(in), optional :: depth
!......................................................................
    integer :: k ! Current depth of the recursive sum.
    integer :: j_k ! Sndex of the counterparts considered at depth `k`.
    logical, dimension(0:n_ctp) :: exclude_next ! Sources excluded at the &
!& next (deeper) level of the recursion.
!______________________________________________________________________

    if (present(depth)) then
       k = depth
    else
       k = 1
    endif

    if (k == 1) then
       exclude_next = exclude
       if (j == 0) then
          res = (1-f_2)*xi(k, j)* &
               rec_sum2_j(j = j, xi = xi, f_2 = f_2, &
               n_ngb = n_ngb, n_ctp = n_ctp, &
               exclude = exclude_next, n_1_eff = n_1_eff, depth = k+1)
       else
          exclude_next(j) = .true.
          if (xi(k, j) /= 0) then
             res = f_2*xi(k, j)/n_1_eff* &
                  rec_sum2_j(j = j, xi = xi, f_2 = f_2, &
                  n_ngb = n_ngb, n_ctp = n_ctp, &
                  exclude = exclude_next, n_1_eff = n_1_eff, depth = k+1)
          else
             res = 0
          endif
       endif
    else if ( k < n_ngb) then
       res = 0
       do j_k = 0, n_ctp
          if (exclude(j_k)) cycle
          exclude_next = exclude
          if (j_k == 0) then
             res = res + (1-f_2)*xi(k, j_k)* &
                  rec_sum2_j(j = j, xi = xi, f_2 = f_2, &
                  n_ngb = n_ngb, n_ctp = n_ctp, &
                  exclude = exclude_next, n_1_eff = n_1_eff, depth = k+1)
          else
             exclude_next(j_k) = .true.
             if (xi(k, j_k) /= 0) then
                res = res + f_2*xi(k, j_k)/(n_1_eff-count(exclude))* &
                     rec_sum2_j(j = j, xi = xi, f_2 = f_2, &
                     n_ngb = n_ngb, n_ctp = n_ctp, &
                     exclude = exclude_next, n_1_eff = n_1_eff, depth = k+1)
             endif
          endif
       enddo
    else if (k == n_ngb) then
       res = 0
       do j_k = 0, n_ctp
          if (exclude(j_k)) cycle
          if (j_k == 0) then
             res = res + (1-f_2)*xi(k, j_k)
          else
             res = res + f_2*xi(k, j_k)/(n_1_eff-count(exclude))
          endif
       enddo
    else ! Occurs when `n_ngb` = 1.
       res = 1
    endif

  end function rec_sum2_j

!======================================================================

  function Steffen_integral(n, x, y, i)

! See Steffen M. (1990), A&A 239, 443.
! Integration routine based on Steffen (1990) interpolation routine.
! Advantages are that the interpolation is smooth, local (contrary to
! splines), fast, without oscillations and that extrema happen only at the
! tabulated values.
! Given
! 1. a function `y(x)`, tabulated for `n` values such as
!    `x(1) <= ... <= x(n)`; and
! 2. an integer `i`,
! `Steffen_integral` is the integral from `x(i)` to `x(i+1)` of the function
! interpolating `y`.
    implicit none
    real(kind=real_type) :: Steffen_integral
    integer, intent(in) :: n, i
    real(kind=real_type), dimension(:), intent(in) :: x, y
!......................................................................
    real(kind=real_type) :: a, b, c, d
    real(kind=real_type) :: h_i, h_im1, h_ip1
    real(kind=real_type) :: s_i, s_im1, s_ip1
    real(kind=real_type) :: p_i, p_ip1, y1_i, y1_ip1
!______________________________________________________________________

    if (i == 1) then
       h_i = x(i+1)-x(i)
       s_i = (y(i+1)-y(i))/h_i
       h_ip1 = x(i+2)-x(i+1)
       s_ip1 = (y(i+2)-y(i+1))/h_ip1
       p_ip1 = (s_i*h_ip1+s_ip1*h_i)/(h_i+h_ip1)
       y1_ip1 = (sign(1._real_type, s_i)+sign(1._real_type, s_ip1)) * &
            min(abs(s_i), abs(s_ip1), 0.5_real_type*abs(p_ip1))
! { Parabola through first 3 points.
       p_i = s_i*(1+h_i/(h_i+h_ip1)) - s_ip1*h_i/(h_i+h_ip1)
       if (p_i*s_i <= 0) then
          y1_i = 0
       else if (abs(p_i) > 2*abs(s_i)) then
          y1_i = 2*s_i
       else
          y1_i = p_i
       endif
!}
    else if (i == n-1) then
       h_i = x(i+1)-x(i)
       s_i = (y(i+1)-y(i))/h_i
       h_im1 = x(i)-x(i-1)
       s_im1 = (y(i)-y(i-1))/h_im1
       p_i = (s_im1*h_i+s_i*h_im1)/(h_im1+h_i)
       y1_i = (sign(1._real_type, s_im1)+sign(1._real_type, s_i)) * &
            min(abs(s_im1), abs(s_i), 0.5_real_type*abs(p_i))
! { Parabola through last 3 points.
       p_ip1 = s_i*(1+h_i/(h_i+h_im1)) - s_im1*h_i/(h_i+h_im1)
       if (p_ip1*s_i <= 0) then
          y1_ip1 = 0
       else if (abs(p_ip1) > 2*abs(s_i)) then
          y1_ip1 = 2*s_i
       else
          y1_ip1 = p_ip1
       endif
! }
    else
       h_i = x(i+1)-x(i)
       s_i = (y(i+1)-y(i))/h_i
       h_im1 = x(i)-x(i-1)
       s_im1 = (y(i)-y(i-1))/h_im1
       h_ip1 = x(i+2)-x(i+1)
       s_ip1 = (y(i+2)-y(i+1))/h_ip1
       p_i = (s_im1*h_i+s_i*h_im1)/(h_im1+h_i)
       p_ip1 = (s_i*h_ip1+s_ip1*h_i)/(h_i+h_ip1)
       y1_i = (sign(1._real_type, s_im1)+sign(1._real_type, s_i)) * &
            min(abs(s_im1), abs(s_i), 0.5_real_type*abs(p_i))
       y1_ip1 = (sign(1._real_type, s_i)+sign(1._real_type, s_ip1)) * &
            min(abs(s_i), abs(s_ip1), 0.5_real_type*abs(p_ip1))
    end if
    a = (y1_i+y1_ip1-2._real_type*s_i)/h_i**2
    b = (3._real_type*s_i-2._real_type*y1_i-y1_ip1)/h_i
    c = y1_i
    d = y(i)
    Steffen_integral = a*h_i**4/4 + b*h_i**3/3 + c*h_i**2/2 + d*h_i

  end function Steffen_integral

!======================================================================

  recursive subroutine refine_grid(n_points, point, &
       n_1, n_2, assoc_data_1)

! Subdivide interval [`point%f`, `point%ptr%f`] until the curvature of
! `point%d` is negligible. The curvature is characterized by the difference 
! between `point%d` at the midpoint of the bin and the average of this 
! quantity at the endpoints of the bin.
    implicit none
    integer, intent(inout) :: n_points
    type(point_list), pointer :: point
    integer, intent(in) :: n_1, n_2
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1
! Local variables.
    real(kind=real_type) :: f_inf, f_med, f_sup, d_inf, d_med, d_sup
    type(point_list), pointer :: new_point, point_med
!______________________________________________________________________

    f_inf = point%f
    d_inf = point%d
    f_sup = point%ptr%f
    d_sup = point%ptr%d
    n_points = n_points + 1
    f_med = (f_inf+f_sup)/2
    d_med = drv_ln_oto_Lh(assoc_data_1, n_2, f_med)
    allocate(new_point)
    new_point%ptr => point%ptr
    new_point%f = f_med
    new_point%d = d_med
    new_point%orig = 0
    point%ptr => new_point
    point_med => point%ptr

    if (abs(d_sup + d_inf - 2 * d_med) > &
         epsilon_ln_oto_Lh*(abs(d_sup) + abs(d_inf) + 2 * abs(d_med))) then
       call refine_grid(n_points, point, n_1, n_2, assoc_data_1)
       call refine_grid(n_points, point_med, n_1, n_2, assoc_data_1)
    endif

  end subroutine refine_grid

!======================================================================

  function drv_ln_oto_Lh(assoc_data_1, n_2, f_1)

    implicit none
    real(kind=real_type) :: drv_ln_oto_Lh
    real(kind=real_type), intent(in) :: f_1
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1
    integer, intent(in) :: n_2
!......................................................................
    integer :: n_1
    real(kind=real_type) :: f_2
    real(kind=real_type), dimension(size(assoc_data_1)) :: prob0
!______________________________________________________________________

    call compute_oto_prob(assoc_data_1, n_2, f_1, prob0 = prob0)
    n_1 = size(assoc_data_1)
    if (n_1 <= n_2) then
       drv_ln_oto_Lh = (n_1*(1-f_1) - sum(prob0(1:n_1))) / (f_1*(1-f_1))
    else
       f_2 = (f_1 * n_1)/n_2
       drv_ln_oto_Lh = (n_1*(1-f_1) - sum(prob0(1:n_1))) / (f_1*(1-f_2))
    endif

  end function drv_ln_oto_Lh

!======================================================================

  subroutine compute_ln_oto_Lh(assoc_data_1, assoc_data_2, &
       n_grid, f_grid, ln_oto_Lh_grid, &
       f_1, ln_oto_Lh)

! Compute `ln_oto_Lh` by integrating its derivative with respect to `f` at
! a given value of `f_1` or (inclusively) for a grid of values `f_grid`.
    implicit none
    integer, intent(inout), optional :: n_grid
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1, assoc_data_2
    real(kind=real_type), dimension(:), pointer, optional :: f_grid, &
         ln_oto_Lh_grid
    real(kind=real_type), intent(in), optional :: f_1
    real(kind=real_type), intent(out), optional :: ln_oto_Lh
!......................................................................
    integer :: n_grid_tmp, n_1, n_2
    real(kind=real_type), dimension(:), allocatable :: f_grid_tmp, &
         ln_oto_Lh_grid_tmp, d_grid_tmp
    real(kind=real_type) :: integr, ln_Lh0, f_max
    type(point_list), pointer :: point, current_point, next_point
    integer :: i, i_grid, n_points
    real(kind=real_type), dimension(:), allocatable :: f, d
    integer, dimension(:), allocatable :: orig
!______________________________________________________________________

    n_1 = size(assoc_data_1)
    n_2 = size(assoc_data_2)

    if (any((/present(f_1), present(ln_oto_Lh)/)) .and. &
         .not.(all((/present(f_1), present(ln_oto_Lh)/)))) then
       write(*,"(a)") "Subroutine `compute_ln_oto_Lh`: parameters `f_1` &
            &and `ln_oto_Lh` must be present or absent jointly."
       stop
    endif
    if (any((/present(n_grid), present(f_grid), present(ln_oto_Lh_grid)/)) &
         .and. .not.(all((/present(n_grid), present(f_grid), &
         present(ln_oto_Lh_grid)/)))) then
       write(*,"(a)") "Subroutine `compute_ln_oto_Lh`: parameters `n_grid`, &
            &`f_grid` and `ln_oto_Lh_grid` must be present or absent jointly."
       stop
    endif
    if (.not.any((/present(f_1), present(ln_oto_Lh), present(n_grid), &
         present(f_grid), present(ln_oto_Lh_grid)/))) then
       write(*,"(a)") "Subroutine `compute_ln_oto_Lh`: at least one of the &
            &sets of parameters {`f_1`, `ln_oto_Lh`} and {`n_grid`, `f_grid`, &
            &`ln_oto_Lh_grid`} should be present. No calculations done."
       return
    endif

! Allocate grids. A temporary grid is created to compute `ln_oto_Lh` at `f_1`
! in case no grid (`f_grid`, `ln_oto_Lh_grid`) is requested.
    if (present(n_grid)) then
       if (n_grid < 1) then
          write(*,"(a,i0,a)") "Subroutine `compute_ln_oto_Lh`: `n_grid` &
               &should be greater than or equal to 1. Stopped."
          stop
       else if (n_grid == 1) then
          n_grid_tmp = 2
       else
          n_grid_tmp = n_grid
       endif
    else
       n_grid_tmp = 2
    endif
    allocate(f_grid_tmp(n_grid_tmp))
    allocate(d_grid_tmp(n_grid_tmp))
    allocate(ln_oto_Lh_grid_tmp(n_grid_tmp))

    if (present(f_grid)) then
       if (associated(f_grid)) then
          deallocate(f_grid)
       endif
       allocate(f_grid(n_grid))
       if (associated(ln_oto_Lh_grid)) then
          deallocate(ln_oto_Lh_grid)
       endif
       allocate(ln_oto_Lh_grid(n_grid))
    endif

! Maximal value of `f_1`.
    f_max = min(1., real(n_2)/n_1)
    if (present(f_1)) then
       if (f_1 > f_max) then
          write(*,"(a,f8.5)") "Subroutine `compute_ln_oto_Lh`: &
               &`f_1` must be less than ", f_max
          stop
       endif
    endif
    f_max = f_max - f_max_shift ! For numerical reasons.

! Value of `drv_ln_oto_Lh` at `f` = 0.
    f_grid_tmp(1) = 0
    d_grid_tmp(1) = 0
    do i = 1, n_1
       d_grid_tmp(1) = d_grid_tmp(1) + &
            assoc_data_1(i)%sum_xi_main/assoc_data_1(i)%xi_main(0)
    enddo
    d_grid_tmp(1) = -n_1 + d_grid_tmp(1)/n_2

!  Value of `drv_ln_oto_Lh` at `f` > 0.
    do i_grid = 2, n_grid_tmp
       f_grid_tmp(i_grid) = (i_grid-1._real_type)/(n_grid_tmp-1)*f_max
       d_grid_tmp(i_grid) = drv_ln_oto_Lh(assoc_data_1, n_2, f_grid_tmp(i_grid))
    enddo

! Create a linked list `point` containing `f_grid_tmp`, `d_grid_tmp`.
    n_points = n_grid_tmp
    allocate(point)
    point%f = f_grid_tmp(1)
    point%d = d_grid_tmp(1)
    point%orig = 1
    current_point => point
    do i_grid = 2, n_grid_tmp
       if (present(f_1)) then
          if (f_1 > f_grid_tmp(i_grid-1) .and. f_1 <= f_grid_tmp(i_grid)) then
             allocate(current_point%ptr)
             current_point => current_point%ptr
             nullify(current_point%ptr)
             current_point%f = f_1
! `current_point%d` should be very close to 0 for `f_1` = `oto_f_1`.
! Not set to 0 for numerical reasons and in case `f_1` /= `oto_f_1`.
             current_point%d = drv_ln_oto_Lh(assoc_data_1, n_2, current_point%f)
             current_point%orig = -1
             n_points = n_points + 1
          endif
       endif
       allocate(current_point%ptr)
       current_point => current_point%ptr
       nullify(current_point%ptr)
       current_point%f = f_grid_tmp(i_grid)
       current_point%d = d_grid_tmp(i_grid)
       current_point%orig = i_grid
    enddo

    if (allocated(d_grid_tmp)) deallocate(d_grid_tmp)

! Value of `ln_oto_Lh` at `f` = 0.
    ln_Lh0 = 0
    do i = 1, n_1
       ln_Lh0 = ln_Lh0 + log(assoc_data_1(i)%xi_main(0))
    enddo
    do i = 1, n_2
       ln_Lh0 = ln_Lh0 + log(assoc_data_2(i)%xi_main(0))
    enddo

! Refine linked list `point` to compute `ln_oto_Lh`.
! The `orig` key is 0 for new points.

    current_point => point
    do i = 1, n_points-1
       next_point => current_point%ptr
       call refine_grid(n_points, current_point, n_1, n_2, assoc_data_1)
       current_point => next_point
    enddo

! Create arrays from the linked list. (More convenient for what follows.)
    allocate(f(n_points))
    allocate(d(n_points))
    allocate(orig(n_points))
    current_point => point
    do i = 1, n_points
       if (associated(current_point)) then
          f(i) = current_point%f
          d(i) = current_point%d
          orig(i) = current_point%orig
       else
          exit
       endif
       current_point => current_point%ptr
    enddo

! Integrate `d` with respect to `f` to get `ln_oto_Lh`.
    ln_oto_Lh_grid_tmp(1) = ln_Lh0
    i_grid = 1
    do i = 1, n_points-1
       if (orig(i) == i_grid) then
          ln_oto_Lh_grid_tmp(i_grid+1) = ln_oto_Lh_grid_tmp(i_grid)
          i_grid = i_grid + 1
       else if (orig(i) == -1) then
          if (present(ln_oto_Lh)) then
             ln_oto_Lh = ln_oto_Lh_grid_tmp(i_grid)
          endif
       endif
       integr = Steffen_integral(n_points, f, d, i) ! Smooth integration
!& from `f(i)` to `f(i+1)`.
       ln_oto_Lh_grid_tmp(i_grid) = ln_oto_Lh_grid_tmp(i_grid) + integr
    enddo
    if (present(f_grid)) then
       if (n_grid > 1) then
          f_grid = f_grid_tmp
          ln_oto_Lh_grid = ln_oto_Lh_grid_tmp
       else ! n_grid = 1.
          f_grid(1) = f_grid_tmp(n_grid_tmp)
          ln_oto_Lh_grid(1) = ln_oto_Lh_grid_tmp(n_grid_tmp)
       endif
    endif

    deallocate(f, d, orig)
    deallocate(f_grid_tmp, ln_oto_Lh_grid_tmp)

! Deallocate `point`.
    current_point => point
    do
       if (.not.associated(current_point)) exit
       next_point => current_point%ptr
       deallocate(current_point)
       current_point => next_point
    enddo

  end subroutine compute_ln_oto_Lh

!======================================================================

  function func_ln_oto_Lh_alt(assoc_data_1, assoc_data_2, f_1)

! Very inefficient. Use only for tests.

    implicit none
    real(kind=real_type) :: func_ln_oto_Lh_alt
    real(kind=real_type), intent(in) :: f_1
    type(assoc_struct), dimension(:), intent(in) :: assoc_data_1, assoc_data_2
!......................................................................
    real(kind=real_type) :: f_2
    integer :: i, n_1, n_2
    real(kind=real_type), dimension(size(assoc_data_1)) :: prob0
!______________________________________________________________________

    write(*,"(a)") "`func_ln_oto_Lh_alt` is very inefficient. &
         &Use it only for tests."

    n_1 = size(assoc_data_1)
    n_2 = size(assoc_data_2)
    f_2 = (f_1*n_1)/n_2
    func_ln_oto_Lh_alt = 0
    do i = 1, n_2
       func_ln_oto_Lh_alt = func_ln_oto_Lh_alt + log(assoc_data_2(i)%xi_main(0))
    enddo
    do i = 1, n_1
       call compute_oto_prob(assoc_data_1, n_2, f_1, &
            prob0 = prob0, i_min = i)
       if (n_1 <= n_2) then
          func_ln_oto_Lh_alt = func_ln_oto_Lh_alt + &
               log((1-f_1)*assoc_data_1(i)%xi_any(1,0)/prob0(i))
       else
          func_ln_oto_Lh_alt = func_ln_oto_Lh_alt + &
               log((1-f_2)*assoc_data_1(i)%xi_any(1,0)/prob0(i))
       endif
    enddo
    if (n_1 > n_2) func_ln_oto_Lh_alt = func_ln_oto_Lh_alt + &
         (n_2-n_1)*log(1-f_2)

  end function func_ln_oto_Lh_alt

end module mod_Aspects
