/*-----------------------------------------------------------------*/
/*! 
  \file arome.c 
  \brief high-level API 
         Computes and return analytical Rossiter-McLaughlin signals
         adapted the CCF or the iodine cell technique.

  \author  G. Boue
           EXOEarths, Centro de Astrofisica, Universidade do Porto.

   Copyright (C) 2012, CAUP
   email of the author : gwenael.boue@astro.up.pt

   This work has been supported by the European Research Council/European
   Community under the FP7 through Starting Grant agreement number 239953, as
   well as from Fundacao para a Ciencia e a Tecnologia (FCT) through program
   Ciencia 2007 funded by FCT/MCTES (Portugal) and POPH/FSE (EC), and in the
   form of grants reference PTDC/CTE-AST/098528/2008, SFRH/BPD/71230/2010, and
   SFRH/BPD/81084/2011.

*/
/*-----------------------------------------------------------------*/

/*----------------------------------------------------------------*/
/* License of the file :

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*----------------------------------------------------------------*/
#include "aromeconfig.h"
#if HAVE_MATH_H
#include <math.h>
#endif
#if HAVE_FLOAT_H
#include <float.h>
#endif
#if HAVE_STDARG_H
#include <stdarg.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_STDIO_H
#include <stdio.h>
#endif
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include "arome.h"

#ifndef SQR
#define SQR(x) (powi(x,2))
#endif /* SQR */

#ifndef MAX
#define MAX(x,y) ((x)>(y)?(x):(y))
#endif /* MAX */

#ifndef arome_error_int
#define arome_error_int(...) {arome_print_error(__VA_ARGS__); return 1;}
#endif

#ifndef arome_error_double
#define arome_error_double(...) {arome_print_error(__VA_ARGS__); return NAN;}
#endif

#ifndef arome_error_ptr
#define arome_error_ptr(...) {arome_print_error(__VA_ARGS__); return NULL;}
#endif

#ifndef arome_error_void
#define arome_error_void(...) {arome_print_error(__VA_ARGS__); return;}
#endif

struct t_arome_sglobal
{
  void (*userfuncerror)(const char *); /*!< pointer to error function */
  int usertypeerror;    /*!< error type  
                           _arome_iContinue : prints the message and the program continues (default)
                           _arome_iExit : prints the message and the program exits (call exit(1))  
                           _arome_iUserfunc : call the C or F2003 function userfunc and the program continues 
                           _arome_iUserFunc90 : call the F90 function userfunc and the program continues */

} arome_sglobal = 
{ NULL, _arome_iNULL};

/***********************/
/* local routines      */
/***********************/
inline double powi(const double m, int n) {double res=n>0?m:1; for(;n>1;n--) {res*=m;} return res;}

static int    solve_kepler_ME(double M, double ecc, double *E);
static int    solve_kepler_LF(double L, double k, double h, double *F);
static double arome_get_orbit_transit_time_eo(t_orbit * const orbit);
static double arome_get_orbit_transit_time_kh(t_orbit * const orbit);
static int    arome_get_orbit_xyz_eo(t_orbit * const orbit, double t, double *x, double *y, double *z);
static int    arome_get_orbit_xyz_kh(t_orbit * const orbit, double t, double *x, double *y, double *z);

static void   arome_print_error(const char *format, ...);
static double funcF00(double x0, double y0, double rx, double ry, double phi);
static double funcF10(double x0, double y0, double rx, double ry, double phi);
static double funcF01(double x0, double y0, double rx, double ry, double phi);
static double funcF20(double x0, double y0, double rx, double ry, double phi);
static double funcF02(double x0, double y0, double rx, double ry, double phi);
static double funcF11(double x0, double y0, double rx, double ry, double phi);

static double funcLimb(double x0, double y0, void *pdata);
static void   dfuncLimb(double x0, double y0, void *pdata, double *Jx, double *Jy);
static void   ddfuncLimb(double x0, double y0, void *pdata, double *Hxx, double *Hyy, double *Hxy);

static double funcIxn(double x0, double y0, void *pdata, int n);
static void   HessIxn(double x0, double y0, void *pdata, int n, double *Hxx, double *Hyy, double *Hxy);

static void   setdefaultparam(t_arome * const parome);
static int    dGconvR(double sig, double v, t_arome * const parome, double *res);
static int    setrotkernel(t_arome * const parome);
static double funcAmp_a0(double x, void *pdata);
static int    setGaussfit_a0(t_arome * const parome);
static double funcIodine_den(double x, void *pdata);
static int    setIodine_den(t_arome * const parome);

static double hypertrunc(double a, double b, double c, double x, int kmax);

static double trapzd(double (*func)(double, void*), double a, double b, void *pdata, int n);
static double qtrap(double (*func)(double, void *), double a, double b, void *pdata);
static double gammln(float xx);
static int    gamma_inc(double a, double x, t_arome * const parome, double *res);
static void   gser(double *gamser, double a, double x, const void * const pdata, double *gln);
static void   gcf(double *gammcf, double a, double x, const void * const pdata, double *gln);







/****************************/
/* orbit routines           */
/****************************/
t_orbit * arome_new_orbit()
/* allocate an orbit structure
*/
{
  t_orbit *orbit;

  orbit = (t_orbit*)malloc(sizeof(t_orbit));
  if (orbit==NULL)
    arome_error_ptr("Can't allocate memory for %lu bytes in function arome_new_orbit.", sizeof(t_orbit));
   
  orbit->isset = 0;
  return orbit;
}


int arome_set_orbit_eo(double per, double sma, double ecc, double om, 
                       double inc, double lambda, t_orbit * const orbit)
/* set the parameters of the orbit
 
   @param per    (in) orbital period in days
   @param sma    (in) semi-major axis in stellar radius
   @param ecc    (in) eccentricity
   @param om     (in) argument of periastron in deg
   @param inc    (in) inclination in deg
   @param lambda (in) projected spin-orbit angle in deg
   @param orbit  (inout) orbit structure
*/
{
  const double deg_to_rad = acos(-1.E0)/180.E0;
  
  if (!orbit)
     arome_error_int("orbit is not yet allocated in function arome_set_orbit_eo.");
  
  orbit->per    = per;
  orbit->sma    = sma;
  orbit->ecc    = ecc;
  orbit->om     = om*deg_to_rad;
  orbit->k      = orbit->ecc*cos(orbit->om);
  orbit->h      = orbit->ecc*sin(orbit->om);
  orbit->inc    = inc*deg_to_rad;
  orbit->lambda = lambda*deg_to_rad;
  orbit->type   = _arome_iEO;
  orbit->isset  = 1;
  
  return 0;
}


int arome_set_orbit_kh(double per, double sma, double k, double h, double inc,
                       double lambda, t_orbit * const orbit)
/* set the parameters of the orbit
 
   @param per    (in) orbital period in days
   @param sma    (in) semi-major axis in stellar radius
   @param k      (in) e*cos(om)
   @param h      (in) e*sin(om)
   @param inc    (in) inclination in deg
   @param lambda (in) projected spin-orbit angle in deg
   @param orbit  (inout) orbit structure
*/
{
  const double deg_to_rad = acos(-1.E0)/180.E0;
  
  if (!orbit)
     arome_error_int("orbit is not yet allocated in function arome_set_orbit_kh.");
  
  orbit->per    = per;
  orbit->sma    = sma;
  orbit->k      = k;
  orbit->h      = h;
  orbit->ecc    = sqrt(k*k+h*h);
  orbit->om     = atan2(h,k);
  orbit->inc    = inc*deg_to_rad;
  orbit->lambda = lambda*deg_to_rad;
  orbit->type   = _arome_iKH;
  orbit->isset  = 1;
  
  return 0;
}


double arome_get_orbit_transit_time(t_orbit * const orbit)
/* computes and return the time of transit assuming that the 
   
   @param orbit  (in)  orbit structure
*/
{
   if (!orbit)
      arome_error_double("orbit is not yet allocated in function arome_get_orbit_transit_time.");
  
   if (!orbit->isset)
      arome_error_double("orbit parameters are not set in arome_get_orbit_transit_time.");
  
   switch(orbit->type)
   {
    case _arome_iEO:
     return arome_get_orbit_transit_time_eo(orbit);
     break;
    case _arome_iKH:
     return arome_get_orbit_transit_time_kh(orbit);
     break;
    default:
     arome_error_double("unknown orbit type (%d) in function arome_get_transit_time.\n", orbit->type);
     break;
   }
   
   return NAN;
}


double arome_get_orbit_transit_time_eo(t_orbit * const orbit)
/* computes and return the time of transit assuming that the 
   passage at the periastron is t0=0
   
   @param orbit  (in)  orbit structure
*/
{
   const double pi=acos(-1.E0);
   double true_anom;
   double ecc_anom;
   
   true_anom = pi/2.E0 - orbit->om;
   ecc_anom = 2.E0*atan(sqrt((1.E0-orbit->ecc)/(1.E0+orbit->ecc))*tan(true_anom/2.E0));
   return orbit->per/(2.E0*pi)*(ecc_anom - orbit->ecc*sin(ecc_anom));
}


double arome_get_orbit_transit_time_kh(t_orbit * const orbit)
/* computes and return the time of transit assuming that the 
   passage at the ascending node is t0=0
   
   @param orbit  (in)  orbit structure
*/
{
   const double pi=acos(-1.E0);
   double ecc_anom0;
   double mean_anom0;
   double true_anom;
   double ecc_anom;
   double mean_anom;
   
   ecc_anom0 = -2.E0*atan(sqrt((1.E0-orbit->ecc)/(1.E0+orbit->ecc))*tan(orbit->om/2.E0));
   mean_anom0 = ecc_anom0 - orbit->ecc*sin(ecc_anom0);
   
   true_anom = pi/2.E0 - orbit->om;
   ecc_anom = 2.E0*atan(sqrt((1.E0-orbit->ecc)/(1.E0+orbit->ecc))*tan(true_anom/2.E0));
   mean_anom = ecc_anom - orbit->ecc*sin(ecc_anom);
   return orbit->per/(2.E0*pi)*(mean_anom-mean_anom0);
}


int arome_get_orbit_xyz(t_orbit * const orbit, double t, double *x, double *y, double *z)
/* computes the planet coordinates on a given orbit at the time t
 
   @param orbit (in)  orbit structure
   @param t     (in)  date in day
   @param x     (out) planet x-coordinate
   @param y     (out) planet y-coordinate
   @param z     (out) planet z-coordinate
*/
{
  int status = 0;
  
  if (!orbit)
     arome_error_int("orbit is not yet allocated in function arome_get_orbit_xyz.");
  
  if (!orbit->isset)
     arome_error_int("orbit parameters are not set in arome_get_orbit_xyz.");
  
  switch(orbit->type)
  {
   case _arome_iEO:
    status = arome_get_orbit_xyz_eo(orbit, t, x, y, z);
    break;
   case _arome_iKH:
    status = arome_get_orbit_xyz_kh(orbit, t, x, y, z);
    break;
   default:
    arome_error_int("unknown orbit type (%d) in function arome_get_orbit_xyz.\n", orbit->type);
  }
  
  return status;
}


int arome_get_orbit_xyz_eo(t_orbit * const orbit, double t, double *x, double *y, double *z)
/* computes the planet coordinates on a given orbit at the time t
   assuming that the passage at the periastron is at t=0
 
   @param orbit (in)  orbit structure
   @param t     (in)  date in day
   @param x     (out) planet x-coordinate
   @param y     (out) planet y-coordinate
   @param z     (out) planet z-coordinate
*/
{
  const double pi = acos(-1.E0);
  double mean_anom;
  double ecc_anom;
  double true_anom;
  double radius_vec;
  double true_lat;
  double node;
  int status = 0;
  
  mean_anom = 2.E0*pi/orbit->per*t;
  status += solve_kepler_ME(mean_anom, orbit->ecc, &ecc_anom);
  true_anom = 2.E0*atan(sqrt((1.E0+orbit->ecc)/(1.E0-orbit->ecc))*tan(ecc_anom/2.E0));
  radius_vec = orbit->sma*(1.E0-SQR(orbit->ecc))/(1.E0+orbit->ecc*cos(true_anom));
  node = pi+orbit->lambda;
  true_lat = orbit->om + true_anom;
  *x = radius_vec*(cos(node)*cos(true_lat)-sin(node)*sin(true_lat)*cos(orbit->inc));
  *y = radius_vec*(sin(node)*cos(true_lat)+cos(node)*sin(true_lat)*cos(orbit->inc));
  *z = radius_vec*sin(true_lat)*sin(orbit->inc);
  
  return status;
}


int arome_get_orbit_xyz_kh(t_orbit * const orbit, double t, double *x, double *y, double *z)
/* computes the planet coordinates on a given orbit at the time t
   assuming that the passage at the ascending node is at t=0
 
   @param orbit (in)  orbit structure
   @param t     (in)  date in day
   @param x     (out) planet x-coordinate
   @param y     (out) planet y-coordinate
   @param z     (out) planet z-coordinate
*/
{
  const double pi = acos(-1.E0);
  double ecc_anom0;
  double mean_lat0;
  double mean_lat;
  double ecc_lat;
  double x1, y1, den1, num1;
  double node;
  int status = 0;
  
  ecc_anom0 = -2.E0*atan(sqrt((1.E0-orbit->ecc)/(1.E0+orbit->ecc))*tan(orbit->om/2.E0));
  mean_lat0 = orbit->om + ecc_anom0 - orbit->ecc*sin(ecc_anom0);
  
  mean_lat = 2.E0*pi/orbit->per*t + mean_lat0;
  status += solve_kepler_LF(mean_lat, orbit->k, orbit->h, &ecc_lat);
  den1 = 1.E0+sqrt(1.E0-orbit->k*orbit->k-orbit->h*orbit->h);
  num1 = orbit->k*sin(ecc_lat)-orbit->h*cos(ecc_lat);
  x1 = cos(ecc_lat)+orbit->h*num1/den1-orbit->k;
  y1 = sin(ecc_lat)-orbit->k*num1/den1-orbit->h;
  node = pi+orbit->lambda;
  *x = orbit->sma*(cos(node)*x1-sin(node)*cos(orbit->inc)*y1);
  *y = orbit->sma*(sin(node)*x1+cos(node)*cos(orbit->inc)*y1);
  *z = orbit->sma*sin(orbit->inc)*y1;
  
  return status;
}


int arome_mget_orbit_xyz(t_orbit * const orbit, double *t, int n, double *x, double *y, double *z)
/* computes the planet coordinates on a given orbit at the times t
   assuming that the passage at the periastron is at t=0
 
   @param orbit (in)  orbit structure
   @param t     (in)  array of dates in day
   @param n     (in)  number of dates
   @param x     (out) array of planet x-coordinate
   @param y     (out) array of planet y-coordinate
   @param z     (out) array of planet z-coordinate
*/
{
  int status = 0;
  
  PRAGMA_IVDEP
  for(int k=0; k<n; k++)
    status += arome_get_orbit_xyz(orbit, t[k], &(x[k]), &(y[k]), &(z[k]));

  return status;
}


void arome_free_orbit(t_orbit * const orbit)
/* release memory allocated with arome_new_orbit
 
   @param orbit (in) orbit structure
*/
{
  free(orbit);
}


int solve_kepler_ME(double M, double ecc, double *E)
/* return the eccentric anomaly computed from the mean anomaly
 
   @param M   (in)  mean anomaly (rad)
   @param ecc (in)  eccentricity
   @param E   (out) eccentric anomaly (rad)
*/
{
   double ERRTOL = 1.E-10;
   double diff;
   double res;
   int niter;
   int niter_max = 10;
   
   res = M + ecc*sin(M)/(1.E0-ecc*cos(M));
   niter = 0;
   do 
   {
     diff = ( res - M - ecc*sin(res) ) / ( 1.E0 - ecc*cos(res) );
     res -= diff;
     niter++;
   } while (fabs(diff) > ERRTOL && niter < niter_max);
   
   if (fabs(diff) > ERRTOL)
      arome_error_int("could not solve the Kepler Eq. for M = %.15E and ecc = %.15E"
                      " (error = %.15E) in solve_kepler_ME. You may want to increase niter_max.", 
                      M, ecc, diff);
   
   *E = res;
   return 0;
}


int solve_kepler_LF(double L, double k, double h, double *F)
/* return the eccentric latitude computed from the mean latitude
 
   @param L   (in)  mean latitude om+M (rad)
   @param k   (in)  e*cos(om)
   @param h   (in)  e*sin(om)
   @param F   (out) eccentric latitude om+E (rad)
*/
{
   double ERRTOL = 1.E-10;
   double diff;
   double res;
   int niter;
   int niter_max = 10;
   
   res = L + (k*sin(L)-h*cos(L))/(1.E0-k*cos(L)-h*sin(L));
   niter = 0;
   do 
   {
     diff = ( res - L - k*sin(res) + h*cos(res) ) / ( 1.E0 - k*cos(res) - h*sin(res) );
     res -= diff;
     niter++;
   } while (fabs(diff) > ERRTOL && niter < niter_max);
   
   if (fabs(diff) > ERRTOL)
      arome_error_int("could not solve the Kepler Eq. for L = %.15E, k = %.15E, and h = %.15E"
                      " (error = %.15E) in solve_kepler_LF. You may want to increase niter_max.", 
                      L, k, h, diff);
   
   *F = res;
   return 0;
}










/****************************/
/* ARoME routines           */
/****************************/
void setdefaultparam(t_arome * const parome)
/* set default parameters to parome
 
   @param parome (in) simulation structure
*/
{
  parome->planet_flag = 0;
  parome->lineprofile_flag = 0;
  parome->fvpbetap_flag = 0;
  parome->RM_init_flag = 0;
  parome->malloc_flag = 0;
  
  /* Numerical Recipes parameters */
  parome->gcf_ITMAX  = 100;
  parome->gcf_EPS    = 3.0e-7;
  parome->gcf_FPMIN  = 1.0e-30;
  parome->gser_ITMAX = 100;
  parome->gser_EPS   = 3.0e-7;
  parome->qtrap_EPS  = 1.0e-5;
  parome->qtrap_JMAX = 20;
}

t_arome * arome_alloc_quad(double u1, double u2)
/* allocate an arome structure for quadratic limb-darkening
   
   @param u1 (in) 1st quadratic limb-darkening
   @param u2 (in) 2nd quadratic limb-darkening
*/
{
  t_arome *parome;
  const double denom = M_PI*(1.0-u1/3.0-u2/6.0);
  int status = 0;

  parome = (t_arome*)malloc(sizeof(t_arome));
  if (parome==NULL)
    arome_error_ptr("Can't allocate memory for %lu bytes in function arome_alloc_quad.", sizeof(t_arome));
  
  parome->type_limb = _arome_iQuad;
  setdefaultparam(parome);
  
  parome->nlimb = 3;
  if (!(parome->limb_coef = (double *)malloc(parome->nlimb*sizeof(double))))
     arome_error_ptr("Can't allocate %d doubles in function arome_alloc_quad.", parome->nlimb);
  if (!(parome->limb_pow  = (int *)malloc(parome->nlimb*sizeof(int))))
  {
     free(parome->limb_coef);
     arome_error_ptr("Can't allocate %d integers in function arome_alloc_quad.", parome->nlimb);
  }
  if (!(parome->kern_coef = (double *)malloc(parome->nlimb*sizeof(double))))
  {
     free(parome->limb_coef);
     free(parome->limb_pow);
     arome_error_ptr("Can't allocate %d doubles in function arome_alloc_quad.", parome->nlimb);
  }
  
  parome->limb_coef[0] = (1.0-u1-u2)/denom;
  parome->limb_coef[1] = (u1+2.0*u2)/denom;
  parome->limb_coef[2] = -u2/denom;
  parome->limb_pow[0]  = 0;
  parome->limb_pow[1]  = 2;
  parome->limb_pow[2]  = 4;
  
  status = setrotkernel(parome);
  if (status) 
  {
    arome_free(parome);
    parome = NULL;
  }
       
  return parome;
}



t_arome * arome_alloc_nl(double c1, double c2, double c3, double c4)
/* allocate an arome structure for non-linear limb-darkening
   
   @param c1 (in) 1st nonlinear limb-darkening
   @param c2 (in) 2nd nonlinear limb-darkening
   @param c3 (in) 3st nonlinear limb-darkening
   @param c4 (in) 4nd nonlinear limb-darkening
*/
{
  t_arome *parome;
  const double denom = M_PI*(1.0-c1/5.0-c2/3.0-3.0*c3/7.0-c4/2.0);
  int status = 0;

  parome = (t_arome*)malloc(sizeof(t_arome));
  if (parome==NULL)
    arome_error_ptr("Can't allocate memory for %lu bytes in function arome_alloc_nl.", sizeof(t_arome));
  
  parome->type_limb = _arome_iNL;
  setdefaultparam(parome);
  
  parome->nlimb = 5;
  if (!(parome->limb_coef = (double *)malloc(parome->nlimb*sizeof(double))))
     arome_error_ptr("Can't allocate %d doubles in function arome_alloc_nl.", parome->nlimb);
  if (!(parome->limb_pow  = (int *)malloc(parome->nlimb*sizeof(int))))
  {
     free(parome->limb_coef);
     arome_error_ptr("Can't allocate %d integers in function arome_alloc_nl.", parome->nlimb);
  }
  if (!(parome->kern_coef = (double *)malloc(parome->nlimb*sizeof(double))))
  {
     free(parome->limb_coef);
     free(parome->limb_pow);
     arome_error_ptr("Can't allocate %d doubles in function arome_alloc_nl.", parome->nlimb);
  }
  
  parome->limb_coef[0] = (1.0-c1-c2-c3-c4)/denom;
  parome->limb_coef[1] = c1/denom;
  parome->limb_coef[2] = c2/denom;
  parome->limb_coef[3] = c3/denom;
  parome->limb_coef[4] = c4/denom;
  parome->limb_pow[0]  = 0;
  parome->limb_pow[1]  = 1;
  parome->limb_pow[2]  = 2;
  parome->limb_pow[3]  = 3;
  parome->limb_pow[4]  = 4;
  
  status = setrotkernel(parome);
  if (status) 
  {
    arome_free(parome);
    parome = NULL;
  }
       
  return parome;
}


void arome_free(t_arome * parome)
/* destructor */
{
  if (parome)
  {
    free(parome->limb_coef);
    free(parome->limb_pow);
    free(parome->kern_coef);
    
    if (parome->malloc_flag)
      arome_mfree(parome);
    
    free(parome);
  }
}


int arome_reset_quad(double u1, double u2, t_arome * const parome)
/* re-initialize the quadratic coefficients
   returns 0 if no errors
   
   @param u1 (in) 1st quadratic limb-darkening
   @param u2 (in) 2nd quadratic limb-darkening
*/
{
  double denom = M_PI*(1.0-u1/3.0-u2/6.0);
  int status = 0;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_reset_quad.");
  
  if (parome->type_limb != _arome_iQuad)
     arome_error_int("\"arome_reset_quad\" can only be used after \"arome_alloc_quad\"");
  
  parome->RM_init_flag = 0;
  
  parome->limb_coef[0] = (1.0-u1-u2)/denom;
  parome->limb_coef[1] = (u1+2.0*u2)/denom;
  parome->limb_coef[2] = -u2/denom;
  
  status = setrotkernel(parome);
  
  return status;
}


int arome_reset_nl(double c1, double c2, double c3, double c4, t_arome * const parome)
/* re-initialize the nonlinear coefficients
   returns 0 if no errors
   
   @param c1 (in) 1st nonlinear limb-darkening
   @param c2 (in) 2nd nonlinear limb-darkening
   @param c3 (in) 3st nonlinear limb-darkening
   @param c4 (in) 4nd nonlinear limb-darkening
*/
{
  double denom = M_PI*(1.0-c1/5.0-c2/3.0-3.0*c3/7.0-c4/2.0);
  int status = 0;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_reset_nl.");
  
  if (parome->type_limb != _arome_iNL)
     arome_error_int("\"arome_reset_nl\" can only be used after \"arome_alloc_nl\"");
  
  parome->RM_init_flag = 0;
  
  parome->limb_coef[0] = (1.0-c1-c2-c3-c4)/denom;
  parome->limb_coef[1] = c1/denom;
  parome->limb_coef[2] = c2/denom;
  parome->limb_coef[3] = c3/denom;
  parome->limb_coef[4] = c4/denom;
  
  status = setrotkernel(parome);
  
  return status;
}


int arome_set_lineprofile(double beta0, double Vsini, double sigma0, double zeta, int Kmax, t_arome * const parome)
/* set the parameters of the line profile
   return 0 if no error
   
   @param beta0  (in) width of the non-rotating star (km/s)
   @param Vsini  (in) projected velocity of the star (km/s)
   @param sigma0 (in) width of the best Gaussian fit (km/s) only used by arome_get_RM_CCF
   @param zeta   (in) macro-turbulence parameter (km/s)
   @param Kmax   (in) order of expansion of the iodine cell technique
   @param parome (inout) simulation structure
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_lineprofile.");
  
   parome->beta0  = beta0;
   parome->Vsini  = Vsini;
   parome->sigma0 = sigma0;
   parome->zeta   = zeta;
   parome->rotational_Kmax  = Kmax;
   parome->lineprofile_flag = 1;
   
   return 0;
}

int arome_set_planet(double Rp, t_arome * const parome)
/* set the planet parameters
   return 0 if no error
   
   @param Rp (in) radius of the planet in stellar radius
   @param parome (inout) simulation structure
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_planet.");
  
   parome->Rp = Rp;
   parome->planet_flag = 1;
   
   return 0;
}

int arome_init_CCF(t_arome * const parome)
/* initialise constants for RM signals computed by a Gaussian fit to the CCF
   return 0 if no error
   
   @param parome (inout) simulation structure
*/
{
  int status = 0;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_init_CCF.");
  
  if (!parome->lineprofile_flag)
    arome_error_int("in function \"arome_init_CCF\", parameters of the line profile are not set.");
  
  status = setGaussfit_a0(parome);
  
  if (!status)
    parome->RM_init_flag |= _arome_iCCF;
  
  return status;
}

int arome_init_iodine(t_arome * const parome)
/* initialise constants for RM signals computed by the iodine cell technique
   return 0 if no error
   
   @param parome (inout) simulation structure
*/
{
  int status = 0;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_init_iodine.");
  
  if (!parome->lineprofile_flag)
    arome_error_int("in function \"arome_init_iodine\", parameters of the line profile are not set.");
  
  status = setIodine_den(parome);
  
  if (!status)
    parome->RM_init_flag |= _arome_iIodine;
  
  return status;
}

int arome_calc_fvpbetap(double x, double y, double z, t_arome * const parome)
/* Computes the flux f, the subplanet velocity vp, and dispersions betapR, betapT
   return 0 if no error
   
   @param x      (in) x coordinate of the planet in stellar radius
   @param y      (in) y coordinate of the planet in stellar radius
   @param z      (in) z coordinate of the planet in stellar radius
   @param parome (inout) simulation structure
*/
{
  const double EPSILON=1.e-10;
  double       r     ;    // largest radius of the planet
  double       rx    ;    // x radius of the rotated projected planet
  double       ry    ;    // y radius of the rotated projected planet
  double       phi0  ;    // atan2(x0, y0)
  double       rho   ;    // sqrt(x0**2+y0**2)
  double       psi   ;    // angle OPA (O=star, P=planet, A=intersection star/planet)
  double       phi1  ;    // limit inf of an arc
  double       phi2  ;    // limit sup of an arc
  double       xbar  ;    // averaged x coordinate of the planet
  double       ybar  ;    // averaged y coordinate of the planet
  double       II    ;    // II = I(xbar, ybar) (mean subplanet intensity)
  double       Hxx   ;    // partial_x partial_x I(x,y)
  double       Hyy   ;    // partial_y partial_y I(x,y)
  double       Hxy   ;    // partial_x partial_y I(x,y)
  double       Hxx2  ;    // partial_x partial_x I(x,y)
  double       Hyy2  ;    // partial_y partial_y I(x,y)
  double       Hxy2  ;    // partial_x partial_y I(x,y)
  double       a00   ;    // area covered by the planet
  double       axx   ;    // averaged of (x-xbar)**2
  double       ayy   ;    // averaged of (y-ybar)**2
  double       axy   ;    // averaged of (x-xbar)(y-ybar)
  double       ff    ;    // fraction of flux oculted by the planet
  double       vv    ;    // subplanet velocity
  double       v2    ;    // square of the subplanet velocity
  double       dmin  ;    // minimal distance between the planet ellipse and the origin
  double       dmax  ;    // maximal distance between the planet ellipse and the origin
  double       dbetaR;    // dispersion in the radial direction
  double       dbetaT;    // dispersion in the radial direction
  double       zetaR2;    // radial dispersion due to macro-turbulence
  double       zetaT2;    // tangential dispersion due to macro-turbulence
  
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_calc_fvpbetap.");
  
  if (!parome->planet_flag)
    arome_error_int("the planet is not initialized in function arome_calc_fvpbetap.");
  if (!parome->lineprofile_flag)
    arome_error_int("the line profile is not initialized in function arome_calc_fvpbetap.");
  
  
  if (z <= 0.0)
  /* the planet is behind the star */
  {
    parome->flux   = 0.0;
    parome->vp     = 0.0;
    parome->betapR = parome->beta0;
    parome->betapT = parome->beta0;
    
    parome->fvpbetap_flag |= _arome_iflux | _arome_ivp | _arome_ibetapR | _arome_ibetapT;
    
    return 0;
  }
  
  /* parameters of the planet */
  rx = ry = r = parome->Rp;
  phi0 = atan2(y, x);
  rho  = sqrt(x*x+y*y);
  dmin = rho-r;
  dmax = rho+r;
  
  if (dmin>=1.0-EPSILON)
  /* the planet doesn't overlap the stellar disk */
  {
    parome->flux   = 0.0;
    parome->vp     = 0.0;
    parome->betapR = parome->beta0;
    parome->betapT = parome->beta0;
    
    parome->fvpbetap_flag |= _arome_iflux | _arome_ivp | _arome_ibetapR | _arome_ibetapT;
  
    return 0;
  }
  
  if (dmax<=1.0)
  /* the planet is completely inside the stellar disk */
  {
    xbar=x;                // int x dxdy / int dxdy
    ybar=y;                // int y dxdy / int dxdy
    
    a00 = M_PI*rx*ry;      // int dxdy
    axx = rx*rx/4.0;       // int (x-x0)**2 dxdy / int dxdy
    ayy = ry*ry/4.0;       // int (y-y0)**2 dxdy / int dxdy
    axy = 0.0;             // int (x-x0)*(y-y0) dxdy / int dxdy
  } 
  else //if (dmax>1.0 && dmin<1.0)
  /* during ingress of egress */
  {
     /* stellar boundary */
     psi   = acos((1.0+rho*rho-r*r)/(2.0*rho)); // angle BSP (see Fig. 1)
     phi1  = phi0-psi; // angle xSB (see Fig. 1)
     phi2  = phi0+psi; // angle xSA (see Fig. 1)
     
     a00  = funcF00(0.0,0.0,1.0,1.0,phi2)-funcF00(0.0,0.0,1.0,1.0,phi1);
     xbar = funcF10(0.0,0.0,1.0,1.0,phi2)-funcF10(0.0,0.0,1.0,1.0,phi1);
     ybar = funcF01(0.0,0.0,1.0,1.0,phi2)-funcF01(0.0,0.0,1.0,1.0,phi1);
     axx  = funcF20(0.0,0.0,1.0,1.0,phi2)-funcF20(0.0,0.0,1.0,1.0,phi1);
     ayy  = funcF02(0.0,0.0,1.0,1.0,phi2)-funcF02(0.0,0.0,1.0,1.0,phi1);
     axy  = funcF11(0.0,0.0,1.0,1.0,phi2)-funcF11(0.0,0.0,1.0,1.0,phi1);
     
     /* planet boundary */
     psi   = acos(-(1.0-rho*rho-r*r)/(2.0*r*rho)); // angle APS (see Fig. 1) in [0,pi]
     phi1  = phi0+M_PI-psi; // angle xPA (see Fig. 1)
     phi2  = phi0+M_PI+psi; // angle xPB (see Fig. 1)
     
     a00  += funcF00(x,y,rx,ry,phi2)-funcF00(x,y,rx,ry,phi1);
     xbar += funcF10(x,y,rx,ry,phi2)-funcF10(x,y,rx,ry,phi1);
     ybar += funcF01(x,y,rx,ry,phi2)-funcF01(x,y,rx,ry,phi1);
     axx  += funcF20(x,y,rx,ry,phi2)-funcF20(x,y,rx,ry,phi1);
     ayy  += funcF02(x,y,rx,ry,phi2)-funcF02(x,y,rx,ry,phi1);
     axy  += funcF11(x,y,rx,ry,phi2)-funcF11(x,y,rx,ry,phi1);
     
     xbar /= a00;
     ybar /= a00;
     axx   = axx/a00 - xbar*xbar;
     ayy   = ayy/a00 - ybar*ybar;
     axy   = axy/a00 - xbar*ybar;
  }
  
  II = funcIxn(xbar,ybar,parome,0);
  HessIxn(xbar,ybar,parome,0,&Hxx,&Hyy,&Hxy);
  ff = a00*(II+0.5*(Hxx*axx+Hyy*ayy+2.0*Hxy*axy));
  
  HessIxn(xbar,ybar,parome,1,&Hxx2,&Hyy2,&Hxy2);
  Hxx2 -= xbar*Hxx;
  Hyy2 -= xbar*Hyy;
  Hxy2 -= xbar*Hxy;
  vv    = xbar + 0.5/II*(Hxx2*axx+Hyy2*ayy+2.0*Hxy2*axy);
  
  HessIxn(xbar,ybar,parome,2,&Hxx2,&Hyy2,&Hxy2);
  Hxx2 -= xbar*xbar*Hxx;
  Hyy2 -= xbar*xbar*Hyy;
  Hxy2 -= xbar*xbar*Hxy;
  v2    = xbar*xbar + 0.5/II*(Hxx2*axx+Hyy2*ayy+2.0*Hxy2*axy);
  
  /* result */
  parome->flux   = ff;
  parome->vp     = vv;
  dbetaR         = sqrt(v2-SQR(vv));
  dbetaT         = dbetaR;
  
  /* set the units */
  parome->vp  *= parome->Vsini;
  dbetaR      *= parome->Vsini;
  dbetaT      *= parome->Vsini;
  
  if (parome->zeta>0.0)
  /* take into account macro turbulence */
  {
    double limb_pow[parome->nlimb];
    double mu2bar = 1.0-xbar*xbar-ybar*ybar;
    
    /* multiply I(x,y) by cos**2(theta) */
    PRAGMA_IVDEP
    for(int k=0; k<parome->nlimb; k++)
    {
      limb_pow[k] = parome->limb_pow[k];
      parome->limb_pow[k] += 4;
    }
    
    HessIxn(xbar,ybar,parome,0,&Hxx2,&Hyy2,&Hxy2);
    Hxx2 -= mu2bar*Hxx;
    Hyy2 -= mu2bar*Hyy;
    Hxy2 -= mu2bar*Hxy;
    zetaR2 = mu2bar + 0.5/II*(Hxx2*axx+Hyy2*ayy+2.0*Hxy2*axy);
    zetaT2 = 1.0-zetaR2;
    
    zetaR2 *= SQR(parome->zeta);
    zetaT2 *= SQR(parome->zeta);
    
    dbetaR = sqrt(SQR(dbetaR)+zetaR2);
    dbetaT = sqrt(SQR(dbetaT)+zetaT2);
    
    /* retrieve the initial limb-darkening law */
    PRAGMA_IVDEP
    for(int k=0; k<parome->nlimb; k++)
      parome->limb_pow[k] = limb_pow[k];
  }
  
  /* add to the width of the non-rotating star */
  parome->betapR = sqrt(SQR(dbetaR)+SQR(parome->beta0));
  parome->betapT = sqrt(SQR(dbetaT)+SQR(parome->beta0));
  
  parome->fvpbetap_flag |= _arome_iflux | _arome_ivp | _arome_ibetapR | _arome_ibetapT;
  
  return 0;
}


int arome_get_RM_CCF_e(t_arome * const parome, double *res)
/* Computes the RM effect measured by the CCF technique.
   v = 1/den * (2*sig0**2/(sig0**2+betap**2))**(3/2)*f*vp*exp(-vp**2/(2*(sig0**2+betap**2)))
   
   @param parome (in)  simulation structure
   @param res    (out) result
*/
{
  double den ;
  double f   ;
  double vp  ;
  double bs2 ;
  double bpR2;
  double bpT2;
  
  *res = NAN;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_get_RM_CCF_e.");
  
  if (!parome->lineprofile_flag)
    arome_error_int("the line profile is not initialized in function arome_get_RM_CCF_e.");
  if (!(parome->RM_init_flag & _arome_iCCF))
    arome_error_int("the CCF is not initialized in function arome_get_RM_CCF_e.");
  if (!(parome->fvpbetap_flag & _arome_iflux))
    arome_error_int("the flux is not initialized in function arome_get_RM_CCF_e.");
  if (!(parome->fvpbetap_flag & _arome_ivp))
    arome_error_int("the subplanet vp is not initialized in function arome_get_RM_CCF_e.");
  if (!(parome->fvpbetap_flag & _arome_ibetapR))
    arome_error_int("the subplanet betapR is not initialized in function arome_get_RM_CCF_e.");
  if (!(parome->fvpbetap_flag & _arome_ibetapT))
    arome_error_int("the subplanet betapT is not initialized in function arome_get_RM_CCF_e.");
  
  den   = parome->Gaussfit_a0;
  f     = parome->flux;
  vp    = parome->vp;
  bs2   = SQR(parome->sigma0);
  bpR2  = SQR(parome->betapR);
  bpT2  = SQR(parome->betapT);
  
  if (f<0.0)
  {
    *res = 0.0;
    return 0;
  }
  
  if (parome->zeta > 0.0)
  /* with macro-turbulence */
  {
    *res = -0.5/den*powi(sqrt(2.0*bs2/(bs2+bpR2)),3)*f*vp*exp(-vp*vp/(2.0*(bs2+bpR2)))
           -0.5/den*powi(sqrt(2.0*bs2/(bs2+bpT2)),3)*f*vp*exp(-vp*vp/(2.0*(bs2+bpT2)));
  }
  else
  /* without macro-turbulence */
  {
    *res = -1.0/den*powi(sqrt(2.0*bs2/(bs2+bpR2)),3)*f*vp*exp(-vp*vp/(2.0*(bs2+bpR2)));
  }
  
  return 0;
}
   


int arome_get_RM_iodine_e(t_arome * const parome, double *res)
/* Computes the RM effect measured by the iodine technique.
   v = 1/den * f * [dG_sig/dv * R](vp)
   where sig**2 = beta0**2 + betap**2
   
   @param parome (in)  simulation structure
   @param res    (out) result
*/
{
  double den    ;
  double f      ;
  double vp     ;
  double beta0  ;
  double betapR ;
  double betapT ;
  double sigR   ;
  double sigT   ;
  double numR   ;
  double numT   ;
  int status = 0;
  
  *res = NAN;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_get_RM_iodine_e.");
  
  if (!parome->lineprofile_flag)
    arome_error_int("the line profile is not initialized in function arome_get_RM_iodine_e.");
  if (!(parome->RM_init_flag & _arome_iIodine))
    arome_error_int("the line profile is not initialized in function arome_get_RM_iodine_e.");
  if (!(parome->fvpbetap_flag & _arome_iflux))
    arome_error_int("the flux is not initialized in function arome_get_RM_iodine_e.");
  if (!(parome->fvpbetap_flag & _arome_ivp))
    arome_error_int("the subplanet vp is not initialized in function arome_get_RM_iodine_e.");
  if (!(parome->fvpbetap_flag & _arome_ibetapR))
    arome_error_int("the subplanet betapR is not initialized in function arome_get_RM_iodine_e.");
  if (!(parome->fvpbetap_flag & _arome_ibetapT))
    arome_error_int("the subplanet betapT is not initialized in function arome_get_RM_iodine_e.");
  
  den    = parome->iodine_den;
  f      = parome->flux;
  vp     = parome->vp;
  beta0  = parome->beta0;
  betapR = parome->betapR;
  betapT = parome->betapT;
  sigR   = sqrt(SQR(beta0)+SQR(betapR));
  sigT   = sqrt(SQR(beta0)+SQR(betapT));
  
  if (parome->zeta > 0.0)
  /* with macro-turbulence */
  {
    status = dGconvR(sigR,vp,parome,&numR);
    if (status) return 1;
    status = dGconvR(sigT,vp,parome,&numT);
    if (status) return 1;
    *res =  0.5/den*f*(numR+numT);
  }
  else
  /* without macro-turbulence */
  {
    status = dGconvR(sigR,vp,parome,&numR);
    if (status) return 1;
    *res = 1.0/den*f*numR;
  }
  
  return 0;
}


int arome_get_RM_mean_e(t_arome * const parome, double *res)
/* Computes the RM effect measured by the weighted mean.
   v = -f*vp/(1-f)
   
   @param parome (in)  simulation structure
   @param res    (out) result
*/
{
  double f   ;
  double vp  ;
  
  *res = NAN;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_get_RM_mean_e.");
  
  if (!(parome->fvpbetap_flag & _arome_iflux))
    arome_error_int("the flux is not initialized in function arome_get_RM_mean_e.");
  if (!(parome->fvpbetap_flag & _arome_ivp))
    arome_error_int("the subplanet vp is not initialized in function arome_get_RM_mean_e.");
  
  f     = parome->flux;
  vp    = parome->vp;
  
  if (f<0.0)
  {
    *res = 0.0;
    return 0;
  }
  
  *res = -f*vp/(1.0-f);
  
  return 0;
}


double arome_get_RM_CCF(t_arome * const parome)
/* Returns the RM value determined by a Gaussian fit to the CCF
   using the values of f, vp, betap previously compted

   @param t_arome (in) simulation structure
*/
{
  double res;
  int status;
  
  if (!parome)
     arome_error_double("arome is not yet allocated in function arome_get_RM_CCF.");
  
  status = arome_get_RM_CCF_e(parome, &res);
  
  if (status)
    res = NAN;
  
  return res;
}

double arome_get_RM_iodine(t_arome * const parome)
/* Returns the RM value determined by the iodine cell technique
   using the values of f, vp, betap previously compted

   @param t_arome (in) simulation structure
*/
{
  double res;
  int status;
  
  if (!parome)
     arome_error_double("arome is not yet allocated in function arome_get_RM_iodine.");
  
  status = arome_get_RM_iodine_e(parome, &res);
  
  if (status)
    res = NAN;
  
  return res;
}

double arome_get_RM_mean(t_arome * const parome)
/* Returns the RM value determined by a weighted mean
   using the values of f, vp previously compted

   @param t_arome (in) simulation structure
*/
{
  double res;
  int status;
  
  if (!parome)
     arome_error_double("arome is not yet allocated in function arome_get_RM_mean.");
  
  status = arome_get_RM_mean_e(parome, &res);
  
  if (status)
    res = NAN;
  
  return res;
}

double arome_get_flux(t_arome * const parome)
/* Returns the value of the flux previously computed
   
   @param parome (in) simulation structure
*/
{
  if (!parome)
     arome_error_double("arome is not yet allocated in function arome_get_flux.");
  
  if (!(parome->fvpbetap_flag & _arome_iflux))
    arome_error_double("flux not yet initialized in function arome_get_flux.");
  
  return parome->flux;
}

double arome_get_vp(t_arome * const parome)
/* Returns the value of the subplanet vp previously computed
   
   @param parome (in) simulation structure
*/
{
  if (!parome)
     arome_error_double("arome is not yet allocated in function arome_get_vp.");
  
  if (!(parome->fvpbetap_flag & _arome_ivp))
     arome_error_double("vp not yet initialized in function arome_get_flux.");
  
  return parome->vp;
}

double arome_get_betapR(t_arome * const parome)
/* Returns the value of the subplanet betapR previously computed
   
   @param parome (in) simulation structure
*/
{
  if (!parome)
     arome_error_double("arome is not yet allocated in function arome_get_betapR.");
  
  if (!(parome->fvpbetap_flag & _arome_ibetapR))
    arome_error_double("betapR not yet initialized in function arome_get_flux.");
  
  return parome->betapR;
}

double arome_get_betapT(t_arome * const parome)
/* Returns the value of the subplanet betapT previously computed
   
   @param parome (in) simulation structure
*/
{
  if (!parome)
     arome_error_double("arome is not yet allocated in function arome_get_betapT.");
  
  if (!(parome->fvpbetap_flag & _arome_ibetapT))
    arome_error_double("betapT not yet initialized in function arome_get_flux.");
  
  return parome->betapT;
}

int arome_set_flux(double flux, t_arome * const parome)
/* set manually the value of the flux
 
   @param flux  (in) flux
   @param arome (inout) simulation structure
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_flux.");
  
   parome->flux = flux;
   parome->fvpbetap_flag |= _arome_iflux;
   
   return 0;
}

int arome_set_vp(double vp, t_arome * const parome)
/* set manually the value of the subplanet vp
 
   @param vp    (in) subplanet vp
   @param arome (inout) simulation structure
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_vp.");
  
   parome->vp = vp;
   parome->fvpbetap_flag |= _arome_ivp;
   
   return 0;
}

int arome_set_betapR(double betapR, t_arome * const parome)
/* set manually the value of the subplanet betapR
 
   @param betapR (in) subplanet betapR
   @param arome  (inout) simulation structure
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_betapR.");
  
   parome->betapR = betapR;
   parome->fvpbetap_flag |= _arome_ibetapR;
   
   return 0;
}

int arome_set_betapT(double betapT, t_arome * const parome)
/* set manually the value of the subplanet betapT
 
   @param betapT (in) subplanet betapT
   @param arome  (inout) simulation structure
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_betapT.");
  
   parome->betapT = betapT;
   parome->fvpbetap_flag |= _arome_ibetapT;
   
   return 0;
}

void arome_print_error(const char *format, ...)
/* print the error message
   ex: arome_print_error("this is an error message");
       arome_print_error("x = %lg is not correct", x);
   
   @param format (in)    format of the error message
   @param ...    (opt)   parameters of the message
*/
{
  typedef void (*t_f90userfuncerror) (const char *, int len); /*!< user function callback  */
  char msg[512];
  t_f90userfuncerror ptrfunc = (t_f90userfuncerror) arome_sglobal.userfuncerror;

  va_list args;

  va_start(args, format);

  switch (arome_sglobal.usertypeerror)
  {
   case _arome_iUserfunc:
    vsprintf(msg, format, args);
    ptrfunc(msg, (int) strlen(msg));    /* give length for F90 functions */
    break;
   case _arome_iExit:
    fprintf(stderr,"AROME ERROR : ");
    vfprintf(stderr,format, args);
    fprintf(stderr,"\n");
    exit(1);
    break;
   case _arome_iContinue:
   default:
    fprintf(stderr,"AROME ERROR : ");
    vfprintf(stderr,format, args);
    fprintf(stderr,"\n");
    break;
  }
  va_end(args);
}

void arome_set_exit_on_error(void)
/* After this call, any error will stop the programme
*/
{
   arome_sglobal.usertypeerror = _arome_iExit;
}

void arome_set_continue_on_error(void)
/* After this call, errors are returned but they do not stop the programme
*/
{
   arome_sglobal.usertypeerror = _arome_iContinue;
}

void arome_set_func_on_error(void (*userfunc)(const char *))
/* After this call, errors are handled by the user function
*/
{
   arome_sglobal.userfuncerror = userfunc;
   arome_sglobal.usertypeerror = _arome_iUserfunc;
}

int arome_malloc(int n, t_arome * const parome)
/* allocate arrays of flux, vp, betapR and betapT
   returns 0 if no error
   
   @param n      (in) size of the arrays
   @param parome (in) simulation structure
*/
{
   if (!parome)
     arome_error_int("arome is not yet allocated in function arome_malloc.");
   
   if (parome->malloc_flag)
     arome_error_int("you must first run \"arome_mfree\" before using \"arome_malloc\".");
   
   parome->mdim = n;
   
   if (!(parome->mflux = (double *)malloc(n*sizeof(double))))
     arome_error_int("Can't allocate %lu doubles in function arome_malloc.", n);
   if (!(parome->mvp = (double *)malloc(n*sizeof(double))))
   {
     free(parome->mflux);
     arome_error_int("Can't allocate %lu doubles in function arome_malloc.", n);
   }
   if (!(parome->mbetapR = (double *)malloc(n*sizeof(double))))
   {
     free(parome->mflux);
     free(parome->mvp);
     arome_error_int("Can't allocate %lu doubles in function arome_malloc.", n);
   }
   if (!(parome->mbetapT = (double *)malloc(n*sizeof(double))))
   {
     free(parome->mflux);
     free(parome->mvp);
     free(parome->mbetapR);
     arome_error_int("Can't allocate %lu doubles in function arome_malloc.", n);
   }
   
   parome->malloc_flag = 1;
   
   return 0;
}

void arome_mfree(t_arome * const parome)
/* free the arrays mflux, mvp, mbetapR, and mbetapT
   
   @param parome (in) simulation structure
*/
{
   if (!parome)
     return;
   
   if (!parome->malloc_flag)
     return;
   
   free(parome->mflux);
   free(parome->mvp);
   free(parome->mbetapR);
   free(parome->mbetapT);
   
   parome->mfvpbetap_flag = 0;
   parome->malloc_flag = 0;
}

int arome_mcalc_fvpbetap(double *x, double *y, double *z, int n, t_arome * const parome)
/* Computes the flux f, the subplanet velocity vp, and dispersions betapR, betapT
   return 0 if no error
   
   @param x      (in) x coordinate of the planet in stellar radius
   @param y      (in) y coordinate of the planet in stellar radius
   @param z      (in) z coordinate of the planet in stellar radius
   @param parome (inout) simulation structure
*/
{
   if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mcalc_fvpbetap.");
   
   if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mcalc_fvpbetap\".");
   
   if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mcalc_fvpbetap\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
   
   for(int k=0; k<n; k++)
   {
     arome_calc_fvpbetap(x[k], y[k], z[k], parome);  
     parome->mflux[k]   = parome->flux;
     parome->mvp[k]     = parome->vp;
     parome->mbetapR[k] = parome->betapR;
     parome->mbetapT[k] = parome->betapT;
   }
   parome->mfvpbetap_flag |= _arome_iflux | _arome_ivp | _arome_ibetapR | _arome_ibetapT;
   parome->fvpbetap_flag = 0;
   
   return 0;
}

int arome_mget_flux(t_arome * const parome, int n, double *flux)
/* Returns the array of the flux previously computed
   
   @param parome (in)  simulation structure
   @param n      (in)  size of flux
   @param flux   (out) output flux
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mget_flux.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mget_flux\".");
  
  if (!(parome->mfvpbetap_flag & _arome_iflux))
    arome_error_int("flux not yet initialized in function arome_mget_flux.");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mget_flux\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  PRAGMA_IVDEP
  for(int k=0; k<n; k++)
    flux[k] = parome->mflux[k];
  
  return 0;
}

int arome_mget_vp(t_arome * const parome, int n, double *vp)
/* Returns the array of the vp previously computed
   
   @param parome (in)  simulation structure
   @param n      (in)  size of vp
   @param vp     (out) output vp
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mget_vp.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mget_vp\".");
  
  if (!(parome->mfvpbetap_flag & _arome_ivp))
    arome_error_int("vp not yet initialized in function arome_mget_vp.");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mget_vp\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  PRAGMA_IVDEP
  for(int k=0; k<n; k++)
    vp[k] = parome->mvp[k];
  
  return 0;
}

int arome_mget_betapR(t_arome * const parome, int n, double *betapR)
/* Returns the array of the betapR previously computed
   
   @param parome (in)  simulation structure
   @param n      (in)  size of betapR
   @param betapR (out) output betapR
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mget_betapR.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mget_betapR\".");
  
  if (!(parome->mfvpbetap_flag & _arome_ibetapR))
    arome_error_int("betapR not yet initialized in function arome_mget_betapR.");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mget_betapR\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  PRAGMA_IVDEP
  for(int k=0; k<n; k++)
    betapR[k] = parome->mbetapR[k];
  
  return 0;
}

int arome_mget_betapT(t_arome * const parome, int n, double *betapT)
/* Returns the array of the betapT previously computed
   
   @param parome (in) simulation structure
   @param n      (in)  size of betapT
   @param betapT (out) output betapT
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mget_betapT.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mget_betapT\".");
  
  if (!(parome->mfvpbetap_flag & _arome_ibetapT))
    arome_error_int("betapT not yet initialized in function arome_mget_betapT.");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mget_betapT\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  PRAGMA_IVDEP
  for(int k=0; k<n; k++)
    betapT[k] = parome->mbetapT[k];
  
  return 0;
}

int arome_mset_flux(double *flux, int n, t_arome * const parome)
/* set manually the value of the flux
 
   @param flux  (in) flux
   @param n     (in) size of flux
   @param arome (inout) simulation structure
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mset_flux.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mset_flux\".");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mset_flux\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  PRAGMA_IVDEP
  for(int k=0; k<n; k++)
     parome->mflux[k] = flux[k];
  
  parome->mfvpbetap_flag |= _arome_iflux;
   
  return 0;
}

int arome_mset_vp(double *vp, int n, t_arome * const parome)
/* set manually the value of the vp
 
   @param vp  (in) vp
   @param n     (in) size of vp
   @param arome (inout) simulation structure
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mset_vp.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mset_vp\".");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mset_vp\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  PRAGMA_IVDEP
  for(int k=0; k<n; k++)
     parome->mvp[k] = vp[k];
  
  parome->mfvpbetap_flag |= _arome_ivp;
   
  return 0;
}

int arome_mset_betapR(double *betapR, int n, t_arome * const parome)
/* set manually the value of the betapR
 
   @param betapR  (in) betapR
   @param n     (in) size of betapR
   @param arome (inout) simulation structure
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mset_betapR.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mset_betapR\".");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mset_betapR\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  PRAGMA_IVDEP
  for(int k=0; k<n; k++)
     parome->mbetapR[k] = betapR[k];
  
  parome->mfvpbetap_flag |= _arome_ibetapR;
   
  return 0;
}

int arome_mset_betapT(double *betapT, int n, t_arome * const parome)
/* set manually the value of the betapT
 
   @param betapT  (in) betapT
   @param n     (in) size of betapT
   @param arome (inout) simulation structure
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mset_betapT.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mset_betapT\".");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mset_betapT\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  PRAGMA_IVDEP
  for(int k=0; k<n; k++)
     parome->mbetapT[k] = betapT[k];
  
  parome->mfvpbetap_flag |= _arome_ibetapT;
   
  return 0;
}

int arome_mget_RM_CCF(t_arome * const parome, int n, double *res)
/* Computes the RM effect measured by the CCF technique.
   v = 1/den * (2*sig0**2/(sig0**2+betap**2))**(3/2)*f*vp*exp(-vp**2/(2*(sig0**2+betap**2)))
   
   @param parome (in)  simulation structure
   @param n      (in)  size of the result
   @param res    (out) result
*/
{
  int nerror = 0;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mget_RM_CCF.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mget_RM_CCF\".");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mget_RM_CCF\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  if (!parome->lineprofile_flag)
    arome_error_int("the line profile is not initialized in function arome_mget_RM_CCF.");
  if (!(parome->RM_init_flag & _arome_iCCF))
    arome_error_int("the CCF is not initialized in function arome_mget_RM_CCF.");
  if (!(parome->mfvpbetap_flag & _arome_iflux))
    arome_error_int("the flux is not initialized in function arome_mget_RM_CCF.");
  if (!(parome->mfvpbetap_flag & _arome_ivp))
    arome_error_int("the subplanet vp is not initialized in function arome_mget_RM_CCF.");
  if (!(parome->mfvpbetap_flag & _arome_ibetapR))
    arome_error_int("the subplanet betapR is not initialized in function arome_mget_RM_CCF.");
  if (!(parome->mfvpbetap_flag & _arome_ibetapT))
    arome_error_int("the subplanet betapT is not initialized in function arome_mget_RM_CCF.");

  for(int k=0; k<n; k++)
  {
     parome->fvpbetap_flag = 0;
     nerror += arome_set_flux(parome->mflux[k], parome);
     nerror += arome_set_vp(parome->mvp[k], parome);
     nerror += arome_set_betapR(parome->mbetapR[k], parome);
     nerror += arome_set_betapT(parome->mbetapT[k], parome);
     nerror += arome_get_RM_CCF_e(parome, &res[k]);
  }
  parome->fvpbetap_flag = 0;
  
  return nerror;
}

int arome_mget_RM_iodine(t_arome * const parome, int n, double *res)
/* Computes the RM effect measured by the iodine technique.
   v = 1/den * f * [dG_sig/dv * R](vp)
   where sig**2 = beta0**2 + betap**2
   
   @param parome (in)  simulation structure
   @param n      (in)  size of the result
   @param res    (out) result
*/
{
  int nerror = 0;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mget_RM_iodine.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mget_RM_iodine\".");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mget_RM_iodine\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  if (!parome->lineprofile_flag)
    arome_error_int("the line profile is not initialized in function arome_mget_RM_iodine.");
  if (!(parome->RM_init_flag & _arome_iIodine))
    arome_error_int("the iodine is not initialized in function arome_mget_RM_iodine.");
  if (!(parome->mfvpbetap_flag & _arome_iflux))
    arome_error_int("the flux is not initialized in function arome_mget_RM_iodine.");
  if (!(parome->mfvpbetap_flag & _arome_ivp))
    arome_error_int("the subplanet vp is not initialized in function arome_mget_RM_iodine.");
  if (!(parome->mfvpbetap_flag & _arome_ibetapR))
    arome_error_int("the subplanet betapR is not initialized in function arome_mget_RM_iodine.");
  if (!(parome->mfvpbetap_flag & _arome_ibetapT))
    arome_error_int("the subplanet betapT is not initialized in function arome_mget_RM_iodine.");

  for(int k=0; k<n; k++)
  {
     parome->fvpbetap_flag = 0;
     nerror += arome_set_flux(parome->mflux[k], parome);
     nerror += arome_set_vp(parome->mvp[k], parome);
     nerror += arome_set_betapR(parome->mbetapR[k], parome);
     nerror += arome_set_betapT(parome->mbetapT[k], parome);
     nerror += arome_get_RM_iodine_e(parome, &(res[k]));
  }
  parome->fvpbetap_flag = 0;
  
  return nerror;
}
   

int arome_mget_RM_mean(t_arome * const parome, int n, double *res)
/* Computes the RM effect measured by the weighted mean.
   v = -f*vp/(1-f)
   
   @param parome (in)  simulation structure
   @param n      (in)  size of the result
   @param res    (out) result
*/
{
  int nerror = 0;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function arome_mget_RM_mean.");
  
  if (!parome->malloc_flag)
     arome_error_int("you must run \"arome_malloc\" before using \"arome_mget_RM_mean\".");
  
  if (parome->mdim != n)
     arome_error_int("n (%d) in \"arome_mget_RM_mean\" is different "
                     "from the allocated size (%d) in arome_malloc.", n, parome->mdim);
  
  if (!(parome->mfvpbetap_flag & _arome_iflux))
    arome_error_int("the flux is not initialized in function arome_mget_RM_mean.");
  if (!(parome->mfvpbetap_flag & _arome_ivp))
    arome_error_int("the subplanet vp is not initialized in function arome_mget_RM_mean.");

  for(int k=0; k<n; k++)
  {
     parome->fvpbetap_flag = 0;
     nerror += arome_set_flux(parome->mflux[k], parome);
     nerror += arome_set_vp(parome->mvp[k], parome);
     nerror += arome_get_RM_mean_e(parome, &res[k]);
  }
  parome->fvpbetap_flag = 0;
  
  return nerror;
}


int arome_set_gcf_ITMAX(int ITMAX, t_arome * const parome)
/* set the parameter ITMAX used in function gcf of the numerical recipes
   returns 0 if no problem
   
   @param ITMAX  (in) new value of the parameter
   @param parome (in) simulation structure
*/
{
   if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_gcf_ITMAX.");
   
   parome->gcf_ITMAX = ITMAX;
   
   return 0;
}

int arome_set_gcf_EPS(double EPS, t_arome * const parome)
/* set the parameter EPS used in function gcf of the numerical recipes
   returns 0 if no problem
   
   @param EPS    (in) new value of the parameter
   @param parome (in) simulation structure
*/
{
   if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_gcf_EPS.");
   
   parome->gcf_EPS = EPS;
   
   return 0;
}

int arome_set_gcf_FPMIN(double FPMIN, t_arome * const parome)
/* set the parameter FPMIN used in function gcf of the numerical recipes
   returns 0 if no problem
   
   @param FPMIN  (in) new value of the parameter
   @param parome (in) simulation structure
*/
{
   if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_gcf_FPMIN.");
   
   parome->gcf_FPMIN = FPMIN;
   
   return 0;
}

int arome_set_gser_ITMAX(int ITMAX, t_arome * const parome)
/* set the parameter ITMAX used in function gser of the numerical recipes
   returns 0 if no problem
   
   @param ITMAX  (in) new value of the parameter
   @param parome (in) simulation structure
*/
{
   if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_gser_ITMAX.");
   
   parome->gser_ITMAX = ITMAX;
   
   return 0;
}

int arome_set_gser_EPS(double EPS, t_arome * const parome)
/* set the parameter EPS used in function gser of the numerical recipes
   returns 0 if no problem
   
   @param EPS    (in) new value of the parameter
   @param parome (in) simulation structure
*/
{
   if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_gser_EPS.");
   
   parome->gser_EPS = EPS;
   
   return 0;
}

int arome_set_qtrap_EPS(double EPS, t_arome * const parome)
/* set the parameter EPS used in function qtrap of the numerical recipes
   returns 0 if no problem
   
   @param EPS    (in) new value of the parameter
   @param parome (in) simulation structure
*/
{
   if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_qtrap_EPS.");
   
   parome->qtrap_EPS = EPS;
   
   return 0;
}

int arome_set_qtrap_JMAX(int JMAX, t_arome * const parome)
/* set the parameter JMAX used in function qtrap of the numerical recipes
   returns 0 if no problem
   
   @param EPS    (in) new value of the parameter
   @param parome (in) simulation structure
*/
{
   if (!parome)
     arome_error_int("arome is not yet allocated in function arome_set_qtrap_JMAX.");
   
   parome->qtrap_JMAX = JMAX;
   
   return 0;
}

int dGconvR(double sig, double v, t_arome * const parome, double *val)
/* Computes [dG_sig/dv * R](v)
   where R is the rotation kernel
   returns 0 if no problem
   
   @param sig   (in) width of the Gaussian
   @param v     (in) coordinates at which the cross-correlation is evaluated
   @param pdata (in) astronomical constants containing the limb-darkening coefficients
   @param res   (out) result
*/
{
  double vsini = parome->Vsini;
  int kmax = parome->rotational_Kmax;
  int nlimb = parome->nlimb;
  double s = sig/vsini;
  double x = v/vsini;
  double x1= -(1.0+x)/s;
  double x2=  (1.0-x)/s;
  double F1, F2;
  double res;
  double map[nlimb];      // (-alpha)_p/p! = (-alpha)(-alpha+1)...(-alpha+p-1)
  double alpha;
  double fac;
  double hyp;
  double pow2;
  int status = 0;
  int p;
  
  if (!parome)
     arome_error_int("arome is not yet allocated in function dGconvR.");
  
  res = 0.0;
  fac = 1.0;
  PRAGMA_IVDEP
  for(int k=0; k<nlimb; k++) map[k] = 1.0;
  for(int m=0; m<=2*kmax; m++)
  {
   p = m/2;
   pow2 = pow(2.0, m/2.0);
   status = gamma_inc(m/2.0+1.0,x1*x1/2.0,parome,&F1);
   if (status) return 1;
   status = gamma_inc(m/2.0+1.0,x2*x2/2.0,parome,&F2);
   if (status) return 1;
   F1 *= pow2;
   F2 *= pow2;
   hyp=0.0;
   switch(m%2)
   {
    case 0:
     PRAGMA_IVDEP
     for(int k=0; k<nlimb; k++)
     {
      if (map[k])
      {
       alpha = (parome->limb_pow[k]+1.0)/2.0;
       hyp += parome->kern_coef[k]*map[k]*hypertrunc(-alpha+p, p+0.5, 0.5, x*x, kmax-p);
      }
     }
     if (p) fac/=p;
     break;
    case 1:
     F1 *= x1 > 0.0 ? 1.0 : -1.0;
     F2 *= x2 > 0.0 ? 1.0 : -1.0;
     for(int k=0; k<nlimb; k++)
     {
      if (map[k])
      {
       alpha = (parome->limb_pow[k]+1.0)/2.0;
       map[k] *= -alpha+p;
       hyp += parome->kern_coef[k]*map[k]*2.0*x*hypertrunc(-alpha+p+1.0, p+1.5, 1.5, x*x, kmax-p-1);
      }
     }
     break;
    default:
     arome_error_int("in dGconvR: m (%d) should be odd or even.", m);
   }
   if (m) fac*=s;
   res += hyp*fac*(F2-F1);
  }
  *val = res / (sqrt(2.0*M_PI)*s*SQR(vsini));
  return 0;
}




double funcF00(double x0, double y0, double rx, double ry, double phi)
/* Computes F_00(phi) (Boue et al., 2012, Tab1) use for the covered surface
   @param x_0 (in) x coordinates of the planet (or of the star)
   @param y_0 (in) y coordinates of the planet (or of the star)
   @param rx  (in) x radius of the planet (or of the star)
   @param ry  (in) y radius of the planet (or of the star)
   @param phi (in) limit of the arc 
*/
{
   return 0.5*(rx*ry*phi+x0*ry*sin(phi)-y0*rx*cos(phi));
}



double funcF10(double x0, double y0, double rx, double ry, double phi)
/* Computes F_10(phi) (Boue et al., 2012, Tab1) use for the averaged velocity
   @param x_0 (in) x coordinates of the planet (or of the star)
   @param y_0 (in) y coordinates of the planet (or of the star)
   @param rx  (in) x radius of the planet (or of the star)
   @param ry  (in) y radius of the planet (or of the star)
   @param phi (in) limit of the arc 
*/
{
   return -x0*y0*rx*cos(phi)
          -0.5*y0*rx*rx*SQR(cos(phi))
          +0.25*x0*rx*ry*(2.0*phi-sin(2.0*phi))
          +1.0/12.0*rx*rx*ry*(3.0*sin(phi)-sin(3.0*phi));
}



double funcF01(double x0, double y0, double rx, double ry, double phi)
/* Computes F_01(phi) (Boue et al., 2012, Tab1) use for the averaged velocity
   @param x_0 (in) x coordinates of the planet (or of the star)
   @param y_0 (in) y coordinates of the planet (or of the star)
   @param rx  (in) x radius of the planet (or of the star)
   @param ry  (in) y radius of the planet (or of the star)
   @param phi (in) limit of the arc 
*/
{
   return  x0*y0*ry*sin(phi)
          +0.5*x0*ry*ry*SQR(sin(phi))
          +0.25*y0*rx*ry*(2.0*phi+sin(2.0*phi))
          -1.0/12.0*rx*ry*ry*(3.0*cos(phi)+cos(3.0*phi));
}



double funcF20(double x0, double y0, double rx, double ry, double phi)
/* Computes F_20(phi) (Boue et al., 2012, Tab1) use for the velocity dispersion
   @param x_0 (in) x coordinates of the planet (or of the star)
   @param y_0 (in) y coordinates of the planet (or of the star)
   @param rx  (in) x radius of the planet (or of the star)
   @param ry  (in) y radius of the planet (or of the star)
   @param phi (in) limit of the arc 
*/
{
   return -x0*x0*y0*rx*cos(phi)
          -x0*y0*rx*rx*SQR(cos(phi))
          +0.25*x0*x0*rx*ry*(2.0*phi-sin(2.0*phi))
          -1.0/12.0*y0*rx*rx*rx*(3.0*cos(phi)+cos(3.0*phi))
          +1.0/6.0*x0*rx*rx*ry*(3.0*sin(phi)-sin(3.0*phi))
          +1.0/32.0*SQR(rx)*rx*ry*(4.0*phi-sin(4.0*phi));
}



double funcF02(double x0, double y0, double rx, double ry, double phi)
/* Computes F_02(phi) (Boue et al., 2012, Tab1) use for the velocity dispersion
   @param x_0 (in) x coordinates of the planet (or of the star)
   @param y_0 (in) y coordinates of the planet (or of the star)
   @param rx  (in) x radius of the planet (or of the star)
   @param ry  (in) y radius of the planet (or of the star)
   @param phi (in) limit of the arc 
*/
{
   return  x0*y0*y0*ry*sin(phi)
          +x0*y0*ry*ry*SQR(sin(phi))
          +0.25*y0*y0*rx*ry*(2.0*phi+sin(2.0*phi))
          +1.0/12.0*x0*ry*ry*ry*(3.0*sin(phi)-sin(3.0*phi))
          -1.0/6.0*y0*rx*ry*ry*(3.0*cos(phi)+cos(3.0*phi))
          +1.0/32.0*SQR(ry)*rx*ry*(4.0*phi-sin(4.0*phi));
}



double funcF11(double x0, double y0, double rx, double ry, double phi)
/* Computes F_11(phi) (Boue et al., 2012, Tab1) use for the velocity dispersion
   @param x_0 (in) x coordinates of the planet (or of the star)
   @param y_0 (in) y coordinates of the planet (or of the star)
   @param rx  (in) x radius of the planet (or of the star)
   @param ry  (in) y radius of the planet (or of the star)
   @param phi (in) limit of the arc 
*/
{
   return  0.25*x0*y0*(2.0*rx*ry*phi+x0*ry*sin(phi)-y0*rx*cos(phi))
          +0.125*SQR(x0*ry*sin(phi))
          -0.125*(y0*y0+ry*ry)*SQR(rx*cos(phi))
          +1.0/48.0*y0*rx*rx*ry*(15.0*sin(phi)-sin(3*phi))
          -1.0/48.0*x0*rx*ry*ry*(15.0*cos(phi)+cos(3*phi));
}



double funcLimb(double x0, double y0, void *pdata)
/* Computes I(x,y)=sum cn*mu**(n/2) at the position (x0,y0)
   @param x0    (in) x coordinates of the planet
   @param y0    (in) y coordinates of the planet
   @param pdata (in) limb-darkening coefficients
*/
{
   t_arome *parome = (t_arome *)pdata;
   const double R   = x0*x0+y0*y0;
   const double mus = sqrt(sqrt(1.0-R));
   double res;
   res = 0.0;
   PRAGMA_IVDEP
   for(int k=0; k<parome->nlimb; k++) 
     res += parome->limb_coef[k]*powi(mus, parome->limb_pow[k]);
   return res;
}



void dfuncLimb(double x0, double y0, void *pdata, double *Jx, double *Jy)
/* Computes the first derivatives of I(x,y)=sum cn*mu**(n/2) at the position (x0,y0)
   @param x0    (in)  x coordinates of the planet
   @param y0    (in)  y coordinates of the planet
   @param pdata (in)  limb-darkening coefficients
   @param Jx    (out) partial_x I(x,y)
   @param Jy    (out) partial_y I(x,y)
*/
{
   t_arome *parome = (t_arome *)pdata;
   const double R   = x0*x0+y0*y0;
   const double mu2 = 1.0-R;
   const double mus = sqrt(sqrt(mu2));
   double dIdR;
   dIdR = 0.0;
   PRAGMA_IVDEP
   for(int k=0; k<parome->nlimb; k++)
     dIdR -= 0.25*parome->limb_pow[k]*parome->limb_coef[k]*powi(mus, parome->limb_pow[k])/mu2;
   *Jx = 2.0*x0*dIdR;
   *Jy = 2.0*y0*dIdR;
}



void ddfuncLimb(double x0, double y0, void *pdata, double *Hxx, double *Hyy, double *Hxy)
/* Computes the second derivatives of I(x,y)=sum cn*mu**(n/2) at the position (x0,y0)
   @param x0    (in)   x coordinates of the planet
   @param y0    (in)   y coordinates of the planet
   @param pdata (in)   limb-darkening coefficient
   @param Hxx   (out)  partial_x partial_x (I(x,y))
   @param Hyy   (out)  partial_y partial_y (I(x,y))
   @param Hxy   (out)  partial_x partial_y (I(x,y))
*/
{
   t_arome *parome = (t_arome *)pdata;
   const double R   = x0*x0+y0*y0;
   const double mu2 = 1.0-R;
   const double mus = sqrt(sqrt(mu2));
   double var;
   double IR;
   double IRR;
   
   IR = 0.0;
   IRR = 0.0;
   PRAGMA_IVDEP
   for(int k=0; k<parome->nlimb; k++)
   {
     var  = 0.25*parome->limb_pow[k]*parome->limb_coef[k]*powi(mus, parome->limb_pow[k])/mu2; 
     IR  -= var;
     IRR += var*(0.25*parome->limb_pow[k]-1.0)/mu2;
   }
   
   *Hxx = 2.0*IR+4.0*x0*x0*IRR;
   *Hyy = 2.0*IR+4.0*y0*y0*IRR;
   *Hxy = 4.0*x0*y0*IRR;
}



int setrotkernel(t_arome * const parome)
/* Computes the coefficients of the rotation kernel R(v) (Eq. B14)
   return 0 if no problem
   
   @param parome (in) limb-darkening coefficient
*/
{
   const double Im2 = M_PI;                   // int[-1,1] 1/sqrt(1-x**2) dx
   const double Im1 = 2.3962804694711844;     // int[-1,1] 1/(1-x**2)**(1/4) dx
   const double Im0 = 2.0;                    // int[-1,1] dx
   const double Ip1 = 1.7480383695280799;     // int[-1,1] (1-x**2)**(1/4) dx
   double *tabI;
   int alpha;
   int ntabI;
   
   if (!parome)
     arome_error_int("arome is not yet allocated in function setrotkernel.");
  
   ntabI = 4;
   for(int k=0; k<parome->nlimb; k++)
     ntabI = MAX(ntabI, parome->limb_pow[k]+4);
   
   tabI = (double *)malloc(ntabI*sizeof(double));
   if (tabI==NULL)
     arome_error_int("can't allocate %d doubles in function setkernelcoef.", ntabI);
   
   tabI += 2;
   tabI[-2] = Im2;
   tabI[-1] = Im1;
   tabI[ 0] = Im0;
   tabI[ 1] = Ip1;
   
   for(int k=2; k<ntabI-2; k++)
     tabI[k] = (double)k/(k+2.0)*tabI[k-4]; // int[-1,1] (1-x**2)**(k/4) dx
   
   PRAGMA_IVDEP
   for(int k=0; k<parome->nlimb; k++)
   {
     alpha = parome->limb_pow[k];
     parome->kern_coef[k] = parome->limb_coef[k]*tabI[alpha];
   }
   
   free(tabI-2);
   
   return 0;
}



double funcAmp_a0(double x, void *pdata)
/* function R(x*Vsini)*Gauss(sig_t, x*Vsini)
   where R(v) is the rotational kernel
   sig_t = sqrt(beta0**2+sigma0**2+zeta**2/2)
   
   this function is used to compute the amplitude a0 of the best Gaussian fit
   @param x     (in) v/(Vsini) going from -1 to 1
   @param pdata (in) limb-darkening coefficient
*/
{
   t_arome *parome = (t_arome *)pdata;
   const double sig2 = SQR(parome->sigma0)+SQR(parome->beta0)+SQR(parome->zeta)/2.0;
   const double mu   = sqrt(1.0-x*x);
   const double smu = sqrt(mu);
   double Rx;
   double Gx;
   
   /* rotation kernel */
   Rx = 0.0;
   PRAGMA_IVDEP
   for(int k=0; k<parome->nlimb; k++)
     Rx += parome->kern_coef[k]*powi(smu, parome->limb_pow[k]);
   Rx *= mu;
   
   /* Gaussian */
   Gx = 1.0/sqrt(2.0*M_PI*sig2)*exp(-SQR(x*parome->Vsini)/(2.0*sig2));
   
   return Rx*Gx;
}




double funcIodine_den(double x, void *pdata)
/* function [R * Gauss'(beta0)]**2(v*Vsini)*x**2 with v = 1/x-1
   where R(v) is the rotational kernel
   beta0 = sqrt(sigma0**2+zeta**2/2)
   
   this function is used to compute the denominator of v_iodine
   @param x     (in) 1/(1+v/(Vsini)) going from 0 to 1
   @param pdata (in) limb-darkening coefficient
*/
{
   t_arome *parome = (t_arome *)pdata;
   double sig2 = SQR(parome->beta0)+SQR(parome->zeta)/2.0;
   double vsini = parome->Vsini;
   double v;
   double res = 0.0;
   int status = 0;
   
   if (x)
   {
     v = 1.0/x-1.0;
     status = dGconvR(sqrt(sig2), v*vsini, parome, &res);
     if (status) return NAN;
     res /= x;
     res = res*res*vsini;
   }
   
   return res;
}


int setGaussfit_a0(t_arome * const parome)
/* computes numerically the amplitude a0 of the
   best Gaussian fit
   @param parome (in) astronomical constants
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function setGaussfit_a0.");
  
  parome->Gaussfit_a0 =  4.0*parome->sigma0*sqrt(M_PI)*qtrap(funcAmp_a0, 0.0, 1.0, parome);
  if (parome->Gaussfit_a0 != parome->Gaussfit_a0) return 1;
  
  return 0;
}


int setIodine_den(t_arome * const parome)
/* computes numerically the denominator of the RM signal
   measured by the iodine technique
   @param parome (in) astronomical constants
*/
{
  if (!parome)
     arome_error_int("arome is not yet allocated in function setIodine_den.");
  
  parome->iodine_den = 2.0*qtrap(funcIodine_den, 0.0, 1.0, parome);
  if (parome->iodine_den != parome->iodine_den) return 1;
  
  return 0;
}


double funcIxn(double x0, double y0, void *pdata, int n)
/* Computes x**n*B(x,y)*I(x,y) at the position (x0,y0)
   @param x0    (in) x coordinates of the planet
   @param y0    (in) y coordinates of the planet
   @param pdata (in) limb-darkening coefficient
   @param n     (in) power of x
*/
{
   t_arome *parome = (t_arome *)pdata;
   double res;
   res = powi(x0, n)*funcLimb(x0,y0,parome);
   return res;
}


void HessIxn(double x0, double y0, void *pdata, int n, double *Hxx, double *Hyy, double *Hxy)
/* Computes the Hessian of x**n*I(x,y) at the position (x0,y0)
   @param x0  (in)   x coordinates of the planet
   @param y0  (in)   y coordinates of the planet
   @param c1  (in)   limb-darkening coefficient
   @param c2  (in)   limb-darkening coefficient
   @param n   (in)   power of x
   @param Hxx (out)  partial_x partial_x (x^nI(x,y))
   @param Hyy (out)  partial_y partial_y (x^nI(x,y))
   @param Hxy (out)  partial_x partial_y (x^nI(x,y))
*/
{
   t_arome *parome = (t_arome *)pdata;
   const double xn  = powi(x0,n);
   double L, Lx, Ly, Lxx, Lyy, Lxy;
   
   L = funcLimb(x0,y0,parome);
   dfuncLimb(x0,y0,parome,&Lx, &Ly);
   ddfuncLimb(x0,y0,parome,&Lxx, &Lyy, &Lxy);
   
   *Hxx = xn*Lxx;
   if (n>0) *Hxx += 2.0*n*powi(x0, n-1)*Lx;
   if (n>1) *Hxx += n*(n-1.0)*powi(x0,n-2)*L;
   *Hyy = xn*Lyy;
   *Hxy = xn*Lxy;
   if (n>0) *Hxy += n*powi(x0,n-1)*Ly;
}


double hypertrunc(double a, double b, double c, double x, int kmax)
/* Return the hypergeometric series F(a,b,c;x) truncated sum_{k=0}^kmax
*/
{
  double aa,bb,cc,fac,res;
  
  res=fac=1.0;
  aa=a;
  bb=b;
  cc=c;
  for(int k=1;k<=kmax;k++)
  {
   fac*=aa*bb/(cc*k)*x;
   res+=fac;
   aa+=1.0;
   bb+=1.0;
   cc+=1.0;
  }
  return res;
}



double trapzd(double (*func)(double, void*), double a, double b, void *pdata, int n)
/* This routine computes the nth stage of refinement of an extended trapezoidal rule.
func is input as a pointer to the function to be integrated between limits a and b, 
also input. When called with n=1, the routine returns the crudest estimate of int[a,b] 
f(x)dx. Subsequent calls with n=2,3,... (in that sequential order) will improve the 
accuracy by adding 2**(n-2) additional interior points.
*/
{
  double x,tnm,sum,del;
  static double s;
  int it,j;

  if (n==1) {
     return (s=0.5*(b-a)*((*func)(a,pdata)+(*func)(b,pdata)));
  } else {
     for (it=1,j=1;j<n-1;j++) it <<=1;
     tnm=it;
     del=(b-a)/tnm;
     x=a+0.5*del;
     for (sum=0.0,j=1;j<=it;j++,x+=del) sum += (*func)(x,pdata);
     s=0.5*(s+(b-a)*sum/tnm);
     return s;
  }
}
//  (C) Copr. 1986-92 Numerical Recipes Software

double qtrap(double (*func)(double, void*), double a, double b, void *pdata)
/* Returns the integral of the function func from a to b. The parameters EPS can be
set to the desired fractional accuracy and JMAX so that 2 to the power JMAX-1 is the
maximum allowed number of steps. Integration is performed by the trapezoidal rule.
*/
{
  t_arome *parome = (t_arome *)pdata;
  double EPS = parome->qtrap_EPS;
  int JMAX = parome->qtrap_JMAX;
  int j;
  double s,olds=0.0;

  for (j=1;j<=JMAX;j++) {
    s=trapzd(func,a,b,pdata,j);
    if (j > 5)
      if (fabs(s-olds) < EPS*fabs(olds) ||
          (s == 0.0 && olds == 0.0)) return s;
    olds=s;
  }
  arome_print_error("Too many steps in function qtrap.");
  return NAN;
}
//  (C) Copr. 1986-92 Numerical Recipes Software




double gammln(float xx)
/* Returns the value ln((Gamma(xx))) for xx > 0.
*/
{
  double x,y,tmp,ser;
  static double cof[6]={76.18009172947146,-86.50532032941677,
     24.01409824083091,-1.231739572450155,
     0.1208650973866179e-2,-0.5395239384953e-5};
  int j;
  
  y=x=xx;
  tmp=x+5.5;
  tmp -= (x+0.5)*log(tmp);
  ser=1.000000000190015;
  for (j=0;j<=5;j++) ser += cof[j]/++y;
  return -tmp+log(2.5066282746310005*ser/x);
}
//  (C) Copr. 1986-92 Numerical Recipes Software

int gamma_inc(double a, double x, t_arome * const parome, double *res)
/* Computes the incomplete gamma function P(a,x)
*/
{
  double gamser,gammcf,gln;
  
  if (x < 0.0 || a <= 0.0)
    arome_error_int("Invalid arguments (%lg,%lg) in routine gammp.", a, x);
  if (x < (a+1.0)) {
     gser(&gamser,a,x,parome,&gln);
     if (gamser != gamser) return 1;
     *res = gamser*exp(gln);
  } else {
     gcf(&gammcf,a,x,parome,&gln);
     if (gammcf != gammcf) return 1;
     *res= (1.0-gammcf)*exp(gln);
  }
  return 0;
}
//  (C) Copr. 1986-92 Numerical Recipes Software

void gser(double *gamser, double a, double x, const void * const pdata, double *gln)
/* Returns the incomplete gamma function P(a,x) evaluated by its series 
representation as gamser. Also returns ln(Gamma(a)) as gln.

this function has been modified to handle the errors.
*/
{
  t_arome *parome = (t_arome *)pdata;
  int ITMAX = parome->gser_ITMAX;
  double EPS = parome->gser_EPS;
  
  int n;
  double sum,del,ap;
  
  *gln=gammln(a);
  if (x <= 0.0) {
     if (x < 0.0) 
     {
        arome_print_error("x (%lg) less than 0 in routine gser", x);
        *gamser = NAN;
        return;
     }
     *gamser=0.0;
     return;
  } else {
     ap=a;
     del=sum=1.0/a;
     for (n=1;n<=ITMAX;n++) {
        ++ap;
        del *= x/ap;
        sum += del;
        if (fabs(del) < fabs(sum)*EPS) {
           *gamser=sum*exp(-x+a*log(x)-(*gln));
           return;
        }
     }
     arome_print_error("a (%lg) too large, ITMAX (%d) too small in routine gser.", a, ITMAX);
     *gamser = NAN;
     return;
  }
}
//  (C) Copr. 1986-92 Numerical Recipes Software

void gcf(double *gammcf, double a, double x, const void * const pdata, double *gln)
/* Returns the incomplete gamma function Q(a,x) evaluated by its continued
fraction representation as gammcf. Also returns ln Gamma(a) as gln.

this function has been modified to handle the errors.
*/
{
  t_arome *parome = (t_arome *)pdata;
  int ITMAX = parome->gcf_ITMAX;
  double EPS = parome->gcf_EPS;
  double FPMIN = parome->gcf_FPMIN;
  
  int i;
  double an,b,c,d,del,h;
  
  *gln=gammln(a);
  b=x+1.0-a;
  c=1.0/FPMIN;
  d=1.0/b;
  h=d;
  for (i=1;i<=ITMAX;i++) {
     an = -i*(i-a);
     b += 2.0;
     d=an*d+b;
     if (fabs(d) < FPMIN) d=FPMIN;
     c=b+an/c;
     if (fabs(c) < FPMIN) c=FPMIN;
     d=1.0/d;
     del=d*c;
     h *= del;
     if (fabs(del-1.0) < EPS) break;
  }
  if (i > ITMAX) 
  {
     arome_print_error("a (%lg) too large, ITMAX (%d) too small in gcf.", a, ITMAX);
     *gammcf = NAN;
     return;
  }
  *gammcf=exp(-x+a*log(x)-(*gln))*h;
}
//  (C) Copr. 1986-92 Numerical Recipes Software
