/*-----------------------------------------------------------------*/
/*! 
  \file csingle.c 
  \brief Example of usage of the ARoME library using the C interface
         This example computes the Rossiter-McLaughlin effect at one given position of the planet
         It returns the value computed simulating 3 different techniques:
           - the CCF technique
           - the iodine cell technique
           - the weighted mean (like Ohta et al. 2005)

  \author  G. Boue
           EXOEarths, Centro de Astrofisica, Universidade do Porto.

   Copyright (C) 2012, CAUP
   email of the author : gwenael.boue@astro.up.pt
   
   This work has been supported by the European Research Council/European
   Community under the FP7 through Starting Grant agreement number 239953, as
   well as from Fundacao para a Ciencia e a Tecnologia (FCT) through program
   Ciencia 2007 funded by FCT/MCTES (Portugal) and POPH/FSE (EC), and in the
   form of grants reference PTDC/CTE-AST/098528/2008, SFRH/BPD/71230/2010, and
   SFRH/BPD/81084/2011.

  History:                                                                

*/
/*-----------------------------------------------------------------*/   

/*----------------------------------------------------------------*/
/* License of the file :

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*----------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include "arome.h"


/*************************/
/* main routine          */
/* just a test           */
/*************************/
int main(int argc, char **argv)
{
   double u1, u2, beta0, Vsini, sigma0, zeta, Rp;
   int Kmax;
   double x0, y0, z0, f, vp, betapR, betapT, v_CCF, v_iodine, v_mean;
   double own_f, own_betapR;
   t_arome *parome;
   int status = 0;
   
   /* planet coordinates in stellar radius */
   x0 = 0.2;
   y0 = 0.0;
   z0 = 5.0;  /* z0 should be positive to have transit */
   
   /* limb-darkening */
   u1 = 0.69;
   u2 = 0.00;
   
   /* line profile */
   beta0  =  2.0;  /* width of the non-rotating star */
   Vsini  = 15.0;  /* Vsini */
   sigma0 =  8.0;  /* width of the best Gaussian fit */
   zeta   =  2.0;  /* macro-turbulence parameter */
   Rp     =  0.1;  /* radius of the planet */
   Kmax   =    4;  /* order of expansion for the Iodine cell technique */
   
   /* you can choose to never exit on error */
   arome_set_continue_on_error();
   
   /* allocation of the ARoME structure */
   parome = arome_alloc_quad(u1, u2);
   
   /* or with a nonlinear limb-darkening law */
   //parome = arome_alloc_nl(c1, c2, c3, c4);
   
   /* set the lineprofile */
   status += arome_set_lineprofile(beta0, Vsini, sigma0, zeta, Kmax, parome);
   
   /* print error message and exit if status != 0 */
   if (status) exit(EXIT_FAILURE);
   
   /* you can choose to always exit on error */
   arome_set_exit_on_error();
   
   /* set the planet parameters */
   status += arome_set_planet(Rp, parome);
   
   /* initialize the techniques you are going to use in the following */
   /* here we initialize both the CCF and the iodine cell techniques */
   /* should be done once the planet and line profile parameters are set */
   status += arome_init_CCF(parome);
   status += arome_init_iodine(parome);
   
   
   /* for all planet positions, do the following -----------------------*/
   /* compute the flux, vp and betap at a given position (x0, y0, z0) of the planet */
   status += arome_calc_fvpbetap(x0,y0,z0, parome);
   
   /* you can have access to the computed flux, subplanet vp, betapR, and betapT */
   f = arome_get_flux(parome);
   vp = arome_get_vp(parome);
   betapR = arome_get_betapR(parome);
   betapT = arome_get_betapT(parome);
   
   printf("------------------------------\n");
   printf("values obtained by ARoME .....\n");
   printf("------------------------------\n");
   printf("f = %.15E\n", f);
   printf("vp = %.15E\n", vp);
   printf("betapR = %.15E\n", betapR);
   printf("betapT = %.15E\n", betapT);
   printf("------------------------------\n\n");
   
   /* you can change any of the flux, vp, betap if you have your own formulae */
   /* let us change the value of the flux and betapR */
   own_f      = 0.01;
   own_betapR = 15.0; /* km/s */
   
   printf("------------------------------\n");
   printf("reset the values of the flux and radial disp...\n");
   printf("------------------------------\n");
   printf("own_f = %.15E\n", own_f);
   printf("own_betapR = %.15E\n", own_betapR);
   printf("------------------------------\n\n");
   
   arome_set_flux(own_f, parome);
   arome_set_betapR(own_betapR, parome);
   
   /* then you can compute the RM signal for the technique(s) already initialized */
   /* you can use the error-handling function with suffix "_e" */
   status += arome_get_RM_CCF_e(parome, &v_CCF);
   
   /* or simply (without the suffix "_e") */
   v_iodine = arome_get_RM_iodine(parome);
   v_mean = arome_get_RM_mean(parome);
   
   printf("------------------------------\n");
   printf("RM signals...\n");
   printf("------------------------------\n");
   printf("v_CCF = %.15E\n", v_CCF);
   printf("v_iodine = %.15E\n", v_iodine);
   printf("v_mean = %.15E\n", v_mean);
   printf("------------------------------\n");
   
   /* end of the loop over the positions of the planet ----------------*/
   
   /* release memory */
   arome_free(parome);
   
   return 0;
}
