;+
; NAME:
;         p3d_misc_choose
;
;         $Id: p3d_misc_choose.pro 125 2010-03-22 09:56:12Z christersandin $
;
; PURPOSE:
;         This routine provides a GUI for choosing an instrument, before
;         launching the main panel GUI.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_choose,defv,verbose=,error=,/debug
;
; INPUTS:
;         defv            - A structure containing default paths and additional
;                           parameters such as the window title for each
;                           instrument.
;
; KEYWORD PARAMETERS:
;         verbose [0]     - Set this variable to 0<=VERBOSE<=3 to provide
;                           different levels of output to the prompt
;                           (VERBOSE==0 results in no output at all). In order
;                           to save the messages, use the LOGFILE and LOGLEVEL
;                           keywords instead.
;         error           - Returns an error code if set. A 0 is returned if
;                           operations are successful.
;         debug           - If this keyword is set (as /DEBUG) then no error
;                           handlers are setup. This is useful when debugging
;                           the code.
; OUTPUTS:
;         pstate          - A pointer containing the chosen instrument
;                           parameters.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         Requires IDL version 6.2 or better.
;
; MODIFICATION HISTORY:
;         01.11.2008 - The first version of this routine was written by
;                      Matthias Reis. /Matthias Reis
;-
PRO p3d_misc_choose_event,event
  compile_opt hidden,IDL2

  widget_control, event.top, get_uvalue=state
  if strlowcase(tag_names(event,/structure_name)) eq 'widget_tab' then return
  widget_control,event.id,get_uvalue=uval

  case uval of
    'OK': begin

      ;; we're done, let's save all the parameters and return

      widget_control,(*state).tparfwid,get_value=tmp
      (*(*state).o).parfile=tmp

      widget_control,(*state).tdatawid,get_value=tmp
      defsysv,'!p3d_data_path',tmp[0L]

      (*(*state).o).okexit=1

      widget_control,event.top, /destroy
    end

    'bingrwid': begin
      widget_control,event.id,get_value=val

      ;; set the text widget's contents to the default values of each
      ;; instrument

      (*state).selected=val

      case val of
        0: begin
          (*(*state).o).title=(*state).defv.flam.title
          widget_control,(*state).tparfwid,set_value=(*state).defv.flam.parfile
        end
        1: begin
          (*(*state).o).title=(*state).defv.imacs.title
          widget_control,(*state).tparfwid, $
                         set_value=(*state).defv.imacs.parfile
        end
        2: begin
          (*(*state).o).title=(*state).defv.larr4k.title
          widget_control,(*state).tparfwid, $
                         set_value=(*state).defv.larr4k.parfile
        end
        3: begin
          (*(*state).o).title=(*state).defv.ppak4k.title
          widget_control,(*state).tparfwid, $
                         set_value=(*state).defv.ppak4k.parfile
        end
        4: begin
          (*(*state).o).title=(*state).defv.larr2k.title
          widget_control,(*state).tparfwid, $
                         set_value=(*state).defv.larr2k.parfile
        end
        5: begin
          (*(*state).o).title=(*state).defv.ppak2k.title
          widget_control,(*state).tparfwid, $
                         set_value=(*state).defv.ppak2k.parfile
        end
        6: begin
          (*(*state).o).title=(*state).defv.spir.title
          widget_control,(*state).tparfwid,set_value=(*state).defv.spir.parfile
        end
        7: begin
          (*(*state).o).title=(*state).defv.vimo.title
          widget_control,(*state).tparfwid,set_value=(*state).defv.vimo.parfile
        end
        8: begin
          (*(*state).o).title=(*state).defv.viru.title
          widget_control,(*state).tparfwid,set_value=(*state).defv.viru.parfile
        end
        9: begin
          (*(*state).o).title=(*state).defv.oviru.title
          widget_control,(*state).tparfwid, $
                         set_value=(*state).defv.oviru.parfile
        end
        10: begin
          (*(*state).o).title=(*state).defv.gmos.title
          widget_control,(*state).tparfwid, $
                         set_value=(*state).defv.gmos.parfile
        end
        11: begin
          (*(*state).o).title=(*state).defv.mpfs3k.title
          widget_control,(*state).tparfwid, $
                         set_value=(*state).defv.mpfs3k.parfile
        end
      endcase ;; val

    end ;; case: 'bingrwid'

    ;;========================================------------------------------
    ;; handlers for 'Browse'-buttons:

    'bdatawid': begin
      widget_control,(*state).tdatawid,get_value=tmp
      ret=dialog_pickfile(title='Please select a directory',path=tmp, $
              /must_exist,/directory,dialog_parent=event.top)
      if ret ne '' then widget_control,(*state).tdatawid,set_value=ret
    end

    'bparfwid': begin
      widget_control,(*state).tparfwid,get_value=tmp
      ret=dialog_pickfile(title='Please select a parameter file',/must_exist, $
              path=tmp,filter='*.prm',dialog_parent=event.top)
      if ret ne '' then widget_control,(*state).tparfwid,set_value=ret
    end

  endcase ;; uval

  return
END ;;; procedure: p3d_misc_choose_event


PRO p3d_misc_choose_cleanup,id
  compile_opt hidden,IDL2

  widget_control,id,get_uvalue=state

  if ptr_valid(state) then ptr_free,state

  return
END ;;; procedure: p3d_misc_choose_cleanup


FUNCTION p3d_misc_choose,defv,verbose=verbose,error=error,debug=debug
  compile_opt hidden,IDL2

  rname='p3d_misc_choose: ' & error=0
  debug=keyword_set(debug)

  bmofile=filepath(subdir=['resource','bitmaps'],'open.bmp')

  ;; Create the widgets
  bbasewid=widget_base(title='p3d: Please choose an instrument',/column, $
               /base_align_center,space=5L,xpad=2L,ypad=2L,/tlb_frame_attr)

  btabbwid=widget_tab(bbasewid)

  bitabwid=widget_base(btabbwid,title='Quick instrument setup',/column, $
      space=2L,xpad=0L,ypad=0L)
  ifu=['Flames [Argus]          :: ESO/VLT/UT2/GIRAFFE         :: /FLAMES', $
       'Imacs                   :: Magellan                    :: /IMACS', $
       'Pmas [larr] (4kx4k CCD) :: Calar Alto 3.5m             :: /LARR4K', $
       'Pmas [ppak] (4kx4k CCD) :: Calar Alto 3.5m             :: /PPAK4K', $
       'Pmas [larr] (2kx4k CCD) :: Calar Alto 3.5m             :: /LARR2K', $
       'Pmas [ppak] (2kx4k CCD) :: Calar Alto 3.5m             :: /PPAK2K', $
       'Spiral                  :: AAT 3.9m/AAOmega            :: /SPIRAL', $
       'Vimos                   :: ESO/VLT/UT3                 :: /VIMOS', $
       'Virus-P (bundle 2)      :: UTexas/Harlan J. Smith 2.7m :: /VIRUS', $
       'Virus-P (bundle 1)      :: UTexas/Harlan J. Smith 2.7m :: /OLDVIRUS', $
       'Gmos                    :: Gemini                      :: /GMOS', $
       'Mpfs (3kx2k CCD)        :: BTA 6m                      :: /MPFS3K']
  bingrwid=cw_bgroup(bitabwid,ifu,column=1L,/exclusive,ids=ids,/return_name, $
      uvalue='bingrwid',space=2L,xpad=0L,ypad=0L,/no_release)
  widget_control,ids[0L],sensitive=debug  ;; FLAMES/Argus
  widget_control,ids[1L],sensitive=debug  ;; Imacs
  widget_control,ids[7L],sensitive=debug  ;; VIMOS
  widget_control,ids[10L],sensitive=debug ;; GMOS
  widget_control,ids[11L],sensitive=debug ;; MPFS

  twid=50L
  bpathwid=widget_base(btabbwid,title='Program and data paths',/column, $
      space=2L,xpad=0L,ypad=0L,/base_align_left)

  bdatawid=widget_base(bpathwid,/row,space=5L,xpad=0L,ypad=0L)
  ldatawid=widget_label(bdatawid,value='p3d Data Path: ',/align_left)
  defsysv,'!p3d_data_path',exists=exists
  val=exists?(!p3d_data_path):''
  tdatawid=widget_text(bdatawid,value=val,xsize=twid,/editable)
  bdatbwid=widget_button(bdatawid,value=bmofile,uvalue='bdatawid',/bitmap)

  bdatawid=widget_base(bpathwid,/row,space=5L,xpad=0L,ypad=0L)
  ldatawid=widget_label(bdatawid,value='p3d Path:      ',/align_left)
  tdatbwid=widget_label(bdatawid,value=!p3d_path)


  bftabwid=widget_base(btabbwid,title='Configuration files',/column, $
      space=2L,xpad=0L,ypad=0L)


  bparfwid=widget_base(bftabwid,/row,space=5L,xpad=0L,ypad=0L)
  lparfwid=widget_label(bparfwid,value='Parameter file:',/align_left)
  tparfwid=widget_text(bparfwid,value=defv.larr2k.parfile,xsize=twid,/editable)
  bparbwid=widget_button(bparfwid,value=bmofile,uvalue='bparfwid',/bitmap)

  bokbuwid=widget_button(bbasewid,value='   OK   ',uvalue='OK')

  odata=ptr_new({okexit:0,title:defv.larr2k.title,parfile:'',squadrant:''})
  state={selected:0L,o:odata,defv:defv,bingrwid:bingrwid,tparfwid:tparfwid, $
         tdatawid:tdatawid}
  pstate=ptr_new(state,/no_copy)

  ;; Centering the application:
  screensize=get_screen_size()
  tmp=widget_info(bbasewid,/geometry)
  widget_control,bbasewid,xoffset=(screensize[0L]-tmp.xsize)/2L
  widget_control,bbasewid,yoffset=(screensize[1L]-tmp.ysize)/2L

  widget_control,bbasewid,/realize
  widget_control,bingrwid,set_value=4L
  widget_control,bbasewid,set_uvalue=pstate

  xmanager,'p3d_misc_choose',bbasewid,cleanup='p3d_misc_choose_cleanup'

  tmp=*odata
  ptr_free,odata

  return,tmp

error_handler:
  message,errmsg,/informational
  return,error
END ;;; procedure: p3d_misc_choose
