;+
; NAME:
;         p3d_wavecal_dispersion_correction
;
;         $Id: p3d_wavecal_dispersion_correction.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;	  This routine contains an algorithm that applies a dispersion
;	  correction on a stacked spectrum image, given a wavelength template,
;	  which is an image of the same size as the input image.
;
; AUTHOR:
;         Christer Sandin and Peter Weilbacher
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: wavelength calibration
;
; CALLING SEQUENCE:
;         p3d_wavecal_dispersion_correction,stack,waves,crval,cdelt,out,dout, $
;             dstack=,incdelt=,incrval=,npix=,skyalign=,/dpath,daxis=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         stack           - A three- (or two-)dimensional array of stacked
;                           image data.
;         waves           - A two-dimensional array of floating point type,
;                           that specifies a value on the wavelength for every
;                           pixel.
;
; KEYWORD PARAMETERS:
;         dstack          - If present, then this is the error in STACK. The
;                           dimensions of DSTACK must be the same as those of
;                           STACK.
;         incdelt         - A decimal scalar specifying the wavelength bin
;                           width (dispersion) [/pixel].
;         incrval         - A decimal scalar specifying the wavelength of the
;                           first pixel [].
;         npix            - A scalar integer specifying the number of
;                           wavelength bins.
;         skyalign        - If set to a scalar string with the name of a file
;                           with a list of telluric lines then the wavelength
;                           solution is aligned with the median offset position
;                           of the telluric lines.
;         dpath           - If this keyword is set then SKYALIGN is passed
;                           through p3d_misc_pathify with dpath set.
;         daxis [1]       - Specifies the dispersion axis of STACK; 1 is the
;                           x-axis, 2 is the y-axis.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         crval           - A decimal scalar that specifies the wavelength of
;                           the first pixel.
;         cdelt           - A decimal scalar that specifies the dispersion per
;                           pixel for output images, which are linearized in
;                           wavelength.
;         out             - A three- (or two-)dimensional array of stacked
;                           images, which have been wavelength calibrated.
;
; OPTIONAL OUTPUTS:
;         dout            - The resulting error of the extraction image OUT.
;                           This is only calculated if DSTACK is set.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         09.07.2009 - Included a method to align the spectra against telluric
;                      lines, which are provided in a line list. /PMW
;         07.11.2008 - Converted from the original routine dispcor of
;                      Thomas Becker. /CS
;-
PRO p3d_wavecal_dispersion_correction,stack_,waves_,crval,cdelt,out,dout, $
        dstack=dstack_,incdelt=cdelt_,incrval=crval_,npix=npix, $
        skyalign=skyalign,dpath=dpath,daxis=daxis,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_wavecal_dispersion_correction: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_wavecal_dispersion_correction'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(daxis) then daxis=1L
  s=size(daxis)
  if s[s[0L]+2L] ne 1L or (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  sid=daxis?2L:1L

  s=size(stack_)
  if s[0L] ne 2L and s[0L] ne 3L or $
    (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='STACK must be a two- or three-dimensional array of floating p' + $
           'oint type.'
    goto,error_handler
  endif
  threedim=s[0L] eq 3L
  nimages=s[0L] eq 2L?1L:s[3L]
  pix0=dblarr(s[sid])

  se=size(dstack_) & ecalc=0L
  if se[se[0L]+2L] ne 0L then begin
    if se[0L] ne s[0L] or se[se[0L]+1L] eq 7L or $
       se[se[0L]+2L] ne s[s[0L]+2L] then begin
      errmsg='DSTACK {'+strtrim(se[se[0L]+1L],2L)+',['+strtrim(se[1L],2L)+ $
             ','+strtrim(se[2L],2L)+']}, if set, must be of the same ' + $
             'dimensions as STACK {'+strtrim(s[s[0L]+1L],2L)+',['+ $
             strtrim(s[1L],2L)+','+strtrim(s[2L],2L)+']}.'
      goto,error_handler
    endif
    ecalc=1L
  endif

  sb=size(waves_)
  if sb[0L] ne 2L or sb[1L] ne s[1L] or sb[2L] ne s[2L] or $
    (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='WAVES must be a two-dimensional array of decimal type, of the ' + $
           'same dimensions as STACK ['+strtrim(s[1L],2L)+','+ $
           strtrim(s[2L],2L)+']; WAVES ['+strtrim(sb[1L],2L)+ $
           ','+strtrim(sb[2L],2L)+'] is invalid.'
    goto,error_handler
  endif

  stack=daxis?stack_:transpose(stack_,[1L,0L,2L])
  s=size(stack)
  if ecalc then dstack=daxis?dstack_:transpose(dstack_,[1L,0L,2L])
  waves=daxis?waves_:transpose(waves_)
  nwaves=n_elements(waves[*,0L])

  sb=size(cdelt_)
  if sb[sb[0L]+2L] ne 0L then begin
    if sb[sb[0L]+2L] ne 1L or $
      (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
      errmsg='INCDELT, if specified, must be a decimal scalar.'
      goto,error_handler
    endif
    cdelt=double(cdelt_)
    cdeltstr=' (set with /INCDELT)'
  endif else begin
    cdelt=mean(double(waves[1L:s[1L]-1L,*]-waves[0L:s[1L]-2L,*]))
    cdeltstr=' (calculated)'
  endelse

  sb=size(crval_)
  if sb[sb[0L]+2L] ne 0L then begin
    if sb[sb[0L]+2L] ne 1L or $
      (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
      errmsg='INCRVAL, if specified, must be a decimal scalar.'
      goto,error_handler
    endif
    crval=double(crval_)
    crvalstr=' (set with /INCRVAL)'
  endif else begin
    crval=max(double(waves[0L,*]))
    crvalstr=' (calculated)'
  endelse

  sb=size(npix)
  if sb[sb[0L]+2L] ne 0L then begin
    if sb[sb[0L]+2L] ne 1L or $
      (sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
      errmsg='NPIX, if specified, must be a scalar integer.'
      goto,error_handler
    endif
    disp1=crval+cdelt*npix
    npixstr=' (set with /NPIX)'
  endif else begin
    disp1=min(double(waves[s[daxis]-1L,*]))
    npix=floor((disp1-crval)/cdelt)
    npixstr=' (calculated)'
  endelse
  outsize=npix

  sb=size(skyalign) & useskyalign=0L
  if sb[sb[0L]+2L] ne 0L then begin
    if sb[sb[0L]+2L] ne 1L or sb[sb[0L]+1L] ne 7L then begin
      errmsg='SKYALIGN must, if specified, be a scalar string with the nam' + $
             'e of an existing file with a list of telluric lines.'
      goto,error_handler
    endif
    if ~file_test(skyalign,/read,/regular) then begin
      errmsg='Not able to read the file SKYALIGN "'+skyalign+ $
             '". Cannot continue!'
      goto,error_handler
    endif

    if nimages eq 1L then begin
      ;; Reading the data file:
      readcol,skyalign,skylines,format='d',comment=';',silent=verbose lt 3, $
              delimiter=' '
      nskylines=n_elements(skylines)
      if ~nskylines then begin
        errmsg=['There are no entries with telluric lines in SKYALIGN="'+ $
                p3d_misc_pathify(skyalign,/dpath)+'".','  Cannot continue!']
        goto,error_handler
      endif ;; ~n_elements(skylines)
      useskyalign=1L

      skyalign=p3d_misc_pathify(skyalign,dpath=keyword_set(dpath))
    endif ;; nimages eq 1L
  endif ;; sb[sb[0L]+2L] ne 0L

  ;;========================================------------------------------
  ;; Logging the input parameters:

  msg='Applying a dispersion correction. Using the following parameters:'
  msg=[msg, $
       '       crval='+string(crval,format='(f14.8)')+' []'+crvalstr+',', $
       '       cdelt='+string(cdelt,format='(f14.8)')+' []'+cdeltstr+',', $
       '        npix='+string( npix,format='(i14)')  +'    '+ npixstr+',', $
       ' useskyalign='+string(useskyalign,format='(i9)')]
  if useskyalign then $
     msg=[msg,'    skyalign="'+skyalign+'" (file name).']
  msg=[msg,'Dispersion axis: '+strtrim(daxis,2L)+'.']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  istr=strtrim(s[2L],2L) & listr=strlen(istr)
  sstr=string(replicate(32b,listr))

  ;;========================================------------------------------
  ;; Fix absolute wavelength shifts by determining mean a shift of the science
  ;; image and applying the reverse to the wavelength map:

  if useskyalign then begin

    ;; Finding the positions of SKYLINES within the spectrum wavelength array:
    idx=value_locate(waves[*,s[2L]/2L+2L],skylines) ;; offsetting by two...
    cdx=where(idx ge 0L and idx lt nwaves-1L,count)
    cstr=replicate(' ',nskylines)
    if count ne 0L then cstr[cdx]='*'

    ;; Logging the lines in the line list:
    sskylines=string(skylines,format='(f10.3)')
    msg=['[Telluric lines calibration] Using the following sky emission li' + $
         'nes','  (those marked with ''*'' are present in the center spect' + $
         'rum):']
    stride=10L
    n=(nskylines-1L)/stride & nmod=(nskylines-1L) mod stride
    for i=0L,n-1L do begin
      tmp='             '
      for j=i*stride,(i+1L)*stride-1L do tmp+=sskylines[j]+cstr[j]+', '
      msg=[msg,tmp]
    endfor
    tmp='             '
    for j=i*stride,i*stride+nmod-1L do tmp+=sskylines[j]+cstr[j]+', '
    tmp+=cstr[j]+sskylines[j]
    msg=[msg,tmp]
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)


    noffsets=0L & offsets=dblarr(s[2L]*nskylines)

    ;; Looping over all spectra:
    smlen=strtrim(strlen(strtrim(s[2L],2L)),2L) & fstr='(i'+smlen+')'
    for k=0L,s[2L]-1L do begin

      ;; Finding positions of all sky emission lines in the current spectrum:
      idx=value_locate(waves[*,k],skylines)

      ;; Loop over all sky lines in the line list:
      for l=0L,nskylines-1L do begin

        ;; n gives the initial position in pixels:
        n=idx[l]

        ;; Skipping lines which are outside the current wavelength range:
        if n lt 0L or n ge nwaves-1L then continue

        if abs(waves[n,k]-skylines[l]) gt cdelt then begin
          msg='[Telluric lines calibration] {spec='+string(k+1L,format=fstr)+ $
              '} Warning: the offset between the sky line '+sskylines[l]+ $
              ' and the wavelength '+ $
              string(waves[k,l],format='(f10.3)')+' is large: '+ $
              string(waves[n,k]-skylines[l],format='(f10.3)')+'.'
          error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
        endif ;; abs(waves[n,k]-skylines[l]) gt cdelt

        posinit=n

        ;; Extracting a a spectrum of width +/-5 pixels around the current
        ;; position, in order to fit a Gaussian:
        nlo=0L>(n-5L) & nhi=(n+5L)<(nwaves-1L)

        ;; Fit a Gaussian, with only a constant background offset (not
        ;; providing any initial estimates):
        if ecalc then dspec=dstack[nlo:nhi,k]
        fit=gaussfit(waves[nlo:nhi,k],stack[nlo:nhi,k],coeff,nterms=4L, $
                measure_errors=dspec)

        if coeff[1L] lt waves[nlo,k] or coeff[1L] gt waves[nhi,k] then begin
          msg='[Telluric lines calibration] {spec='+string(k+1L,format=fstr)+ $
              '} Warning: The Gaussian fit gave an unlikely center position'+ $
              ') '+string(format='(f10.3)',coeff[1L])+') for the sky line a'+ $
              't '+sskylines[l]+', and the starting center value of '+ $
              string(format='(f10.3)',waves[n,k])+'.'
          error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
          continue
        endif ;; coeff[1L] lt waves[nlo,k]) or coeff[1L] gt waves[nhi,k]

        ;; Determining the offset to the real value, using the fit center:
        offset=skylines[l]-coeff[1L]

        ;; Discarding too large offsets; because, they imply that the skyline
        ;; was affected by something else (for example a cosmic ray or object
        ;; emission):
        if abs(offset) gt cdelt then begin
          msg='[Telluric lines calibration] {spec='+string(k+1L,format=fstr)+ $
              '} Warning: Discarding the large offset '+ $
              string(offset,format='(f10.3)')+' for the sky line at '+ $
              sskylines[l]+'.'
          error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
          continue
        endif ;; abs(offset) gt cdelt

        ;; Storing the offset value:
        offsets[noffsets++]=offset

      endfor ;; l=0L,nskylines-1L
    endfor ;; k=0L,s[2L]-1L

    msg='[Telluric lines calibration] Found a total of '+ $
        strtrim(noffsets,2L)+' valid sky line offset values.'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

    if noffsets gt 0L then begin
      ;; apply the median offset (computed over all valid elements in the
      ;; offset vector to all calibrated wavelengths
      ;;XXX this could be wrong if stack/waves contain more than one exposure?!
      offsetmedian=median(offsets[where(offsets ne 0d0)],/even)

      msg='[Telluric lines calibration] Applying a median offset of '+ $
          string(format='(f10.3)',offsetmedian)+ $
          ' to the wavelength array of every spectrum.'
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

      waves+=offsetmedian
    endif ;; noffsets gt 0L
  endif ;; useskyalign


  ;;========================================------------------------------
  ;; Calculating and applying the mapping from pixel coordinates to wavelength:

  pixsize=cdelt/(double(waves[1L:s[1L]-1L,*])- $
                 double(waves[0L:s[1L]-2L,*])) 
  outpix0=ceil(((double(waves[0L,*])-crval)>0d0)/cdelt) 
  outwave0=crval+double(outpix0)*cdelt

  ;; Looping over all spectra:
  for k=0L,s[2L]-1L do begin
    pos=where(double(waves[*,k]) le outwave0[k]) & pos=max(pos)>0L
    waverest=(outwave0[k]-double(waves[pos,k]))/ $
             (double(waves[pos+1L,k])-double(waves[pos,k]))
    pix0[k]=pos+waverest+pixsize[pos,k]/2d0
  endfor

  pixpos=dblarr(outsize,s[2L])

  for k=0L,s[2L]-1L do begin 
    pixpos[outpix0[k],k]=pix0[k]
    for L=outpix0[k]+1L,outsize-1L do pixpos[L,k]= $
      (pixpos[L-1L,k]+pixsize[long(pixpos[L-1L,k]),k])<(s[1L]-2L) 
  endfor

  out=fltarr(outsize,s[2L],nimages)
  if ecalc then dout=out

  ;;=============================--------------------
  ;; Looping over all spectra:

  for L=0L,s[2L]-1L do begin

    lower=pixpos[*,L]-pixsize[pixpos[*,L],L]/2d0 & lowfix= ceil(lower)
    upper=pixpos[*,L]+pixsize[pixpos[*,L],L]/2d0 & uppfix=floor(upper-1d0)

    lowlow=(lowfix-1L)>0L         & lowrest=lowfix-lower
    uppupp=(uppfix+1L)<(s[1L]-1L) & upprest=upper-uppfix-1L

    ;;=============================--------------------
    ;; Looping over the relevant spectrum bins:

    for k=outpix0[L],outsize-1L do begin
      if pixpos[k,L] lt s[1L]-2L then begin
        if uppfix[k] lt lowfix[k]-1L then begin

          ;;=============================--------------------
          ;; When cdelt>pixel_size:

          tmp=(lowrest[k]+upprest[k]-1L)
          out[k,L,*]=tmp*stack[lowlow[k],L,*]
          if ecalc then dout[k,L,*]=tmp^2*dstack[lowlow[k],L,*]^2

          ;;=============================--------------------
          ;; Logging the performed operations:

          if ~k then if logunit[1L] ge 2L then begin
            lstr=strtrim(L+1L,2L) & llstr=strlen(lstr)
            tmp_=strmid(sstr,0L,listr-llstr)+lstr
            msg=' Spectrum '+tmp_+'/'+istr+ $
                ' :: 0-pixel ['+string(lower[k],format='(f7.2)')+ $
                '] :: added the pixel value fraction '+ $
                string(format='(f5.3)',tmp)+' ['+ $
                strtrim(lowlow[k],2L)+']  :: cdelt>size(pixel).'
            error=p3d_misc_logger(msg,logunit,rname=rname,loglevel=2, $
                verbose=verbose ge 2)
          endif

        endif else begin
          tmpstr=' :: '

          ;;=============================--------------------
          ;; When cdelt<pixel_size:

          if uppfix[k] ge lowfix[k] then begin
            out[k,L,*]=total(stack[lowfix[k]:uppfix[k],L,*],1L)
            if ecalc then $
              dout[k,L,*]=total(dstack[lowfix[k]:uppfix[k],L,*]^2,1L)

            if ~k then if logunit[1L] ge 2L then $
              tmpstr=' :: 1.0 ['+strtrim(lowfix[k],2L)+':'+ $
                                 strtrim(uppfix[k],2L)+'] :: '
          endif

          ;;=============================--------------------
          ;; Summing up the remainder:

          out[k,L,*]+=lowrest[k]*stack[lowlow[k],L,*]
          out[k,L,*]+=upprest[k]*stack[uppupp[k],L,*]
          if ecalc then begin
            dout[k,L,*]+=lowrest[k]^2*dstack[lowlow[k],L,*]^2
            dout[k,L,*]+=upprest[k]^2*dstack[uppupp[k],L,*]^2
          endif

          ;;=============================--------------------
          ;; Logging the performed operations:

          if ~k then begin
            if logunit[1L] ge 2L then begin
              lstr=strtrim(L+1L,2L) & llstr=strlen(lstr)
              tmp=strmid(sstr,0L,listr-llstr)+lstr
              msg=' Spectrum '+tmp+'/'+istr+ $
                  ' :: 0-pixel ['+string(lower[k],format='(f7.2)')+ $
                  '] :: added the pixel value fraction '+ $
                  string(lowrest[k],format='(f5.3)')+' ['+ $
                  strtrim(lowlow[k],2L)+']'+tmpstr+ $
                  string(upprest[k],format='(f5.3)')+' ['+ $
                  strtrim(uppupp[k],2L)+'] :: total '+ $
                  string(lowrest[k]+upprest[k],format='(f5.3)')+'.'
              error=p3d_misc_logger(msg,logunit,rname=rname,loglevel=2, $
                  verbose=verbose ge 2)
            endif
          endif ;; ~k
        endelse ;; uppfix[k] lt lowfix[k]-1L
      endif ;; pixpos[L,k] lt s[1L]-2L
    endfor ;; k=outpix0[L],outsize-1L

  endfor ;; L=0L,s[2L]-1L

  ;; Preparing the output:
  if daxis eq 2L then begin
    out=transpose( out,[1L,0L,2L])
    if ecalc then dout=transpose(dout,[1L,0L,2L])
  endif ;; daxis eq 2L

  if ecalc then dout=sqrt(dout)

  if nimages eq 1L then begin
    out=reform(out)
    if ecalc then dout=reform(dout)
  endif ;; nimages eq 1L

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_wavecal_dispersion_correction
