;+
; NAME:
;         p3d_misc_getinformation
;
;         $Id: p3d_misc_getinformation.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         The purpose of this routine is to return instrument-specific values
;         on the CCD gain and CCD readout noise. The values are searched for
;         either in the instrument-specific (and specified) file, or in the
;         user parameter file.
;
;         Regardless if the parameters exist in the data file header or in the
;         user parameter file it is mandatory to specify all the following
;         keywords in the instrument-specific header keywords file:
;           GAIN, GAINUNIT, RDNOISE, RDNUNIT.
;
;         The values are returned if either:
;          1) The header keyword CRDNOIx are found in the header. In this case
;               there is no checking for units, CRDNOI and GAIN are expected to
;               be of the correct units (i.e. ADU and e-/ADU). This is the case
;               with images that have been combined by p3d using
;               p3d_misc_imcombine.
;          2) Alternatively, if the header keyword NCOMBINE is found in a 
;               specified master bias file then the error of every pixel value
;               of the master bias can be calculated, and thereafter also other
;               errors. In this case the gain and readout noise values are
;               read from the data header, or if the USERPARFILE keyword file
;               is specified from that file (if the values are present
;               there). 
;         In the former case the name of the keyword parameter in the data
;         header is looked up in KWRDLIST. Thereafter the value is read from
;         the data header.
;
;         In the second case the value of the respective parameter is read from
;         USERPARFILE. Accounting for instruments with several detectors the
;         optional INDEX keyword specifies the detector (INDEX must be >=1 if
;         it is specified).
;
;
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_getinformation,filename,kwrdlist,masterbias=,dbias=,gain=, $
;             rdnoise=,userparfile=,index=,kwgain=,topwid=,logunit=, $
;             verbose=,error=,/debug,/help
;
; INPUTS:
;         filename        - A scalar string that contains the name of an object
;                           data file. The header of this file is scanned for
;                           the gain and the readout noise. These are the
;                           values that are returned in GAIN and RDNOISE.
;                           The instrument-specific names of the keywords are
;                           at first looked up in the keywords file
;                           (GAIN, GAINUNIT, RDNOISE, RDNUNIT). Thereafter
;                           those keywords are used when scanning the file
;                           header. GAINUNIT and RDNUNIT are used to ensure
;                           that the correct unit is returned of all
;                           quantities (ADU).
;         kwrdlist        - A scalar string with the name of a file that
;                           contains instrument-specific keyword names for use
;                           with p3d.
;
; KEYWORD PARAMETERS:
;         masterbias      - A scalar string that contains the name of a
;                           master bias file. The header of this file is
;                           scanned for the CCD gain, the readout noise, and
;                           the number or image (n), that were combined to
;                           create the master bias (NCOMBINE). The pixel error
;                           is thereafter calculated as: rdnoise/sqrt(n).
;                           The instrument-specific names of the keywords are
;                           at first looked up in the keywords file
;                           (GAIN, GAINUNIT, RDNOISE, RDNUNIT). Thereafter
;                           those keywords are used when scanning the file
;                           header. GAINUNIT and RDNUNIT are used to ensure
;                           that the correct unit is returned of all
;                           quantities (ADU).
;         dbias           - A scalar decimal value, that returns the error of
;                           every pixel in the master bias image, in ADU.
;         gain            - A scalar decimal value, that returns the CCD gain,
;                           in units of e-/ADU.
;         rdnoise         - A scalar decimal value, that returns the CCD
;                           readout noise value, in units of ADU.
;         userparfile     - A scalar string with the name of a file with user-
;                           defined parameters. The following parameters are
;                           read here from this file (if they are present):
;                             gainx, rdnoisex
;                           If they are present then the values are not read
;                           from the data header. The suffix 'x' is provided by
;                           the input keyword INDEX, if it is present.
;         index ['']      - If the gain and readout noise are read from 
;                           USERPARFILE then an optional numerical suffix can
;                           be provided in INDEX (in order to account for
;                           instruments with several detectors). This INDEX
;                           must satisfy INDEX>=1.
;         kwgain          - Unless CRDNOI1 is set this keyword returns the name
;                           of the keyword used with the chosen detector
;                           (of KWRDLIST).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
;-
PRO p3d_misc_getinformation,filename,kwrdlist,masterbias=masterbias, $
        dbias=dbias,gain=gain,rdnoise=rdnoise,userparfile=userparfile, $
        index=index_,kwgain=kwgain,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_getinformation: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_getinformation'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  useuserpar=0L
  if n_elements(userparfile) ne 0L then begin
    if size(userparfile,/type) ne 7L then begin
      errmsg='USERPARFILE must, if specified, be of string type.'
      goto,error_handler
    endif
    if userparfile ne '' then begin
      if ~file_test(userparfile,/read,/regular) then begin
        errmsg='The file USERPARFILE "'+userparfile+'" does not exist.'
        goto,error_handler
      endif

      readcol,userparfile,uparname,uparvalue,format='a,a',comment=';', $
          silent=verbose lt 3,delimiter=' '

      if n_elements(uparname) gt 0L then useuserpar=1L
    endif ;; userparfile ne ''
  endif ;; n_elements(userparfile) ne 0L

  uindex=~n_elements(index_)?0L:1L
  if uindex then begin
    s=size(index_)
    if s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L then begin
      errmsg='INDEX must be of (any) integer type; INDEX>=1.'
      goto,error_handler
    endif
    if index_ lt 1L then begin
      errmsg='INDEX must be of (any) integer type; INDEX>=1.'
      goto,error_handler
    endif
  endif ;; uindex
  index=uindex?strtrim(index_,2L):''

  ;;========================================------------------------------
  ;; Checking if the dispersion axis flipping of the master bias file and the
  ;; data file agree:

  hdr=headfits(filename)
  if n_elements(masterbias) ne 0L then begin
    bhdr=headfits(filename)
    tmp1=fxpar( hdr,'P3DDFLIP')
    tmp2=fxpar(bhdr,'P3DDFLIP')
    if tmp1 ne tmp2 then begin
      errmsg='The dispersion axis flipping of the data file and the master' + $
             ' bias differ {P3DDFLIP}.'
      goto,error_handler
    endif
  endif ;; n_elements(masterbias) ne 0L
  
  ;;========================================------------------------------
  ;; First retrieving information on the name of the keywords in the header:

  kwgain=p3d_misc_get_hdr_kwrd(kwrdlist,'GAIN',topwid=topwid, $
      verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  if kwgain eq '-1' then begin
    errmsg='[getinformation] Cannot read the GAIN keyword in the keyword list.'
    goto,error_handler
  endif
  kugain=p3d_misc_get_hdr_kwrd(kwrdlist,'GAINUNIT',topwid=topwid, $
      verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  if kugain eq '-1' then begin
    errmsg='[getinformation] Cannot read the GAINUNIT keyword in the keywo' + $
           'rd list.'
    goto,error_handler
  endif

  kwrdn=p3d_misc_get_hdr_kwrd(kwrdlist,'RDNOISE',topwid=topwid, $
      verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  if kwrdn eq '-1' then begin
    errmsg='[getinformation] Cannot read the RDNOISE keyword in the keywor' + $
           'd list.'
    goto,error_handler
  endif
  kurdn=p3d_misc_get_hdr_kwrd(kwrdlist,'RDNUNIT',topwid=topwid, $
      verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  if kurdn eq '-1' then begin
    errmsg='[getinformation] Cannot read the RDNUNIT keyword in the keywor' + $
           'd list.'
    goto,error_handler
  endif

  ;;=================-------------
  ;; Logging the information:

  msg=['Read the following keywords from the instrument-specific header ke' + $
       'ywords file:','     KW(GAIN)="'+kwgain+'" ['+kugain+'],', $
       '  KW(RDNOISE)="'+kwrdn +'" ['+kurdn +'].']
  error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose ge 1L)
  if error ne 0 then return

  ;;========================================------------------------------
  ;; Checking for the existence of the CRDNOI (readout-noise) keyword of data,
  ;; which has been combined from several blocks with different values on the
  ;; readout noise (the gain has in this case already been normalized):

  tmp1=fxpar(hdr,'CRDNOI1')
  if tmp1 ne '0' then begin

    ;;=======================================----------------------
    ;; Retrieving RON from the MASTERBIAS header first (and the gain):

    if n_elements(masterbias) ne 0L then begin
      bhdr=headfits(masterbias)

      dbias_=p3d_misc_fitsstrex(bhdr,'NCOMBINE',topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
      if error ne 0 then return

      if dbias_ ne -1L then begin

        tmp1=fxpar(bhdr,'CRDNOI1')
        notready=1L & i=1L
        rdnoise=[double(tmp1)]
        while notready do begin
          tmp=fxpar(bhdr,'CRDNOI'+strtrim(i+1L,2L))
          if tmp ne '0' then begin
            rdnoise=[rdnoise,double(tmp)]
            i++
          endif else notready=0L
        endwhile ;; notready

        ;; Calculating the error of the pixels in the masterbias:
        dbias=rdnoise/sqrt(dbias_)

        ;; Reading the readout noise and gain values from the data header:
        notready=1L & i=1L
        rdnoise=[double(tmp1)]
        while notready do begin
          tmp=fxpar(hdr,'CRDNOI'+strtrim(i+1L,2L))
          if tmp ne '0' then begin
            rdnoise=[rdnoise,double(tmp)]
            i++
          endif else notready=0L
        endwhile ;; notready

        notready=1L & i=1L

        gain=p3d_misc_fitsstrex(hdr,kwgain,topwid=topwid, $
                 logunit=logunit,verbose=verbose,error=error,debug=debug)
        if error ne 0 then return

        ;;=================-------------
        ;; Logging the information:

        nbl=n_elements(gain)
        msg='Read the following values from the data header:'
        for j=0L,nbl-1L do begin
          msg=[msg,'  block '+strtrim(j+1L,2L)+'/'+strtrim(nbl,2L)+ $
               ' :: dbias='+string(  dbias[j],format='(e10.3)')+' [ADU]'+ $
               ', rdnoise='+string(rdnoise[j],format='(e10.3)')+' [ADU].']
        endfor ;;  j=0L,nbl-1L
        msg=[msg,'  gain='   +string(gain,format='(e10.3)')+' [e-/ADU].']
        error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
            verbose=verbose ge 1L)

        ;; All done, returning:
        return

      endif ;; dbias_ ne -1L
    endif ;; n_elements(masterbias) ne 0L
  endif ;; tmp1 ne '0' and tmp2 ne '0'

  ;;=======================================----------------------
  ;; Master bias:

  if n_elements(masterbias) ne 0L then begin
    bhdr=headfits(masterbias)

    dbias_=p3d_misc_fitsstrex(bhdr,'NCOMBINE',topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return

    if dbias_ ne -1L then begin

      ;;=======================================----------------------
      ;; Retrieving GAIN and RON from the master bias file header at first:

      gsrc='masterbias header' & override=0L
      if useuserpar then begin
        p3d_misc_read_params,uparname,uparvalue,'gain'+index,ugain,/upo, $
            type='float',logunit=logunit,topwid=topwid,verbose=verbose, $
            error=error,debug=debug
        if error ne 0 then return
        if n_elements(ugain) eq 1L then begin
          override=1L
          gain=ugain
        endif
      endif ;; useuserpar
      if ~override then begin
        gain=p3d_misc_fitsstrex(bhdr,kwgain,topwid=topwid,logunit=logunit, $
            verbose=verbose,error=error,debug=debug)
        if error ne 0 then return
        if gain eq '-1' or size(gain,/type) eq 7L then begin
          tmp='gain'+(index eq ''?'':'['+index+']')
          errmsg=['Could not read any value on the gain in the master bias' + $
                  ' file header.',' Please specify a decimal value using t' + $
                  'he user parameter file {'+tmp+'}.']
          goto,error_handler
        endif ;; gain eq -1L
      endif else gsrc='user parameter file'

      if ~strpos(kugain,'ADU') then gain=1d0/gain

      rsrc='masterbias header' & override=0L
      if useuserpar then begin
        p3d_misc_read_params,uparname,uparvalue,'rdnoise'+index,urdnoise, $
            /upo,type='float',logunit=logunit,topwid=topwid,verbose=verbose, $
            error=error,debug=debug
        if error ne 0 then return
        if n_elements(urdnoise) eq 1L then begin
          override=1L
          rdnoise=urdnoise
        endif
      endif ;; useuserpar
      if ~override then begin
        rdnoise=p3d_misc_fitsstrex(bhdr,kwrdn,topwid=topwid,logunit=logunit, $
            verbose=verbose,error=error,debug=debug)
        if error ne 0 then return
        if rdnoise eq '-1' or size(rdnoise,/type) eq 7L then begin
          tmp='rdnoise'+(index eq ''?'':'['+index+']')
          errmsg=['Could not read any value on the readout noise in the ma' + $
                  'ster bias file header.',' Please specify a decimal valu' + $
                  'e using the user parameter file {'+tmp+'}.']
          goto,error_handler
        endif ;; rdnoise eq -1L
      endif else rsrc='user parameter file'
      if strpos(kurdn,'ADU') ne 0L then rdnoise/=gain

      ;; Calculating the error of the pixels in the master bias:
      dbias=rdnoise/sqrt(dbias_)

      if ~finite(gain) or ~finite(rdnoise) or ~finite(dbias) then begin
        errmsg='One of gain ['+strtrim(gain,2L)+'], rdnoise ['+ $
               strtrim(rdnoise,2L)+'], or dbias ['+strtrim(dbias,2L)+ $
               '] is not finite. Cannot continue!'
        goto,error_handler
      endif ;; ~finite(gain) or ~finite(rdnoise) or ~finite(dbias)

      if gain lt 0d0 or rdnoise lt 0d0 or dbias lt 0d0 then begin
        errmsg='One of gain ['+strtrim(gain,2L)+'], rdnoise ['+ $
               strtrim(rdnoise,2L)+'] or dbias ['+strtrim(dbias,2L)+ $
               '] is <0. Cannot continue!'
        goto,error_handler
      endif ;; gain lt 0d0 or rdnoise lt 0d0 or dbias lt 0d0

      ;;=================-------------
      ;; Logging the information:

      msg=['Read the following values:', $
           '    dbias='+string(dbias,format='(e10.3)')+' [ADU],', $
           '     gain='+string( gain,format='(e10.3)')+' [e-/ADU] {'+gsrc+ $
           '},', $
           '  rdnoise='+string(rdnoise,format='(e10.3)')+' [ADU]    {'+ $
           rsrc+'}.']
      error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
          verbose=verbose ge 1L)
      if error ne 0 then return

    endif ;; dbias_ ne -1L
  endif ;; n_elements(masterbias) ne 0L

  ;;=======================================----------------------
  ;; Retrieving GAIN and RON from the data header:

  gsrc='data header' & override=0L
  if useuserpar then begin
    p3d_misc_read_params,uparname,uparvalue,'gain'+index,uugain,/upo, $
        type='float',logunit=logunit,topwid=topwid,verbose=verbose, $
        error=error,debug=debug
    if error ne 0 then return
    if n_elements(uugain) eq 1L then begin
      override=1L
      gain=uugain
    endif
  endif ;; useuserpar
  if ~override then begin
    gain=p3d_misc_fitsstrex(hdr,kwgain,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
    if gain eq '-1' or size(gain,/type) eq 7L then begin
      tmp='gain'+(index eq ''?'':'['+index+']')
      errmsg=['Could not read any value on the gain in the data file heade' + $
              'r.',' Please specify a decimal value using the user paramet' + $
              'er file {'+tmp+'}.']
      goto,error_handler
    endif ;; rdnoise eq -1L
  endif else gsrc='user parameter file'
  if ~strpos(kwgain,'ADU') then gain=1d0/gain

  rsrc='data header'
  if useuserpar then begin
    p3d_misc_read_params,uparname,uparvalue,'rdnoise'+index,uurdnoise,/upo, $
        type='float',logunit=logunit,topwid=topwid,verbose=verbose, $
        error=error,debug=debug
    if error ne 0 then return
    if n_elements(uurdnoise) eq 1L then begin
      override=1L
      rdnoise=uurdnoise
    endif
  endif ;; useuserpar
  if ~override then begin
    rdnoise=p3d_misc_fitsstrex(hdr,kwrdn,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
    if rdnoise eq '-1' or size(rdnoise,/type) eq 7L then begin
      tmp='rdnoise'+(index eq ''?'':'['+index+']')
      errmsg=['Could not read any value on the readout noise in the data f' + $
              'ile header.',' Please specify a decimal value using the use' + $
              'r parameter file {'+tmp+'}.']
      goto,error_handler
    endif ;; rdnoise eq -1L
  endif else rsrc='user parameter file'
  if strpos(kwrdn,'ADU') ne 0L then rdnoise/=gain

  if ~finite(gain) or ~finite(rdnoise) then begin
    errmsg='Either gain ['+strtrim(gain,2L)+'] or rdnoise ['+ $
           strtrim(rdnoise,2L)+'] is not finite. Cannot continue!'
    goto,error_handler
  endif ;; ~finite(gain) or ~finite(rdnoise)

  if gain lt 0d0 or rdnoise lt 0d0 then begin
    errmsg='Either gain ['+strtrim(gain,2L)+'] or rdnoise ['+ $
           strtrim(rdnoise,2L)+'] is <0. Cannot continue!'
    goto,error_handler
  endif ;; gain lt 0d0 or rdnoise lt 0d0

  ;;=================-------------
  ;; Logging the information:

  msg=['Read the following values:', $
       '     gain='+string(gain   ,format='(e10.3)')+' [e-/ADU] {'+gsrc+'},', $
       '  rdnoise='+string(rdnoise,format='(e10.3)')+' [ADU]    {'+rsrc+'}.']
  error=p3d_misc_logger(msg,logunit,rname=rname, $
      topwid=topwid,verbose=verbose ge 1L)
  if error ne 0 then return

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
            verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_getinformation
