;+
; NAME:
;         p3d_misc_colortable
;
;         $Id: p3d_misc_colortable.pro 83 2010-03-05 11:33:06Z olestreicher $
;
; PURPOSE:
;         Loads a colortable, according to the input, and also defines a set of
;         pre-defined colors for use with the p3d suite of tools.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_colortable,colortable,bottom=,index=,indv=,define=,name=, $
;             colors=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         colortable [-1] - A scalar string or integer that specifies which
;                           colortable should be loaded. If COLORTABLE is a
;                           string then the file with this filename is loaded
;                           by loadct. If it is an integer then there are the
;                           following cases:
;                               -1:: Loads the Sauron colortable (default)
;                             0-40:: Loads the corresponding colortable that is
;                                   defined by loadct.
;                           Any other number will result in an error being
;                           issued.
;
; KEYWORD PARAMETERS:
;         bottom [max(index)+1] -
;                           A scalar integer that specifies the lower 
;                           index of the colortable that is used in the 
;                           scaling of the colormap. The upper index is always 
;                           the maximum value (!d.table_size). This value
;                           should be set >=2 (5 is the optimum choice) to 
;                           allow some room for the reserved colors of P3D.
;         index [0,1,2,3,4] - 
;                           An array of integers specifying which color
;                           indices should be used for the reserved colors. It
;                           does not make sense to make index larger than five
;                           elements. Also note that the largest index is used
;                           to determine BOTTOM (in the default state). Try to
;                           choose the lowest number of indices possible.
;         indv            - A five-element array of integers that specifies
;                           which color indices should be used for the P3D
;                           reserved colors. If INDEX has >=5 elements, then
;                           INDV is equal to INDEX[0:4]. If INDEX has fewer
;                           elements then all higher indices of INDV are set to
;                           the highest value of INDEX. For instance, with
;                           INDEX=[0,1,2] INDV=[0,1,2,2,2].
;         define          - If this keyword is set then a set of reserved
;                           colors are loaded into the colortable. Up to five
;                           colors are loaded, if the number of elements in
;                           INDEX permits. The colors are:
;                             index[0L]:: black
;                             index[1L]:: white
;                             index[2L]:: red
;                             index[3L]:: green
;                             index[4L]:: dark green
;         name            - This output keyword returns the name of the loaded
;                           colortable.
;         colors          - This output keyword structure returns the loaded
;                           color arrays as:
;                           colors.red, colors.green, and colors.blue.
;         grn             - This output keyword returns the loaded green array.
;         blu             - This output keyword returns the loaded blue array.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         Modifies the colortable.
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         05.02.2009 - Initial programming. /CS
;         
;-
PRO p3d_misc_colortable,colortable,bottom=bottom,index=index,indv=indv, $
        define=define,name=name,colors=colors,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  error=0 & rname='p3d_misc_colortable: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_colortable'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  ;; The Sauron colortable is the default:
  if ~n_elements(colortable) then colortable=-1L

  sb=size(colortable)
  if sb[sb[0L]+2L] ne 1L then begin
    errmsg='COLORTABLE must be set to a scalar integer or string.'
    goto,error_handler
  endif
  if sb[sb[0L]+1L] eq 7L then begin
    if ~file_test(colortable,/regular,/read) then begin
       errmsg='The file pointed at by COLORTABLE ["'+colortable+ $
              '"] does not exist.'
       goto,error_handler
    endif
  endif else if sb[sb[0L]+1L] ge 4L and sb[sb[0L]+1L] le 11L then begin
    errmsg='COLORTABLE must be set to an integer scalar.'
    goto,error_handler
  endif
  if sb[sb[0L]+1L] ne 7L and $
     (colortable lt -1L or colortable gt 40L) then begin
     errmsg='-1<=COLORTABLE<=40 is not satisfied; COLORTABLE='+ $
         strtrim(colortable,2L)+'.'
    goto,error_handler
  endif


  nindex=n_elements(index)
  if ~nindex then index=[0L,1L,2L,3L,4L]
  sb=size(index)
  if sb[sb[0l]+1L] ge 4L and sb[sb[0L]+1L] le 11L then begin
    errmsg='INDEX must be set to an array of integers;' + $
           ' 0<=value<!d.table_size-2.'
    goto,error_handler
  endif
  if min(index) lt 0L or max(index) ge !d.table_size-2L then begin
    errmsg='INDEX must be set to an array of integers;' + $
           ' 0<=value<!d.table_size-2.'
    goto,error_handler
  endif

  nindex=sb[sb[0L]+2L]
  indv=lonarr(5L)
  case nindex of
    1: indv[*]=index[0L]
    2: indv[1L:4L]=index[1L]
    3: begin
      indv[0L]=index[0L]
      indv[1L]=index[1L]
      indv[2L:4L]=index[2L]
    end
    4: begin
      indv[0L]=index[0L]
      indv[1L]=index[1L]
      indv[2L]=index[2L]
      indv[3L:4L]=index[3L]
    end
    else: begin
      indv[0L]=index[0L]
      indv[1L]=index[1L]
      indv[2L]=index[2L]
      indv[3L]=index[3L]
      indv[4L]=index[4L]
    end
  endcase

  if ~n_elements(bottom) then bottom=max(index)+1L
  sb=size(bottom)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0l]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='BOTTOM must be set to a scalar integer;' + $
           ' 0<=value<=!d.table_size-1.'
    goto,error_handler
  endif
  if bottom lt 0L or bottom gt !d.table_size-1L then begin
    errmsg='BOTTOM must be set to a scalar integer;' + $
           ' 0<=value<=!d.table_size-1.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Loading the color table:

  if colortable eq -1L then begin
    p3d_misc_sauron_colormap,ncolors=!d.table_size-bottom,bottom=bottom
    name='Sauron'
  endif else if size(colortable,/type) eq 7L then begin
    loadct,file=colortable,bottom=bottom,ncolors=!d.table_size-bottom, $
           silent=verbose lt 3
    name='User defined'
  endif else begin
    loadct,colortable,bottom=bottom,ncolors=!d.table_size-bottom, $
           silent=verbose lt 3
    loadct,get_names=names
    name=names[colortable]
  endelse

  ;;========================================------------------------------
  ;; Defining colors:

  if keyword_set(define) then begin
    if nindex ge 1L then tvlct,  0b,  0b,  0b,index[0L] ;; Black
    if nindex ge 2L then tvlct,255b,255b,255b,index[1L] ;; White
    if nindex ge 3L then tvlct,255b,  0b,  0b,index[2L] ;; Red
    if nindex ge 4L then tvlct,  0b,255b,  0b,index[3L] ;; Green
    if nindex ge 5L then tvlct,  0b,170b,  0b,index[4L] ;; Dark Green
  endif

  ;;========================================------------------------------
  ;; Reading the current color table and returning the arrays:

  tvlct,red,green,blue,/get
  colors={red:red,green:green,blue:blue}

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
                        verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_colortable
