;+
; NAME:
;         p3d_extract_optimal_mpd
;
;         $Id: p3d_extract_optimal_mpd.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         This routine extracts spectra from a two-dimensional spectrum image
;         using the multi-profile deconvolution approach of
;         Sharp & Birchall 2010, PASA, in press.
;
;         The spectra are extracted using (pre-calculated) input profiles (over
;         the cross-dispersion axis) to sum up the flux at every wavelength
;         bin. The input image is turned into an extracted spectrum image,
;         where every spectrum is placed in an individual row.
;
;         Depending on the number of neighbor profiles that are fitted a diffe-
;         rent method is used to minimize the residual to find the intensities.
;         If only one neighbor profile is used on either side of the reference
;         profile then a tri-diagonal system of equations can be solved. If the
;         number (MPDNPROF) is larger than 1 then it is possible to use either:
;
;         o A band-diagonal sparse matrix solver (MPDMETHOD='band-diagonal')
;
;         o Or solving by singular value decomposition (SVD, MPDMETHOD='svd';
;           cf. Numerical Recipes, 2nd ed., Sect. 2.6).
;
;           NOTE! This method has not yet been optimized for sparse arrays and
;                 is therefore extremely slow.
;
;         If either one of these latter two methods is used then the maximum
;         number of iterations is MPDITMAX. If the SVD-method is used then all
;         singular values which are smaller than MPDSVDLIM are set to 0 before
;         calculating the intensities.
;
;         The following parameters are read from the user parameter file:
;          mpdnprof [1] ::
;               A scalar integer specifying the number of neigbor profiles that
;               are used on each side of the current profile. A tri-diagonal
;               solver can only be used if this value is equal to 1.
;          mpdmethod ['band-diagonal'] ::
;               A scalar string specifying the solver to use with this method.
;               The default is a band-diagonal solver (that is more stable than
;               the built-in function TRISOL for tri-diagonal systems). This
;               value can be set to: 'band-diagonal', 'tri-diagonal', or 'svd'.
;               Where SVD is the singular value decomposition method.
;          mpdthreshold [1d-10] ::
;               A scalar decimal value that is used with the band-diagonal
;               solution method. As the IDL manual page for SPRSIN tells this
;               keyword sets the criterion for deciding the absolute magnitude
;               of the elements to be retained in sparse storage mode.
;          mpdtolerance [1d-10] ::
;               A scalar decimal value that is used with the band-diagonal
;               solution method. As the IDL manual page for LINBCG tells this
;               keyword specifies the desired convergence tolerance.
;          mpdsvdlim [1d-10] ::
;               A scalar decimal value that is used with the singular value
;               decomposition. All singular values that are smaller than this
;               value are set to 0.
;          mpditmax [20] ::
;               A scalar integer that  is used with the singular value
;               decomposition. As the IDL manual page for SVDC tells this
;               keyword specifies the maximum number of iterations.
;          pmultfac [10] ::
;               A scalar integer specifying a subsampling factor that is used
;               when calculating the profile.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: spectrum extraction
;
; CALLING SEQUENCE:
;         p3d_extract_optimal_mpd,im,dim,lprofs,out,dout,detsec=,rdnoise=, $
;             proffun=,/ecalc,userparfile=,stawid=,topwid=, $
;             logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         im              - A two-dimensional array, that holds the spectra
;                           that will be extracted here. The dispersion axis
;                           must be the x-axis.
;         dim             - A two-dimensional array of the same dimensions as
;                           IM, with the errors of IM. DIM must be present.
;                           The variance of IM is DIM.
;         lprofs [sp=size(lprofs)]
;                         - A three-dimensional array, that for every spectrum
;                           bin, of every spectrum, holds the fitting
;                           parameters of a cross-dispersion profile.
;
; KEYWORD PARAMETERS:
;         detsec          - A four-element (columns) -by- number of blocks
;                           (rows) integer array that specifies the detector
;                           region to use on the CCD for each block. RDNOISE
;                           must have as many elements as DETSEC has rows.
;         rdnoise         - A decimal scalar or decimal array that specifies
;                           the readout noise. RDNOISE must have as many
;                           elements as DETSEC has rows.
;         proffun         - A scalar string with the name of the function to
;                           use when (re-)calculating the line profile.
;         ecalc           - If this keyword is set then output errors are
;                           calculated.
;         userparfile     - A scalar string specifying the name of an optional
;                           user parameter file, that could contain any of the
;                           keywords that are described in the routine
;                           description.
;         stawid          - If set, then various messages are written to the
;                           p3d GUI status line (this must be the widget id of
;                           that label widget).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - A two-dimensional array of the extracted spectra
;                           with the same dimensions as the first two
;                           dimensions of LPROFS.
;         dout            - A two-dimensional array of the extracted spectra
;                           with the same dimensions as OUT. This is the error
;                           of OUT. DOUT is onlyt present if ECALC is set.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
;-
PRO p3d_extract_optimal_mpd,im,dim_,lprofs,out,dout,detsec=detsec, $
        rdnoise=rdnoise_,proffun=proffun_,ecalc=ecalc, $
        userparfile=userparfile,stawid=stawid,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_extract_optimal_mpd: '
  if ~n_elements(verbose) then verbose=0
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)
  loglevel=~n_elements(logunit)?0L:logunit[1L]

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_extract_optimal_mpd'
    return
  endif ;; keyword_set(help) or ~n_params()

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(im)
  if ~s[s[0L]+2L] or s[0L] ne 2L or $
     (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='IM [1] must be set to a two-dimensional array of decimal type.'
    goto,error_handler
  endif
  ncim=s[2L]

  se=size(dim_)
  if ~se[se[0L]+2L] or (se[se[0L]+1L] ge 6L and se[se[0L]+1L] le 11L) or $
     (se[se[0L]+2L] ge 1L and (se[0L] ne s[0L] or se[1L] ne s[1L] or $
                               se[se[0L]+2L] ne s[s[0L]+2L])) then begin
    errmsg='DIM [2] {'+strtrim(se[se[0L]+1L],2L)+',['+strtrim(se[1L],2L)+ $
           ','+strtrim(se[2L],2L)+']} must be of the same ' + $
           'dimensions as IM {'+strtrim(s[s[0L]+1L],2L)+',['+ $
           strtrim(s[1L],2L)+','+strtrim(s[2L],2L)+']}.'
    goto,error_handler
  endif
  dim=dim_

  sp=size(lprofs)
  if ~sp[sp[0L]+2L] or sp[0L] ne 3L or $
     (sp[sp[0L]+1L] ge 6L and sp[sp[0L]+1L] le 11L) or $
     (sp[0L] eq 3L and (sp[1L] ne s[1L])) then begin
    errmsg='LPROFS [3] must be set to a three-dimensional array of decimal' + $
           ' type, with as any elements in the first dimension as IM.'
    goto,error_handler
  endif
  nwl =sp[1L]
  nprf=sp[2L]

  tmp='scalar'
  nblocks=1L
  if n_elements(detsec) ne 0L then begin
    sd=size(detsec)
    nblocks=(sd[0L] eq 1L)?1L:sd[2L]
    if (sd[0L] eq 1L and sd[1L] ne 4L) or $
       (sd[0L] eq 2L and sd[1L] ne 4L) or $
       (sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L) then begin
      errmsg='DETSEC must have the dimensions [4,NBLOCKS], not ['+ $
             strtrim(sd[1L],2L)+','+strtrim(sd[2L],2L)+'].'
      goto,error_handler
    endif
    tmp='4x'+strtrim(nblocks,2L)+'-element array'
  endif ;; nblocks gt 1L

  sb=size(rdnoise_)
  if ~sb[sb[0L]+2L] or $
     (sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='RDNOISE must be set to a decimal '+tmp+'; RDNOISE>0.'
    goto,error_handler
  endif
  if min(rdnoise_) lt 0d0 or sb[sb[0L]+2L] ne nblocks then begin
    errmsg='RDNOISE must be set to a decimal '+tmp+'; RDNOISE>0.'
    goto,error_handler
  endif

  sb=size(proffun_)
  if sb[sb[0L]+2L] ne 1L or sb[sb[0L]+1L] ne 7L then begin
    errmsg='PROFFUN must be set to a scalar string with the name of the fu' + $
           'nction to use.'
    goto,error_handler
  endif

  proffun=strlowcase(proffun_)
  case proffun of
    'gaussian':      iidx=2L
    'lorentzian':    iidx=2L
    'gauss/lorentz': iidx=3L
    'doublegauss':   iidx=4L
    else: begin
      errmsg=['PROFFUN must be one of the four options:', $
              '  "gaussian", "lorentzian", "gauss/lorentz", "doublegauss"', $
              ' PROFFUN="'+proffun_+'" is not a valid option.']
      goto,error_handler
    end
  endcase ;; proffun

  if n_elements(userparfile) ne 0L then begin
    sp=size(userparfile)
    if sp[sp[0L]+2L] ne 1L or sp[sp[0L]+1L] ne 7L then begin
      errmsg='USERPARFILE must, if set, be a scalar string.'
      goto,error_handler
    endif
    if userparfile[0L] ne '' then begin
      if ~file_test(userparfile,/regular,/read) then begin
        errmsg='USERPARFILE, cannot read the file "'+userparfile+'".'
        goto,error_handler
      endif

      ;; Reading the user parameter file data:
      uparname_='' & uparvalue_=''
      readcol,userparfile,uparname_,uparvalue_,format='a,a',comment=';', $
          silent=verbose lt 3,delimiter=' '
      if n_elements(uparname_) ne 0L then begin
        uparname=uparname_ & uparval=uparvalue_
      endif ;; nelements(uparname_) ne 0L
    endif ;; userparfile[0L] ne ''
  endif ;; n_elements(userparfile) ne 0L

  ecalc=keyword_set(ecalc)

  ;;========================================------------------------------
  ;;========================================------------------------------
  ;; Reading various MPD-related parameters:

  ;;====================---------------
  ;; Profile subsampling factor:

  p3d_misc_read_params,uparname,uparval,'pmultfac',pmultfac,/upo,/a0, $
      type='integer',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(pmultfac) then pmultfac=10L

  sp=size(pmultfac)
  if ~sp[sp[0L]+2L] or $
     (sp[sp[0L]+1L] ge 4L and sp[sp[0L]+1L] le 11L) then begin
    errmsg='PMULTFAC must be set to a scalar integer; PMULTFAC>0.'
    goto,error_handler
  endif
  if pmultfac le 0L then begin
    errmsg='PMULTFAC must be set to a scalar integer; PMULTFAC>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; Number of neighbor spectra to use when deconvolving the intensities:

  p3d_misc_read_params,uparname,uparval,'mpdnprof',mpdnprof,/upo,/a0, $
      type='integer',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(mpdnprof) then mpdnprof=1L
  if mpdnprof le 0L then begin
    errmsg='MPDNPROF must be set to a scalar integer; MPDNPROF>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; Selecting the multi-profile deconvolution solution method to use:

  p3d_misc_read_params,uparname,uparval,'mpdmethod',mpdmethod,/upo,/a0, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(mpdmethod) then mpdmethod='band-diagonal'
  mpdmethod=strlowcase(strtrim(mpdmethod,2L))
  if mpdmethod ne 'svd' and mpdmethod ne 'band-diagonal' and $
     mpdmethod ne 'tri-diagonal' then begin
    errmsg='MPDMETHOD must be either  ["band-diagonal"], "tri-diagonal", o' + $
           'r "svd".'
    goto,error_handler
  endif

  ;;====================---------------
  ;; The maximum number of possible iterations when using a band-diagonal or
  ;; singular value decomposition solution:

  p3d_misc_read_params,uparname,uparval,'mpditmax',mpditmax,/upo,/a0, $
      type='integer',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(mpditmax) then mpditmax=20L
  if mpditmax le 0L then begin
    errmsg='MPDITMAX must be set to a scalar integer; MPDITMAX>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; When using a band-diagonal solution method, matrix entries with a value
  ;; smaller than this threshold will be set to 0:

  p3d_misc_read_params,uparname,uparval,'mpdthreshold',mpdthreshold,/upo,/a0, $
      type='float',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(mpdthreshold) then mpdthreshold=1d-10
  if mpdthreshold le 0d0 then begin
    errmsg='MPDTHRESHOLD must be set to a scalar decimal value; MPDTHRESHO' + $
           'LD>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; When using a band-diagonal solution method matrix the following value is
  ;; used as a tolerance to determine when the solution is final:

  p3d_misc_read_params,uparname,uparval,'mpdtolerance',mpdtolerance,/upo,/a0, $
      type='float',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(mpdtolerance) then mpdtolerance=1d-10
  if mpdtolerance le 0d0 then begin
    errmsg='MPDTOLERANCE must be set to a scalar decimal value; MPDTOLERAN' + $
           'CE>0.'
    goto,error_handler
  endif

  ;;====================---------------
  ;; When using a singular value decomposition solution the singular values,
  ;; which are smaller than this limiting value, are set to 0:

  p3d_misc_read_params,uparname,uparval,'mpdsvdlim',mpdsvdlim,/upo,/a0, $
      type='float',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then return
  if ~n_elements(mpdsvdlim) then mpdsvdlim=1d-10
  if mpdsvdlim le 0d0 then begin
    errmsg='MPDSVDLIM must be set to a scalar decimal value; MPDSVDLIM>0.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;;========================================------------------------------
  ;; Preparing the calculations

  c=reform(lprofs[*,*,0L])          ;; The center positions
  dist=median(c[*,1L:nprf-1L]-c[*,0L:nprf-2L]) ;; The median spectrum sep.
  mpdprofwidth=ceil(dist*(mpdnprof+0.5d0))
  n=2L*mpdprofwidth+1L              ;; The profile pixel width
  c=round(c) & clo=c-mpdprofwidth

  out=dblarr(nwl,nprf)
  if ecalc then dout=out

  ;; The array that will hold the normalized line profiles:
  x0=rebin(dindgen(n)-mpdprofwidth,n,pmultfac)
  x1=rebin(dindgen(1L,pmultfac)/pmultfac,n,pmultfac)
  x=x0+x1

  ;;========================================------------------------------
  ;; Logging the used parameters:

  msg='Using the extraction method: Multi-profile deconvolution'
  msg=[msg,'Using the following extraction parameters:',' profwidth='+ $
       string(mpdprofwidth,format='(f9.3)')+ $
       ' :: spectrum extraction half width.']
  msg=[msg,'  Profile subsampling factor='+ $
       string(format='(i9)',pmultfac)+'.']
  msg=[msg,'       method='+mpdmethod, $
           '     mpdnprof='+string(mpdnprof,format='(i10)'), $
           '     mpditmax='+string(mpditmax,format='(i10)')]
  if mpdmethod eq 'band-diagonal' then begin
    msg=[msg,' mpdthreshold='+string(format='(e10.3)',mpdthreshold), $
             ' mpdtolerance='+string(format='(e10.3)',mpdtolerance)]
  endif
  if mpdmethod eq 'svd' then $
     msg=[msg,'    mpdsvdlim='+string(format='(e10.3)',mpdsvdlim)]
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  ;;========================================------------------------------
  ;; Resizing RDNOISE to full frame arrays in order to easily sum up
  ;; values of spectra which extend across several CCD blocks:

  rdnoise=dblarr(nwl,ncim)
  case nblocks of
    1: rdnoise+=rdnoise_
    else: begin
      xmint=min(detsec[0L:1L,*])
      ymint=min(detsec[2L:3L,*])
      for j=0L,nblocks-1L do begin
        xoff=min(detsec[0L:1L,j]) ne xmint?xmint:0L
        yoff=min(detsec[2L:3L,j]) ne ymint?ymint:0L

        xmin=min(detsec[0L:1L,j])-xmint-xoff
        xmax=max(detsec[0L:1L,j])-xmint-xoff
        ymin=min(detsec[2L:3L,j])-ymint-yoff
        ymax=max(detsec[2L:3L,j])-ymint-yoff

        rdnoise[xmin:xmax,ymin:ymax]=rdnoise_[j]
      endfor ;; j=0L,nblocks-1L
    end ;; else
  endcase ;; nblocks

  ;;========================================------------------------------
  ;; Looping over all wavelength bins:

  itimes=dblarr(nwl)
  wlstr='/'+strtrim(nwl,2L)
  i=-1L & while ++i lt nwl do begin
    itimes[i]=systime(1L)

    ;;==============================--------------------
    ;; Output of information:

    if ~(i mod 50L) then begin
      tmp=strtrim(i+1L,2L)+'-'+strtrim((i+50L)<nwl,2L)
      msg='[Optimal spectrum extraction] w.l.bins '+tmp+wlstr
      if i gt 0L then begin
        tmp=total(itimes[0L:i-1L])
        tmp=strtrim(round(tmp*(double(nwl)/i-1d0)),2L)
        msg+=' :: remaining time: ~'+strtrim(tmp,2L)+'s.'
      endif else msg+='.'
      if usestawid then widget_control,stawid,set_value=msg
      ;; Print to the console, if verbose=1 (otherwise it's printed below)
      error=p3d_misc_logger(msg,rname=rname,verbose=verbose eq 1)
    endif ;; ~(i mod 50L)

    ;;==============================--------------------
    ;; Extracting and normalizing pixel-based cross-dispersion profiles for
    ;; every spectrum of the current wavelength bin:

    prof=dblarr(nprf,ncim)
    j=-1L & while ++j lt nprf do begin
      if lprofs[i,j,iidx] eq 0d0 then continue

      lo=(mpdprofwidth-c[i,j])>0L
      hi=(mpdprofwidth-c[i,j]+ncim-1L)<(n-1L)

      xarr=x+c[i,j]
      if lo gt 0L or hi lt n-1L then xarr=xarr[lo:hi,*]

      f=p3d_misc_profunction(xarr,lprofs[i,j,*],proffun=proffun,/nobg, $
            topwid=topwid,error=error,verbose=verbose)
      if error ne 0 then return

      if pmultfac gt 1L then f=total(f,2L)/pmultfac
      prof[j,(clo[i,j]>0L):((clo[i,j]+n-1L)<(ncim-1L))]=f/total(f)
    endwhile ;; ++j lt nprf

    ;;==============================--------------------
    ;; Solving the linear system of equations either using a tri-diagonal, a
    ;; band-diagonal, or singular value decomposition (SVD):
    ;;   Actually finding the solution that minimizes a residual

    case mpdmethod of
      'tri-diagonal': begin
        ;;==============================--------------------
        ;; Populating matrices:

         bj=dblarr(nprf)
        ckj=dblarr(3L,nprf)
        if ecalc then begin
           ej=dblarr(nprf)
          ekj=dblarr(3L,nprf)
        endif

        j=-1L & while ++j lt nprf do begin
          fl= clo[i,j]>0L
          fh=(clo[i,j]+n-1L)<(ncim-1L)

          ;; Eqs.(9) and (17):
          bj[j]=total(prof[j,fl:fh]*  im[i,fl:fh]/dim[i,fl:fh]^2)
          if ecalc then $
             ej[j]=total(prof[j,fl:fh]*(dim[i,fl:fh]^2-rdnoise[i,fl:fh]^2))

          k=((j-1L)>0L)-1L & kmax=(j+1L)<(nprf-1L)
          ki=~j?0L:-1L
          while ++k le kmax do begin  ;; ;; Eq.(9) & Eq.(17)
            tmp=prof[j,fl:fh]*prof[k,fl:fh]
            if ecalc then ekj[ki,j]=total(tmp)
            ckj[++ki,j]=total(temporary(tmp)/dim[i,fl:fh]^2)
          endwhile ;; ++k le kmax
        endwhile ;; ++j lt nprf

        ;; Tri-diagonal:
        arr=trisol(reform(ckj[0L,*]),reform(ckj[1L,*]),reform(ckj[2L,*]),bj)
        if ecalc then $
           err=trisol(reform(ekj[0L,*]),reform(ekj[1L,*]),reform(ekj[2L,*]),ej)
      end ;; 'tri-diagonal'
      else: begin

        ;;==============================--------------------
        ;; Populating matrices:

        bj=dblarr(nprf) & ckj=dblarr(nprf,nprf)
        if ecalc then begin
          ej=dblarr(nprf) & ekj=dblarr(nprf,nprf)
        endif

        j=-1L & while ++j lt nprf do begin
          fl= clo[i,j]>0L
          fh=(clo[i,j]+n-1L)<(ncim-1L)

          ;; Eqs.(9) and (17):
          bj[j]=total(prof[j,fl:fh]*  im[i,fl:fh]/dim[i,fl:fh]^2)
          if ecalc then $
             ej[j]=total(prof[j,fl:fh]*(dim[i,fl:fh]^2-rdnoise[i,fl:fh]^2))

          k=((j-mpdnprof)>0L)-1L & kmax=(j+mpdnprof)<(nprf-1L)
          while ++k le kmax do begin
            a=prof[j,fl:fh]*prof[k,fl:fh]
            if ecalc then ekj[k,j]=total(a)
            ckj[k,j]=total(temporary(a)/dim[i,fl:fh]^2)
          endwhile ;; ++k le kmax
        endwhile ;; ++j lt nprf

        ;;==============================--------------------
        ;; Using either a band-diagonal, or singular value decomposition:

        case mpdmethod of
          'band-diagonal': begin
            ;; Solving for the intensities:
            xinit=dblarr(nprf)+1d0
            tmp=sprsin(ckj,threshold=mpdthreshold)
            arr=linbcg(temporary(tmp),bj,xinit,tol=mpdtolerance, $
                    iter=viter,itmax=mpditmax)

            ;; Solving for the variances:
            if ecalc then begin
              xinit=dblarr(nprf)+1d0
              tmp=sprsin(ekj,threshold=mpdthreshold)
              err=linbcg(temporary(tmp),ej,xinit,tol=mpdtolerance, $
                         iter=eiter,itmax=mpditmax)
            endif ;; ecalc
          end ;; 'band-diagonal'
          else: begin ;; SVD
            ;; Solving for the intensities:
            svdc,ckj,w,u,v,itmax=mpditmax
            idx=where(w lt mpdsvdlim,count)
            w=1/w & if count ne 0L then w[idx]=0d0
            arr=u##diag_matrix(w)##(transpose(v)##transpose(bj))

            if ecalc then begin
              ;; Solving for the variances:
              svdc,ekj,w,u,v,itmax=mpditmax
              idx=where(w lt mpdsvdlim,count)
              w=1/w & if count ne 0L then w[idx]=0d0
              err=u##diag_matrix(w)##(transpose(v)##transpose(ej))
            endif ;; ecalc
          end ;; else
        endcase ;; mpdmethod
      end ;; else
    endcase ;; mpdmethod

    out[i,*]=arr
    if ecalc then dout[i,*]=sqrt(err)

    ;;==============================--------------------
    ;; Logging information about the iterations:

    if verbose ge 2 or loglevel ge 2 then begin
      msg=string(format='("  wavelength bin ",i4,"/",i4)',i+1L,nwl)
      if n_elements(viter) ne 0L then begin
        if ecalc then begin
          msg+=' ::      #intensity,variance iter.=' + $
               string(format='(i2,", ",i2)',viter,eiter)+'.'
        endif else begin
          msg+=' ::      #intensity iter.=' + $
               string(format='(i2,", ",i2)',viter)+'.'
        endelse ;; ecalc
      endif ;; n_elements(viter) ne 0L
      error=p3d_misc_logger(msg,logunit,loglevel=2,rname=rname, $
                verbose=verbose ge 2)
    endif ;; verbose ge 2 or loglevel ge 2

    itimes[i]=systime(1L)-itimes[i]
  endwhile ;; ++i lt nwl

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_extract_optimal_mpd
