;+
; NAME:
;         p3d_cobjex
;
;         $Id: p3d_cobjex.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         A more detailed description is pending.
;
;         This program is called from the p3d GUI, but can also be used as a 
;         separate program.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: main routine
;
; CALLING SEQUENCE:
;         p3d_cobjex,filename,parfile,out,bswout,icstr=,ostr=,masterbias=, $
;             tracemask=,dispmask=,flatfield=,crmask=,waveprompt=,skyalign=, $
;             /savee3d,/sbsw,shiftreffile=,trcshiftfile=,dispshiftfile=, $
;             objshiftfile=,userparfile=,/exmonitor,ofilename=, $
;             bsw_ofilename=,opath=,detector=,sfx=,/compress,logfile=, $
;             loglevel=,stawid=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         filename        - An array of strings that specifies the names of the
;                           raw data files, which are used when creating the
;                           extracted object file.
;         parfile         - A scalar string that specifies the name of a file
;                           with instrument-specific setup parameters.
;
; KEYWORD PARAMETERS:
;         icstr ['_imcmb'] - A scalar string with the image-comb. specific
;                           string that is used to create the output filename.
;         ostr ['_oextr'] - A scalar string with the object-data specific
;                           string that is used to create the output filename.
;         masterbias      - A scalar string with the name of a master bias
;                           file. This file is required if optimal extraction
;                           is used.
;         tracemask       - A scalar string with the name of a trace mask file.
;         dispmask        - A scalar string with the name of a dispersion mask.
;                           (If a separate dispersion mask is wanted for the
;                            flat field then the flat-field header must contain
;                            the entry IMDMK, which value is used when applying
;                            the dispersion correction to the fiber flat.)
;         flatfield       - A scalar string with the name of a flat field.
;         crmask          - A scalar string with the name of a cosmic ray mask.
;                           Note! This mask is only used if optimal extraction
;                                 is switched on in the user parameter file.
;         waveprompt [0]  - This keyword determines if the the user should be
;                           queried for changing CRVAL, CDELT, and NPIX
;                           interactively (they can all be set using the user
;                           parameter file).
;         skyalign        - A scalar string that contains the name of a
;                           linelist file with sky emission lines
;                           (cf. p3d_wavecal_dispersion_correction).
;         savee3d         - If this keyword is set then the output data is
;                           saved in the E3D format in addition to the usual
;                           RSS format (not yet implemented).
;         sbsw            - This keyword returns the beam switch setting;
;                             sbsw=1: beam switch data == true.
;         shiftreffile    - n/a.
;         trcshiftfile    - n/a.
;         dispshiftfile   - n/a.
;         objshiftfile    - n/a.
;         userparfile     - A scalar string with the name of a file with user-
;                           defined parameters. The following parameters are
;                           read here from this file:
;                             'methodimcombine':: see p3d_misc_imcombine.pro
;                             'methodextract':: see p3d_misc_extract_method.pro
;                             'detsec':: see p3d_misc_detsec.pro
;         ofilename       - This keyword returns the full name of the created
;                           extracted object file.
;         exmonitor       - If this keyword is set then a line profiles viewer
;                           is shown after all fitting is done when using
;                           optimal extraction.
;         bsw_ofilename   - n/a.
;         opath ['.']     - A scalar string that specifies the path, where the
;                           output data is saved.
;         detector [0]    - A scalar integer that specifies the currently
;                           selected detector; DETECTOR is a zero-based value.
;         sfx [.fits]     - A scalar string specifying the file ending (without
;                           a trailing compression suffix, such as .gz or .Z).
;         compress [0]    - If this keyword is set then the output data file is
;                           compressed (using gzip).
;         logfile         - If this routine is launched as a separate program
;                           then this keyword specifies the name of the log
;                           file that will be created.
;         loglevel [1]    - If this routine is launched as a separate program,
;                           and LOGFILE is used, then this keyword specifies
;                           the logging level; [1],2,3 (the higher the value
;                           the more output).
;         stawid          - If set, then various messages are written to the
;                           p3d GUI status line (this must be the widget id of
;                           that label widget).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         out             - Upon a successful execution this variable contains
;                           the extracted data when the routine exits.
;         bswout          - Upon a successful execution this variable contains
;                           the extracted beam switch data when the routine
;                           exits.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; MODIFICATION HISTORY:
;         03.09.2009 - Routine introduced (moved conts. from p3d_gui). /CS
;
;-
PRO p3d_cobjex_waveprompt_event,event
  compile_opt hidden,IDL2

  rname='p3d_cobjex_waveprompt_event: '

  ;;========================================------------------------------
  ;; Get the state information:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  on_ioerror,reset_label

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Handling tracking events first:

  tmpstrname=strlowcase(tag_names(event,/structure_name))
  if tmpstrname eq 'widget_tracking' then begin
    if event.enter then begin  
      infostr=widget_info(event.id,/uname)
      widget_control,(*state).lstatwid,set_value=infostr
    endif else widget_control,(*state).lstatwid,set_value=''

    ;; Nothing more to do, returning:
    return
 endif

  if tmpstrname eq 'widget_kbrd_focus' then if event.enter then return

  ;;========================================------------------------------
  ;; Get the user value of the event structure:

  widget_control,event.id,get_uvalue=uval,get_value=val
  s=size(val)

  case uval of
    'crval': begin
      tmp=double(val[0L])
      if tmp lt 0d0 then begin
        widget_control,(*state).tcrvawid,set_value=strtrim((*state).crval,2L)
      endif else (*state).crval=tmp
      widget_control,event.id,set_value=strtrim(tmp,2L)
    end
    'cdelt': begin
      tmp=double(val[0L])
      if tmp lt 0d0 then begin
        widget_control,(*state).tcdelwid,set_value=strtrim((*state).cdelt,2L)
      endif else (*state).cdelt=tmp
      widget_control,event.id,set_value=strtrim(tmp,2L)
    end
    'npix': begin
      tmp=long(val[0L])
      if tmp lt 0L then begin
        widget_control,(*state).tnpixwid,set_value=strtrim((*state).npix,2L)
      endif else (*state).npix=tmp
      widget_control,event.id,set_value=strtrim(tmp,2L)
    end
    'ok': widget_control,event.top,/destroy
  endcase ;; uval

  return

reset_label:
  case event.id of
    (*state).tcrvawid: begin
      tmp=strtrim((*state).crval,2L)
      tmpstr='Incorrect value; must be a decimal value.'
    end
    (*state).tcdelwid: begin
      tmp=strtrim((*state).cdelt,2L)
      tmpstr='Incorrect value; must be a decimal value.'
    end
    (*state).tnpixwid: begin
      tmp=strtrim((*state).npix,2L)
      tmpstr='Incorrect value; must be an integer.'
    end
  endcase
  widget_control,event.id,set_value=tmp
  widget_control,(*state).lstatwid,set_value=tmpstr

  return
END ;;; procedure: p3d_cobjex_waveprompt_event


PRO p3d_cobjex_waveprompt,crval=crval,cdelt=cdelt,npix=npix, $
        topwid=topwid,verbose=verbose,error=error,debug=debug
  compile_opt hidden,IDL2

  rname='p3d_cobjex_waveprompt: '

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Setting up a widget hierarchy:

  bbasewid=widget_base(/column,/base_align_center,group_leader=topwid, $
      tlb_frame_attr=1L+4L,/floating,title='p3d_cobjex: waveprompt')

  bbas0wid=widget_base(bbasewid,/column,space=0L,xpad=0L,ypad=0L,/frame)

  bbas1wid=widget_base(bbas0wid,/row,/base_align_center)
  tlabewid=widget_label(bbas1wid,value='   0-Wavelength [crval]:')
  tcrvawid=widget_text(bbas1wid,value=strtrim(crval,2L),uvalue='crval', $
      xsize=20L,uname='Sets the wavelength of pixel 1.',/tracking_events, $
      /editable,/kbrd_focus_events)

  bbas2wid=widget_base(bbas0wid,/row,/base_align_center)
  tlabewid=widget_label(bbas2wid,value='   C-Dispersion [cdelt]:')
  tcdelwid=widget_text(bbas2wid,value=strtrim(cdelt,2L),uvalue='cdelt', $
      xsize=20L,uname='Sets the constant dispersion.',/tracking_events, $
      /editable,/kbrd_focus_events)

  bbas3wid=widget_base(bbas0wid,/row,/base_align_center)
  tlabewid=widget_label(bbas3wid,value='# Wavelength bins [npix]:')
  tnpixwid=widget_text(bbas3wid,value=strtrim(npix,2L),uvalue='npix', $
      xsize=20L,uname='Sets the number of wavelength bins.',/tracking_events, $
      /editable,/kbrd_focus_events)

  lstatwid=widget_label(bbasewid,/sunken_frame,value='t',uname='Widget tool'+ $
      ' information line.',/tracking_events)

  bdonewid=widget_button(bbasewid,value='  OK  ',uvalue='ok', $
      uname='Press this button when ready.',/tracking_events)

  widget_control,bbasewid,/realize

  ;; Re-sizing the status line widget:
  tmp=widget_info(bbasewid,/geometry)
  widget_control,lstatwid,xsize=tmp.xsize-2L*tmp.xpad
  widget_control,lstatwid,set_value=widget_info(lstatwid,/uname)

  ;; Centering the tool:
  ttp=widget_info(topwid,/geometry)
  xpos=ttp.xoffset+ttp.xsize/2L-tmp.xsize/2L
  ypos=ttp.yoffset+ttp.ysize/2L-tmp.ysize/2L
  widget_control,bbasewid,tlb_set_xoffset=xpos
  widget_control,bbasewid,tlb_set_yoffset=ypos

  sstate={lstatwid:lstatwid,bdonewid:bdonewid,tcrvawid:tcrvawid, $
          tcdelwid:tcdelwid,tnpixwid:tnpixwid,crval:crval,cdelt:cdelt, $
          npix:npix,verbose:verbose,debug:debug}

  ;; Storing the state information in the widget tree:
  pstate=ptr_new(sstate,/no_copy)
  widget_control,bbasewid,set_uvalue=pstate

  xmanager,'p3d_cobjex_waveprompt',bbasewid

  ;; Returning the set values:
  crval=(*pstate).crval & cdelt=(*pstate).cdelt & npix=(*pstate).npix

  ;; Freeing the memory of the state pointer:
  ptr_free,pstate

  return
END ;;; procedure: p3d_cobjex_waveprompt


PRO p3d_cobjex,filename,parfile,out,bswout,icstr=icstr,ostr=ostr, $
        masterbias=masterbias,tracemask=tracemask,dispmask=dispmask, $
        flatfield=flatfield,crmask=crmask,waveprompt=waveprompt, $
        skyalign=skyalign,savee3d=savee3d,sbsw=sbsw, $
        shiftreffile=shiftreffile,trcshiftfile=traceshiftfile, $
        dispshiftfile=dispshiftfile,objshiftfile=objectshiftfile, $
        userparfile=userparfile,ofilename=ofilename,exmonitor=exmonitor, $
        bsw_ofilename=bsw_ofilename, $
        opath=opath_,detector=d,sfx=sfx,compress=compress,logfile=logfile, $
        loglevel=loglevel,stawid=stawid,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_cobjex: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_cobjex'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  logfileisopened=0L
  if ~n_elements(logunit) and n_elements(logfile) eq 1L then begin
    if size(logfile,/type) ne 7L then begin
      errmsg='LOGFILE must, if specified, be a scalar string with the name' + $
             ' of a file to write.'
      goto,error_handler
    endif

    openw,logunit,logfile,/get_lun,error=error
    if error ne 0 then begin
      errmsg='Could not open a logfile with the name "'+logfile+'".'
      goto,error_handler
    endif
    logfileisopened=1L

    if ~n_elements(loglevel) then loglevel=1L
    s=size(loglevel)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='LOGLEVEL must be a scalar integer; 1<=LOGLEVEL<=3.'
      goto,error_handler
    endif
    if loglevel lt 1L or loglevel gt 3L then begin
      errmsg='LOGLEVEL must be a scalar integer; 1<=LOGLEVEL<=3.'
      goto,error_handler
    endif

    logunit=[logunit,loglevel]
  endif ;; ~n_elements(logunit) and n_elements(logfile) eq 1L

  s=size(filename) & n=s[s[0L]+2L]
  if ~n or s[s[0L]+1L] ne 7L then begin
    errmsg='FILENAME [1] must be specified, and be of string type.'
    goto,error_handler
  endif
  for i=0L,n-1L do begin
    if ~file_test(filename[i],/read,/regular) then begin
      errmsg='Cannot read or find the file "'+filename[i]+'" ['+ $
          strtrim(i+1L,2L)+'/'+strtrim(n,2L)+']'
      goto,error_handler
    endif
  endfor

  if size(parfile,/type) ne 7L then begin
    errmsg='PARFILE [2] must be specified; as the scalar filename of an in' + $
           'strument-specific parameter file.'
    goto,error_handler
  endif
  if ~file_test(parfile,/read,/regular) then begin
    errmsg='The file PARFILE "'+parfile+'" does not exist.'
    goto,error_handler
  endif

  if ~n_elements(ostr) then ostr='_oextr'
  if size(ostr,/type) ne 7L or n_elements(ostr) ne 1L then begin
    errmsg='OSTR must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  usembiasfile=0L
  if n_elements(masterbias) eq 1L then begin
    if size(masterbias,/type) ne 7L then begin
      errmsg='MASTERBIAS must, if set, be of string type.'
      goto,error_handler
    endif
    if ~file_test(masterbias,/read,/regular) then begin
      errmsg='The file MASTERBIAS "'+masterbias+'" does not exist.'
      goto,error_handler
    endif
    usembiasfile=1L
  endif

  if ~n_elements(tracemask) or size(tracemask,/type) ne 7L then begin
    errmsg='TRACEMASK must be set and be of string type.'
    goto,error_handler
  endif
  if ~file_test(tracemask,/read,/regular) then begin
    errmsg='The file TRACEMASK "'+tracemask+'" does not exist.'
    goto,error_handler
  endif

  usedispmask=0L
  if n_elements(dispmask) eq 1L then begin
    if size(dispmask,/type) ne 7L then begin
      errmsg='DISPMASK must, if set, be of string type.'
      goto,error_handler
    endif
    if ~file_test(dispmask,/read,/regular) then begin
      errmsg='The file DISPMASK "'+dispmask+'" does not exist.'
      goto,error_handler
    endif
    usedispmask=1L
  endif ;; n_elements(dispmask) eq 1L

  useflatfield=0L
  if n_elements(flatfield) eq 1L then begin
    if size(flatfield,/type) ne 7L then begin
      errmsg='FLATFIELD must, if set, be of string type.'
      goto,error_handler
    endif
    if ~file_test(flatfield,/read,/regular) then begin
      errmsg='The file FLATFIELD "'+flatfield+'" does not exist.'
      goto,error_handler
    endif
    useflatfield=1L
  endif ;; n_elements(flatfield) eq 1L

  usecrmask=0L
  if n_elements(crmask) eq 1L then begin
    if size(crmask,/type) ne 7L then begin
      errmsg='CRMASK must, if set, be of string type.'
      goto,error_handler
    endif
    if ~file_test(crmask,/read,/regular) then begin
      errmsg='The file CRMASK "'+crmask+'" does not exist.'
      goto,error_handler
    endif
    usecrmask=1L
  endif ;; n_elements(crmask) eq 1L

  if n_elements(userparfile) ne 0L then begin
    if size(userparfile,/type) ne 7L then begin
      errmsg='USERPARFILE must, if specified, be of string type.'
      goto,error_handler
    endif

    if userparfile ne '' then begin
      if ~file_test(userparfile,/read,/regular) then begin
        errmsg='The file USERPARFILE "'+userparfile+'" does not exist.'
        goto,error_handler
      endif

      ;; Reading the user parameter file:
      readcol,userparfile,uparname,uparvalue,format='a,a', $
          silent=verbose lt 3,delimiter=' ',comment=';'
    endif ;; userparfile ne ''
  endif ;; n_elements(userparfile) ne 0L

  if n_elements(opath_) ne 0L then begin
    if size(opath_,/type) ne 7L then begin
      errmsg='OPATH must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(opath_,/directory,/read) then begin
      errmsg='The directory OPATH "'+opath_+'" does not exist.'
      goto,error_handler
    endif
  endif else begin
    cd,current=opath_
  endelse
  opath=opath_
  idx=strpos(opath,path_sep(),/reverse_search)
  if idx ne strlen(opath)-1L then opath+=path_sep()

  if ~n_elements(d) then d=0L
  s=size(d)
  if s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L then begin
    errmsg='DETECTOR must be of (any) integer type; DETECTOR>=0.'
    goto,error_handler
  endif
  if d lt 0L then begin
    errmsg='DETECTOR must be of (any) integer type; DETECTOR>=0.'
    goto,error_handler
  endif

  if ~n_elements(sfx) then sfx='.fits'
  if size(sfx,/type) ne 7L or n_elements(sfx) ne 1L then begin
    errmsg='SFX must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  compress=keyword_set(compress)
  savee3d=keyword_set(savee3d)

  ;;========================================------------------------------
  ;; Routine information:

  msg=['','==============================--------------------', $
          '========== Extracting object data', $
          '==========   '+systime(), $
          '==============================--------------------']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 2)

  ;;========================================------------------------------
  ;; Reading instrument-specific parameters:

  ppath=strmid(parfile,0L,strpos(parfile,path_sep(),/reverse_search)+1L)
  parname='' & parvalue=''
  readcol,parfile,parname,parvalue,format='a,a',delimiter=' ', $
      silent=verbose lt 3,comment=';'

  p3d_misc_read_params,parname,parvalue,'daxis',daxis,/must_exist,/nou, $
      type='integer',logunit=logunit,topwid=topwid, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2 (not '+strtrim(daxis,2L)+').'
    goto,error_handler
  endif

  p3d_misc_read_params,parname,parvalue,'keywordsfile',kwrdlist,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  kwrdlist=ppath+kwrdlist
  if ~file_test(kwrdlist,/read,/regular) then begin
    errmsg='The file KWRDLIST "'+kwrdlist+'" does not exist.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Combining the data files, at first the data is combined accounting for if
  ;; the number of blocks>1, and if there are more than 3 (groups of) files:

  if n_elements(filename) eq 2L then begin
    errmsg='You must select either 1, or >=3 (sets of) raw data images.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Combining the data files, at first the data is combined accounting for if
  ;; the number of blocks>1, and if there are more than 3 (groups of) files:

  title='master object '
  p3d_misc_imcombine_wrapper,filename,ofilename,ofsfx,ostr=icstr, $
      detsec=detsec,parfile=parfile,userparfile=userparfile,opath=opath, $
      sfx=sfx,xstr=xstr,dsh=dsh,compress=compress,title=title,stawid=stawid, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  filename=ofilename
  fsfx=ofsfx

  ;; Writing information to the status line:
  if usestawid and dsh then begin
     tmpstr='[Object ex.] Wrote the master object file "'+ $
            file_basename(ofilename)+'".'
     widget_control,stawid,set_value=tmpstr
  endif

  ;;========================================------------------------------
  ;; Reading some more parameters:

  if ~n_elements(exmonitor) then begin
    p3d_misc_read_params,parname,parvalue,'exmonitor',exmonitor, $
        uparname=uparname,uparvalue=uparvalue,logunit=logunit,topwid=topwid, $
        verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    exmonitor=~n_elements(exmonitor)?1L:(strlowcase(exmonitor) eq 'yes'?1L:0L)
  endif
  exmonitor=keyword_set(exmonitor)

  ;;=============================--------------------
  ;; Composing output filenames:

  tmpstr=opath+file_basename(filename[0L])
  ofilename     =tmpstr+ostr       +xstr+fsfx[0L]
  tmp=strpos(ofilename,'.gz')
  if compress then begin
    if tmp eq -1L then ofilename+='.gz'
  endif else begin
    if tmp ne -1L then ofilename=strmid(ofilename,0L,tmp)
  endelse

  ofilename_e3d =tmpstr+ostr+'_e3d'+xstr+fsfx[0L]
  tmp=strpos(ofilename_e3d,'.gz')
  if compress then begin
    if tmp eq -1L then ofilename_e3d+='.gz'
  endif else begin
    if tmp ne -1L then ofilename_e3d=strmid(ofilename_e3d,0L,tmp)
  endelse

  ofilenamee    =tmpstr+ostr+'_err'+xstr+fsfx[0L]
  tmp=strpos(ofilenamee,'.gz')
  if compress then begin
    if tmp eq -1L then ofilenamee+='.gz'
  endif else begin
    if tmp ne -1L then ofilenamee=strmid(ofilenamee,0L,tmp)
  endelse

  ofilename_bsw =tmpstr+ostr+'_bsw'+xstr+fsfx[0L]
  tmp=strpos(ofilename_bsw,'.gz')
  if compress then begin
    if tmp eq -1L then ofilename_bsw+='.gz'
  endif else begin
    if tmp ne -1L then ofilename_bsw=strmid(ofilename_bsw,0L,tmp)
  endelse

  ofilename_bsw_e3d=tmpstr+ostr+'_bsw_e3d'+xstr+fsfx[0L]
  tmp=strpos(ofilename_bsw_e3d,'.gz')
  if compress then begin
    if tmp eq -1L then ofilename_bsw_e3d+='.gz'
  endif else begin
    if tmp ne -1L then ofilename_bsw_e3d=strmid(ofilename_bsw_e3d,0L,tmp)
  endelse

  ofilename_bswe=tmpstr+ostr+'_bsw_err'+xstr+fsfx[0L]
  tmp=strpos(ofilename_bswe,'.gz')
  if compress then begin
    if tmp eq -1L then ofilename_bswe+='.gz'
  endif else begin
    if tmp ne -1L then ofilename_bswe=strmid(ofilename_bswe,0L,tmp)
  endelse

  ;; Determining which method to use for spectrum extraction:
  method=p3d_misc_extract_method(userparfile,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug)
  if error ne 0 then goto,logfile_close
  if method eq 'optimal' and usecrmask then crmaskfile=crmask

  ;; Writing information to the status line: 
  if usestawid then begin
    tmpstr='[Object ex.] Trimming data and extracting spectra ['+method+'].'
    widget_control,stawid,set_value=tmpstr
  endif

  ;; Shall a bias be subtracted?:
  ecalc=0L
  if usembiasfile then begin
    ;; The following doesn't work with data where the
    ;; bias has already been subtracted...must think....
    p3d_misc_getinformation,filename+fsfx,kwrdlist,dbias=dbias,gain=gain, $
        rdnoise=rdnoise,masterbias=masterbias,userparfile=userparfile, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close

    if min(dbias) gt 0d0 then ecalc=1L
  endif ;; usembiasfile

  ;;==============================--------------------
  ;; Loading the object data, the trace mask, and the bias image (if it is 
  ;; loaded), in order to extract the same region on the CCD, which was
  ;; trimmed of prescan and overscan regions:

  p3d_extract_prepare_extraction,filename+fsfx,tracemask,kwrdlist, $
      extracttrace,object,objshift,bias,sbsw,xbin=xbin,ybin=ybin, $
      crmaskfile=crmaskfile,crmaskdata=crmasko,userparfile=userparfile, $
      masterbias=masterbias,trcshiftfile=traceshiftfile, $
      dispshiftfile=dispshiftfile,objshiftfile=objectshiftfile, $
      parfile=parfile,xsize=xsize,ysize=ysize,x0=x0,y0=y0,daxis=daxis, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  p3d_misc_retrieve_lprofs,tracemask,lprofs,proffun,method=method, $
      opath=opath,topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then goto,logfile_close

; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ;; TEST TEST TEST
; ndetector=1L
; gapfile=strarr(ndetector)
; for i=0L,ndetector-1L do begin
;   str=ndetector eq 1L?'':'_'+strtrim(i+1L,2L)
;   p3d_misc_read_params,parname,parvalue,'gapfile'+str,gapfilename, $
;       /must_exist,uparname=uparname,uparvalue=uparvalue, $
;       topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
;   if error ne 0 then goto,logfile_close
;   gapfile[i]=gapfilename
; endfor
; tmp=strpos(gapfile,path_sep())
; if tmp[0L] eq -1L then $
;    gapfile=ppath+gapfile ;; adding the path of the parameter file
;
; for i=0L,ndetector-1L do begin
;   if ~file_test(gapfile[i],/regular,/read) then begin
;     errmsg='The gaps array file "'+gapfile[i]+'" does not exist.'
;     goto,error_handler
;   endif
; endfor
;
; readcol,gapfile[d],gaps,format='l',silent=verbose lt 3,comment=';'
; p3d_tracing_calculate_lprofs,filename+fsfx,masterbias,tracemask,cdoffset, $
;     dmbias=dbias,gaps=gaps,xstr=xstr,xbin=xbin,ybin=ybin,gain=gain, $
;     rdnoise=rdnoise,detector=d,kwrdlist=kwrdlist,parfile=parfile, $
;     userparfile=userparfile,/fitcenter,fixprofs=lprofs, $
;     daxis=daxis,stawid=stawid,topwid=topwid,logunit=logunit, $
;     verbose=verbose,error=error,debug=debug
; if error ne 0 then goto,logfile_close
;
; for j=0,381 do lprofs[*,j,0L]+=lprofs[*,j,0L]-cdoffset[j] ;; TEST TEST
;
; ;; TEST TEST TEST
; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  ;;==============================--------------------
  ;; Extracting all spectra from the raw data image by integrating up the
  ;; contribution of all contributing pixels using the trace mask:

  bin=daxis?xbin:ybin
  p3d_extract,object,extracttrace,extract,extract_bsw,dout=dextract, $
      bdout=dextract_bsw,bias=bias,bin=bin,dbias=dbias,crmask=crmasko, $
      gain=gain,rdnoise=rdnoise,userparfile=userparfile,detsec=detsec, $
      parfile=parfile,bsw=sbsw,xstr=xstr,method=method,lprofs=lprofs, $
      proffun=proffun,trcshift=objshift[1L],exmonitor=exmonitor, $
      subtitle=file_basename(filename+fsfx),daxis=daxis,stawid=stawid, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  sbsw=sbsw ne 0L
  s=size(extract)

  ;;==============================--------------------
  ;;==============================--------------------
  ;; Normalizing the flat-field data:

  if useflatfield then begin

    ;; Writing information to the status line: 
    if usestawid then begin
      tmp=~usedispmask?'and applying ':''
      tmpstr='[Object ex.] {flat field} Preparing '+tmp+'the data.'
      widget_control,stawid,set_value=tmpstr
    endif ;; stawid

    msg='Preparing the flat-field data.'

    flatextract=readfits(flatfield,fhdr,silent=verbose lt 3,/no_unsigned)

    ;; Testing the dispersion-axis flipping state of the object data and the
    ;; flat-field mask:
    thdr=headfits(filename+fsfx)
    tmp1=fxpar(thdr,'P3DDFLIP')
    tmp2=fxpar(fhdr,'P3DDFLIP')
    if tmp1 ne tmp2 then begin
      errmsg='The dispersion axis flipping of the object-data file and the' + $
             ' flat-field mask differ {P3DDFLIP}.'
      goto,error_handler
    endif

    ;; Attempting to read information on the used dispersion mask:
    useflatdispmask=0L
    ffdmk=fxpar(fhdr,'IMDMK')
    i=1L & more=1L
    while more do begin
      tmp=fxpar(fhdr,'IMDMK'+strtrim(i++,2L))
      if tmp ne '0' then ffdmk+=strtrim(tmp,2L) else more=0L
    endwhile
    if ffdmk ne '0' then $
       if file_test(ffdmk,/read,/regular) then useflatdispmask=1L

    ;; Attempting to read the errors file:
    dfile=fxpar(fhdr,'IMERR')
    i=1L & more=1L
    while more do begin
      tmp=fxpar(fhdr,'IMERR'+strtrim(i++,2L))
      if tmp ne '0' then dfile+=strtrim(tmp,2L) else more=0L
    endwhile

    etmp=''
    fecalc=0L
    if ecalc then begin
      dff=file_test(dfile,/read,/regular)
      if ~dff then begin
        ;; Making another attempt, with a slightly different filename:
        tmp=strpos(flatfield,sfx,/reverse_search)
        fffx=strmid(flatfield,tmp)
        dfile=strmid(flatfield,0L,tmp)+'_err'+fffx
        if file_test(dfile,/read,/regular) then dff=1L
      endif
      if dff then begin
        dflatextract=readfits(dfile,silent=verbose lt 3,/no_unsigned)
        fecalc=1L
        etmp=' (errors are calculated)'
      endif else begin
        msg='[object] Cannot read the flat-field errors file, skipping it!'
        ret=dialog_message(msg,/center,/information,dialog_parent=topwid)
        etmp=' (errors are calculated, but are ignored with the flat field)'
      endelse
    endif ;; ecalc

    p3d_misc_read_params,parname,parvalue,'smowidth_ff',smowid,/must_exist, $
        type='integer',uparname=uparname,uparvalue=uparvalue,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    smowid/=(daxis?xbin:ybin)

    p3d_misc_read_params,parname,parvalue,'deg_polyfit_ff',deg,/must_exist, $
        type='integer',uparname=uparname,uparvalue=uparvalue,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close

    error=p3d_misc_logger(msg,logunit,loglevel=2,rname=rname, $
        verbose=verbose ge 2)

    p3d_flatfield,flatextract,flat,dflat,din=dflatextract,smowid=smowid, $
        deg=deg,/normalize,daxis=daxis, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close

    pos=where(flat eq 0.0,count)
    if count ne 0L then begin
      flat[pos]=1.0
      if n_elements(dbias) ne 0L then dflat[pos]=1.0
    endif

    ;; No wavelength calibration: flat-fielding the data here:
    if ~usedispmask then begin
      if fecalc then begin
        dextract    =sqrt(dextract^2/flat^2+dflat^2*(extract/flat^2)^2)
        if sbsw and n_elements(dextract_bsw) ne 0L then $
           extract_bsw=sqrt(dextract_bsw^2/flat^2+ $
                            dflat^2*(extract_bsw/flat^2)^2)
      endif ;; fecalc
      extract/=flat
      if sbsw then extract_bsw/=flat

      msg='[Flat field] No wavelength calibration, normalizing the' + $
          ' object data'+etmp+'.'
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
    endif ;; ~usedispmask

  endif ;; useflatfield

  ;;==============================--------------------
  ;; Writing a fits file with the extracted data of the object file:

  ;; Writing information to the status line: 
  if usestawid then begin
    tmp=sbsw?'and bsw-data ':'' & tmps=sbsw?'(s)':''
    tmpstr='[Object ex.] Writing the extracted data '+tmp+'file'+tmps+'.'
    widget_control,stawid,set_value=tmpstr
  endif ;; stawid

  dstr=strtrim(daxis,2L)

  hdr=headfits(filename[0L]+fsfx[0L])
  fxhmake,tmphdr,extract
  fxaddpar,hdr,'NAXIS1',s[1L]
  fxaddpar,hdr,'NAXIS2',s[2L]
  fxaddpar,hdr,'BITPIX',fxpar(tmphdr,'BITPIX')
  fxaddpar,hdr,'BZERO',0
  fxaddpar,hdr,'CRVAL'+dstr,0L
  fxaddpar,hdr,'CDELT'+dstr,1L
  fxaddpar,hdr,'CRPIX'+dstr,1L
  fxaddpar,hdr,'CTYPE'+dstr,'AWAV' ;; for ds9
  fxaddpar,hdr,'SHIFTREF',shiftreffile
  if sbsw then begin
    if fxpar(hdr,'BS_MODE') ne 0L then begin
      msg=['The fits keyword exptime is valid for the total exposure (sky ' + $
           '+ object)!', $
           'Please, remember to correct this for flux calibration!']
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
    endif else fxaddpar,hdr,'EXPTIME',fxpar(hdr,'EXPTIME')/2.
  endif
  fxaddpar,hdr,'IMTYPE','p3d: extracted object data'

  ;; Adding information on the input files:
  p3d_misc_fitsparsplit,'IMRAW',filename+fsfx,hdr,' Raw data filename', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  p3d_misc_fitsparsplit,'IMTRC',tracemask[0L],hdr,' Trace mask filename', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  if usembiasfile then begin
    p3d_misc_fitsparsplit,'IMMBI',masterbias[0L],hdr,' Masterbias filename', $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
  endif ;; usembiasfile

  if useflatfield then begin
    p3d_misc_fitsparsplit,'IMFFL',flatfield[0L],hdr,' Flat-field filename', $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
  endif ;; useflatfield

  etmp=' ' & eetmp='".'
  if ecalc then begin
    p3d_misc_fitsparsplit,'IMERR',ofilenamee,hdr,' Error data filename', $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    etmp=', and its error, ' & eetmp='", and'
  endif ;; ecalc

  writefits,ofilename,extract,hdr,compress=compress
  if ecalc then begin
    fxaddpar,hdr,'IMTYPE','p3d: extracted object data - error'
    writefits,ofilenamee,dextract,hdr,compress=compress
  endif
  tmpstr=useflatfield and ~usedispmask?', and flat fielded,':''
  msg=['Wrote the extracted'+tmpstr+' object data'+etmp+'to:', $
       '  "'+p3d_misc_pathify(ofilename,/dpath)+eetmp]
  if ecalc then msg=[msg,'  "'+p3d_misc_pathify(ofilenamee,/dpath)+'".']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  if sbsw then begin
    fxaddpar,hdr,'IMTYPE','p3d: extracted object data'
    if n_elements(dextract_bsw) ne 0L then begin
      p3d_misc_fitsparsplit,'IMERR',ofilename_bswe,hdr, $
          ' Flat error filename',topwid=topwid,logunit=logunit, $
          verbose=verbose,error=error,debug=debug
      if error ne 0 then goto,logfile_close
    endif
    writefits,ofilename_bsw,extract_bsw,hdr,compress=compress

    if n_elements(dextract_bsw) ne 0L then begin
      fxaddpar,hdr,'IMTYPE','p3d: extracted object data - error'
      writefits,ofilename_bswe,dextract_bsw,hdr,compress=compress
    endif

    msg=['Wrote the extracted'+tmpstr+' bsw object data '+etmp+ $
         'to:','  "'+p3d_misc_pathify(ofilename_bsw,/dpath)+eetmp]
    if ecalc then $
       msg=[msg,'  "'+p3d_misc_pathify(ofilename_bswe,/dpath)+'".']
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  endif ;; sbsw

  ;; Saving the data in the E3d-format:
  if savee3d then begin
    ;; Retrieving the name(s) of the position tables:
    str=d eq 0L?'':'_'+strtrim(d,2L)
    p3d_misc_read_params,parname,parvalue,'postable'+str,/must_exist,/nou, $
        posfile,topwid=topwid,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    posfile=file_dirname(parfile,/mark_directory)+posfile

    ;; Reading the fiber position table:
    p3d_misc_read_postable,postable=posfile,rownum=frow, $
        xpos=fxpos,ypos=fypos,lens_size=lenssize,shape=fshape,/science, $
        verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close

    ;; Creating an array that only consists of the science spectra:
    oute3d=daxis eq 1L?extract[*,frow-1L]:extract[frow-1L,*]

    ;; Calling the e3d writing routine:
    p3d_misc_e3dio_write,oute3d,fxpos,fypos,ofilename_e3d,shape=fshape, $
        size=lenssize[0L],topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug
    if error ne 0 then goto,logfile_close
  endif ;; savee3d

  ;;==============================--------------------
  ;;==============================--------------------
  ;; Applying the dispersion correction on the data:

  if usedispmask then begin

    ;; Writing information to the status line: 
    if usestawid then begin
      tmpstr='[Object ex.] {Wavel.calib.} Preparing and applying the dispe' + $
             'rsion mask.'
      widget_control,stawid,set_value=tmpstr
    endif ;; stawid

    msg='Preparing the dispersion mask for wavelength calibration.'

    nodisperse=0L

    hdrdmk=headfits(dispmask)

    ;; Checking if the grating/grism setup of the object data
    ;; file and the dispersion mask file are the same:
    tmpgp=p3d_misc_get_hdr_kwrd(kwrdlist,'GROTPOS',topwid=topwid, $
        verbose=verbose,error=error,debug=debug)
    if error ne 0 then goto,logfile_close
    grotdmk=tmpgp ne '-1'?float(fxpar(hdrdmk,tmpgp)):-1
    grot=tmpgp ne '-1'?float(fxpar(hdr,tmpgp)):-1
    if useflatfield then grotffl=tmpgp ne '-1'?float(fxpar(fhdr,tmpgp)):-1

    tmpgi=p3d_misc_get_hdr_kwrd(kwrdlist,'GRAT_ID',topwid=topwid, $
        verbose=verbose,error=error,debug=debug)
    if error ne 0 then goto,logfile_close

    gratiddmk=fxpar(hdrdmk,tmpgi) & sgratiddmk=gratiddmk
    gratid=fxpar(hdr,tmpgi)       & sgratid=gratid

    if useflatfield then begin
      gratidffl=fxpar(fhdr,tmpgi)  & sgratidffl=gratidffl
    endif

    ;; Attempting to convert the strings to longs:
    on_ioerror,use_string
    gratiddmk=long(gratiddmk)
    gratid=long(gratid)
    if useflatfield then gratidffl=long(gratidffl)
use_string:
    on_ioerror,NULL
    if grotdmk ne grot or gratiddmk ne gratid then begin
            
      msg=['Warning! Dispersion mask settings do not match the object!', $
           'mask   - GRAT_ID: '+strtrim(sgratiddmk,2L)+'  GROTPOS: ' + $
           strtrim(grotdmk,2L), $
           'object - GRAT_ID: '+strtrim(sgratid,2L)   +'  GROTPOS: ' + $
           strtrim(grot,2L)]
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
    endif ;; grotdmk ne grot or gratiddmk ne gratid

    if useflatfield then begin
      if grotffl ne grot or gratidffl ne gratid then begin
            
        msg=['Warning! Flat-field settings do not match the object!', $
           'mask   - GRAT_ID: '+strtrim(sgratidffl,2L)+'  GROTPOS: ' + $
           strtrim(grotffl,2L), $
           'object - GRAT_ID: '+strtrim(sgratid,2L)   +'  GROTPOS: ' + $
           strtrim(grot,2L)]
        error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
      endif ;; grotffl ne grot or gratidffl ne gratid
    endif ;; useflatfield

    ;;==============================--------------------
    ;; Unfolding the dispersion mask:

    ;; The following lines work poorly with MPFS, and so I've made a dirty
    ;; trick to make the dispersion mask size work also with this IFU.
    ;; There might be some trouble...so I leave the old code snippet in for
    ;; now.
;    if daxis then begin
;      xwidth=(xsize/1024L*1024L) eq xsize?xsize/1024L:xsize/1024L+1L
;    endif else begin
;      ywidth=(ysize/1024L*1024L) eq ysize?ysize/1024L:ysize/1024L+1L
;    endelse
;    dispnpixels=daxis?(1024L*xwidth):(1024L*ywidth)

    s=size(extract)
    dispnpixels=daxis?s[1L]*xbin:s[2L]*ybin

    p3d_wavecal_unfold_dispmask_file,dispmask,dispnpixels,disp, $
        bin=bin,daxis=daxis,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug
    if error ne 0 then goto,logfile_close

    ;;==============================--------------------
    ;; Calculating a dispersion solution (unless NODISPERSE==1):

    if ~nodisperse then begin

      ;; Calculating initial values for CRVAL and CDELT:
      sd=size(disp)
      disp=congrid(disp,sd[1L]/(daxis?xbin:1L),sd[2L]/(daxis?1L:ybin),/interp)
      if daxis then begin
        s=size(extract)
        aperpix=mean(disp[1L:s[1L]-1L,*]-disp[0L:s[1L]-2L,*])
        disp0=max(disp[0L,*])
      endif else begin 
        s=size(extract)
        aperpix=mean(disp[*,1L:s[2L]-1L]-disp[*,0L:s[2L]-2L])
        disp0=max(disp[*,0L])
      endelse

      ;;==============================--------------------
      ;; Reading the values of CRVAL, CDELT, and NPIX from the user
      ;; parameter file, if they exist:

      crval=disp0 & cdelt=aperpix
      npix=round((min(daxis?disp[s[daxis]-1L,*]: $
                            disp[*,s[daxis]-1L])-crval)/cdelt)
      userstr=' {default} '

      if userparfile ne '' then begin
        if uparname[0L] ne '' then begin
          err=0

          p3d_misc_read_params,uparname,uparvalue,'crval',crval,/upo, $
              type='float',found=found,topwid=topwid,logunit=logunit, $
              verbose=verbose,error=error,debug=debug
          if error ne 0 then goto,logfile_close
          if ~found then crval=disp0

          p3d_misc_read_params,uparname,uparvalue,'cdelt',cdelt,/upo, $
              type='float',found=found,topwid=topwid,logunit=logunit, $
              verbose=verbose,error=error,debug=debug
          if error ne 0 then goto,logfile_close
          if ~found then cdelt=aperpix

          p3d_misc_read_params,parname,parvalue,'npix',npix,/upo, $
              type='integer',found=found,topwid=topwid,logunit=logunit, $
              verbose=verbose,error=error,debug=debug
          if error ne 0 then goto,logfile_close
          if ~found then npix=round((min(daxis? $
              disp[s[daxis]-1L,*]:disp[*,s[daxis]-1L])-crval)/cdelt)
        endif ;; uparname[0L] ne ''
      endif ;; userparfile

      msg='[createobject] Using [crval,cdelt,npix]=['+ $
          strtrim(crval,2L)+','+strtrim(cdelt,2L)+','+strtrim(npix,2L)+'].'

      ;;==============================--------------------
      ;; Prompting the user for the first pixel wavelength, the
      ;; dispersion (wavelength bin width) and the number of pixels:

      if keyword_set(waveprompt) then begin
        p3d_cobjex_waveprompt,crval=crval,cdelt=cdelt,npix=npix, $
            topwid=topwid,verbose=verbose,error=error,debug=debug
        if error ne 0 then goto,logfile_close

        msg='[createobject] Selected {prompted}: [crval,cdelt,npix]=['+ $
            strtrim(crval,2L)+','+strtrim(cdelt,2L)+','+strtrim(npix,2L)+'].'
      endif ;; waveprompt

      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

      ;;==============================--------------------
      ;; Checking if the user has specified any line list with telluric
      ;; lines in the user parameter file:

      if userparfile ne '' and ~n_elements(skyalign) then begin
        if ~file_test(userparfile,/regular,/read) then begin
          errmsg='The user parameter file "'+userparfile+'" does not exist.'
          goto,error_handler
        endif

        if parname[0L] ne '' then begin
          p3d_misc_read_params,uparname,uparvalue,'skyalign',skyalign__, $
              found=found,topwid=topwid,logunit=logunit, $
              verbose=verbose,error=error,debug=debug
          if error ne 0 then goto,logfile_close

          if found then begin
            if ~file_test(skyalign__,/read,/regular) then begin
              msg=['The entry ''skyalign'' in the user parameter file does' + $
                   ' not point','at any file that can be used, reverting t' + $
                   'o the default behavior.',' skyalign="'+ $
                   p3d_misc_pathify(skyalign__,/dpath)+'".']
              ret=dialog_message(msg,/center,/information,dialog_parent=topwid)
            endif else skyalign=skyalign__
          endif ;; found
        endif ;; parname[0L] ne ''
      endif ;; userparfile ne '' and skyalign eq ''

      ;;==============================--------------------
      ;; Applying the dispersion correction to the extracted object data:

      msg='Wavelength calibrating the extracted object data.'
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
      if n_elements(skyalign) ne 0L then $
         if skyalign ne '' then iskyalign=skyalign

      p3d_wavecal_dispersion_correction,extract,disp,crval,cdelt,tmp, $
          dtmp,dstack=dextract,incrval=crval,incdelt=cdelt,npix=npix, $
          skyalign=iskyalign,daxis=daxis,topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then goto,logfile_close
      extract=tmp
      if n_elements(dtmp) ne 0L then dextract=dtmp

      ;; Applying the dispersion correction to the flat-field data:
      if useflatfield then begin
        msg='Wavelength calibrating the extracted flat-field data.'
        error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

        ;; Using a dedicated dispersion mask for the flat field, if such a one
        ;; exists:
        if useflatdispmask then begin
          p3d_wavecal_unfold_dispmask_file,ffdmk,dispnpixels,fdisp, $
              bin=bin,daxis=daxis,topwid=topwid,logunit=logunit, $
              verbose=verbose,error=error,debug=debug
          if error ne 0 then goto,logfile_close
          sf=size(fdisp)
          fdisp=congrid(fdisp,sf[1L]/(daxis?xbin:1L), $
                              sf[2L]/(daxis?1L:ybin),/interp)
        endif else fdisp=disp

        p3d_wavecal_dispersion_correction,flat,fdisp,crval,cdelt,oflat, $
            dtmp,dstack=dflat,incrval=crval,incdelt=cdelt,npix=npix, $
            daxis=daxis,topwid=topwid,logunit=logunit,verbose=verbose, $
            error=error,debug=debug
        if error ne 0 then goto,logfile_close
        flat=oflat
        if n_elements(dtmp) ne 0L then dflat=dtmp

        ;; Flatfielding the extracted data here:
        if n_elements(dbias) ne 0L and fecalc then $
           dextract=sqrt(dextract^2/flat^2+dflat^2*(extract/flat^2)^2)
        extract/=flat

        msg='Normalized the object data with the flat-field data.'
        error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

      endif ;; useflatfield

      ;;==============================--------------------
      ;; Writing a fits file with the extracted, wavelength calibrated,
      ;; and flat-fielded data of the object file:

      if usestawid then begin
        tmp=sbsw?'and bsw-data ':'' & tmps=sbsw?'(s)':''
        tmpstr='[Object ex.] Writing the extracted data '+tmp+'file'+tmps+'.'
        widget_control,stawid,set_value=tmpstr
      endif ;; stawid

      dstr=strtrim(daxis,2L)

      s=size(extract)
      fxhmake,tmphdr,extract
      fxaddpar,hdr,'BITPIX',fxpar(tmphdr,'BITPIX')
      fxaddpar,hdr,'BZERO',0
      fxaddpar,hdr,'NAXIS1',s[1L]
      fxaddpar,hdr,'NAXIS2',s[2L]
      fxaddpar,hdr,'CRVAL'+dstr,crval
      fxaddpar,hdr,'CDELT'+dstr,cdelt
      fxaddpar,hdr,'CRPIX'+dstr,1L
      fxaddpar,hdr,'CTYPE'+dstr,'AWAV' ;; for ds9
      fxaddpar,hdr,'CUNIT'+dstr,'ANGSTROM'

      ;; Adding information on the input files:
      p3d_misc_fitsparsplit,'IMRAW',filename+fsfx,hdr,' Raw data filename', $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then goto,logfile_close

      p3d_misc_fitsparsplit,'IMTRC',tracemask[0L],hdr,' Trace mask filename', $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then goto,logfile_close

      p3d_misc_fitsparsplit,'IMDMK',dispmask[0L],hdr,' Disp.mask filename', $
          topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
      if error ne 0 then goto,logfile_close

      etmp=' ' & eetmp='".'
      if ecalc then begin
        p3d_misc_fitsparsplit,'IMERR',ofilenamee,hdr,' Error data filename', $
            topwid=topwid,logunit=logunit,verbose=verbose,error=error, $
            debug=debug
        if error ne 0 then goto,logfile_close
        etmp=', and its error, ' & eetmp='", and'
      endif

      writefits,ofilename,extract,hdr,compress=compress
      if n_elements(dextract) ne 0L then begin
        fxaddpar,hdr,'IMTYPE','p3d: extracted object data - error'
        writefits,ofilenamee,dextract,hdr,compress=compress
      endif

      ;; Saving the data in the E3d-format:
      if savee3d then begin
        ;; Creating an array that only consists of the science spectra:
        oute3d=daxis eq 1L?extract[*,frow-1L]:extract[frow-1L,*]

        ;; Calling the e3d writing routine:
        p3d_misc_e3dio_write,oute3d,fxpos,fypos,ofilename_e3d,shape=fshape, $
            size=lenssize[0L],topwid=topwid,logunit=logunit,verbose=verbose, $
            error=error,debug=debug
        if error ne 0 then goto,logfile_close
      endif ;; savee3d

      tmpstr=useflatfield?', and flat-fielded,':''
      msg=['Wrote the extracted'+tmpstr+' object data'+etmp+'to:', $
           '  "'+p3d_misc_pathify(ofilename,/dpath)+eetmp]
      if ecalc then msg=[msg,'  "'+p3d_misc_pathify(ofilenamee,/dpath)+'".']
      error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

      ;;==============================--------------------
      ;; Applying the dispersion correction also to the sky image of
      ;; beam switched data, when such data exists:

      if sbsw then begin
        msg='Applying the dispersion correction to the extracted object data.'
        error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

        p3d_wavecal_dispersion_correction,extract_bsw,disp,crval,cdelt, $
            tmp,dtmp,dstack=dextract_bsw,incrval=crval,incdelt=cdelt, $
            npix=npix,daxis=daxis,topwid=topwid,logunit=logunit, $
            verbose=verbose,error=error,debug=debug
        if error ne 0 then goto,logfile_close

        extract_bsw=tmp
        if n_elements(dtmp) ne 0L then dextract_bsw=dtmp

        if useflatfield then begin
          if n_elements(dbias) ne 0L then $
             dextract_bsw=sqrt(dextract_bsw^2/flat^2+ $
                               dflat^2*(extract_bsw/flat^2)^2)
          extract_bsw/=flat
        endif

        fxaddpar,hdr,'IMTYPE','p3d: extracted object data'

        etmp=' ' & eetmp='".'
        if ecalc then begin
          p3d_misc_fitsparsplit,'IMERR',ofilename_bswe,hdr, $
              ' Error data filename',topwid=topwid, $
              logunit=logunit,verbose=verbose,error=error,debug=debug
          if error ne 0 then goto,logfile_close
          etmp=', and its error, ' & eetmp='", and'
        endif

        fxaddpar,hdr,'IMTYPE','p3d: extracted bsw object data'
        writefits,ofilename_bsw,extract_bsw,hdr,compress=compress
        if n_elements(dextract_bsw) ne 0L then begin
          fxaddpar,hdr,'IMTYPE','p3d: extracted bsw object data - error'
          writefits,ofilename_bswe,dextract_bsw,hdr,compress=compress
        endif

        ;; Saving the data in the E3d-format:
        if savee3d then begin
          ;; Creating an array that only consists of the science spectra:
          oute3d=daxis eq 1L?extract_bsw[*,frow-1L]:extract_bsw[frow-1L,*]

          ;; Calling the e3d writing routine
          p3d_misc_e3dio_write,oute3d,fxpos,fypos,ofilename_e3d_bsw, $
              shape=fshape,size=lenssize[0L],topwid=topwid,logunit=logunit, $
              verbose=verbose,error=error,debug=debug
          if error ne 0 then goto,logfile_close
        endif ;; savee3d

        msg=['Wrote the extracted'+tmpstr+' bsw object data'+etmp+'to:', $
             '  "'+p3d_misc_pathify(ofilename,/dpath)+eetmp]
        if ecalc then $
           msg=[msg,'  "'+p3d_misc_pathify(ofilenamee,/dpath)+'".']
        error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
      endif ;; sbsw

    endif ;; ~nodisperse
  endif ;; usedispmask

  out=temporary(extract)
  if n_elements(extract_bsw) ne 0L then bswout=extract_bsw
  if n_elements(ofilename_bsw) eq 1L then bsw_ofilename=ofilename_bsw

  ;; Writing information to the status line:
  if usestawid then begin
    tmpstr='[Object ex.] Done.'
    widget_control,stawid,set_value=tmpstr
  endif

  ;; Closing the logfile, if it was opened here:
  if logfileisopened then free_lun,logunit[0L]

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
logfile_close:
  if logfileisopened then free_lun,logunit[0L]
  return
END ;; procedure: p3d_cobjex
