/*------------------------------------------------------------------------------
NAME
	fileinfo.c - list statistics for files
 
DESCRIPTION
	'fileinfo' reads the listed lidar data FITS files and prints the start
	time, end time, number of hours of data, and the mean number of photon 
	counts per laser shot.
  
USAGE
	fileinfo [options] files...
 
PARAMETERS
	files...			a list of lidar FITS files (separated by white space)

OPTIONS
	-h, -help			print help for this command
	-t					don't print the table header
 
REVISION
	100907	P. Hickson
	140515  [CDS] SIMPLE or XTENSION

COPYRIGHT
	Copyright (c) 2010 by P. Hickson. No part of this program may be copied or 
	released to a third party without written permission.
------------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>

#include "lidar.h"

// Function prototypes
void 
	parse_args(int,char**),
	print_help();

// External variables
extern char
	*optarg;
extern int	
	errno,
	optind,
	optopt,
	opterr,
	optreset;

// Global variables
char	
	*prog_name;
int
	header		= 1,
	nfiles		;

//------------------------------------------------------------------------------
#define OPTIONS	"ht"

void	parse_args(int,char**);

void	usage() {
	
	fprintf(stderr,"Usage: %s [options] files...\n",prog_name);
	fprintf(stderr,"where options include:\n");
	fprintf(stderr,"  -h, -help print help\n");
	fprintf(stderr,"  -t        don't print the table header\n");
	exit(1);
}
void	parse_args(int argc, char **argv) {
	
	int		c;
	
	prog_name = argv[0];
	while ((c = getopt(argc,argv,OPTIONS)) != -1) {
		switch (c) {
			case 'h':
				print_help();
				break;
			case 't':
				header = 0;
				break;
			default:
				usage();
		}
	}
	nfiles = argc-optind;
	if (nfiles < 1) usage();
}
//------------------------------------------------------------------------------
void print_help() {
	
	printf("\nNAME\n");
	printf("\tfileinfo.c - list statistics for files\n");
	putchar('\n');
	printf("DESCRIPTION\n");
	printf("\t\'fileinfo\' reads the listed lidar data FITS files and prints the start\n");
	printf("\ttime, end time, number of hours of data, and the mean number of photon\n");
	printf("\tcounts per laser shot.\n");
	putchar('\n');
	printf("USAGE\n");
	printf("\tfileinfo [options] files...\n");
	putchar('\n');
	printf("PARAMETERS\n");
	printf("\tfiles...          a list of lidar FITS files (separated by white space)\n");
	putchar('\n');				
	printf("OPTIONS\n");
	printf("\t-h, -help         print help for this command\n");
	printf("\t-t                don't print a table header\n");
	putchar('\n');
	printf("EXAMPLES\n");
	printf("\tfileinfo /data/lidar/fits/*.fits\n");
	printf("\t - print file information for all FITS files in the \'/data/lidar/fits/\'\n");
	printf("\t   directory\n");
	putchar('\n');				
	printf("\tfileinfo 2010*.fits\n");
	printf("\t - print file information for all fits files in the current directory\n");
	printf("\t   having names that start with \'2010\'.\n");
	putchar('\n');
	printf("REVISION\n");
	printf("\t110502\n");
	putchar('\n');					
	exit(0);
}
//------------------------------------------------------------------------------
int main(int argc,char **argv) {

	char		header[FITS_HDR_SIZ],filename[STRSIZE];
	int			blk,blocks,count,dsize,fd,i,error;
	double		start_time,end_time;
	Fits		fits;
	
	// Read the command-line options.
	parse_args(argc,argv);
	
	// Print a header
	if (header) {
		printf("#-------------------------------------------------\n");
		printf("# FILE         UTC_START UTC_END   HOURS  <COUNTS>\n");
		printf("#-------------------------------------------------\n");
	}	
	
	// Loop through the file list
	for (;optind<argc;optind++) {
		
		error = 0;
		
		// Open the data file.
		if ((fd = open(argv[optind],O_RDONLY)) == -1) {
			fprintf(stderr,"%s: cannot open %s\n",prog_name,argv[optind]);
			exit(-1);
		}
		
		// Extract the file name from the complete path.
		for (i=strlen(argv[optind])-1;i>=0;i--) {
			if (argv[optind][i] == '/') break;
		}
		strncpy(filename,argv[optind]+i+1,STRSIZE);
		
		// Read the data
		count = 0;
		for (blk=0;;blk++) {
			
			// Read and decode the FITS header.
			if (read(fd,header,FITS_HDR_SIZ) != FITS_HDR_SIZ) {
				
				// End of file, get out of the loop.
				break;
			}
			;
			
			if (read_fits_header(header,&fits) != 0) {
				error = 1;
				break;
			};
			
			// Find the start time
			if (blk == 0) {
				start_time = fits.hour+(fits.min+fits.sec/60.0)/60.0;
			}
			
			// Find the number of photons
			count += fits.naxis1;
			
			// Find the number of data blocks.
			blocks = (2*fits.naxis1)/FITS_BLOCK_SIZ;
			if (2*fits.naxis1 > FITS_BLOCK_SIZ*blocks) blocks++;
			dsize = FITS_BLOCK_SIZ*blocks;
			
			// Skip over the data
			if (lseek(fd,dsize,SEEK_CUR) == -1) {
				error = 2;
				break;
			}
		}
		if (blk > 0) end_time = fits.hour+(fits.min+fits.sec/60.0)/60.0;
		
		// Print the results.
		if (blk == 0 || error) {
			printf("%s FITS file format error\n",filename);
		}
		else {
			printf("%s %7.4f %7.4f %7.4f  %8.1f\n",filename,start_time,end_time,
				   blk/3600.0,count/((double)blk*NSHOTS));

		}
		
		// Close the data file.
		close(fd);		
	}
}
#include "fits_read_header.c"
