/*------------------------------------------------------------------------------
NAME
	density.c - generate a binned density map for a specified night
 
DESCRIPTION
	'density' reads the specified data file and prints an ASCII listing of 
	photon counts binned to a specified resolution. Since the data are 
	dominated by photon noise, the standard error for the count in each bin 
	is the square root of the count. Each line of output data has the following 
	format:
 
	UT_seconds	count[0] count[1] count[2] .... count[nbins-1]
  
USAGE
	density [options] [directory/]date > output_file
 
PARAMETERS
	'file' is the date to be processed, in the form 'yyyymmdd'. If the data
	files for this night are not in the current directory, 'files' must 
	be preceeded by a path to the directory that contains the data files
	(eg. /lidar/data/20090924 ). It is not necessary to add the extension
	_00.fits, _01.fits, etc. The program will find all FITS data files
	for the indicated date in the specified directory.
 
 OPTIONS
	-a		asize		size of altitude bin (m) [90]
    -h, -help			print help for this command
	-l		lower		altitude of lowest bin (m) [80000]
	-n		nbins		number of altitude bins [278]
	-t		tsize		size of time bin (s) [1]
	-v		level		level of verbosity [0]
 
REVISION
	110502	P. Hickson
	140515  [CDS] SIMPLE or XTENSION

COPYRIGHT
	Copyright (c) 2011 by P. Hickson. No part of this program may be copied or 
	released to a third party without written permission.
------------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>

#include "lidar.h"

// Function prototypes
void 
	parse_args(int,char**),
	print_help();

// External variables
extern char
	*optarg;
extern int	
	errno,
	optind,
	optopt,
	opterr,
	optreset;

// Global variables
u_int		
	*abins;
char
	*prog_name;
u_short
	*data;
int
	*bins		,
	abin		= 180,
	nbins		= 200,
	ndirs		= 0,
	nfiles		= 0,
	tbin		= 10,
	verbose		= 0;
double
	hbin		= 180.0,
	lower		= 80000;

//------------------------------------------------------------------------------
#define OPTIONS	"a:hl:n:t:v:"

void	parse_args(int,char**);
char	*prog_name;
int		update = 0;

void	usage()
{
	fprintf(stderr,"Usage: %s [options] file\n",prog_name);
	fprintf(stderr,"where options include:\n");
	fprintf(stderr,"  -a abin   altitude bin size (ns) [180]\n");
	fprintf(stderr,"  -h, -help print help for this command\n");
	fprintf(stderr,"  -l lower  altititude of lowest bin (m) [80000]\n");
	fprintf(stderr,"  -n nbins  number of bins [200]\n");
	fprintf(stderr,"  -t tbin   time bin size (s) [10]\n");
	fprintf(stderr,"  -v level  level of verbosity [0]\n");
	exit(1);
}
void	parse_args(int argc, char **argv) {
	
	int		c;
	
	prog_name = argv[0];
	while ((c = getopt(argc,argv,OPTIONS)) != -1) {
		switch (c) {
			case 'a':
				abin = atof(optarg);
				break;
			case 'h':
				print_help();
				break;
			case 'l':
				lower = atof(optarg);
				break;
			case 'n':
				nbins = atoi(optarg);
				break;
			case 't':
				tbin = (int)(atoi(optarg)+0.5);
				break;
			case 'v':
				verbose = atoi(optarg);
				break;
			default:
				usage();
		}
	}
	nfiles = argc-optind;
	if (nfiles != 1) usage();
}
//------------------------------------------------------------------------------
void print_help() {
	
	printf("\nNAME\n");
	printf("\tdensity.c - generate a binned density map for a specified night\n");
	putchar('\n');
	printf("DESCRIPTION\n");
	printf("\t\'density\' reads the specified data file and prints an ASCII listing of\n");
	printf("\tphoton counts binned to a specified resolution. Since the data are\n");
	printf("\tdominated by photon noise, the standard error for the count in each bin\n");
	printf("\tis the square root of the count. Each line of output data has the following\n"); 
	printf("\tformat:\n");
	putchar('\n');
	printf("\tUT_seconds	count[0] count[1] count[2] .... count[nbins-1]\n");
	putchar('\n');
	printf("USAGE\n");
	printf("\tdensity [options] [directory/]date > output_file\n");
	putchar('\n');
	printf("PARAMETERS\n");
	printf("\t'file' is the date to be processed, in the form 'yyyymmdd'. If the data\n");
	printf("\tfiles for this night are not in the current directory, 'files' must\n");
	printf("\tbe preceeded by a path to the directory that contains the data files\n");
	printf("\t(eg. /lidar/data/20090924 ). It is not necessary to add the extension\n");
	printf("\t_00.fits, _01.fits, etc. The program will find all FITS data files\n");
	printf("\tfor the indicated date in the specified directory.\n");
	putchar('\n');				
	printf("OPTIONS\n");
	printf("\t-a    asize    size of altitude bin (m) [90]\n");
	printf("\t-h, -help      print help for this command\n");
	printf("\t-l    lower    altitude of lowest bin (m) [80000]\n");
	printf("\t-n    nbins    number of altitude bins [278]\n");
	printf("\t-t    tsize    size of time bin (s) [1]\n");
	printf("\t-v    level    level of verbosity [0]\n");
	putchar('\n');				
	printf("EXAMPLES\n");
	printf("\tdensity /data/lidar/fits/20091008 > 20091008_density.txt\n");
	printf("\t - computes a density map using data from all fits files for the UT date\n");
	printf("\t   2009-10-08 (ie. files having names of the form \'20091008_XX.fits\' located\n");
	printf("\t   in the directory \'/data/lidar/fits/\') using the default parameters\n");
	printf("\t   (shown in square brackets above under OPTIONS) and writes the result\n");
	printf("\t   to the text file \'20091008_density.txt\'.\n");
	putchar('\n');				
	printf("\tdensity -a 100 -n 200 -t 10 20091008 > 20091008_100_200_20.txt\n");
	printf("\t - computes a density map for UT 2009-10-08, assuming that the fits files\n");
	printf("\t   are located in the current directory, using 200 altitude bins of width\n");
	printf("\t   100 m and 10 sec time bins. The starting altitude is the 80 km default.\n");
	printf("\t   The result is written to the text file \'20091008_100_200_20.txt\'.\n");
	putchar('\n');
	printf("REVISION\n");
	printf("\t110403	P. Hickson\n");
	putchar('\n');					
	exit(0);
}
//------------------------------------------------------------------------------
int main(int argc,char **argv) {

	char		date[9],s[STRSIZE],dir[STRSIZE],file_name[STRSIZE],
				header[FITS_HDR_SIZ];
	u_short		*dp;
	int			bin,blocks,channel,counts,dsize,i,j,k,l,n,err,fd,seq,start_time,
				time;
	double		alt;
	Fits		fits;
	
	// Read the command-line options.
	parse_args(argc,argv);
	
	// Allocate storage
	if ((abins = (int*)malloc(nbins*sizeof(int))) == NULL) {
		fprintf(stderr,"density: error allocating memory\n");
		exit(-1);
	}
	if ((data = (u_short*)malloc(2*MAXCOUNT*NSHOTS)) == NULL) {
		fprintf(stderr,"density: error allocating memory\n");
		exit(-1);
	}
	
	// Determine the directory, and the date to be processed.
	for (i=strlen(argv[optind]);i>=0;i--) {
		if (argv[optind][i] == '/') {
			break;
		}
	}
	if (i >= 0) {
		strncpy(dir,argv[optind],i+1);
		dir[i+1] = '\0';
	}
	else {
		dir[0] = '\0';
	}
	strncpy(date,argv[optind]+i+1,8);
	date[8] = '\0';
	start_time = 0;

	// Loop through file names
	for (seq=0;seq<MAX_SEQ_NO;seq++) {
		
		// Create the file name
		sprintf(file_name,"%s%s_%02d.fits",dir,date,seq);
	
		// Open the data file.
		if ((fd = open(file_name,O_RDONLY)) == -1) {
			fprintf(stderr,"%s: cannot open %s\n",prog_name,file_name);
			exit(-1);
		}
		
//		fprintf(stderr,"processing %s\n",file_name);
		
		// Start processing the data.
		for (i=0;;i++) {
			
			for (k=0;k<nbins;k++) abins[k] = 0;
			for (j=0;j<tbin;j++) {
				
				// Read and decode the FITS header.
				if (read(fd,header,FITS_HDR_SIZ) != FITS_HDR_SIZ) {
					exit(0);
				}
				read_fits_header(header,&fits);
				
				// Find the time
				time = fits.hour*3600+fits.min*60+fits.sec;
				if (j == 0) {
					start_time = time;
				} 
				
				// Check to see if there was an interruption due to laser
				// shutdown. If the time of this record is later than the
				// time bin size, this is the start of a later data acquisistion
				// so break out.
				else if (time > start_time+tbin) {
					break;
				}
				
				// Find the number of data blocks.
				blocks = (2*fits.naxis1)/FITS_BLOCK_SIZ;
				if (2*fits.naxis1 > FITS_BLOCK_SIZ*blocks) blocks++;
				dsize = FITS_BLOCK_SIZ*blocks;
				
				// Read the data.
				if (read(fd,data,dsize) != dsize) {
					fprintf(stderr,"%s: error reading %s\n",prog_name,file_name);
					exit(-1);
				}

				dp = data;
				for (k=0;k<NSHOTS;k++) {
					
					// Process the photons.
					for (l=0;l<fits.count[k];l++) {
						bin = 0x3fff & *dp;
						channel = 0x0003 & (*dp++ >> 14);
						if (bin < fits.nbin0) {
							alt = fits.alt+(bin*fits.bsize0+fits.bstart0)
								*M_PER_NS;
						}
						else {
							alt = fits.alt+((bin-fits.nbin0)*fits.bsize1
							+fits.bstart1)*M_PER_NS;
						}
						n = (alt-lower)/abin;
						if (n >=0 && n < nbins) {
							abins[n]++;
						}
					}
				}
			}
			
			// Print data for this time bin
			printf("%5d ",start_time);
			for (k=0;k<nbins;k++) {
				printf("%7d ",abins[k]);
			}
			putchar('\n');
		}		
			
	// Close the data file.
	close(fd);
	}
	
	// Free allocated storage.
	free(data);
	free(bins);
}
#include "fits_read_header.c"
