#!/bin/bash
##
## p3d_tool_makevm.sh
##
## $Id: p3d_tool_makevm.sh 79 2010-03-04 14:24:25Z christersandin $
##
## --
##
## p3d: a general data-reduction tool for fiber-fed IFSs
##
## Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, see <http://www.gnu.org/licenses>.
##
## Additional permission under GNU GPL version 3 section 7
##
## If you modify this Program, or any covered work, by linking or
## combining it with IDL (or a modified version of that library),
## containing parts covered by the terms of the IDL license, the
## licensors of this Program grant you additional permission to convey
## the resulting work.
##
## --
##
## The purpose of this script is to run the IDL script that creates a
## SAV-file that can be used with the IDL virtual machine
## ('routines/p3d_tool_makevm.pro'), in order to run p3d without a
## license.
##
## NOTE! It is necessary to have an IDL license to create the binary
##       with script. Executing the binary thereafter does not require
##       a license.
##
## p3d can then be started using the VM with the command:
##   idl -vm=<path>/p3d.sav [UN*X]
## Another possibility is to use the supplied script 'p3d_vm.sh'.
##
## Requirements (that are all checked for in advance) are:
##  1) IDL must be present, and the environmental variable $IDL_DIR
##     set (to the idl root directory).
##  2) IDL must be version >=6.2. If a python binary is present the
##     version is checked for automatically.
##  3) This script must be executed from the p3d-directory. Hence
##     the program 'p3d.pro' must be found in the current directory.
##     The IDL Astronomy User's Libray (astro-lib) must be placed in
##     the current directory under ./astro-lib (that is where the
##     subroutines are expected to be found). The entire astro-lib
##     package can be copied to this location or alternatively a
##     symbolic link can be created to point at the directory where
##     it has been placed (the link must be called 'astro-lib' and
##     the routines must be found in that directory [i.e. not under
##     any sub-directory]).
##
##  The IDL Astronomy User's Library can be downloaded from:
##    http://idlastro.gsfc.nasa.gov/homepage.html
##
##  The fitting routine mpfit.pro and mpcurvefit.pro can be
##    downloaded from: http://purl.com/net/mpfit
##
##  This script is designed to work with the following platforms:
##    MacOSX, Sun Solaris, Linux

screxe=${0##*/}

# Checking if p3d.pro is found in the current directory:
if [ ! -r ./p3d.pro ]; then
    echo "$screxe: Cannot find ./p3d.pro. Cannot continue!"
    exit 1
fi

# Checking if $IDL_DIR is set:
if [ -z "$IDL_DIR" ]; then
    echo "$screxe: The IDL variable \$IDL_DIR is not set, it must be."
    echo "$screxe:"
    echo "$screxe: \$IDL_DIR must be set to point at the IDL directory."
    echo "$screxe:   Example (bash): export IDL_DIR=/opt/itt/idl_6.2"
    echo "$screxe:           (tcsh): setenv IDL_DIR /opt/itt/idl_6.2"
    echo "$screxe: Cannot continue!"
    exit 1
fi

# Checking if the IDL binary exists:
if [ ! -x ${IDL_DIR}/bin/idl ]; then
    echo "$screxe: Cannot find the IDL binary '\$IDL_DIR/bin/idl'."
    echo "$screxe: Cannot continue!"
    exit 1
fi

# Checking if the IDL version is correct:
if [ \( -x `which python` \) -a \( -f ${IDL_DIR}/version.txt \) ]; then
    idlversion=`cat ${IDL_DIR}/version.txt`
    versionok=`echo $idlversion | python -c 'a=input() >= 6.2; print a'`
    if [ $versionok = False ]; then
	echo "$screxe: The used IDL version $idlversion < 6.2."
	echo "$screxe: Cannot continue!"
	exit 1
    fi
fi

# Checking if the contrib directory exists:
if [ ! -d ./contrib ]; then
    echo "$screxe: Cannot find the contributing files directory in ./contrib."
    echo "$screxe:"
    echo "$screxe: Cannot continue!"
    exit 1
fi

# Checking if the NASA astro-lib directory exists:
if [ ! -d ./contrib/astro-lib ]; then
    echo "$screxe: Cannot find the astro-lib package in ./contrib/astro-lib/."
    echo "$screxe:"
    echo "$screxe: Either copy the files to the current directory,"
    echo "$screxe:   or make a symbolic link to wherever you have"
    echo "$screxe:   placed these routines. Example:"
    echo "$screxe:   ln -s /path/to/astro/lib/ contrib/astro-lib"
    echo "$screxe:"
    echo "$screxe: Cannot continue!"
    exit 1
fi

# Checking the mpfit routines exist:
if [ \( ! -r ./contrib/mpfit.pro \) -o \
     \( ! -r ./contrib/mpcurvefit.pro \) ] ; then
    echo "$screxe: Cannot find the necessary mpfit routines in ./contrib/."
    echo "$screxe:"
    echo "$screxe: Either copy the files to the current directory,"
    echo "$screxe:   or make a symbolic link to wherever you have"
    echo "$screxe:   placed these routines. Example:"
    echo "$screxe:   ln -s /path/to/markwardt/pro/ contrib"
    echo "$screxe:"
    echo "$screxe: Cannot continue!"
    exit 1
fi

echo "$screxe: Launching IDL and generating a binary file."

export p3d_path=`pwd`

# Removing any previous binary file:
if [ -f ./vm/p3d.sav ]; then
    if [ -w ./vm/p3d.sav ]; then
        rm ./vm/p3d.sav
    else
        echo "$screxe: ./vm/p3d.sav exists, but cannot be removed."
        echo "$screxe:"
        echo "$screxe: Cannot continue!"
        exit 1
    fi
fi

${IDL_DIR}/bin/idl ${IDL_OPTS} -e '@routines/p3d_tool_makevm'

# Checking if p3d.pro is found in the current directory:
if [ -f ./vm/p3d.sav ]; then
    echo "$screxe: Generated an IDL virtual machine binary (./vm/p3d.sav)."
    echo "$screxe:   You can start p3d using the IDL virtual machine"
    echo "$screxe:   with the command '\${p3d_path}/vm/p3d_vm.sh'."
else
    echo "$screxe: An error occurred, could not generate"
    echo "$screxe:   the p3d IDL virtual machine binary."
    echo "$screxe:   Please see the output above for possible"
    echo "$screxe:   sources of errors."
fi
