;+
; NAME:
;         p3d_misc_profunction
;
;         $Id: p3d_misc_profunction.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         Calculates a functional value, or an array of values, using a defined
;         set of input parameters.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:    
;         fun=p3d_misc_profunction(x,p,proffun=,/nobg,topwid=, $
;                 logunit=,error=,verbose=)
;
; INPUTS:   
;         x             - A one-dimensional array with the abscissae of Y.
;         p             - A one-dimensional array with the parameter values
;                         that are used when calculating the functional return
;                         value. Each function uses its own parameters.
;
; KEYWORDS: 
;         proffun       - A scalar string with the name of the function to
;                         use when calculating the line profile.
;         nobg          - If this keyword is set then the background is not
;                         added.
;         topwid        - If set, then error messages are displayed using
;                         DIALOG_MESSAGE, using this widget id as
;                         DIALOG_PARENT, instead of MESSAGE.
;         logunit       - Messages are saved to the file pointed to by this
;                         logical file unit, if it is defined.
;         error         - Returns an error code if set
;         verbose       - Show more information on what is being done
;
; OUTPUTS:
;         fun           - An array of the same properties as X with the
;                         evaluated function.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
;-
FUNCTION p3d_misc_profunction,x,p,proffun=proffun,nobg=nobg, $
             topwid=topwid,logunit=logunit,error=error,verbose=verbose
  compile_opt hidden,IDL2

  error=0 & rname='p3d_misc_profunction: '

  if ~n_elements(proffun) then proffun='gaussian'
  nobg=keyword_set(nobg) & usebg=~nobg

  case proffun of
    'lorentzian': $
       fun=p[2L]/(0.5d0*p[1L]*!dpi)/ $
               (1d0+((x-p[0L])/(0.5d0*p[1L]))^2)+usebg*(x*p[4L]+p[3L])
    'gauss/lorentz': begin
       t=0.5d0*((x-p[0L])/p[1L])^2
       fun=p[3L]*exp((p[2L]-1d0)*t)/(1d0+p[2L]*t*1.44269504088896d0) + $
               usebg*(x*p[5L]+p[4L])
    end
    'doublegauss': $
       fun=usebg*(x*p[6L]+p[5L])+p[4L]/sqrt(2d0*!dpi)* $
             (      exp(-0.5d0*((x-p[0L])/p[1L])^2)/p[1L] + $
              p[3L]*exp(-0.5d0*((x-p[0L])/p[2L])^2)/p[2L])
    else: $
       fun=p[2L]/(p[1L]*sqrt(2d0*!dpi))* $
             exp(-((x-p[0L])/p[1L])^2/2d0)+usebg*(x*p[4L]+p[3L])
  endcase ;; proffun

  return,fun

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return,error
END ;;; function: p3d_misc_profunction
