;+
; NAME:
;         p3d_misc_e3dio_read
;
;         $Id: p3d_misc_e3dio_read.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         Open a FITS file in E3D format and return an RSS array, and
;         position table vectors and spaxel shape and size
;
; AUTHOR:
;         Joris Gerssen
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:    
;         p3d_misc_e3dio_read,fname,rss=,xpos=,ypos=,size=,shape=, $
;             topwid=,logunit=,error=,verbose=,/debug,/help 
;
; PARAMETERS:   
;         fname         - (string) name of the e3d FITS file
;
; KEYWORDS: 
;         rss           - (float) 2D array
;         xpos          - (float) 1D array with spaxel x-positions
;         ypos          - (float) 1D array with spaxel y-positions
;         size          - (float) size of a spaxel
;         shape         - (int)   spaxel shape. 0: square,  1: circular 
;         topwid        - If set, then error messages are displayed using
;                         DIALOG_MESSAGE, using this widget id as
;                         DIALOG_PARENT, instead of MESSAGE.
;         logunit       - Messages are saved to the file pointed to by this
;                         logical file unit, if it is defined.
;         error         - Returns an error code if set
;         verbose       - Show more information on what is being done
;         debug         - The error handler is not setup if debug is set.
;         help          - Show this routine documentation, and exit
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; EXAMPLE:
;         p3d_misc_e3dio_read,'pmas_example.e3d',/verbose
;         p3d_misc_e3dio_read,'pmas_example.e3d',rss=rss,xpos=xpos,ypos=ypos
;
; MODIFICATION HISTORY:
;          02.09.2009 - Adapted to p3d from an earlier version by JAG
;
;-
PRO p3d_misc_e3dio_read,fname,rss=rss,xpos=xpos,ypos=ypos,size=size, $
        shape=shape,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden, IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_misc_e3dio_read: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_e3dio_read'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  ;; determine the number of spectra in the E3D file
  data_table = headfits(fname, exten=1) 
  nrow       = sxpar(data_table, "NAXIS2")  

  ;; determine the number of pixels per spectrum
  extension = 1
  fxbopen, lun,  fname, extension      
  fxbread, lun, spec_len,  4, 1             

  ;; define output RSS
  rss = fltarr(spec_len, nrow)

  ;; define vectors storing the position table information
  xpos = fltarr(nrow)
  ypos = fltarr(nrow)

  ;; loop over the rows/spectra in the E3D extension and store the
  ;; data in the the RSS and position table arrays
  for k = 1L, nrow do begin 
    fxbread, lun, xx,  6, k   
    fxbread, lun, yy,  7, k   
    fxbread, lun, zz,  10, k   
    xpos[k-1]   = xx
    ypos[k-1]   = yy
    rss[*, k-1] = zz
  endfor
  fxbclose, lun

  ;; TEMP - save the RSS to a FITS file, to do: add header
  ;;writefits, 'rss.fits', rss

  ;; Get the spaxel shape and size information
  extension = 2
  fxbopen, lun, fname, extension

  errmsg = ''
  fxbread, lun, dummy, 1, 1, errmsg=errmsg
  if (errmsg eq '') then begin
    fxbread, lun, tmp_shape, 2, 1
    fxbread, lun, size,  3, 1
    fxbclose, lun
  endif else begin 
    print, 'No data group binary table present'
    print, 'Using the default spaxel size (1.0) and shape (S)'
    size = 1.0
    tmp_shape = 'S'
  endelse

  if tmp_shape eq 'S' then shape = 0
  if tmp_shape eq 'C' then shape = 1

 ;; Set error flag if spaxel shape is neither circular nor square
  if ~(tmp_shape eq 'S' or tmp_shape eq 'C') then begin
    error=1
    errmsg= 'Shape of spaxels is neither square nor circular.'
    goto, error_handler
  endif


  if verbose ge 1 then begin
    msg='Read E3D image :: '+ string(format='(a)', fname)
    error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
                          verbose=verbose)
  endif


  if verbose ge 1 then begin
    msg = 'Number of spaxels/spectra:: ' + string(nrow, format='(I5)')
    error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
                          verbose=verbose)
    msg = 'Spectrum length (pixels)::  '  + string(spec_len, format='(I5)')
    error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
                          verbose=verbose)
    msg = 'Spaxel shape::              ' + string(tmp_shape, format='(A5)')
    error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
                          verbose=verbose)
    msg = 'Spaxel size::               ' + string(size, format='(f5.2)')
    error=p3d_misc_logger(msg,logunit,rname=rname,topwid=topwid, $
                          verbose=verbose)
  endif

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
end ;; procedure: p3d_misc_e3dio_read

