;+
; NAME:
;         p3d_imv
;
;         $Id: p3d_imv.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         Displays raw image data using widgets.
;
;         Optionally, if the image data header contains the keyword RAWPFX
;         then the file pointed at by that keyword is loaded and shown as raw
;         data.
;
;         Also optionally, if the image contains the keyword TRCPFX then the
;         file pointed at by that keyword is used as a trace mask to overlay
;         the raw data file (RAWPFX). In this case the original image is not
;         used.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: main routine
;
; CALLING SEQUENCE:
;         p3d_imv,filename,colmin,colmax,trcpfx=,rawpfx=,rawpath=, $
;             imagepath=,title=,bottom=,xsize=,ysize=,xscrsize=,yscrsize=, $
;             group_leader=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         filename        - The filename of the data to view. This must be a
;                           fits-file. If a trace data file is specified in
;                           this file, using the fits header keyword TRCPFX
;                           (see below) then this data is not used, but the
;                           raw data is (RAWPFX).
;         colmin [min(array)] - The lower cut for colors.
;         colmax [max(array)] - The upper cut for colors.
;
; KEYWORD PARAMETERS:
;         trcpfx ['IMTRC']- This is the header keyword that is used in
;                           FILENAME to mark the trace mask filename, if that
;                           has been used. If the tracemask file is found then
;                           the data in filename is not used. Instead the raw
;                           data (RAWPFX) is used. The raw data will be shown
;                           in the first tab and the raw data with an overlayed
;                           trace mask in the next tab.
;         rawpfx ['IMRAW','IMCMB000'] -
;                           This is the header keyword that is used in
;                           FILENAME to mark the raw data filename, if that
;                           has been used. If it has been used then the raw
;                           data is loaded in a separate tab, along with the
;                           resulting data in the next tab.
;                           If the file using the first prefix is not found
;                           then the second prefix is used to find a raw file,
;                           along with the keyword RAWPATH. In this case the
;                           file marked by RAWPFX must not contain the path.
;         rawpath ['IMPATH'] -
;                           This header keyword gives the raw data path in case
;                           the second prefix of RAWPFX is used (see above).
;         imagepath       - A scalar string that, if set, is used as a path for
;                           the trace mask and raw data images (instead of
;                           whatever path is found in the header of FILENAME.
;         title ['Image Display'] -
;                           A string that is used in the title of the base
;                           widget.
;         bottom [0]      - The lower index used in the color table.
;         xsize           - X-size of the displayed image
;         ysize           - Y-size of the displayed image
;         xscrsize [min(1400,xsize)] - Defines the x-size of the draw widget.
;         yscrsize [min(800,ysize)]  - Defines the y-size of the draw widget.
;         group_leader    - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         12.06.2009 - Introduced. /CS
;
;-
PRO p3d_imv_overplot,state
  compile_opt hidden,IDL2

  rname='p3d_imv_overplot: '

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=(*state).top
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the direction:

  s=size((*state).timage)
  daxis=s[1L] eq (*state).imsize[1L]?1L:2L
  nspec=s[daxis?2L:1L]

  ;;========================================------------------------------
  ;; Overplotting the image with the trace data:

  device,get_decomposed=decomposed
  if decomposed then device,decomposed=0L

  xrange=[0,s[1]] & yrange=[0,(*state).imsize[daxis?2L:1L]]
  plot,[0],/nodata,/noerase,xmargin=[0,0],ymargin=[0,0], $
       xstyle=1+4,ystyle=1+4, $
       xtickformat='(a1)',ytickformat='(a1)',yrange=yrange,xrange=xrange

  x=dindgen(s[daxis])

  i=-1L
  while ++i lt nspec do begin
    if daxis then begin
      data=(*state).timage[*,i]
      plots,x,data,color=2L
    endif else begin
      data=reform((*state).timage[i,*])
      plots,data,x,color=2L
    endelse
  endwhile ;; ++i lt nspec-1L

  if decomposed then device,decomposed=decomposed

  return

error_handler:
  error=p3d_misc_logger(errmsg,rname=rname,topwid=top,verbose=verbose,/error)
  return
END ;;; procedure: p3d_imv_overplot


PRO p3d_imv_draw,state,error=error
  compile_opt hidden,IDL2

  rname='p3d_imv_draw: ' & error=0

  ;; Drawing the raw data:
  if (*state).useraw then begin
    wset,(*state).wid.irraw
    p3d_display_tele,(*state).rimage,(*state).min,(*state).max, $
        (*state).xsize,(*state).ysize,bottom=(*state).cbottom, $
        topwid=(*state).top,verbose=(*state).verbose, $
        error=error,debug=(*state).debug
    if error ne 0 then return
  endif

  ;; Drawing the results data:
  wset,(*state).wid.idraw
  p3d_display_tele,(*state).image,(*state).min,(*state).max, $
      (*state).xsize,(*state).ysize,bottom=(*state).cbottom, $
      topwid=(*state).top,verbose=(*state).verbose, $
      error=error,debug=(*state).debug
  if error ne 0 then return

  ;; Drawing the trace mask-overplotted data:
  if (*state).usetrace then begin

    wset,(*state).wid.itraw
    p3d_display_tele,(*state).image,(*state).min,(*state).max, $
        (*state).xsize,(*state).ysize,bottom=(*state).cbottom, $
        topwid=(*state).top,verbose=(*state).verbose, $
        error=error,debug=(*state).debug
    if error ne 0 then return

    plot_image=tvrd(true=3L)

    p3d_imv_overplot,state
    if error ne 0 then return

    oplo_image=tvrd(true=3L)

    ;; Calculating a transparency:
    out=oplo_image*(*state).alpha+plot_image*(1d0-(*state).alpha)

    ;; Showing image:
    p3d_display_tele,out,(*state).min,(*state).max, $
        (*state).xsize,(*state).ysize,true=3L,bottom=(*state).cbottom, $
        topwid=(*state).top,verbose=(*state).verbose, $
        error=error,debug=(*state).debug
    if error ne 0 then return
    
  endif ;; end if; usetrace

  return
END ;;; procedure: p3d_imv_draw


PRO p3d_imv_event,event
  compile_opt hidden,IDL2

  rname='p3d_imv_event: '

  ;;========================================------------------------------
  ;; Get the state information:

  widget_control,event.top,get_uval=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Get the user value of the event structure:

  widget_control,event.id,get_uval=uval,/hourglass

  ;; Extracting a widget "index" for those widgets which have defined one:
  index=-1L
  if n_elements(uval) ne 0L  then begin
    tmppos=strpos(uval,'_',/reverse_search)
    if tmppos ne -1L then begin
      index=float(strmid(uval,tmppos+1L))
      uval=strmid(uval,0L,tmppos)
    endif
  endif

  ;; Viewport events:
  if strlowcase(tag_names(event,/structure_name)) eq 'widget_draw' then begin
    case event.type of
      2L: begin
        tmp=string(format='(i4,",",i4)',event.x+1L,event.y+1L)
        widget_control,(*state).wid.xypos,set_value=tmp

        ;; Image statistics:
        w=1L
        si=event.id eq (*state).wid.drraw? $
           size((*state).rimage):size((*state).image)
        x1=(event.x-w)>0L & x2=(event.x+w)<(si[1L]-1L)
        y1=(event.y-w)>0L & y2=(event.y+w)<(si[2L]-1L)
        if event.id eq (*state).wid.drraw then begin
          dat=[0,0,0];(*state).rimage[x1:x2,y1:y2]
        endif else begin
          dat=(*state).image[x1:x2,y1:y2]
        endelse
        min=min(dat,max=max) & mean=mean(dat) & median=median(dat)
        std=stddev(dat)
        if ~(event.id eq (*state).wid.drraw) then begin
          fmtstr=si[si[0L]+1L] ne 4L and si[si[0L]+1L] ne 5L? $
                 '(4(f7.1,","),f7.1)':'(4(g9.3,","),g9.3)'
          tmp=string(format=fmtstr,min,max,mean,median,std)
          widget_control,(*state).wid.xystat,set_value=tmp
        endif
        return
      end
      3L: begin
        case event.id of
          (*state).wid.drraw: begin
            widget_control,(*state).wid.drraw,get_draw_view=dview
            widget_control,(*state).wid.ddraw,set_draw_view=dview
            if (*state).usetrace then $
               widget_control,(*state).wid.dtraw,set_draw_view=dview
          end
          (*state).wid.ddraw: begin
            widget_control,(*state).wid.ddraw,get_draw_view=dview
            if (*state).useraw then $
               widget_control,(*state).wid.drraw,set_draw_view=dview
            if (*state).usetrace then $
               widget_control,(*state).wid.dtraw,set_draw_view=dview
          end
          (*state).wid.dtraw: begin
            widget_control,(*state).wid.dtraw,get_draw_view=dview
            if (*state).useraw then $
               widget_control,(*state).wid.drraw,set_draw_view=dview
            widget_control,(*state).wid.ddraw,set_draw_view=dview
          end
        endcase ;; event.id
        return
      end ;; 3
      else: begin
        ;; nothing...
      end
    endcase ;; event.type
  endif ;; strlowcase(tag_names(event,/structure_name) eq 'widget_draw'

  case uval of

    'tab': begin
    end ;; case: 'tab'

    'draw': begin
      if event.press and (event.key ge 5L and event.key le 8L) then begin
        ;; Changing the viewport:
        widget_control,(*state).wid.ddraw,get_draw_view=dview

        shift=event.modifiers eq 1L?20L:5L

        case event.key of
          5L: dview[0L]-=shift
          6L: dview[0L]+=shift
          7L: dview[1L]+=shift
          8L: dview[1L]-=shift
        endcase
        widget_control,(*state).wid.ddraw,set_draw_view=dview
        if (*state).useraw then $
           widget_control,(*state).wid.drraw,set_draw_view=dview
        if (*state).usetrace then $
           widget_control,(*state).wid.dtraw,set_draw_view=dview

      endif
    end ;; case: 'draw'

    'alphaslider': begin
      widget_control,event.id,get_value=val
 
      widget_control,(*state).wid.talph,set_value=strtrim(val,2L)

      (*state).alpha=val
      p3d_imv_draw,state
    end ;; case: 'alphaslider'

    'alpha': begin
      widget_control,event.id,get_value=val

      defvalue=(*state).alpha
      defstr='The entered value must be of decimal type.'
      on_ioerror,format_error

      tmp=double(val[0L])
      if tmp ne defvalue and tmp ge 0d0 and tmp le 1d0 then begin
        (*state).alpha=tmp
        widget_control,(*state).wid.dalph,set_value=tmp
        p3d_imv_draw,state
      endif else begin
        tmp=strtrim((*state).alpha,2L)
        widget_control,(*state).wid.talph,set_value=tmp
      endelse
    end ;; case: 'alpha'

    'colminmax': begin
      widget_control,event.id,get_value=val

      defvalue=~index?(*state).min:(*state).max
      defstr='The entered value must be of decimal type.'
      on_ioerror,format_error

      tmp=double(val[0L])
      if tmp ne defvalue then begin
        if ~index then (*state).min=tmp else (*state).max=tmp

        (*state).slid=-1L

        widget_control,(*state).wid.dcold,set_droplist_select=0L

        image=(*state).transpose?transpose((*state).image):(*state).image
        xsize=(*state).transpose?(*state).ysize:(*state).xsize
        ysize=(*state).transpose?(*state).xsize:(*state).ysize
        mini=(*state).min & maxi=(*state).max

        p3d_imv_draw,state
      endif

    end ;; case: 'colminmax'

    'minmaxlimits': begin
      widget_control,event.id,get_value=val
      if event.index ge 0L then begin
        val=val[event.index]

        (*state).slid=event.index

        ;; Calculating the clipping range:
        p3d_misc_dataclip,(*state).image,range,percentage=val, $
            topwid=(*state).top,logunit=logunit,verbose=verbose,error=error, $
            debug=(*state).debug
        if error ne 0 then return

        minval=range[0L]
        maxval=range[1L]

        ;; Setting the range widget values:
        widget_control,(*state).wid.tcolm,set_value=strtrim(minval,2L)
        widget_control,(*state).wid.tcoln,set_value=strtrim(maxval,2L)

        (*state).min=minval
        (*state).max=maxval

      endif else begin
        minval=(*state).min
        maxval=(*state).max
      endelse

      image=(*state).transpose?transpose((*state).image):(*state).image
      xsize=(*state).transpose?(*state).ysize:(*state).xsize
      ysize=(*state).transpose?(*state).xsize:(*state).ysize

      p3d_imv_draw,state
    end ;; case: 'minmaxlimits'

    'transpose': begin
      (*state).transpose=~(*state).transpose

      image=(*state).transpose?transpose((*state).image):(*state).image
      xsize=(*state).transpose?(*state).ysize:(*state).xsize
      ysize=(*state).transpose?(*state).xsize:(*state).ysize
      mini=(*state).min & maxi=(*state).max

      widget_control,(*state).wid.ddraw,draw_xsize=xsize,draw_ysize=ysize

      ;; Resizing the window if it is too large:
      screensize=get_screen_size()
      bgeom=widget_info(event.top,/geometry)
      dgeom=widget_info((*state).wid.ddraw,/geometry)

      maxxsize=screensize[0L]-50L
      xdiff=(bgeom.xsize-maxxsize)>0L
      xscrsize=((dgeom.xsize-xdiff)<xsize)+2L

      maxysize=screensize[1L]-100L
      ydiff=(bgeom.ysize-maxysize)>0L
      yscrsize=((dgeom.ysize-ydiff)<ysize)+2L

      widget_control,event.top,update=0L
      widget_control,(*state).wid.ddraw,xsize=xscrsize,ysize=yscrsize

      wset,(*state).wid.idraw
      p3d_display_tele,image,mini,maxi,xsize,ysize,bottom=(*state).cbottom, $
          topwid=(*state).top,verbose=(*state).verbose,error=error, $
          debug=(*state).debug

      if (*state).usetrace then begin
        p3d_imv_overplot,state
        if error ne 0 then return
      endif

      widget_control,event.top,update=1L
    end

    'print': begin
      if (*state).strucps[0L].filename eq 'idl.ps' then $
         (*state).strucps[0L].filename=!p3d_data_path+'p3d.ps'

      ;; CMPS_FORM does not work any good at all...
;      forminfo=cmps_form(cancel=canceled,create=create, $
;          defaults=(*state).strucps,parent=event.top)

      if ~canceled then begin
        if create then begin

          idx=widget_info((*state).wid.bdtab,/tab_current)
          if (*state).usetrace then begin
            winid=~idx?(*state).wid.irraw:(*state).wid.itraw
          endif else if (*state).useraw then begin
            winid=~idx?(*state).wid.irraw:(*state).wid.idraw
          endif else begin
            winid=(*state).wid.idraw
          endelse

          wset,winid
          out=tvrd(true=3L)

          thisdevice=!d.name
          set_plot,'ps'
          device,_extra=forminfo

          p3d_display_tele,out,(*state).min,(*state).max, $
              (*state).xsize,(*state).ysize,true=3L,bottom=(*state).cbottom, $
              topwid=event.top,verbose=(*state).verbose, $
              error=error,debug=(*state).debug
          if error ne 0 then return

          device,/close
          set_plot,thisdevice
          (*state).strucps=forminfo
        endif else begin ;; ~canceled
          (*state).strucps=forminfo
        endelse
      endif ;; ~canceled

    end ;; case: 'print'

    'viewfitsheader': begin

      case index of
        0: filename=(*state).rawfile
        1: filename=(*state).tracefile
        else: filename=(*state).filename
      endcase
      xdisplayfile,filename,done_button='Exit',/grow,text=headfits(filename), $
          group=event.top,title=p3d_misc_pathify(filename,/dpath)

    end ;; case: 'viewfitsheader'

    'exit': begin
      widget_control,event.top,/destroy
      return
    end ;; case: 'exit'

    else: begin
      tmpstr='There is no handler for widget UVALUE="'+uval+'".'
      print,tmpstr
    end

  endcase ;; uval

  return

format_error:
  widget_control,event.id,set_value=strtrim(defvalue,2L)
  on_ioerror,NULL

  return
END ;;; end of: p3d_imv_event


PRO p3d_imv_resize,event
  compile_opt hidden,IDL2

  ;; Get the state information:
  widget_control,event.top,get_uvalue=state

  ;; Calculating the relative change in the size of the GUI:
  widget_control,event.id,tlb_get_size=base_size
  xchange=base_size[0L]-(*state).base_size[0L]
  ychange=base_size[1L]-(*state).base_size[1L]

  dgeom=widget_info((*state).wid.ddraw,/geometry)
  xsize=((dgeom.scr_xsize+xchange)>(*state).min_size[0L])<(*state).imsize[1L]
  ysize=((dgeom.scr_ysize+ychange)>(*state).min_size[1L])<(*state).imsize[2L]

  ;; Updating the size of the draw widget:
  widget_control,(*state).wid.ddraw,xsize=xsize,ysize=ysize

  ;; Updating the base size values in the state structure:
  widget_control,event.id,tlb_get_size=base_size
  (*state).base_size=base_size

  return
END ;;; procedure: p3d_imv_resize


PRO p3d_imv_cleanup,id
  compile_opt hidden,IDL2

  ;; Get the state information:
  widget_control,id,get_uvalue=state

  if (*state).verbose ge 3 then print,'p3d_imv: cleanup.'

  if ptr_valid(state) then ptr_free,state

  return
END ;;; end of: p3d_imv_cleanup


PRO p3d_imv,filename,colmin,colmax,trcpfx=trcpfx,rawpfx=rawpfx, $
        imagepath=imagepath, $
        title=title,alpha=alpha,bottom=bottom,xsize=xsize,ysize=ysize, $
        isplitbin=isplitbin,imageorder=imageorder,daxis=daxis, $
        xscrsize=xscrsize,yscrsize=yscrsize,group_leader=group_leader, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL version <6.2, cannot continue!.'
  error=0 & rname='p3d_imv: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_imv'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=group_leader
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(daxis) then daxis=1L
  sd=size(daxis)
  if sd[sd[0L]+2L] ne 1L or $
    (sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif

  s=size(filename)
  if s[s[0L]+2L] ne 1L or s[s[0l]+1L] ne 7L then begin
    errmsg='Argument 0, FILENAME, must be set to the filename of an existi' + $
           'ng image to view.'
    goto,error_handler
  endif
  if ~file_test(filename,/regular,/read) then begin
    errmsg='Argument 0, FILENAME, must be set to the filename of an existi' + $
           'ng image to view.'
    goto,error_handler
  endif
 
  ;; Loading the data:
  image=readfits(filename,hdr,silent=verbose lt 3,/no_unsigned)
  hdr_in=hdr

  if ~n_elements(trcpfx) then trcpfx='IMTRC'
  s=size(trcpfx)
  if s[s[0L]+2L] ne 1L or s[s[0l]+1L] ne 7L then begin
    errmsg='TRCPFX must, if set, be a scalar string.'
    goto,error_handler
  endif

  if ~n_elements(rawpfx) then rawpfx=['IMRAW','IMCMB000']
  s=size(rawpfx)
  if ~s[s[0L]+2L] or s[s[0L]+2L] gt 2L or s[s[0l]+1L] ne 7L then begin
    errmsg='RAWPFX must, if set, be a one- or two-dimensional string (array).'
    goto,error_handler
  endif

  if ~n_elements(rawpath) then rawpath='IMPATH'
  s=size(rawpath)
  if s[s[0L]+2L] ne 1L or s[s[0l]+1L] ne 7L then begin
    errmsg='RAWPATH must, if set, be a scalar string.'
    goto,error_handler
  endif

  s=size(imagepath) & useimagepath=0L
  if s[s[0L]+2L] eq 1L then begin
    if s[s[0l]+1L] ne 7L then begin
      errmsg='IMAGEPATH must, if set, be the name of an existing directory.'
      goto,error_handler
    endif
    if ~file_test(imagepath,/directory) then begin
      errmsg='IMAGEPATH must, if set, be the name of an existing directory.'
      goto,error_handler
    endif
    tmp=strpos(imagepath,path_sep(),/reverse_search)
    if tmp ne strlen(imagepath)-1L then imagepath+=path_sep()
    useimagepath=1L
  endif ;; s[s[0L]+2L] eq 1L

  s=size(image)
  usetrace=0L & timage=0L
  useraw=0L & rimage=0L

  ;; Checking for Craig Markwardts routine CMPS_FORM:
  tmp=file_search(strsplit(!path,path_sep(/search_path),/extract)+ $
          path_sep()+'cmps_form.pro')
  opsform=tmp ne ''?1L:0L
  strucps=0L
  if opsform then begin
    preselect='A4 Portrait (color)'
;    strucps=cmps_form(/initialize,select=preselect)
;    help,strucps,/str
  endif

  ;;==============================---------------
  ;; Checking if there is any information about the image raw data:

  rawfile=fxpar(hdr,rawpfx[0L])
  i=1L & more=1L
  while more do begin
    tmp=fxpar(hdr,rawpfx[0L]+strtrim(i++,2L))
    if tmp ne '0' then rawfile+=strtrim(tmp,2L) else more=0L
  endwhile

  rawfile2=fxpar(hdr,'IMRA2')
  i=1L & more=1L
  while more do begin
    tmp=fxpar(hdr,'IMRA2'+strtrim(i++,2L))
    if tmp ne '0' then rawfile2+=strtrim(tmp,2L) else more=0L
  endwhile

  if rawfile ne '0' then begin
    if useimagepath then rawfile=imagepath+file_basename(rawfile)

    ;; Also attempting the FILENAME directory, and its parent:
    curpath=file_dirname(filename,/mark_directory)
    crawfile=curpath+file_basename(rawfile)
    curpathup=file_dirname(curpath,/mark_directory)
    curawfile=curpathup+file_basename(rawfile)
    if file_test(rawfile,/regular,/read) then begin
      rimage=readfits(rawfile,rhdr,silent=verbose lt 3,/no_unsigned)
      useraw=1L
    endif else if file_test(crawfile,/regular,/read) then begin
      rimage=readfits(crawfile,rhdr,silent=verbose lt 3,/no_unsigned)
      useraw=1L
    endif else if file_test(curawfile,/regular,/read) then begin
      rimage=readfits(curawfile,rhdr,silent=verbose lt 3,/no_unsigned)
      useraw=1L
    endif else begin
      msg=['Could not find the path of the raw file that is specified in t' + $
           'he image:','  "'+file_dirname(rawfile)+'"', $
           'also not in the directory of FILENAME or its parent directory.', $
           'Would you like to pick a directory manually?']
      ret=dialog_message(msg,dialog_parent=topwid,/center,/question)
      if ret eq 'Yes' then begin
        title='Please select a directory.'
        simagepath=dialog_pickfile(title=title,dialog_parent=topwid, $
            path=!p3d_data_path,/directory,/must_exist)
        if simagepath ne '' then begin
          useimagepath=1L
          rawfile=simagepath+file_basename(rawfile)
        endif
      endif ;; ret eq 'Yes'

      if file_test(rawfile,/regular,/read) then begin
        rimage=readfits(rawfile,rhdr,silent=verbose lt 3,/no_unsigned)
        useraw=1L
      endif
    endelse ;; file_test(rawfile,/regular,/read)

    trawfile=file_basename(rawfile)

    ;; Loading the second raw data image (if it exists):
    if rawfile2 ne '0' then begin
      if useimagepath then rawfile2=imagepath+file_basename(rawfile2)

      ;; Also attempting the FILENAME directory, and its parent:
      curpath2=file_dirname(filename,/mark_directory)
      crawfile2=curpath2+file_basename(rawfile2)
      curpathup2=file_dirname(curpath2,/mark_directory)
      curawfile2=curpathup2+file_basename(rawfile2)

      if file_test(rawfile2,/regular,/read) then begin
        rimage2=readfits(rawfile2,rhdr2,silent=verbose lt 3,/no_unsigned)
      endif else if file_test(crawfile2,/regular,/read) then begin
        rimage2=readfits(crawfile2,rhdr2,silent=verbose lt 3,/no_unsigned)
      endif else if file_test(curawfile2,/regular,/read) then begin
        rimage2=readfits(curawfile2,rhdr2,silent=verbose lt 3,/no_unsigned)
      endif else begin
        msg=['Could not find the path of the raw file that is specified in' + $
             ' the image:','  "'+file_dirname(rawfile2)+'"', $
             'also not in the directory of FILENAME or its parent directory.',$
             'Would you like to pick a directory manually?']
        ret=dialog_message(msg,dialog_parent=topwid,/center,/question)
        if ret eq 'Yes' then begin
          title='Please select a directory.'
          simagepath=dialog_pickfile(title=title,dialog_parent=topwid, $
              path=!p3d_data_path,/directory,/must_exist)
          if simagepath ne '' then begin
            useimagepath=1L
            rawfile2=simagepath+file_basename(rawfile2)
          endif
        endif ;; ret eq 'Yes'
        if file_test(rawfile2,/regular,/read) then begin
          rimage2=readfits(rawfile2,rhdr2,silent=verbose lt 3,/no_unsigned)
        endif
      endelse ;; file_test(rawfile2,/regular,/read)

      if n_elements(isplitbin) eq 0L then begin
        errmsg='With two raw images ISPLITBIN must be given as an input to' + $
               ' p3d_imv, it is not.'
        goto,error_handler
      endif
      if n_elements(imageorder) eq 0L then begin
        errmsg='With two raw images IMAGEORDER must be given as an input t' + $
               'o p3d_imv, it is not.'
        goto,error_handler
      endif
      if n_elements(daxis) eq 0L then begin
        errmsg='With two raw images DAXIS must be given as an input to p3d' + $
               '_imv, it is not.'
        goto,error_handler
      endif

      si=size(rimage)
      ;; Combining the images:
      out=rimage
      if ~imageorder then begin
        if daxis then begin
          out[isplitbin:si[daxis]-1L,*]=rimage2[isplitbin:si[daxis]-1L,*]
        endif else begin
          out[*,isplitbin:si[daxis]-1L]=rimage2[*,isplitbin:si[daxis]-1L]
        endelse
      endif else begin
        if daxis then begin
          out[       0L:isplitbin-1L,*]=rimage2[       0L:isplitbin-1L,*]
        endif else begin
          out[*,       0L:isplitbin-1L]=rimage2[*,       0L:isplitbin-1L]
        endelse
      endelse ;; ~imageorder
      rimage=out
      trawfile=file_basename(rawfile)+'+'+file_basename(rawfile2)
    endif ;; rawfile2 ne '0'
  endif ;; rawfile ne '0'

  ;; Making a second attempt:
  if ~useraw and n_elements(rawpfx) eq 2L then begin
    rawfile=fxpar(hdr,rawpfx[1L])
    i=1L & more=1L
    while more do begin
      tmp=fxpar(hdr,rawpfx[1L]+strtrim(i++,2L))
      if tmp ne '0' then rawfile+=strtrim(tmp,2L) else more=0L
    endwhile
    if rawfile ne '0' then begin
      rawdpath=fxpar(hdr,rawpath)
      i=1L & more=1L
      while more do begin
        tmp=fxpar(hdr,rawpath+strtrim(i++,2L))
        if tmp ne '0' then rawdpath+=strtrim(tmp,2L) else more=0L
      endwhile
      if rawdpath ne '0' then begin
        if useimagepath then rawdpath=imagepath
        if file_test(rawdpath+rawfile,/regular,/read) then begin
          rawfile=rawdpath+rawfile
          rimage=readfits(rawfile,rhdr,silent=verbose lt 3,/no_unsigned)
          useraw=1L
        endif
      endif ;; rawpath ne '0'
      trawfile=file_basename(rawfile)
    endif ;; rawfile ne '0'
  endif ;; rawfile eq '0' and n_elements(rawpfx) eq 2L

  ;;==============================---------------
  ;; Checking if there is any information about the image trace mask:

  tracefile=''
  if useraw then begin
    tracefile=fxpar(hdr,trcpfx)
    i=1L & more=1L
    while more do begin
      tmp=fxpar(hdr,trcpfx+strtrim(i++,2L))
      if tmp ne '0' then tracefile+=strtrim(tmp,2L) else more=0L
    endwhile
    if tracefile ne '0' then begin
      if useimagepath then tracefile=imagepath+file_basename(tracefile)

      ;; Also attempting the FILENAME directory, and its parent:
      curpath=file_dirname(filename,/mark_directory)
      ctracefile=curpath+file_basename(tracefile)
      curpathup=file_dirname(curpath,/mark_directory)
      cutracefile=curpathup+file_basename(tracefile)

      if file_test(tracefile,/regular,/read) then begin
        timage=readfits(tracefile,thdr,silent=verbose lt 3,/no_unsigned)
        usetrace=1L
      endif else if file_test(ctracefile,/regular,/read) then begin
        timage=readfits(ctracefile,thdr,silent=verbose lt 3,/no_unsigned)
        usetrace=1L
      endif else if file_test(cutracefile,/regular,/read) then begin
        timage=readfits(cutracefile,thdr,silent=verbose lt 3,/no_unsigned)
        usetrace=1L
      endif
    endif ;; tracefile ne '0'

    ;; Making a second attempt:
    if strpos(filename,'_trace') ne -1L then begin
      timage=image & thdr=hdr
      tracefile=filename[0L]
      usetrace=1L
    endif
  endif ;; useraw

  ;;==============================---------------
  ;; Swapping arrays in case the data should be overlayed with the used
  ;; trace mask:

  if usetrace and useraw then begin
    image=rimage & hdr=rhdr
    s=size(image)
    useraw=0L & rimage=0L & rhdr=0L
  endif

  if ~n_elements(title) or size(title,/type) ne 7L then title='Image Display'
  title=p3d_misc_pathify(title,/dpath)

  if ~n_elements(bottom) then bottom=0L
  if ~n_elements(ysize) then ysize=s[2L]
  if ~n_elements(xsize) then xsize=s[1L]
  if ~n_elements(colmin) then colmin=min(image)
  if ~n_elements(colmax) then colmax=max(image)
  if ~n_elements(alpha) then alpha=0.5d0
  sa=size(alpha)
  if sa[sa[0L]+2L] ne 1L or $
    (sa[sa[0L]+1L] ge 6L and sa[sa[0L]+1L] le 11L) then begin
    errmsg='ALPHA, if set, must be a scalar decimal value; 0<=ALPHA<=1.0.'
    goto,error_handler
  endif
  if alpha lt 0d0 or alpha gt 1d0 then begin
    errmsg='ALPHA, if set, must be a scalar decimal value; 0<=ALPHA<=1.0.'
    goto,error_handler
  endif

  screensize=get_screen_size()
  if ~keyword_set(xscrsize) then xscrsize=xsize<(screensize[0L]-50L)
  if ~keyword_set(yscrsize) then yscrsize=ysize<(screensize[1L]-70L)

  ;;========================================------------------------------
  ;; Setting up a widget tool to show the image:

  tele_win_xsize=xsize & tele_win_ysize=ysize

  ;; A widget-ID structure:
  wid={bbase:0L,bfith:lonarr(3L),bpsfm:0L,bexit:0L,dcold:0L, $
       bdtab:0L,bdtbb:0L,brtbb:0L,bttbb:0L, $
       ddraw:0L,drraw:0L,dtraw:0L,dalph:0L,xypos:0L,xystat:0L, $
       talph:0L,tcolm:0L,tcoln:0L,idraw:0L,irraw:0L,itraw:0L}

  wid.bbase=widget_base(title=title,/column,group_leader=group_leader, $
      mbar=mbbarwid,/base_align_center,space=2L,xpad=2L,ypad=2L, $
      /tlb_size_events,resource_name='p3d')
  top=widget_info(group_leader,/valid_id)?group_leader:wid.bbase

  ;;===========
  ;; File menu:

  mfilewid=widget_button(mbbarwid,value='File',/menu, $
      event_pro='p3d_imv_event')

  if useraw or usetrace then begin
    wid.bfith[0L]=widget_button(mfilewid, $
        value='View the raw data fits header', $
        uvalue='viewfitsheader_0',uname='Click this button to view the ' + $
        'raw data fits header.',tracking_events=track)
  endif

  if usetrace then begin
    wid.bfith[1L]=widget_button(mfilewid, $
        value='View the trace mask fits header', $
        uvalue='viewfitsheader_1',uname='Click this button to view the ' + $
        'trace mask fits header.',tracking_events=track)
  endif

  if ~usetrace or (strtrim(tracefile,2L) ne filename[0L]) then begin
    wid.bfith[2L]=widget_button(mfilewid, $
        value='View the data fits header', $
        uvalue='viewfitsheader_2',uname='Click this button to view the ' + $
        'data file fits header.',tracking_events=track)
  endif

  if opsform then begin
    wid.bpsfm=widget_button(mfilewid,value='Print',uvalue='print', $
        uname='Click this button to make a hardcopy.',accelerator='Ctrl+P', $
        tracking_events=track)
  endif
  wid.bexit=widget_button(mfilewid,/separator,value='Exit',uvalue='exit', $
      uname='Click this button to exit this tool.', $
      tracking_events=track,accelerator='Ctrl+Q')


  ;;===========
  ;; The drawing area:

  if useraw or usetrace then begin
    wid.bdtab=widget_tab(wid.bbase,uvalue='tab',location=0L,/align_center, $
        xoffset=0L,yoffset=0L)
  endif

  ;; The raw data draw widget:
  if useraw then begin
    title='Raw data ('+trawfile+')'
    wid.brtbb=widget_base(wid.bdtab,title=title,xpad=0L,ypad=0L,space=0L, $
        /column,resource_name='field2')
    wid.drraw=widget_draw(wid.brtbb,uvalue='draw',xsize=xsize,ysize=ysize, $
        /scroll,x_scroll_size=xscrsize,y_scroll_size=yscrsize,retain=2L, $
        event_pro='p3d_imv_event',/keyboard_events,/motion_events, $
        viewport_events=useraw or usetrace,graphics_level=1L)
  endif

  widg=wid.bbase
  if useraw or usetrace then begin
    title=(usetrace?'Raw data ('+trawfile+')': $
                    'Result ('+file_basename(filename)+')')
    wid.bdtbb=widget_base(wid.bdtab,title=title,xpad=0L,ypad=0L,space=0L, $
        /column,resource_name='field2')
    widg=wid.bdtbb
  endif
  wid.ddraw=widget_draw(widg,uvalue='draw',xsize=xsize,ysize=ysize, $
      /scroll,x_scroll_size=xscrsize,y_scroll_size=yscrsize,retain=2L, $
      event_pro='p3d_imv_event',/keyboard_events,/motion_events, $
      viewport_events=useraw or usetrace,graphics_level=1L)

  ;; The tracing data draw widget:
  if usetrace then begin
    title='Raw data with overlayed trace mask ('+file_basename(tracefile)+')'
    wid.bttbb=widget_base(wid.bdtab,title=title,xpad=0L,ypad=0L,space=0L, $
        /column,resource_name='field2')
    wid.dtraw=widget_draw(wid.bttbb,uvalue='draw',xsize=xsize,ysize=ysize, $
        /scroll,x_scroll_size=xscrsize,y_scroll_size=yscrsize,retain=2L, $
        event_pro='p3d_imv_event',/keyboard_events,/motion_events, $
        viewport_events=useraw or usetrace,graphics_level=1L)
  endif

  ;; The control panel:
  bconpwid=widget_base(wid.bbase,/row,/align_center,/base_align_center, $
      space=20L,xpad=0L,ypad=0L,event_pro='p3d_imv_event')

  bcolmwid=widget_base(bconpwid,/row,/base_align_center, $
      space=0L,xpad=0L,ypad=0L)

  if usetrace then begin
     lalphwid=widget_label(bcolmwid,value='Alpha:')
     wid.dalph=cw_fslider(bcolmwid,/drag,format='(f4.2)',suppress_value=1L, $
         minimum=0.0,maximum=1.0,value=alpha,uvalue='alphaslider',xsize=200L)
     wid.talph=widget_text(bcolmwid,value=strtrim(alpha,2L),xsize=5L, $
         uvalue='alpha',/editable,resource_name='field')
  endif

  colminlabel=widget_label(bcolmwid,value='Value range:',/align_left)
  wid.tcolm=widget_text(bcolmwid,value=strtrim(colmin,2L),xsize=8L, $
      uvalue='colminmax_0',/editable,resource_name='field')
  wid.tcoln=widget_text(bcolmwid,value=strtrim(colmax,2L),xsize=8L, $
      uvalue='colminmax_1',/editable,resource_name='field')

  droplistvalue=['100%','99%','98%','97%','95%','90%','80%']
  wid.dcold=widget_droplist(bconpwid,value=droplistvalue, $
      uvalue='minmaxlimits',uname='Selects the pixel value range that is u' + $
      'sed when scaling the color map.')
  slid=3L & slidp=droplistvalue[slid]

  bposmwid=widget_base(bconpwid,/row,/base_align_center, $
      space=0L,xpad=0L,ypad=0L)
  lxyposwid=widget_label(bposmwid,value='X,Y:')
  pxsize=9L*!d.x_ch_size
  pysize=long(1.2*!d.y_ch_size)
  wid.xypos=widget_label(bposmwid,value='', $
                         xsize=pxsize,ysize=pysize,/sunken_frame)

  lxystawid=widget_label(bposmwid,value=' statistics:')
  pxsize=49L*!d.x_ch_size
  wid.xystat=widget_label(bposmwid,value='', $
                         xsize=pxsize,ysize=pysize,/sunken_frame)

  ;;========================================------------------------------
  ;; Realizing the tool:

  widget_control,wid.bbase,/realize

  if useraw then widget_control,wid.bbase,set_tab_current=2L

  widget_control,wid.dcold,set_droplist_select=slid
  if useraw then begin
    widget_control,wid.drraw,get_value=tmp
    wid.irraw=tmp
  endif
  widget_control,wid.ddraw,get_value=tmp & wid.idraw=tmp
  if usetrace then begin
    widget_control,wid.dtraw,get_value=tmp
    wid.itraw=tmp
  endif

  if slid ge 0L then begin
    val=slidp

    ;; Calculating the clipping range:
    p3d_misc_dataclip,image,range,percentage=val, $
        topwid=top,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    minval=range[0L]
    maxval=range[1L]

    ;; Setting the range widget values:
    widget_control,wid.tcolm,set_value=strtrim(minval,2L)
    widget_control,wid.tcoln,set_value=strtrim(maxval,2L)
  endif else begin
    minval=colmin
    maxval=colmax
  endelse

  ;;========================================------------------------------
  ;; Resizing the window if it is too large:

  bageom=widget_info(wid.bbase,/geometry)
  d1geom=widget_info(wid.ddraw,/geometry)

  maxxsize=screensize[0L]-200L
  xsize_new=long(xsize<(maxxsize-(bageom.xsize-d1geom.xsize)+2L))
  xdiff=xsize_new-d1geom.xsize & xsize=xsize_new

  maxysize=screensize[1L]-200L
  ysize_new=long(ysize<(maxysize-(bageom.ysize-d1geom.ysize)+2L))
  ydiff=ysize_new-d1geom.ysize & ysize=ysize_new

  if useraw or usetrace then begin
    c1geom=widget_info(wid.bdtbb,/geometry)
    bxsize=c1geom.xsize+xdiff
    bysize=c1geom.ysize+ydiff
  endif

  if useraw then begin
    ;; The children of the tab widget must also be resized:
    widget_control,wid.brtbb,xsize=bxsize,ysize=bysize
    widget_control,wid.drraw,xsize= xsize,ysize= ysize
  endif

  if useraw or usetrace then widget_control,wid.bdtbb,xsize=bxsize,ysize=bysize
  widget_control,wid.ddraw,xsize= xsize,ysize= ysize

  if usetrace then begin
    ;; The children of the tab widget must also be resized:
    widget_control,wid.bttbb,xsize=bxsize,ysize=bysize
    widget_control,wid.dtraw,xsize= xsize,ysize= ysize
  endif

  ;;========================================------------------------------
  ;; Storing the widget size in the state structure:

  tmp=widget_info(wid.bbase,/geometry)
  base_size=[tmp.xsize,tmp.ysize]
  min_size=[400L,400L]

  state={image:image,rimage:rimage,timage:timage,imsize:s, $
         useraw:useraw,usetrace:usetrace,min:minval,max:maxval, $
         xsize:tele_win_xsize,ysize:tele_win_ysize,alpha:alpha, $
         base_size:base_size,min_size:min_size,strucps:strucps, $
         slid:slid,slidp:droplistvalue,wid:wid,cbottom:bottom, $
         tracefile:tracefile,rawfile:rawfile,filename:filename[0L], $
         transpose:0L,top:top,debug:debug,verbose:verbose}

  ;; Storing state information:
  pstate=ptr_new(state,/no_copy)
  widget_control,wid.bbase,set_uvalue=pstate

  p3d_imv_draw,pstate

  ;; Managing events from the widget tool:
  xmanager,'p3d_imv',wid.bbase,/no_block, $
      cleanup='p3d_imv_cleanup',event_handler='p3d_imv_resize'

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=top, $
                        verbose=verbose,/error)
  return
END ;;; procedure: p3d_imv
