;+
; NAME:
;         p3d_cdmask
;
;         $Id: p3d_cdmask.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         A more detailed description is pending.
;
;         This program is called from the p3d GUI, but can also be used as a 
;         separate program.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: main routine
;
; CALLING SEQUENCE:
;         p3d_cdmask,filename,parfile,saved,out,icstr=,ostr=,oostr=, $
;             masterbias=,tracemask=,userparfile=,ofilename=,opath=, $
;             detector=,sfx=,/compress,dbin=,track=,/exmonitor,logfile=, $
;             loglevel=,stawid=,topwid=,logunit=,verbose=, $
;             error=,/debug,/help,_extra=
;
; INPUTS:
;         filename        - An array of strings that specifies the names of the
;                           raw data files, which are used when creating the
;                           dispersion mask.
;         parfile         - A scalar string that specifies the name of a file
;                           with instrument-specific setup parameters.
;
; KEYWORD PARAMETERS:
;         icstr ['_imcmb'] - A scalar string with the image-comb. specific
;                           string that is used to create the output filename.
;         ostr ['_dmask'] - A scalar string with the dispersion-mask specific
;                           string that is used to create the output filename.
;         oostr ['_oextr'] - A scalar string with the object-data specific
;                           string that is used to create the output filename.
;         masterbias      - A scalar string with the name of a master bias
;                           file. This file is required if optimal extraction
;                           is used.
;         tracemask       - A scalar string with the name of a trace mask file.
;         userparfile     - A scalar string with the name of a file with user-
;                           defined parameters.
;         ofilename       - This keyword returns the full name of the created
;                           dispersion mask file.
;         opath ['.']     - A scalar string that specifies the path, where the
;                           output data is saved.
;         detector [0]    - A scalar integer that specifies the currently
;                           selected detector; DETECTOR is a zero-based value.
;         sfx [.fits]     - A scalar string specifying the file ending (without
;                           a trailing compression suffix, such as .gz or .Z).
;         compress [0]    - If this keyword is set then the output data file is
;                           compressed (using gzip).
;         dbin [1]        - This parameter determines if the input data is re-
;                           binned on the dispersion axis (DBIN=2||3), or if
;                           the data is kept as is (DBIN=1). A good reason to
;                           rebin the data is if all pixels should fit on the
;                           screen. The different values allowed are:
;                             DBIN=1: The data is not rebinned.
;                             DBIN=2: The data is rebinned by a factor 2.
;                             DBIN=3: The data is rebinned by a factor 2 until
;                                     all pixels fit on the screen.
;         track [1]       - If this keyword is set then the dispersion mask
;                           creation GUI is set up with a status line.
;         exmonitor       - If this keyword is set then a line profiles viewer
;                           is shown after all fitting is done when using
;                           optimal extraction.
;         logfile         - If this routine is launched as a separate program
;                           then this keyword specifies the name of the log
;                           file that will be created.
;         loglevel [1]    - If this routine is launched as a separate program,
;                           and LOGFILE is used, then this keyword specifies
;                           the logging level; [1],2,3 (the higher the value
;                           the more output).
;         stawid          - If set, then various messages are written to the
;                           p3d GUI status line (this must be the widget id of
;                           that label widget).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         saved           - An integer scalar, that is set to 1 if a dispersion
;                           mask was created, otherwise it is set to 0.
;         out             - Upon a successful execution this variable contains
;                           the dispersion mask (data) when the routine exits.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; MODIFICATION HISTORY:
;         02.09.2009 - Routine introduced (moved conts. from p3d_gui). /CS
;
;-
PRO p3d_cdmask,filename,parfile,saved,out,icstr=icstr,ostr=ostr,oostr=oostr, $
        masterbias=masterbias,tracemask=tracemask,userparfile=userparfile, $
        ofilename=ofilename,opath=opath,detector=d,sfx=sfx,compress=compress, $
        dbin=dbin,track=track,exmonitor=exmonitor,logfile=logfile, $
        loglevel=loglevel,stawid=stawid,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,help=help,_extra=extra_
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_cdmask: '
  if ~n_elements(verbose) then verbose=0
  if ~n_elements(topwid) then topwid=0L
  usestawid=~n_elements(stawid)?0L:widget_info(stawid,/valid_id)
  debug=keyword_set(debug)
  saved=0

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_cdmask'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  logfileisopened=0L
  if ~n_elements(logunit) and n_elements(logfile) eq 1L then begin
    if size(logfile,/type) ne 7L then begin
      errmsg='LOGFILE must, if specified, be a scalar string with the name' + $
             ' of a file to write.'
      goto,error_handler
    endif

    openw,logunit,logfile,/get_lun,error=error
    if error ne 0 then begin
      errmsg='Could not open a logfile with the name "'+logfile+'".'
      goto,error_handler
    endif
    logfileisopened=1L

    if ~n_elements(loglevel) then loglevel=1L
    s=size(loglevel)
    if s[s[0L]+2L] ne 1L or $
      (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
      errmsg='LOGLEVEL must be a scalar integer; 1<=LOGLEVEL<=3.'
      goto,error_handler
    endif
    if loglevel lt 1L or loglevel gt 3L then begin
      errmsg='LOGLEVEL must be a scalar integer; 1<=LOGLEVEL<=3.'
      goto,error_handler
    endif

    logunit=[logunit,loglevel]
  endif ;; ~n_elements(logunit) and n_elements(logfile) eq 1L

  s=size(filename) & n=s[s[0L]+2L]
  if ~n or s[s[0L]+1L] ne 7L then begin
    errmsg='FILENAME [1] must be specified, and be of string type.'
    goto,error_handler
  endif
  for i=0L,n-1L do begin
    if ~file_test(filename[i],/read,/regular) then begin
      errmsg='Cannot read or find the file "'+filename[i]+'" ['+ $
          strtrim(i+1L,2L)+'/'+strtrim(n,2L)+']'
      goto,error_handler
    endif
  endfor

  if size(parfile,/type) ne 7L then begin
    errmsg='PARFILE [2] must be specified; as the scalar filename of an in' + $
           'strument-specific parameter file.'
    goto,error_handler
  endif
  if ~file_test(parfile,/read,/regular) then begin
    errmsg='The file PARFILE "'+parfile+'" does not exist.'
    goto,error_handler
  endif

  if ~n_elements(ostr) then ostr='_dmask'
  if size(ostr,/type) ne 7L or n_elements(ostr) ne 1L then begin
    errmsg='OSTR must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  if ~n_elements(oostr) then oostr='_oextr'
  if size(oostr,/type) ne 7L or n_elements(oostr) ne 1L then begin
    errmsg='OOSTR must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  usembiasfile=0L
  if n_elements(masterbias) eq 1L then begin
    if size(masterbias,/type) ne 7L then begin
      errmsg='MASTERBIAS [3] must be of string type.'
      goto,error_handler
    endif
    if ~file_test(masterbias,/read,/regular) then begin
      errmsg='The file MASTERBIAS "'+masterbias+'" does not exist.'
      goto,error_handler
    endif
    usembiasfile=1L
  endif

  if ~n_elements(tracemask) or size(tracemask,/type) ne 7L then begin
    errmsg='TRACEMASK must be set; and be of string type.'
    goto,error_handler
  endif
  if ~file_test(tracemask,/read,/regular) then begin
    errmsg='The file TRACEMASK "'+tracemask+'" does not exist.'
    goto,error_handler
  endif

  useuserpar=0L
  if n_elements(userparfile) ne 0L then begin
    if size(userparfile,/type) ne 7L then begin
      errmsg='USERPARFILE must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(userparfile,/read,/regular) and userparfile ne '' then begin
      errmsg='The file USERPARFILE "'+userparfile+'" does not exist.'
      goto,error_handler
    endif
    if userparfile ne '' then useuserpar=1L

    if useuserpar then $
    readcol,userparfile,uparname,uparvalue,format='a,a',delimiter=' ', $
        silent=verbose lt 3,comment=';'
  endif ;; n_elements(userparfile) ne 0L

  if n_elements(opath) ne 0L then begin
    if size(opath,/type) ne 7L then begin
      errmsg='OPATH must, if specified, be of string type.'
      goto,error_handler
    endif
    if ~file_test(opath,/directory,/read) then begin
      errmsg='The directory OPATH "'+opath+'" does not exist.'
      goto,error_handler
    endif
  endif else begin
    cd,current=opath
  endelse
  idx=strpos(opath,path_sep(),/reverse_search)
  if idx ne strlen(opath)-1L then opath+=path_sep()

  if ~n_elements(d) then d=0L
  s=size(d)
  if s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L then begin
    errmsg='DETECTOR must be of (any) integer type; DETECTOR>=0.'
    goto,error_handler
  endif
  if d lt 0L then begin
    errmsg='DETECTOR must be of (any) integer type; DETECTOR>=0.'
    goto,error_handler
  endif

  if ~n_elements(sfx) then sfx='.fits'
  if size(sfx,/type) ne 7L or n_elements(sfx) ne 1L then begin
    errmsg='SFX must, if specified, be a scalar of string type.'
    goto,error_handler
  endif

  compress=keyword_set(compress)

  ;;========================================------------------------------
  ;; Routine information:

  msg=['','==============================--------------------', $
          '========== Creating a dispersion mask', $
          '==========   '+systime(), $
          '==============================--------------------']
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 2)

  ;;========================================------------------------------
  ;; Reading instrument-specific parameters:

  ppath=strmid(parfile,0L,strpos(parfile,path_sep(),/reverse_search)+1L)
  parname='' & parvalue=''
  readcol,parfile,parname,parvalue,format='a,a',delimiter=' ', $
      silent=verbose lt 3,comment=';'

  p3d_misc_read_params,parname,parvalue,'daxis',daxis,/must_exist,/nou, $
      type='integer',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2 (not '+strtrim(daxis,2L)+').'
    goto,error_handler
  endif

  p3d_misc_read_params,parname,parvalue,'ndetector',ndetector,/nou, $
      type='integer',topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  if ~n_elements(ndetector) then ndetector=1L

  p3d_misc_read_params,parname,parvalue,'keywordsfile',kwrdlist,/must_exist, $
      uparname=uparname,uparvalue=uparvalue,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  kwrdlist=ppath+kwrdlist
  if ~file_test(kwrdlist,/read,/regular) then begin
    errmsg='The file KWRDLIST "'+kwrdlist+'" does not exist.'
    goto,error_handler
  endif

  p3d_misc_read_params,parname,parvalue,'gratingsfile_dm',gratingfile, $
      uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  if n_elements(gratingfile) eq 1L then begin
    gratingfile=ppath+gratingfile
    if ~file_test(gratingfile,/regular,/read) then begin
      errmsg='The grating file "'+gratingfile+'" does not exist.'
      goto,error_handler
    endif
  endif ;; n_elements(gratingfile) eq 1L

  str=ndetector eq 1L?'':'_'+strtrim(d+1L,2L)
  p3d_misc_read_params,parname,parvalue,'deadfibersfile'+str,deadfibersfile, $
      uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  if n_elements(deadfibersfile) eq 1L then begin
    deadfibersfile=ppath+deadfibersfile
    if ~file_test(deadfibersfile,/regular,/read) then begin
      errmsg='The dead fibers file "'+deadfibersfile+'" does not exist.'
      goto,error_handler
    endif

    readcol,deadfibersfile,deadfibers_array,sdeadfibers_,format='l,a', $
        delimiter=' ',silent=verbose lt 3,comment=';'
    if n_elements(deadfibers_array) ge 1L then begin
       deadfibers=deadfibers_array

      for i=0L,n_elements(sdeadfibers_)-1L do begin
        tmp=sdeadfibers_[i]
        iterate=1L & while iterate do begin
          idx=strpos(tmp,'_')
          if idx ge 0L then strput,tmp,' ',idx
          if idx eq -1L then iterate=0L
        endwhile ;; iterate
        sdeadfibers_[i]=tmp
      endfor ;; i=0L,n_elements(sdeadfibers_)-1L
      sdeadfibers=sdeadfibers_
    endif
  endif ;; n_elements(deadfibersfile) eq 1L

  posfile=strarr(ndetector)
  for i=0L,ndetector-1L do begin
    str=ndetector eq 1L?'':'_'+strtrim(i+1L,2L)
    p3d_misc_read_params,parname,parvalue,'postable'+str,/must_exist,/nou, $
        postable,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    posfile[i]=postable
  endfor
  posfile=ppath+posfile ;; adding the path of the parameter file

  ;; Reading the position table(s):
  fpost=ptrarr(ndetector)
  for i=0L,ndetector-1L do begin
    p3d_misc_read_postable,postable=posfile[i],rownum=rownum,id=id, $
        xpos=xpos,ypos=ypos,lens_size=lenssize,shape=shape, $
        nvertices=nvertices,/science, $
        verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    if ~n_elements(lenssize) then lenssize=1.0 ;; temporary!!

    fpost[i]=ptr_new({file:posfile[i],shape:shape,n:nvertices, $
                     fpos:rownum,fid:id,fpx:xpos,fpy:ypos,fles:lenssize})
  endfor ;; i=0L,n_elements(posfile)-1L

  p3d_misc_read_params,parname,parvalue,'lampsfile_dm',lampsfile, $
      uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  if n_elements(lampsfile) eq 1L then begin
    if ~file_test(ppath+lampsfile,/regular,/read) then begin
      errmsg='The lamps file "'+ppath+lampsfile+'" does not exist.'
      lampsfile=''
    endif else lampsfile=ppath+lampsfile
  endif else lampsfile=''

  ;;========================================------------------------------
  ;; Combining the data files, at first the data is combined accounting for if
  ;; the number of blocks>1, and if there are more than 3 (groups of) files:

  title='master arc '
  p3d_misc_imcombine_wrapper,filename,ofilename,ofsfx,ostr=icstr, $
      detsec=detsec,/dmask,parfile=parfile,userparfile=userparfile, $
      opath=opath,sfx=sfx,xstr=xstr,dsh=dsh,compress=compress,title=title, $
      dflip=dflip,stawid=stawid,topwid=topwid,logunit=logunit, $
      verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  filename=ofilename
  fsfx=ofsfx

  ;; Writing information to the status line:
  if usestawid and dsh then begin
    tmpstr='[Disp. mask] Wrote the master arc file "'+ $
           file_basename(strjoin(ofilename,', '))+'".'
    widget_control,stawid,set_value=tmpstr
  endif

  ;;========================================------------------------------
  ;; Reading some more parameters:

  if useuserpar then begin
    if ~n_elements(exmonitor) then begin
      p3d_misc_read_params,uparname,uparvalue,'exmonitor',exmonitor,/upo, $
          logunit=logunit,topwid=topwid,verbose=verbose,error=error,debug=debug
      if error ne 0 then goto,logfile_close
      exmonitor=~n_elements(exmonitor)?0L: $
                (strlowcase(exmonitor) eq 'yes'?1L:0L)
    endif
    exmonitor=keyword_set(exmonitor)

    p3d_misc_read_params,uparname,uparvalue,'dbin',dbin_,/upo,type='integer', $
        logunit=logunit,topwid=topwid,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
    if n_elements(dbin_) eq 1L then dbin=dbin_
  endif ;; useuserpar

  ;;========================================------------------------------
  ;; Determining which method to use for spectrum extraction. Since there
  ;; is no real reason to use anything but standard extraction with a
  ;; dispersion mask creation this option is not documented:

  method=p3d_misc_extract_method(userparfile,keyword='methodextract_dm', $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then goto,logfile_close
 
  nfil=n_elements(filename)
  ofilename1 =strarr(nfil)
  ofilename1e=strarr(nfil)
  ofilename  =strarr(nfil)
  ofilenamee =strarr(nfil)

  for ii=0L,nfil-1L do begin
    tmpstr=opath+file_basename(filename[ii])
    ofilename1[ii]=tmpstr+oostr    +xstr+fsfx[ii]
    tmp=strpos(ofilename1[ii],'.gz')
    if compress then begin
      if tmp eq -1L then ofilename1[ii]+='.gz'
    endif else begin
      if tmp ne -1L then ofilename1[ii]=strmid(ofilename1[ii],0L,tmp)
    endelse

    ofilename1e[ii]=tmpstr+oostr+'_err'+xstr+fsfx[ii]
    tmp=strpos(ofilename1e[ii],'.gz')
    if compress then begin
      if tmp eq -1L then ofilename1e[ii]+='.gz'
    endif else begin
      if tmp ne -1L then ofilename1e[ii]=strmid(ofilename1e[ii],0L,tmp)
    endelse

    ofilename  [ii]=tmpstr+ ostr       +xstr+fsfx[ii]
    tmp=strpos(ofilename[ii],'.gz')
    if compress then begin
      if tmp eq -1L then ofilename[ii]+='.gz'
    endif else begin
      if tmp ne -1L then ofilename[ii]=strmid(ofilename[ii],0L,tmp)
    endelse

    ofilenamee [ii]=tmpstr+ostr+'_err'+xstr+fsfx[ii]
    tmp=strpos(ofilenamee[ii],'.gz')
    if compress then begin
      if tmp eq -1L then ofilenamee[ii]+='.gz'
    endif else begin
      if tmp ne -1L then ofilenamee[ii]=strmid(ofilenamee[ii],0L,tmp)
    endelse
  endfor ;; ii=0L,n_elements(filename)-1L

  ;;==============================---------------
  ;; Retrieving names of the data files which contain the line lists:

  p3d_wavecal_set_linelist,filename[0L]+fsfx[0L],lampsfile,linelists,lamps, $
      kwrdlist=kwrdlist,userparfile=userparfile,/prompt,wsep=opath, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  ;;==============================---------------
  ;; Writing information to the status line:

  if usestawid then begin
    tmpstr='[Dispersion mask] Extracting spectra in the file "'+ $
           file_basename(filename[0L]+fsfx[0L])+'".'
    widget_control,stawid,set_value=tmpstr
  endif

  ;;==============================---------------
  ;; Loading the arc data, the trace mask, and the bias (if it is
  ;; loaded), in order to extract the same region on the CCD:

  if usembiasfile then begin
    p3d_misc_getinformation,filename[0L]+fsfx[0L],kwrdlist,dbias=dbias, $
        gain=gain,rdnoise=rdnoise,masterbias=masterbias, $
        userparfile=userparfile, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
  endif

  p3d_extract_prepare_extraction,filename[0L]+fsfx[0L],tracemask,kwrdlist, $
      disptrace,object,objshift,bias,bsw,userparfile=userparfile, $
      xbin=xbin,ybin=ybin,masterbias=masterbias,parfile=parfile,daxis=daxis, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  if n eq 2L then begin
    if usestawid then begin
      tmpstr='[wavecal] Extracting spectra in the file "'+ $
             file_basename(filename[1L]+fsfx[1L])+'".'
      widget_control,stawid,set_value=tmpstr
    endif

    p3d_extract_prepare_extraction,filename[1L]+fsfx[1L],tracemask,kwrdlist, $
        disptrace,object2,objshift2,bias2,bsw2,userparfile=userparfile, $
        xbin=xbin,ybin=ybin,masterbias=masterbias,parfile=parfile,daxis=daxis,$
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then goto,logfile_close
  endif ;; n eq 2L

  bin=daxis?ybin:xbin ;; cross-dispersion binning parameter

  ;;==============================---------------
  ;; Extracting all spectra from the raw data image by integrating up the
  ;; contribution of all contributing pixels using the trace mask:

  p3d_misc_retrieve_lprofs,tracemask,lprofs,proffun,method=method, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  p3d_extract,object,disptrace,dispextract,dout=ddispextract,bias=bias, $
      bin=bin,dbias=dbias,gain=gain,rdnoise=rdnoise,detsec=detsec, $
      trcshift=objshift[1L],parfile=parfile,xstr=xstr,method=method, $
      lprofs=lprofs,proffun=proffun,exmonitor=exmonitor, $
      subtitle=file_basename(filename[0L]+fsfx[0L]),daxis=daxis, $
      stawid=stawid,topwid=topwid, $
      logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  if n eq 2L then begin
    p3d_extract,object2,disptrace,dispextract2,dout=ddispextract2, $
        bias=bias,bin=bin,dbias=dbias,gain=gain,rdnoise=rdnoise, $
        detsec=detsec,trcshift=objshift[1L],parfile=parfile,xstr=xstr, $
        method=method,lprofs=lprofs,proffun=proffun,exmonitor=exmonitor, $
        subtitle=file_basename(filename[1L]+fsfx[1L]),daxis=daxis, $
        stawid=stawid,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug
    if error ne 0 then goto,logfile_close
  endif ;; n eq 2L

  ;;==============================---------------
  ;; Preparing and writing an output file with the extracted data:

  hdr=headfits(filename[0L]+fsfx[0L])
  fxhmake,tmphdr,dispextract
  s=size(dispextract)
  fxaddpar,hdr,'NAXIS1',s[1L]
  fxaddpar,hdr,'NAXIS2',s[2L]
  fxaddpar,hdr,'BITPIX',fxpar(tmphdr,'BITPIX')
  fxaddpar,hdr,'BZERO',0
  fxaddpar,hdr,'IMTYPE','p3d: extracted arc'

  ;; Adding information on the input files:
  p3d_misc_fitsparsplit,'IMRAW',filename[0L]+fsfx[0L],hdr, $
      ' Raw data filename',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then goto,logfile_close
  if n eq 2L then begin
    p3d_misc_fitsparsplit,'IMRA2',filename[1L]+fsfx[1L],hdr, $
        ' Raw data filename',topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug
    if error ne 0 then goto,logfile_close
  endif
  p3d_misc_fitsparsplit,'IMTRC',tracemask[0L],hdr, $
      ' Trace mask filename',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then goto,logfile_close
  if n_elements(biasfile) ne 0L then $
     p3d_misc_fitsparsplit,'IMMBI',masterbias[0L],hdr, $
         ' Master bias filename',topwid=topwid,logunit=logunit, $
         verbose=verbose,error=error,debug=debug
  p3d_misc_fitsparsplit,'IMERR',ofilename1e[0L],hdr, $
      ' Error data filename',topwid=topwid,logunit=logunit,verbose=verbose, $
      error=error,debug=debug
  if error ne 0 then goto,logfile_close

  writefits,ofilename1[0L],dispextract,hdr,compress=compress
  if n_elements(ddispextract) ne 0L then begin
    fxaddpar,hdr,'IMTYPE','p3d: extracted arc - error'
    writefits,ofilename1e[0L],ddispextract,hdr,compress=compress
  endif

  ;; Writing information to the status line:
  if usestawid then begin
    tmpstr='[wavecal] Writing the file with the extracted spectra; "'+ $
           file_basename(ofilename1[0L])+'".'
    widget_control,stawid,set_value=tmpstr
  endif

  if n_elements(filename) eq 2L then begin
    hdr2=hdr
    fxhmake,tmphdr2,dispextract2
    s=size(dispextract2)
    fxaddpar,hdr2,'BITPIX',fxpar(tmphdr2,'BITPIX')
    fxaddpar,hdr2,'BZERO',0
    fxaddpar,hdr2,'IMTYPE','p3d: extracted arc'

    ;; Adding information on the input files:
    p3d_misc_fitsparsplit,'IMERR',ofilename1e[1L],hdr2, $
        ' Error data filename',topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug
    if error ne 0 then goto,logfile_close

    writefits,ofilename1[1L],dispextract2,hdr2,compress=compress
    if n_elements(ddispextract2) ne 0L then begin
      fxaddpar,hdr2,'IMTYPE','p3d: extracted arc - error'
      writefits,ofilename1e[1L],ddispextract2,hdr2,compress=compress
    endif

    ;; Writing information to the status line:
    if usestawid then begin
      tmpstr='[wavecal] Writing the file with the extracted spectra; "'+ $
             file_basename(ofilename1[1L])+'".'
      widget_control,stawid,set_value=tmpstr
    endif

  endif ;; n_elements(filename) eq 2L

  ;;==============================---------------
  ;; Writing information to the status line:

  if usestawid then begin
    tmpstr='[wavecal] Starting the dispersion mask GUI'
    widget_control,stawid,set_value=tmpstr
  endif

  ;;==============================---------------
  ;; Reading additional parameters, that are used when preparing the
  ;; dispersion mask:

  p3d_misc_read_params,parname,parvalue,'linewidth_dm',linewidth,/must_exist, $
      type='integer',uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  linewidth/=daxis?xbin:ybin

  p3d_misc_read_params,parname,parvalue,'refdist_dm',refdist,/must_exist, $
      type='integer',uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  p3d_misc_read_params,parname,parvalue,'nrows_dm',nrows,/must_exist, $
      type='integer',uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  p3d_misc_read_params,parname,parvalue,'residualcut_dm',residualcut, $
      /must_exist,type='float',uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  p3d_misc_read_params,parname,parvalue,'polynomialorder_dm',xdeg, $
      /must_exist,type='integer',uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close

  p3d_misc_read_params,parname,parvalue,'fwhm_tr',fwhm, $
      /must_exist,type='float',uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  fwhm/=daxis?xbin:ybin

  p3d_misc_read_params,parname,parvalue,'centermethod_dm',cmethod, $
      /must_exist,uparname=uparname,uparvalue=uparvalue, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then goto,logfile_close
  cmethod=cmethod[0L]

  ;;==============================---------------
  ;; Starting the wavelength calibration gui:

;;idx=0L ;; very temporary
  posfile=(*fpost[d]).file
  p3d_wavecal_dispmask_gui,ofilename1,linelists,gratingfile, $
      ofilename,linewidth=linewidth,refdist=refdist,nrows=nrows, $
      kwrdlist=kwrdlist,xdeg=xdeg,saved=saved,residualcut=residualcut, $
      fwhm=fwhm,method=cmethod,postable=posfile,deadfibers=deadfibers, $
      sdeadfibers=sdeadfibers,userparfile=userparfile,dflip=dflip, $
      compress=compress,dbin=dbin,track=track,daxis=daxis,topwid=topwid, $
      logunit=logunit,verbose=verbose,error=error,debug=debug,_extra=extra_
  if error ne 0 then goto,logfile_close

  ofilename=ofilename[0L]
  if saved then begin
    dispmask=readfits(ofilename,silent=verbose lt 3,/no_unsigned)
    out=temporary(dispmask)
  endif

  ;; Closing the logfile, if it was opened here:
  if logfileisopened then free_lun,logunit[0L]

  ;; Writing information to the status line:
  if usestawid then begin
    tmpstr='[Dispersion mask] Done.'
    widget_control,stawid,set_value=tmpstr
  endif

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
logfile_close:
  if logfileisopened then free_lun,logunit[0L]
  return
END ;; procedure: p3d_cdmask
