;+
; NAME:
;         p3d_rss
;
;         $Id: p3d_rss.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         This is a graphical widget-based tool for the inspection of reduced
;         row-stacked spectra, of so-called 3D-spectroscopy data.
;
;         The widget tool is divided into 4 regions. These regions each show
;         the following information:
;
;         1) The spectrum image (top left area)
;
;          The spectrum image shows the input data as an image with one row
;          for each spectrum; the dispersion-direction is shown in the
;          x-direction (see the description for the keyword DAXIS if this is
;          not the case).  A vertical bar marks the wavelength bin that is
;          used to show a spatial map (cf. 2.). When the mouse pointer is
;          dragged across this image then another vertical bar indicates the
;          wavelength position of the mouse in this image, as well as in the
;          spectrum plot (4.), where the wavelength is also shown as a number.
;
;          The position of the vertical bar indicating the selected
;          wave-length bin can be moved by clicking at a new position,
;          dragging the bar (with the left mouse button), entering a new value
;          for the bin in the text widget below the image, clicking either the
;          decrement or the increment button next to the text widget (key
;          accelerators 'Ctrl+G' and 'Ctrl+H'), or using the left and right
;          arrow keys.
;
;          The pixel value range that is used to scale the used colormap is
;          indicated with the two text widgets below the image, to the left;
;          these values can be changed freely. It is also possible to let
;          P3D_RSS calculate the ranges based on a histogram-calculation where
;          a certain percentage of the pixel values are used to set the lower
;          and upper limiting values. Use the droplist widget next to the
;          value text widgets, or the functional keys F1-F8, to use this
;          function. The same scaling of the colormap is used with the
;          spectrum image and the spatial map (2.).
;
;          The narrow image areas just to the left and right of the spectrum
;          image indicate, with white lines on a black background, which image
;          rows are currently used to create the spectrum shown in the
;          spectrum plot. Note that the number of lines will only be >1 if the
;          'Sum' or 'Average' spectrum plot modes are used (cf. 2.). Another
;          spectrum can be selected by clicking in these areas; a spectrum
;          will only be shown if it is marked as a science spectrum in the
;          spatial element position table.
;
;
;         2) A spatial map at one wavelength (top right area)
;
;          The spatial map shows an image of the currently selected wavelength
;          bin as it appears on the sky. In order to put every spatial element
;          in the right place, and use the right shape, it is necessary to use
;          the correct position table file as input to P3D_RSS. The plotting
;          mode ('Single', 'Sum', or 'Average') can be selected with the
;          droplist widget below the spatial map. The accelerators 'Ctrl+F1',
;          'Ctrl+F2' and 'Ctrl+F3' can be used to choose between the
;          respective mode - the active mode is indicated with an asterisk in
;          the 'Spatial Map' menu bar.
;
;          -single- ::
;
;           If the plotting mode is chosen as 'Single' (this is the default)
;           then the currently selected spatial element is indicated with a
;           red-black disk. The image row and element ID are indicated in the
;           two text widgets below the spatial map. Another element can be
;           chosen by a left click in the image. Alternatively it is possible
;           to enter another image row number, or an ID in the text
;           widgets. The spatial element of the previous image row can also be
;           chosen with the accelerator 'Ctrl+O' and the next with
;           'Ctrl+B'. Likewise the previous element ID can be selected with
;           the accelerator 'Ctrl+P' and the next with 'Ctrl+N'. As a last
;           method the spatial elements immediately to the left, right, up,
;           and down of the currently selected element can be chosen using the
;           arrow keys.
;
;           By default the spatial map shows only the value of one wavelength
;           bin, but if the bandwidth is set to a value >0 then every spatial
;           element in the image shows an average of 2*BANDWIDTH+1 wavelength
;           bins. The wider vertical bar is shown in the spectrum image (1.).
;
;           If the right mouse button is clicked inside the image in 'Single'
;           mode then it is possible to change the direction using the
;           popup-menu. By default the spatial map shows East to the top and
;           West to the left.
;
;          -sum- & -average ::
;           If the plotting mode is chosen as 'Sum' then spatial elements must
;           first be selected by clicking in the image. The resulting -summed-
;           spectrum is then shown in the spectrum plot (3.). If the plotting
;           mode is instead 'Average' then the spectrum plot shows the average
;           of the selected spatial elements.
;
;           Elements are selected by clicking with the left mouse button, and
;           de-selected with the right mouse button; for square-shaped
;           elements the marked elements are shown as red-black squares, and
;           with circular elements they are shown as red-black disks. All
;           spatial elements can be selected by clicking on the button 'all'
;           (also the accelerator 'Ctrl+T') and all can be de-selected by
;           clicking on the button 'none' (also the accelerator 'Ctrl+Y'). The
;           total number of selected spatial elements is always indicated in
;           the label widget below the spatial map. The same selected spatial
;           elements are also indicated in the vertical wavelength bar in the
;           spectrum image (seen more easily with a non-zero BANDWIDTH) and in
;           the narrow image area between the spectrum image and the spatial
;           map.
;
;          -subtract-/-restore- ::
;           The currently selected spectrum/spectra can be subtracted from the
;           entire spectrum image. Useful for instance when checking what a
;           sky subtraction could yield. This functionality is accessed by
;           pressing the button 'Subtract' (key accelerator 'Ctrl+D') at any
;           time. The original spectrum can thereafter be restored by pressing
;           the same button again ('Restore').
;
;
;         3) A row of ten -saved- spatial maps (middle row)
;
;          The row of images is initially not initiated. By clicking on one of
;          the buttons below every image area in the row the current spatial
;          map is stored in the selected slot; there are ten such slots. The
;          wavelength bin range used to create every spatial map is then also
;          indicated on the respective button. Every image can, at any time,
;          be over-written with a new spatial map by clicking on the button
;          again.
;
;          The selected set of spatial maps in the row can be saved using the
;          function 'Save spatial maps' in the File menu (the accelerator
;          'Ctrl+W' can also be used).
;
;
;         4) A spectrum plot (bottom image area)
;
;          The currently selected spectrum is shown in the bottom image area.
;          When the mouse pointer is moved across the plot the wavelength at
;          that position is indicated with a vertical bar, both in the
;          spectrum plot and in the spectrum image (1.).
;
;          -errors- ::
;           By default no error bars are shown. If the error of IMAGE was
;           entered (in DIMAGE) then the 'Show errors' button (in the lower
;           left part of the control panel) becomes sensitive. Pressing it
;           will show error bars for all wavelength bins in the spectrum
;           image. The button then also changes name to 'Hide
;           errors'. Pressing this button will repeatedly turn on or off the
;           error bars.
;
;          -x-range- ::
;           The wavelength range is by default the full range, but it can be
;           changed manually through, e.g., the two text widgets ("x:") below
;           the image. The full wavelength range can always be re-selected by
;           pressing the 'Reset' button (or using the accelerator 'Ctrl+A').
;
;          -shift- ::
;           The shown wavelength range can also be shifted by pressing the
;           increment and decrement buttons below the spectrum plot
;           ("x-shift:"; the same functionality is achieved with the
;           respective accelerator 'Ctrl+C' and 'Ctrl+V'). Alternatively, the
;           left and right arrow keys can be used. The stride that is used
;           with the shifting is by default selected as 1/10 of the full
;           wavelength range, it can also be changed by entering a new value
;           in the text widget between the decrement and increment
;           buttons. Again, alternatively, it can be decreased/increased by 5%
;           by pressing 'Ctrl+left'/'Ctrl+right'.
;
;          -zoom- ::
;           It is possible to zoom in or out in the wavelength range using the
;           corresponding buttons below the spectrum plot ("x-zoom:",
;           down-arrow [zoom in] and up-arrow [zoom out]; the same
;           functionality is again achieved using the accelerators 'Ctrl+Z'
;           and 'Ctrl+X', respectively).  Alternatively, the up and down arrow
;           keys can be used. By default 90% (111%) of the currently visible
;           wavelength range will be used when zooming in (out). This value
;           can be changed by entering another value in the text widget
;           between the zoom-in/out buttons. Again, alternatively, the
;           percentage can be changed in steps of 1% by pressing
;           'Ctrl+down'/'Ctrl+up'.
;
;          -y-range- ::
;           The y-range is by default selected to be a histogram-calculated
;           range using 98% of the pixel values in the shown spectrum and
;           wavelength range. This range can be changed by using any of the
;           other values in the droplist widget below the spectrum plot; it is
;           also possible to use any of the accelerators
;           'Shift+F1'-'Shift+F8'. A range can also be selected manually by
;           entering any (decimal) value into the two text widgets next to the
;           droplist widget. If a value is specified manually, then the
;           histogram-calculation is switched off (until it is selected
;           again).
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: main routine
;
; CALLING SEQUENCE:
;         p3d_rss,image,posfile,dimage=,hdr=,colorcut=,specrange=,datapath=, $
;             orientation=,colortable=,bottom=,cindex=,cindv=,title=, $
;             fitsheadertitle=,tracking_events=,daxis=,group_leader=, $
;             logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         image           - A two-dimensional array of decimal type. This
;                           variable contains the data to show in row-stacked-
;                           spectra (RSS) format. One dimension holds the
;                           dispersion data (DAXIS) and the other specifies
;                           how many spectra there are in the image.
;         posfile         - A scalar string that specifies the filename of the
;                           spatial element position file. This file must
;                           follow a specific format,
;                           cf. p3d_misc_read_postable.
;
; KEYWORD PARAMETERS:
;         dimage          - A two-dimensional array of decimal type. This
;                           variable contains the error of IMAGE to show in
;                           row-stacked-spectra (RSS) format. In order to show
;                           the error bars in the spectrum plot it is necessary
;                           to press the 'Show errors' button in the lower left
;                           corner of the control panel (this button then
;                           changes name to 'Hide errors').
;         hdr             - A string array with the FITS-file header of the
;                           data provided with image. This header is primarily
;                           used to read the wavelength information through the
;                           CRVALx and CDELTx keywords (x=DAXIS). This header
;                           is also used if the spectrum or spectrum images
;                           (slices) are saved within P3D_RSS.
;         colorcut        - A two-element array of decimal type that specifies
;                           the lower and upper values that are used when
;                           scaling the color map for the spectrum image view
;                           and the spatial map view. If no value is
;                           specified then the min-max range of the values
;                           within 98% of all pixel values is used as default.
;         datapath [.]    - A scalar string that specifies in which directory
;                           spectra and spectrum images are to be saved.
;         specrange       - A two-element array of decimal type that specifies
;                           the lower and upper values that are used when
;                           scaling the color map for the spectrum plot. If no
;                           value is specified then the min-max range of the
;                           values within 98% of all pixel values in the
;                           currently shown spectrum is used as default.
;         orientation [1] - A scalar integer that specifies the orientation in
;                           the spatial map;
;                           0<=ORIENTATION<=3. The default value 1
;                           results in North being upwards, and East leftwards.
;         colortable [-1] - A scalar integer or string. If COLORTABLE is a
;                           string, then this string must point to a color
;                           table file that can be read by loadct. If it is an
;                           integer then the integer must be given in the range
;                           -1,0,...,40. While -1 loads the 'Sauron'
;                           colortable, the other values use the respective
;                           colormap as defined by loadct.
;         bottom          - A scalar integer that specifies the bottom
;                           colormap index to use with the data. Note,
;                           0<=BOTTOM<=!D.TABLE_SIZE-1.
;         cindex          - An array of integers specifying the color indices
;                           that are used as reserved colors. CINDEX ideally
;                           has 5 elements, P3D_RSS uses 5 reserved colors. If
;                           the number of elements is smaller, then the upper
;                           range of reserved colors will use the largest
;                           index available (also see CINDV).
;         cindv           - An array of 5 integers specifying which of the
;                           indices in CINDEX are used as the reserved colors:
;                           If CINDEX has 5 elements then:
;                              cindv=[CINDEX[0],CINDEX[1],CINDEX[2],
;                                     CINDEX[3],CINDEX[4]]
;                           If CINDEX has 3 elements then:
;                              cindv=[CINDEX[0],CINDEX[1],CINDEX[2],
;                                     CINDEX[2],CINDEX[2]]
;         title ['p3d Spectrum Viewer']
;                         - A string which is used in the title of the
;                           base widget.
;         fitsheadertitle ['Fits header']
;                         - This string is used when displaying the fits
;                           header.
;         tracking_events [1] - If this keyword is set then hints on the
;                           function of the various widgets and actions are
;                           given on a status line.
;         daxis [1]       - Defines the dispersion direction dimension of
;                           IMAGE. The default, 1, is in the x-direction.
;         group_leader    - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE. Also, if the
;                           parent tool is killed or closed, then so is
;                           P3D_RSS.
;         logunit         - Messages are saved to the file pointed to by this  
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         None, unless a spectrum or a (set of) spectrum image(s) is saved.
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
;         Note! The position table -must- have the format specified above,
;              otherwise it is not well defined what is seen in the
;              spatial map.
;
;         Note! Compared to previous versions, this version does not use any
;              common blocks. This means that you can run as many instances of
;              this program as you want simultaneously, they will not affect
;              each other.
;
; EXAMPLE:
;         fxread,'reduced_data.fits,data,header
;         postable='larr_positions.dat'
;         p3d_rss,data,postable,hdr=header,colortable=-1,/verbose
;
; MODIFICATION HISTORY:
;         18.11.2008 - Converted from the original routine p3de_cube_viewer of
;                      Thomas Becker and Petra Bhm. /CS
;         30.01.2009 - A first, working, version of this program is completed.
;         05.02.2009 - All parts of this tool, but 1, are now operational.
;
;-
PRO p3d_rss_find_closest_element,x,y,xsize,ysize,id,rownum, $
        fiberpos=p,orientation=orientation,pos=pos
  compile_opt hidden,IDL2

  rname='p3d_rss_find_closest_element: '

  ;; This routine finds the fiber id (&row number) of the fiber that is
  ;; closest to the (device) coordinates [x,y]:

  ;; Calculating the range - in data space:
  minx=p.xrange[0L] & spanx=p.xrange[1L]-minx
  miny=p.yrange[0L] & spany=p.yrange[1L]-miny
  span=max([spanx,spany]) ;; accounting for an aspect/=1.0
  minval=min([minx,miny]) ;;

  offy=((spanx-spany)/2L*xsize/span)>0L
  offx=((spany-spanx)/2L*ysize/span)>0L

  ;; Accounting for the orientation of the image:
  case orientation of
    0: begin ;; North is right, East is up:
      newx=y-offx
      newy=x-offy
    end
    2: begin ;; North is left, East is down:
      newx=(ysize-y-offx)>0L
      newy=(xsize-x-offy)>0L
    end
    3: begin ;; North is down, East is right:
      newx=       (x -offx)>0L
      newy=((ysize-y)-offy)>0L
    end
    else: begin ;; Default: North is up, East is left:
      newx=xsize-((x-offx)>0L)
      newy=       (y-offy)>0L
    end
  endcase ;; end case: orientation

  ;; Finding out the relative coordinates:
  tmpx=newx/double(xsize)*span+minval
  tmpy=newy/double(ysize)*span+minval

  ;; Square symbols require a 0.5 offset:
  offset=p.shape?0.0:0.5

  ;; Finding the fiber that is closest to the selected position:
  tmp=min((p.x+offset-tmpx)^2+(p.y+offset-tmpy)^2,pos)

      id=p.id [pos]
  rownum=p.row[pos]

  return
END ;;; procedure: p3d_rss_find_closest_element


PRO p3d_rss_plot,state,x,xvar=xvar,xobandw=xobandw, $
        imagebar=imagebar,imageplot=imageplot,imagebands=imagebands, $
        imagesmap=imagesmap,imagerowspatmap=imagerowspatmap, $
        oldelement=oldelement,imagesingleelement=imagesingleelement, $
        singleelement=singleelement,noerase=noerase, $
        specan=specan,specplot=specplot,arrow=arrow,topwid=topwid
  compile_opt hidden,IDL2

  rname='p3d_rss_plot: '

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; General setup:

  xobandw=keyword_set(xobandw)

  s=(*state).si

  tmppm=!p.multi
  !p.multi=0 ;; Only one diagram per figure

  ;;========================================------------------------------
  ;; Changing the decomposed state, if necessary:

  device,get_decomposed=decomposed
  if decomposed then device,decomposed=0L

  ;;========================================------------------------------
  ;; Saving the current colors, loading the used ones:

  tvlct,red,green,blue,/get
  tvlct,(*state).colors.red,(*state).colors.green,(*state).colors.blue

  ;;========================================------------------------------
  ;; Drawing the spatial map:

  rowimages=n_elements(imagerowspatmap) ne 0L?1:0
  if keyword_set(imagesmap) or rowimages then begin

    ;; Choosing either one of the row images, or the main spatial map:
    winid=rowimages?(*state).wid.isad[imagerowspatmap]:(*state).wid.isma
    wset,winid

    ;; Summing up all spectra of the selected bandwidth:
    binlow=((*state).mcol-(*state).bandwidth)>0L<((*state).nbins-1L)
    binupp=((*state).mcol+(*state).bandwidth)>0L<((*state).nbins-1L)

    marr=total((*state).image[binlow:binupp,*],1L,/double)

    ;; Saving the array in case the spectrum will be saved:
    if ptr_valid((*state).marr) then ptr_free,(*state).marr
    (*state).marr=ptr_new(marr)
    (*state).marrbin=[binlow,binupp]

    plotmarks=(*state).addmode and ~rowimages
    donoterase=n_elements(noerase) eq 1L?1L:plotmarks

    idx=where((*state).pos.id eq (*state).mfid)
    indicatecurrent=[idx[0L]]
    if n_elements(oldelement) eq 1L then begin
     idx=where((*state).pos.id eq oldelement)
     indicatecurrent=idx[0L] ne indicatecurrent[0L]? $
                     [indicatecurrent,idx[0L]]:[indicatecurrent,-1L]
    endif

    marrmean=marr/(1d0+2d0*(*state).bandwidth)

    p3d_display_telescl_postable,marrmean, $
        rowimages?(*state).rxsize:(*state).mxsize, $
        rowimages?(*state).rysize:(*state).mysize, $
        orientation=(*state).orientation,indicatecurrent=indicatecurrent, $
        locut=(*state).mcolmin,hicut=(*state).mcolmax,/science, $
        noerase=donoterase,posfile=(*state).pos.file, $
        rownum=(*state).pos.row,xpos=(*state).pos.x,ypos=(*state).pos.y, $
        lens_size=(*state).pos.lsize,arrow=arrow,shape=(*state).pos.shape, $
        plotmarks=plotmarks,marks=(*state).pos.marked, $
        single=singleelement,cindex1=(*state).cindex[(*state).cindv[0L]], $
        cindex2=(*state).cindex[(*state).cindv[2L]],bottom=(*state).cbottom, $
        topwid=(*state).wid.bbas,verbose=(*state).verbose,error=error, $
        debug=(*state).debug
    if error ne 0 then return

    if rowimages then begin
      ;; Saving the spectrum:
      (*state).msave[*,imagerowspatmap]=marr

      ;; Saving the image data in arrays (for later saving):
      lbin=(*state).wavarr[binlow]- $
           0.5d0*((*state).wavarr[binlow+1L]-(*state).wavarr[binlow])
      ubin=(*state).wavarr[binupp]+ $
           0.5d0*((*state).wavarr[binupp]-(*state).wavarr[binupp-1L])

      (*state).mwaves[*,imagerowspatmap]=[lbin,ubin]

    endif else begin

      if (*state).addmode then begin

        ;; Using the summed spectrum:
        (*state).spec=(*state).addspec
        if (*state).serr then (*state).spee=(*state).addspee

        if (*state).meanspec then begin
          n=total((*state).pos.marked)
          (*state).meannumspec=n gt 0L?double(n):1d0
        endif else begin
          (*state).meannumspec=1d0
        endelse

      endif else begin ;; (*state).addmode

        ;; Copying the spectrum to use from the image:
        *(*state).curspec=(*state).image[*,(*state).mrow]

        ;; Using the single spectrum:
        (*state).spec=(*state).curspec
        (*state).meannumspec=1d0

        if (*state).serr then begin
          *(*state).curspee=(*state).dimage[*,(*state).mrow]
          (*state).spee=(*state).curspee
        endif
      endelse ;; (*state).addmode

    endelse
  endif ;; keyword_set(imagesmap)


  ;;========================================------------------------------
  ;; The spectrum image:

  if keyword_set(imageplot) then begin
    wset,(*state).wid.imdi

    p3d_display_tele,(*state).image,(*state).colmin,(*state).colmax, $
        xpos=(*state).xoflo,bottom=(*state).cbottom,topwid=(*state).wid.bbas, $
        verbose=(*state).verbose,error=error,debug=(*state).debug
    if error ne 0 then return

  endif ;; keyword_set(imageplot)


  ;;========================================------------------------------
  ;; The spectrum image; erasing the wavelength bin indicator:

  if keyword_set(imagebar) then begin
    wset,(*state).wid.imdi

    ;; Redrawing the image at the previous position:
    for i=0L,n_elements(xvar)-1L do begin

      if xobandw then begin
        lower=((*state).mcol-xvar[i])>0L<((*state).nbins-1L)
        upper=((*state).mcol+xvar[i])>0L<((*state).nbins-1L)
      endif else begin
        lower=(xvar[i]-(*state).bandwidth)>0L<((*state).nbins-1L)
        upper=(xvar[i]+(*state).bandwidth)>0L<((*state).nbins-1L)
      endelse

      p3d_display_tele,(*state).image[lower:upper,*], $
          (*state).colmin,(*state).colmax,xpos=lower+(*state).xoflo,ypos=0L, $
          bottom=(*state).cbottom,topwid=(*state).wid.bbas, $
          verbose=(*state).verbose,error=error,debug=(*state).debug
      if error ne 0 then return
    endfor

    ;; Plotting two bars showing the wavelength range that is visible in the
    ;; wavelength plot:
    tmp=(*state).nspec-1L
    plots,[(*state).plothig,(*state).plothig]+(*state).xoflo,[0L,tmp], $
        linestyle=2L,/device
    plots,[(*state).plotlow,(*state).plotlow]+(*state).xoflo,[0L,tmp], $
        linestyle=2L,/device

  endif ;; keyword_set(imagebar)


  ;;========================================------------------------------
  ;; The spectrum image; drawing the wavelength bin indicator:

  if keyword_set(imageplot) or keyword_set(imagebar) then begin
    wset,(*state).wid.imdi

    ;; Redrawing the image at the new position:
    lower=((*state).mcol-(*state).bandwidth)>0L<((*state).nbins-1L)
    upper=((*state).mcol+(*state).bandwidth)>0L<((*state).nbins-1L)

    ;; Drawing a wavelength bin bar (of specified bandwidth):
    if n_elements(imagesingleelement) eq 1L then begin
      ;; Only one spectrum is redrawn:
      plots,[lower,upper]+(*state).xoflo, $
          [imagesingleelement,imagesingleelement], $
          color=(*state).cindex[(*state).cindv[3L]],/device
    endif else begin
      ;; Redrawing the wavelength bin bar for all spectra:
      if lower eq upper then begin
        plots,[lower,lower]+(*state).xoflo,[0L,(*state).nspec-1L], $
              color=(*state).cindex[(*state).cindv[3L]],/device
      endif else begin
        polyfill,[lower,upper,upper,lower]+(*state).xoflo, $
                 [0L,0L,(*state).nspec-1L,(*state).nspec-1L], $
                 color=(*state).cindex[(*state).cindv[3L]],/device
      endelse
    endelse

    ;; Marking the marked spectra with a darker shade of green:
    n=n_elements(*(*state).impos)
    skip=0L & if n eq 1L then skip=(*(*state).impos)[0L] eq -1L?1L:0L
    if ~skip then begin
      for k=0L,n-1L do begin
        m=(*(*state).impos)[k]
        plots,[lower,upper]+(*state).xoflo,[m,m], $
            color=(*state).cindex[(*state).cindv[4L]],/device
      endfor
    endif

    ;; Also making marks in the assisting line indicator widgets:
    polyfill,[0L,(*state).xofwid,(*state).xofwid,0L], $
        color=(*state).cindex[(*state).cindv[0L]],/device, $
        [0L,0L,(*state).nspec-1L,(*state).nspec-1L]
    polyfill,[0L,(*state).xofwid,(*state).xofwid,0L]+(*state).xofhi, $
        color=(*state).cindex[(*state).cindv[0L]],/device, $
        [0L,0L,(*state).nspec-1L,(*state).nspec-1L]
    if ~skip then begin
      for k=0L,n-1L do begin
        m=(*(*state).impos)[k]
        plots,[0L,(*state).xofwid]+(*state).xofhi, $
            [m,m],color=(*state).cindex[(*state).cindv[1L]],/device
        plots,[0L,(*state).xofwid], $
            [m,m],color=(*state).cindex[(*state).cindv[1L]],/device
      endfor
    endif

  endif ;; keyword_set(imageplot) or keyword_set(imagebar)


  ;;========================================------------------------------
  ;; Drawing the bar indicating over which wavelength bin the mouse pointer
  ;; currently is:

  if keyword_set(imagebands) then begin
    wset,(*state).wid.imdi

    ;; Redrawing the image at the previously position of the wavelength indic.:
    if (*state).xsave gt -1d0 then begin
      tmp=round(((*state).xsave-(*state).wav0bn)/(*state).wavdsp)
      if tmp ge 0L and tmp lt (*state).nbins then begin

        ;; Overdrawing the previous position of the wavelength bin indicator:
        p3d_display_tele,(*state).image[tmp,*], $
            xpos=tmp+(*state).xoflo,ypos=0L, $
            (*state).colmin,(*state).colmax,bottom=(*state).cbottom, $
            topwid=(*state).wid.bbas, $
            verbose=(*state).verbose,error=error,debug=(*state).debug
        if error ne 0 then return

      endif
    endif

    ;; Drawing a band at the currently selected wavelength:
    tmp=n_elements(x) eq 1L?round((x-(*state).wav0bn)/(*state).wavdsp):-1L
    if tmp ge 0L and tmp lt (*state).nbins then $
       plots,[tmp,tmp]+(*state).xoflo,[0L,(*state).nspec-1L], $
             color=(*state).cindex[(*state).cindv[1L]],/device

  endif ;; keyword_set(imagebands)


  ;;========================================------------------------------
  ;; Determining the y-range of the spectrum plot (manually set values are
  ;; defined elsewhere):

  if (keyword_set(specplot) or keyword_set(specan)) and $
     ~(*state).smanual then begin

    ;; Getting the percentage from the droplist widget:
    widget_control,(*state).wid.dspc,get_value=val
    idx=widget_info((*state).wid.dspc,/droplist_select)
    val=val[idx]

    factor=1d0/(*state).meannumspec

    idx=where((*state).wavarr ge (*state).wavlow and $
              (*state).wavarr le (*state).wavhig,count)
    if ~count then begin
      (*state).specymin=0.0
      (*state).specymax=1.0
    endif else begin
      spec=(*(*state).spec)[idx]
      if val eq '100%' then begin
        (*state).specymin=min(spec*factor,max=tmp)
        (*state).specymax=tmp
      endif else begin
        ;; Calculating the clipping range (histogram):
        p3d_misc_dataclip,spec*factor,range,percentage=val, $
            topwid=topwid,verbose=verbose,error=error,debug=debug
        if error ne 0 then return

        (*state).specymin=range[0L]
        (*state).specymax=range[1L]
      endelse
    endelse

    ;; Updating the y-range widgets:
    widget_control,(*state).wid.tpyl,set_value=strtrim((*state).specymin,2L)
    widget_control,(*state).wid.tpyh,set_value=strtrim((*state).specymax,2L)

  endif


  ;;========================================------------------------------
  ;; Plotting the selected spectrum:

  if keyword_set(specplot) then begin
    wset,(*state).wid.iplo

    data=*(*state).spec/(*state).meannumspec
    plot,(*state).wavarr,data,psym=10L, $
        color=(*state).cindex[(*state).cindv[1L]],charsize=(*state).charsize, $
        /xstyle,xmargin=[10.0,0.5],xrange=[(*state).wavlow,(*state).wavhig], $
        /ystyle,ymargin=[ 2.0,2.0],yrange=[(*state).specymin,(*state).specymax]

    if (*state).serr and (*state).showerrors then begin
      err=(*state).addmode?sqrt(*(*state).spee):*(*state).spee
      err/=(*state).meannumspec;sqrt((*state).meannumspec)
      errplot,(*state).wavarr,data-err,data+err,width=0.002
    endif
  endif ;; keyword_set(specplot)


  ;;========================================------------------------------
  ;; Plotting the selected spectrum - reannotating:

  if keyword_set(specan) then begin
    wset,(*state).wid.iplo

    ;; Over-writing the previous annotation with the lowermost color of the
    ;; color table - if such a one has been drawn:
    xs=(*state).xsave
    if xs gt -1d0 then begin
      str=strtrim(xs,2L)+(*state).angstr
      plots,[xs,xs],[(*state).specymin,(*state).specymax], $
            color=(*state).cindex[(*state).cindv[0L]],/data
;      tmp=0.5*(*state).plotdlt*(*state).wavdsp
;      polyfill,[xs-tmp,xs+tmp,xs+tmp,xs-tmp],[(*state).specymin, $
;          (*state).specymin,(*state).specymax,(*state).specymax],color=(*state).cblack,/data
      xyouts,[xs],[(*state).specymax*1.01-(*state).specymin*.01],str, $
          color=(*state).cindex[(*state).cindv[0L]],/data, $
          alignment=0.5,charsize=(*state).charsize,charthick=20.0
    endif

    ;; Plotting the spectrum:
    data=*(*state).spec/(*state).meannumspec
    plot,(*state).wavarr,data,noerase=1L,psym=10L, $
        color=(*state).cindex[(*state).cindv[1L]],charsize=(*state).charsize, $
        /xstyle,xmargin=[10.0,0.5],xrange=[(*state).wavlow,(*state).wavhig], $
        /ystyle,ymargin=[ 2.0,2.0],yrange=[(*state).specymin,(*state).specymax]

    if (*state).serr and (*state).showerrors then begin
      err=(*state).addmode?sqrt(*(*state).spee):*(*state).spee
      err/=(*state).meannumspec;sqrt((*state).meannumspec)
      errplot,(*state).wavarr,data-err,data+err,width=0.002
    endif ;; (*state).serr

    ;; Annotating with the current value of the wavelength:
    if n_elements(x) eq 1L then begin
      if x ge (*state).wavlow and x le (*state).wavhig then begin
        str=strtrim(x,2L)+(*state).angstr
        plots,[x,x],[(*state).specymin,(*state).specymax], $
              color=(*state).cindex[(*state).cindv[2L]],/data
        xyouts,[x,x],[(*state).specymax*1.01-(*state).specymin*.01],str, $
            color=(*state).cindex[(*state).cindv[1L]],/data,alignment=0.5, $
            charsize=(*state).charsize
      endif
    endif

  endif ;; keyword_set(specan)

  ;;========================================------------------------------
  ;; Restoring the colors:

  tvlct,red,green,blue

  ;; Restoring the decomposed state:
  if decomposed then device,decomposed=decomposed

  ;;========================================------------------------------
  ;; Resetting various plotting variables:

  !p.multi=tmppm

  ;; Re-sensitizing the colortable menu button if xloadct is closed:
  if ~xregistered('xloadct') then $
     widget_control,(*state).wid.bcol,sensitive=1L

  return
END ;;; procedure: p3d_rss_plot


PRO p3d_rss_waverange_shift,index,state
  compile_opt hidden,IDL2

  ;; Storing the location of the previous markers:
  prevcol=[(*state).plotlow,(*state).plothig]

  if ~index then begin
    ;; Shifting down:
    (*state).plothig-=round((*state).plotdlt)
    (*state).plotlow-=round((*state).plotdlt)
    (*state).plotcen-=(*state).plotdlt
  endif else begin
    ;; Shifting up:
    (*state).plothig+=round((*state).plotdlt)
    (*state).plotlow+=round((*state).plotdlt)
    (*state).plotcen+=(*state).plotdlt
  endelse

  (*state).wavlow=(*state).wav0bn+(*state).wavdsp*(*state).plotlow
  (*state).wavhig=(*state).wav0bn+(*state).wavdsp*(*state).plothig

  p3d_rss_plot,state,xvar=prevcol,/imagebar,/specplot

  ;; Updating widgets:
  tmp=(*state).plotdlt*(*state).wavdsp
  widget_control,(*state).wid.tprl,set_value=strtrim((*state).wavlow,2L)
  widget_control,(*state).wid.tprh,set_value=strtrim((*state).wavhig,2L)
  widget_control,(*state).wid.tpsh,set_value=strtrim(tmp,2L)

  ;; Sensitizing the wavelength range reset button:
  widget_control,(*state).wid.bprr,sensitive=1L
  widget_control,(*state).wid.bspr,sensitive=1L

  return
END ;;; procedure: p3d_rss_waverange_shift


PRO p3d_rss_waverange_zoom,index,state
  compile_opt hidden,IDL2

  ;; Storing the location of the previous markers:
  prevcol=[(*state).plotlow,(*state).plothig]

  ztmp=0.5d0*(*state).zoomfac*0.01d0
  factor=index?1d0/ztmp:ztmp ;; zooming out, or in
  tmp=(*state).plotcen+factor*((*state).plotlow-(*state).plotcen)
  (*state).plotlow=round(tmp)
  tmp=(*state).plotcen+factor*((*state).plothig-(*state).plotcen)
  (*state).plothig=round(tmp)

  (*state).wavlow=(*state).wav0bn+(*state).wavdsp*(*state).plotlow
  (*state).wavhig=(*state).wav0bn+(*state).wavdsp*(*state).plothig

  ;; Also changing the shift factor:
  (*state).plotdlt*=factor

  p3d_rss_plot,state,xvar=prevcol,/imagebar,/specplot

  ;; Updating widgets:
  widget_control,(*state).wid.tprl,set_value=strtrim((*state).wavlow,2L)
  widget_control,(*state).wid.tprh,set_value=strtrim((*state).wavhig,2L)
  widget_control,(*state).wid.bprr,sensitive=1L
  widget_control,(*state).wid.bspr,sensitive=1L

  tmp=(*state).wavdsp*(*state).plotdlt
  str=strtrim(tmp,2L)
  tmpstr='Press this button to shift the shown wavelength range by '
  tmp_=' ['+(*state).wavunit+'].'
  widget_control,(*state).wid.bpld,set_uname=tmpstr+'-'+str+tmp_
  widget_control,(*state).wid.bplu,set_uname=tmpstr+    str+tmp_
  widget_control,(*state).wid.tpsh,set_value=strtrim(tmp,2L)

  return
END ;;; procedure: p3d_rss_waverange_zoom


PRO p3d_rss_xloadct,data=state
  compile_opt hidden,IDL2

  ;; Loading the colors to use:
  tvlct,red,green,blue,/get
  (*state).colors={red:red,green:green,blue:blue}

  p3d_rss_plot,state,/imageplot,/imagesmap,/specplot, $
      topwid=(*state).wid.bbas

  return
END ;;; procedure: p3d_rss_xloadct


PRO p3d_rss_event,event
  compile_opt hidden,IDL2

  rname='p3d_rss_event: '

  ;;========================================------------------------------
  ;; Getting the state structure:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Getting the user value of the event id:

  widget_control,event.id,get_uvalue=uval

  ;; Extracting a widget "index" for those widgets which have defined one:
  index=-1L
  if n_elements(uval) ne 0L  then begin
    tmppos=strpos(uval,'_',/reverse_search)
    if tmppos ne -1L then begin
      index=float(strmid(uval,tmppos+1L))
      uval=strmid(uval,0L,tmppos)
    endif

  endif

  ;;========================================------------------------------
  ;; Handling widget tracking and context events first:

  tmpstrname=strlowcase(tag_names(event,/structure_name))
  if tmpstrname eq 'widget_tracking' then begin
    if event.enter then begin  
      infostr=widget_info(event.id,/uname)
      widget_control,(*state).wid.lsta,set_value=infostr
      if n_elements(uval) eq 1L then begin
        case uval of
          'spectrumplot':  wset,(*state).wid.iplo
          'spectrumimage': wset,(*state).wid.imdi
          'spatmap':       wset,(*state).wid.isma
          else:
        endcase
      endif
    endif else widget_control,(*state).wid.lsta,set_value=''

    ;; Showing a directional arrow in the object window when entering
    ;; the widget:
    if event.id eq (*state).wid.dspm then begin
      p3d_rss_plot,state,/imagesmap,arrow=event.enter and ~(*state).addmode
      (*state).enteredspatmap=event.enter
    endif

    ;; Nothing more to do, returning:
    if n_elements(uval) eq 1L then begin
      if uval ne 'spectrumplot' and uval ne 'spectrumimage' and $
         uval ne 'orientation' then return

      ;; Also skipping widget_tracking events of the orientation sel. widget:
      if uval eq 'orientation' and tmpstrname eq 'widget_tracking' then return
    endif else return

  endif  ;; strlowcase(tag_names(event,/structure_name)) eq 'widget_tracking'

  if tmpstrname eq 'widget_kbrd_focus' then begin
    if event.enter then return
  endif

  if event.id eq (*state).wid.dspm then begin
    if event.release eq 4b and ~(*state).addmode then $
       widget_displaycontextmenu,event.id,event.x,event.y,(*state).wid.bdco
  endif ;; 'widget_context'

  ;;========================================------------------------------
  ;; Searching for the correct handler for the current event:

  case uval of

    ;;===============================================
    ;; Draw widget events of the spectrum plot panel:

    'spectrumplot': begin
      tmp=strlowcase(tag_names(event,/structure_name))

      ;; Moving inside the draw widget:
      if tmp eq 'widget_draw' then begin

        wset,(*state).wid.iplo
        if event.press and ~event.modifiers and $
          (event.key eq 5L or event.key eq 6L) then begin
          ;; Left-Right keyboard arrows: shifting the wavelength range:
          upordown=event.key eq 5L?0L:1L
          p3d_rss_waverange_shift,upordown,state

          case upordown of
            0L: begin
              tmp1='left' & tmp2='lower'
            end
            1L: begin
              tmp1='right' & tmp2='higher'
            end
          endcase
          tmpstr='The "'+tmp1+'" arrow key was pressed, shifting to a '+ $
            tmp2+' range.'
          widget_control,(*state).wid.lsta,set_value=tmpstr
        endif else if event.press and ~event.modifiers and $
          (event.key eq 7L or event.key eq 8L) then begin
          ;; Up-down keyboard arrows: zooming the wavelength range:
          inorout=event.key eq 8L?0L:1L
          p3d_rss_waverange_zoom,inorout,state

          case inorout of
            0L: begin
              tmp1='down' & tmp2='in'
            end
            1L: begin
              tmp1='up' & tmp2='out'
            end
          endcase
          tmpstr='The "'+tmp1+'" arrow key was pressed; zoomed '+tmp2+'.'
          widget_control,(*state).wid.lsta,set_value=tmpstr
        endif else if event.press and event.modifiers eq 2L and $
          (event.key eq 5L or event.key eq 6L) then begin
          ;; Ctrl+left-right keyboard arrows: changing shifting factor:

          defvalue=(*state).plotdlt*(*state).wavdsp
          factor=event.key eq 5L?-0.05d0:0.05d0

          tmpq=defvalue*(1d0+factor)
          tmpp=double(tmpq/(*state).wavdsp)

          if round(tmpp) gt 0d0 then begin
            (*state).plotdlt=tmpp

            case event.key of
              5L: begin
                tmp1='down' & tmp2='decreasing'
              end
              6L: begin
                tmp1='up' & tmp2='increasing'
              end
            endcase
            tmpstr='The "Ctrl+'+tmp1+'" key combination was pressed, '+tmp2+ $
              ' the shift factor by 5%.'
            widget_control,(*state).wid.lsta,set_value=tmpstr

            str=strtrim(tmpq,2L)
            tmpstr='Press this button to shift the shown wavelength range by '
            tmp_=' ['+(*state).wavunit+'].'
            widget_control,(*state).wid.bpld,set_uname=tmpstr+'-'+str+tmp_
            widget_control,(*state).wid.bplu,set_uname=tmpstr+    str+tmp_

            widget_control,(*state).wid.tpsh, $
                set_value=string(tmpq,format='(f5.1)')
          endif

        endif else if event.press and event.modifiers eq 2L and $
          (event.key eq 7L or event.key eq 8L) then begin
          ;; Ctrl+up-down keyboard arrows: changing zooming factor:

          factor=event.key eq 7L?1d0:-1d0
          tmpq=(*state).zoomfac+factor

          if tmpq gt 0d0 and tmpq lt 1d2 then begin
            (*state).zoomfac=tmpq

            widget_control,(*state).wid.tzom, $
                set_value=string((*state).zoomfac,format='(f5.1)')
            case event.key of
              8L: begin
                tmp1='down' & tmp2='decreasing'
              end
              7L: begin
                tmp1='up' & tmp2='increasing'
              end
            endcase
            tmpstr='The "Ctrl+'+tmp1+'" key combination was pressed, '+tmp2+ $
                ' the zooming factor by 1%.'
            widget_control,(*state).wid.lsta,set_value=tmpstr
          endif

        endif else begin
          ;; Mouse cursor:
          cursor,x,y,/nowait,/data
          p3d_rss_plot,state,x,/imagebar,/imagebands,/specan
          (*state).xsave=x
        endelse
      endif ;; tmp eq 'widget_draw'

      ;; Covering the cases when the draw widget is entered and left:
      if tmp eq 'widget_tracking' then begin
        wset,(*state).wid.iplo
        if event.enter then begin
          cursor,x,y,/nowait,/data
          p3d_rss_plot,state,x,/imagebar,/imagebands,/specan
          (*state).xsave=x
        endif else begin
          ;; The widget area has been left, redrawing the image without
          ;; the wavelength position bar:
          if (*state).xsave gt -1d0 then $
             p3d_rss_plot,state,/imagebar,/imagebands,/specan
          (*state).xsave=-1d0
        endelse
      endif
    end ;; case: 'spectrumplot'

    ;;================================================
    ;; Draw widget events of the spectrum image panel:

    'spectrumimage': begin
      tmp=strlowcase(tag_names(event,/structure_name))

      ;; The widget area is left, redrawing the spectrum plot without
      ;; a wavelength position bar:
      if tmp eq 'widget_tracking' then begin
        if ~event.enter then begin
          p3d_rss_plot,state,/imagebar,/imagebands,/specan
          (*state).xsave=-1d0
        endif
      endif ;; tmp eq 'widget_tracking'

      if tmp eq 'widget_draw' then begin
        x=event.x-(*state).xoflo
        y=event.y

        ;; An arrow key was pressed:
        if event.press and (event.key eq 5L or event.key eq 6L) then begin

          oldcol=(*state).mcol
          tmp=(*state).mcol+(event.key eq 5L?-1L:1L)

          if tmp ge 0L and tmp le (*state).nbins-1L then begin
            (*state).mcol=tmp
            p3d_rss_plot,state,/imagesmap,/specan ;; /SPECAN -must- be present!

            ;; Sensitizing the increment and decrement buttons:
            tmp=(*state).nbins-1L
            sens=(*state).mcol lt tmp?1L:0L
            widget_control,(*state).wid.bbir,sensitive=sens
            widget_control,(*state).wid.bsip,sensitive=sens
            sens=(*state).mcol le 0L?0L:1L
            widget_control,(*state).wid.bbil,sensitive=sens
            widget_control,(*state).wid.bsim,sensitive=sens

            ;; Updating the wavelength bin text widget:
            tmpstr=strtrim((*state).mcol,2L)
            widget_control,(*state).wid.tbic,set_value=tmpstr

            case event.key of
              5L: begin
                tmp1='left' & tmp2='lower'
              end
              6L: begin
                tmp1='right' & tmp2='higher'
              end
            endcase
            tmpstr='The "'+tmp1+'" arrow key was pressed, shifting to a'+ $
              ' '+tmp2+' wavelength bin.'
            widget_control,(*state).wid.lsta,set_value=tmpstr

          endif

        endif ;; event.release and (event.key eq 5L or event.key eq 6L)

        replot=y ge 0L and y le (*state).nspec-1L and $
               x ge 0L and x le (*state).nbins-1L?1L:0L

        if event.press   and ~event.key then (*state).pressed=1L
        if event.release then (*state).pressed=0L

        if (*state).pressed then begin

          ;; Only handling events occuring inside the widget:
          if y ge 0L and y le (*state).nspec-1L then begin

            if x ge 0L and x le (*state).nbins-1L then begin
              replot=1L

              ;; Redrawing the spatial map for the new wavelength bin:
              oldcol=(*state).mcol
              (*state).mcol=x
              p3d_rss_plot,state,/imagesmap,/specan ;; /SPECAN -must- be here!

              ;; Updating the wavelength bin text widget:
              tmpstr=strtrim((*state).mcol,2L)
              widget_control,(*state).wid.tbic,set_value=tmpstr

            endif else if ~(*state).addmode and x ge -(*state).xoflo and $
                          x le (*state).nbins+(*state).xofwid-1L then begin

              tmp=(*state).mrow
              idx=where((*state).pos.row eq y,count)

              if count eq 1L then begin
                oldelement=(*state).mfid
                olssingleelement=*(*state).impos
                (*state).mfid=(*state).pos.id[idx]
                (*state).mrow=(*state).pos.row[idx]

                if ptr_valid((*state).impos) then ptr_free,(*state).impos
                (*state).impos=ptr_new((*state).mrow)

                p3d_rss_plot,state,/imagesmap,/specan

                ;; Updating widgets:
                sensitive=(*state).mfid gt min((*state).pos.id)?1L:0L
                widget_control,(*state).wid.bmom,sensitive=sensitive
                widget_control,(*state).wid.bmrm,sensitive=sensitive
                sensitive=(*state).mfid lt max((*state).pos.id)?1L:0L
                widget_control,(*state).wid.bmop,sensitive=sensitive
                widget_control,(*state).wid.bmrp,sensitive=sensitive
                widget_control,(*state).wid.tmod, $
                  set_value=strtrim((*state).mfid,2L)
                widget_control,(*state).wid.tmor, $
                  set_value=strtrim((*state).mrow,2L)

                p3d_rss_plot,state,xvar=oldcol, $
                    /imagebar,/imagebands,/specplot,/specan, $
                    imagesingleelement=oldsingleelement,oldelement=oldelement

              endif ;; count eq 1L
            endif

          endif
        endif ;; (*state).pressed

        ;; The spectrum image and plot are redrawn for the current wavelength:
        if replot then begin
          p3d_rss_plot,state,(*state).wavarr[x],xvar=oldcol, $
              /imagebar,/imagebands,/specan
          (*state).xsave=(*state).wavarr[x]
        endif
      endif ;; tmp eq 'widget_draw'

    end ;; case: 'spectrumimage'

    ;;================================================
    ;; Wavelength bin widgets:

    ;; Text widget:
    'imagebinnumber':begin
      widget_control,event.id,get_value=val

      defvalue=(*state).mcol
      defstr='The entered value must be an integer; 0<=value<='+ $
             strtrim((*state).nbins-1L,2L)+'.'
      on_ioerror,format_error
      val=long(val[0L])
      on_ioerror,NULL

      if val ne (*state).mcol and $
         val ge 0L and val le (*state).nbins-1L then begin
        oldcol=(*state).mcol
        (*state).mcol=val
        p3d_rss_plot,state,xvar=oldcol,/imagebar,/imagesmap

        ;; Sensitizing the increment and decrement buttons:
        if val gt                0L then sens=1L else sens=0L
        widget_control,(*state).wid.bbil,sensitive=sens
        widget_control,(*state).wid.bsim,sensitive=sens
        if val lt (*state).nbins-1L then sens=1L else sens=0L
        widget_control,(*state).wid.bbir,sensitive=sens
        widget_control,(*state).wid.bsip,sensitive=sens
      endif

      widget_control,(*state).wid.tbic,set_value=strtrim((*state).mcol,2L)
    end ;; case: 'imagebinnumber'

    ;; Decrement/Increment buttons:
    'imagebinincrementdecrement': begin

      newval=~index?((*state).mcol-1L)>0L: $
             ((*state).mcol+1L)<((*state).nbins-1L)
      if newval ne (*state).mcol then begin
        oldcol=(*state).mcol & (*state).mcol=newval
        p3d_rss_plot,state,xvar=oldcol,/imagebar,/imagesmap

        ;; Sensitizing the increment and decrement buttons:
        tmp=(*state).nbins-1L
        sens=newval lt tmp?1L:0L
        widget_control,(*state).wid.bbir,sensitive=sens
        widget_control,(*state).wid.bsip,sensitive=sens
        sens=newval le 0L?0L:1L
        widget_control,(*state).wid.bbil,sensitive=sens
        widget_control,(*state).wid.bsim,sensitive=sens

        ;; Updating the text widget:
        widget_control,(*state).wid.tbic,set_value=strtrim((*state).mcol,2L)
      endif
    end ;; end case: 'imagebinincrementdecrement'

    ;;================================================
    ;; Wavelength bandwidth:

    'bandwidth': begin
      widget_control,event.id,get_value=val

      defvalue=(*state).mcol
      defstr='The entered value must be an integer; 0<=value<=100.'
      on_ioerror,format_error
      val=long(val[0L])
      on_ioerror,NULL

      if val ge 0L and val le 100L then begin
        xvar=(*state).bandwidth
        (*state).bandwidth=val[0L]

        p3d_rss_plot,state,xvar=xvar,/xobandw,/imagebar,/imagesmap

        ;; Sensitizing the increment and decrement buttons:
        sens=val[0L] gt  0L
        widget_control,(*state).wid.bbwl,sensitive=sens
        sens=val[0L] lt 100L
        widget_control,(*state).wid.bbwr,sensitive=sens
      endif

      ;; Updating the text widget:
      widget_control,(*state).wid.tbib,set_value=strtrim((*state).bandwidth,2L)

    end ;; case: bandwidth

    'spatialmapbandwidthchange': begin

      case index of
        0: newval=((*state).bandwidth-1L)>0L    ;; decrease
        1: newval=((*state).bandwidth+1L)<100L  ;; increase
      endcase

      if newval ne (*state).bandwidth then begin
        xvar=(*state).bandwidth
        (*state).bandwidth=newval

        p3d_rss_plot,state,xvar=xvar,/xobandw,/imagebar,/imagesmap

        ;; Updating widgets:
        sens=newval lt 100L
        widget_control,(*state).wid.bbwr,sensitive=sens
        sens=newval gt 0L
        widget_control,(*state).wid.bbwl,sensitive=sens
      endif

      ;; Updating the text widget:
      widget_control,(*state).wid.tbib,set_value=strtrim((*state).bandwidth,2L)

    end ;; case: 'spatialmapbandwidthchange'

    ;;================================================
    ;; The use-a-single--use-many-spectra-droplist:

    'addmode': begin

       ;; Was the event generated by a (button widget) accelerator, or by the
       ;; droplist widget?
      if tmpstrname eq 'widget_button' then begin

        ;; Updating the droplist widget:
        if widget_info((*state).wid.bmod,/valid_id) then $
           widget_control,(*state).wid.bmod,set_droplist_select=index
      endif else begin
        index=event.index
      endelse

      ;; Updating the menu bar buttons to show which mode is currently used:
      for i=0L,2L do begin
         widget_control,(*state).wid.bsva[i],get_value=uval
         idx=strpos(uval,'* ')
         uval=i eq index?(idx eq -1L?'* '+strmid(uval,2L):uval): $
                         (idx eq -1L?uval:'  '+strmid(uval,2L))
         widget_control,(*state).wid.bsva[i],set_value=uval
      endfor

      case index of
        0: begin
          (*state).addmode=0L
          (*state).meanspec=0L
        end
        1: begin
          (*state).addmode=1L
          (*state).meanspec=0L
        end
        2: begin
          (*state).addmode=1L
          (*state).meanspec=1L
        end
      endcase
      
      ;; Resetting the value of IMPOS (indicating which elements have been
      ;; selected):
      if ptr_valid((*state).impos) then ptr_free,(*state).impos
      if (*state).addmode then begin
        idx=where((*state).pos.marked,count)
        (*state).impos=ptr_new(~count?-1L:(*state).pos.row[idx])
      endif else begin
        (*state).impos=ptr_new((*state).mrow)
      endelse

      p3d_rss_plot,state,xvar=(*state).mcol,/imagebar,/imagesmap,/specplot, $
          arrow=(*state).enteredspatmap and ~(*state).addmode

      ;; Updating related widgets:
      if (*state).addmode then begin
        widget_control,(*state).wid.tmod,sensitive=0L
        widget_control,(*state).wid.tmor,sensitive=0L
        widget_control,(*state).wid.lmnm,sensitive=1L
        widget_control,(*state).wid.bara[0L],sensitive=1L
        widget_control,(*state).wid.bara[1L],sensitive=1L
        if widget_info((*state).wid.bmar,/valid_id) then $
           widget_control,(*state).wid.bmar,sensitive=1L

        widget_control,(*state).wid.dspm,set_uname='In the spatial map: pr' + $
            'ess the left mouse button to add an additional element to the' + $
            'shown spectrum. The right mouse button unselects an element.'

      endif else begin
        widget_control,(*state).wid.tmod,sensitive=1L
        widget_control,(*state).wid.tmor,sensitive=1L
        widget_control,(*state).wid.lmnm,sensitive=0L
        widget_control,(*state).wid.bara[0L],sensitive=0L
        widget_control,(*state).wid.bara[1L],sensitive=0L
        if widget_info((*state).wid.bmar,/valid_id) then $
           widget_control,(*state).wid.bmar,sensitive=0L

        widget_control,(*state).wid.dspm,set_uname='This spatial map shows' + $
            ' a spatial image of the data for the selected wavelength (spa' + $
	    'n); right-click to change the orientation.'
      endelse

    end ;; case: 'addmode'

    ;;================================================
    ;; Spectrum plot: the wavelength range widgets:

    'wavelowhigh': begin
      widget_control,event.id,get_value=val
      
      defvalue=index?(*state).wavhig:(*state).wavlow
      defstr='The entered value must be of decimal type.'
      on_ioerror,format_error
      tmp=float(val)
      on_ioerror,NULL

      if tmp ne (*state).wavhig and tmp ne (*state).wavlow then begin
        if index then begin
          ;; high:
          (*state).wavhig=tmp
          oplothig=(*state).plothig
          (*state).plothig=round(((*state).wavhig-(*state).wav0bn)/ $
                                 (*state).wavdsp)
        endif else begin
          ;; low:
          (*state).wavlow=tmp
          oplotlow=(*state).plotlow
          (*state).plotlow=round(((*state).wavlow-(*state).wav0bn)/ $
                                 (*state).wavdsp)
        endelse

        (*state).plotcen=0.5d0*((*state).plothig+(*state).plotlow)

        p3d_rss_plot,state,xvar=oplotlow,/imagebar,/specplot

        ;; Updating widgets:
        widget_control,(*state).wid.bprr,sensitive=1L
        widget_control,(*state).wid.bspr,sensitive=1L
      endif

    end ;; case: 'wavelow'

    'wavereset':  begin

      ;; Using the entire wavelength range, that is available in the data:
      (*state).wavlow=min((*state).wavarr)
      (*state).wavhig=max((*state).wavarr)

      prevcol=[(*state).plotlow,(*state).plothig]
      (*state).plotlow=0L
      (*state).plothig=(*state).nbins-1L
      (*state).plotcen=0.5*((*state).plothig+(*state).plotlow)

      p3d_rss_plot,state,xvar=prevcol,/imagebar,/specplot

      ;; Setting text widget values:
      widget_control,(*state).wid.tprl,set_value=strtrim((*state).wavlow,2L)
      widget_control,(*state).wid.tprh,set_value=strtrim((*state).wavhig,2L)

      ;; Unsensitizing the button (it can only be reset once):
      widget_control,(*state).wid.bprr,sensitive=0L
      widget_control,(*state).wid.bspr,sensitive=0L

    end ;; case: 'wavereset'

    ;;================================================
    ;; Spectrum plot: the ordinate range widgets:

    'spectrumyrangedroplist': begin

      ;; Determining if the event came from the droplist or the menu buttons:
      if tmpstrname eq 'widget_button' then begin
        val=strtrim(index,2L)+'%'
        case index of
          100: choice=1L
          99.5: choice=2L
          99: choice=3L
          98: choice=4L
          97: choice=5L
          95: choice=6L
          90: choice=7L
          80: choice=8L
          70: choice=9L
        endcase

        ;; Updating the droplist widget:
        widget_control,(*state).wid.dspc,set_droplist_select=choice
      endif else begin
        ;; Extracting the percentage (droplist):
        widget_control,event.id,get_value=val
        val=val[event.index]
        choice=event.index
      endelse

      case choice of
        0:    (*state).smanual=1L ;; Values are not changed:
        else: (*state).smanual=0L
      endcase

      ;; Drawing the image anew:
      p3d_rss_plot,state,/specplot

    end ;; case: 'spectrumyrangedroplist'

    'spectrumyrange': begin
      widget_control,event.id,get_value=val

      case index of
        0: defvalue=(*state).specymin
        1: defvalue=(*state).specymax
      endcase

      defstr='The entered value must be of decimal type.'
      on_ioerror,format_error
      tmp=float(val)
      on_ioerror,NULL

      if tmp ne (*state).specymax and tmp ne (*state).specymin then begin

        (*state).smanual=1L ;; Making manual y-range settings

        case index of
          0: (*state).specymin=tmp
          1: (*state).specymax=tmp
        endcase

        p3d_rss_plot,state,/specplot

        ;; Sensitizing the reset button:
        widget_control,(*state).wid.dspc,set_droplist_select=0L ;; user

      endif

    end ;; case: 'spectrumyrange'

    ;;================================================
    ;; Spectrum plot: the wavelength range shifting widgets:

    'wavesupdown': begin
      p3d_rss_waverange_shift,index,state
    end ;; case: 'wavesupdown'

    'wavesval': begin
      widget_control,event.id,get_value=val

      defvalue=long((*state).plotdlt*(*state).wavdsp)
      defstr='The entered value must be of integer type.'
      on_ioerror,format_error
      tmp=double(val[0L])
      on_ioerror,NULL

      if tmp ge 0L and tmp le (*state).nbins-1L and tmp ne defvalue then begin
        (*state).plotdlt=double(tmp/(*state).wavdsp)

        str=strtrim(tmp,2L)
        tmpstr='Press this button to shift the shown wavelength range by '
        tmp_=' ['+(*state).wavunit+'].'
        widget_control,(*state).wid.bpld,set_uname=tmpstr+'-'+str+tmp_
        widget_control,(*state).wid.bplu,set_uname=tmpstr+    str+tmp_
      endif

    end ;; case: 'wavesval'

    ;;================================================
    ;; Spectrum plot: the wavelength zooming widgets:

    'wavezoominout': begin
      p3d_rss_waverange_zoom,index,state
    end ;; case: 'wavezoominout'

    'wavezoomfactor': begin
      widget_control,event.id,get_value=val
      notok=1L
      on_ioerror,zoomfac_error
      tmp=double(val)
      on_ioerror,NULL

      notok=0L
      if tmp le 0d0 or tmp ge 1d2 then notok=1L
      zoomfac_error:
      if notok then begin
        widget_control,(*state).wid.tzom, $
                       set_value=string((*state).zoomfac,format='(f5.1)')
        widget_control,(*state).wid.lsta,set_value='The entered value must' + $
            ' be of decimal type; 0.0<value<100.0.'
      endif else begin
        (*state).zoomfac=tmp
        ;; Reformatting the value:
        widget_control,(*state).wid.tzom, $
                       set_value=string((*state).zoomfac,format='(f5.1)')
      endelse

    end ;; case: 'wavezoomfactor'

    ;;================================================
    ;; The spatial map:

    'spatmap': begin

      ;; If one of the mouse buttons was pressed:
      if event.press ne 0b and ~event.key then begin
        (*state).mpressed=1b
        (*state).buttonmode=event.press
      endif

      ;; If one of the mouse buttons was released, then we exit this mode:
      if event.release ne 0b then (*state).mpressed=0b

      ;; Arrow keys:
      if event.release and ~(*state).mpressed and ~(*state).addmode and $
        event.key ge 5L and event.key le 8L then begin

        x=event.x
        y=event.y

        ;; Only using events originating inside the draw widget area:
        if x ge 0L and x lt (*state).mxsize-1L and $
           y ge 0L and y lt (*state).mysize-1L then begin


          idx=where((*state).pos.row eq (*state).mrow)
          val=0.45*(*state).pos.lsize[idx] & val=val[0L]

          ;; Finding the element that is closest to the clicked position:
          case event.key of
            5L: begin ;;  east of the current position
              xoff= val & yoff= 0.0 & tmp1='left' & tmp2='to the left'
            end
            6L: begin ;;  west of the current position
              xoff=-val & yoff= 0.0 & tmp1='right' & tmp2='to the right'
            end
            7L: begin ;; north of the current position
              xoff= 0.0 & yoff= val & tmp1='up' & tmp2='upwards'
            end
            8L: begin ;; south of the current position
              xoff= 0.0 & yoff=-val & tmp1='down' & tmp2='downwards'
            end
          endcase

          ;;Accounting for the orientation:
          case (*state).orientation of
            0: begin
              xof= yoff
              yof=-xoff
            end
            1: begin
              xof= xoff
              yof= yoff
            end
            2: begin
              xof=-yoff
              yof= xoff
            end 
            3: begin
              xof=-xoff
              yof=-yoff
            end
          endcase

          cx=(*state).pos.x[idx] & cx=cx[0L]
          cy=(*state).pos.y[idx] & cy=cy[0L]

          sep=((*state).pos.x-(cx+xof))^2 + $
              ((*state).pos.y-(cy+yof))^2
          idx=sort(sep)
          pos=idx[1L] ;; not using the current index (0)

          mfid=(*state).pos.id[pos]
          mrow=(*state).pos.row[pos]

          oldelement=(*state).mfid
          olssingleelement=*(*state).impos
          (*state).mrow=mrow
          (*state).mfid=mfid

          if ptr_valid((*state).impos) then ptr_free,(*state).impos
          (*state).impos=ptr_new(mrow)

          p3d_rss_plot,state,/imagebar,/imagesmap, $
              imagesingleelement=oldsingleelement,oldelement=oldelement, $
              arrow=(*state).enteredspatmap,/specplot

          ;; Updating the widgets showing the current spatial elements:
          widget_control,(*state).wid.tmod,set_value=strtrim(mfid,2L)
          widget_control,(*state).wid.tmor,set_value=strtrim(mrow,2L)

          sensitive=(*state).mfid gt min((*state).pos.id)?1L:0L
          widget_control,(*state).wid.bmom,sensitive=sensitive
          widget_control,(*state).wid.bmrm,sensitive=sensitive
          sensitive=(*state).mfid lt max((*state).pos.id)?1L:0L
          widget_control,(*state).wid.bmop,sensitive=sensitive
          widget_control,(*state).wid.bmrp,sensitive=sensitive

          tmpstr='The "'+tmp1+'" arrow key was pressed, shifting to the'+ $
            ' closest spectrum '+tmp2+'.'
          widget_control,(*state).wid.lsta,set_value=tmpstr

        endif ;; x ge 0L and x lt (*state).mxsize-1L and $
              ;; y ge 0L and y lt (*state).mysize-1L
      endif ;; *arrow keys*

      ;; Only acting if a button was pressed:
      if (*state).mpressed then begin
        x=event.x
        y=event.y

        ;; Only using events originating inside the draw widget area:
        if x ge 0L and x lt (*state).mxsize-1L and $
           y ge 0L and y lt (*state).mysize-1L then begin

          ;; Finding the element that is closest to the clicked position:
          p3d_rss_find_closest_element,x,y, $
              (*state).mxsize,(*state).mysize,mfid,mrow,pos=pos, $
              fiberpos=(*state).pos,orientation=(*state).orientation

          ;; Are spatial elements added, or are they chosen one at a time:
          if (*state).addmode then begin

            ;;==========-----
            ;; Adding the selected element to the stack of selected elements:
            ;;==========-----

            previousmark=(*state).pos.marked[pos]
            ;; The element is added only if a button was pressed:
            newmark=long((*state).buttonmode eq 1b)
            used=newmark-previousmark
            (*state).pos.marked[pos]=newmark

            ;; Saving an array with the image rows of the selected elements:
            idx=where((*state).pos.marked,count)
            if ptr_valid((*state).impos) then ptr_free,(*state).impos
            (*state).impos=ptr_new(~count?-1L:(*state).pos.row[idx])

            ;; Adding or subtracting the current spectrum to the summed
            ;; spectrum array. For the first click the spectrum is added, on
            ;; the second click it is subtracted:
            marr=(*state).image[*,mrow]
            *(*state).addspec+=used*marr
            if (*state).serr then begin
              dmarr=(*state).dimage[*,mrow]
              *(*state).addspee+=used*dmarr^2
            endif

            p3d_rss_plot,state,/imagebar,/imagesmap,singleelement=pos

            ;; Updating the widget that shows how many spectra are summed:
            tmp=string(total((*state).pos.marked),format='(i5)')
            widget_control,(*state).wid.lmnm,set_value=tmp

          endif else begin

            ;;==========-----
            ;; Selecting a single element:
            ;;==========-----

            if event.press gt 1b then return ;; only accepting left clicks

            oldelement=(*state).mfid
            olssingleelement=*(*state).impos
            (*state).mrow=mrow
            (*state).mfid=mfid

            if ptr_valid((*state).impos) then ptr_free,(*state).impos
            (*state).impos=ptr_new(mrow)

            p3d_rss_plot,state,/imagebar,/imagesmap, $
                imagesingleelement=oldsingleelement,oldelement=oldelement, $
                arrow=(*state).enteredspatmap

            ;; Updating the widgets showing the current spatial elements:
            widget_control,(*state).wid.tmod,set_value=strtrim(mfid,2L)
            widget_control,(*state).wid.tmor,set_value=strtrim(mrow,2L)

            sensitive=(*state).mfid gt min((*state).pos.id)?1L:0L
            widget_control,(*state).wid.bmom,sensitive=sensitive
            widget_control,(*state).wid.bmrm,sensitive=sensitive
            sensitive=(*state).mfid lt max((*state).pos.id)?1L:0L
            widget_control,(*state).wid.bmop,sensitive=sensitive
            widget_control,(*state).wid.bmrp,sensitive=sensitive

          endelse ;; (*state).addmode

          p3d_rss_plot,state,/specplot

        endif ;; x ge 0L and x lt (*state).mxsize-1L and $
              ;; y ge 0L and y lt (*state).mysize-1L
      endif ;; (*state).mpressed

    end ;; case: 'spatmap'

    'spatmapmarkallnone': begin
      if (*state).addmode then begin

        ;; Was the event generated by a (button widget) accelerator, or by the
        ;; droplist widget?
        if tmpstrname eq 'widget_button' then begin

          ;; Updating the droplist widget:
          if widget_info((*state).wid.bmar,/valid_id) then $
             widget_control,(*state).wid.bmar,set_droplist_select=index

        endif else index=event.index

        ;; index=0 (all), =1 (none)
        (*state).pos.marked=index?0b:1b

        ;; Saving an array with the image rows of the selected elements:
        if ptr_valid((*state).impos) then ptr_free,(*state).impos
        (*state).impos=ptr_new(index?-1L:(*state).pos.row)

        if ~index then begin
          ;; Adding or subtracting the current spectrum to the summed
          ;; spectrum array. For the first click the spectrum is added, on
          ;; the second click it is subtracted:

          marr=dblarr((*state).nbins)
          dmarr=marr
          for i=0L,n_elements((*state).pos.row)-1L do begin
            if (*state).pos.marked[i] then begin
              idx=(*state).pos.row[i]
              marr+=(*state).image[*,idx]
              if (*state).serr then dmarr+=(*state).dimage[*,idx]^2
            endif
          endfor
          *(*state).addspec=reform(marr)
          if (*state).serr then *(*state).addspee=reform(dmarr)
        endif else begin
          *(*state).addspec=fltarr((*state).nbins)
          if (*state).serr then *(*state).addspee=*(*state).addspec
        endelse

        p3d_rss_plot,state,/imagebar,/imagesmap,/specplot

        ;; Updating the widget that shows how many spectra are summed:
        tmp=string(total((*state).pos.marked),format='(i5)')
        widget_control,(*state).wid.lmnm,set_value=tmp

      endif ;; (*state).addmode
    end ;; case: 'spatmapmarkallnone'

    'spatialmapselectid': begin
      widget_control,event.id,get_value=val
      
      defvalue=(*state).mfid
      defstr='The entered value must be of integer type.'
      on_ioerror,format_error
      tmp=long(val[0L])
      on_ioerror,NULL

      idx=where((*state).pos.id eq tmp,count)
      if ~count then begin
        ;; Resetting the text widget:
        widget_control,(*state).wid.tmod,set_value=strtrim((*state).mfid,2L)

        ;; Updating the status widget:
        tmpstr='There is no spatial element with the ID '+strtrim(tmp,2L)+'.'
        widget_control,(*state).wid.lsta,set_value=tmpstr
      endif else if tmp ne (*state).mfid then begin

        oldelement=(*state).mfid
        olssingleelement=*(*state).impos
        (*state).mfid=tmp
        (*state).mrow=(*state).pos.row[idx]

        if ptr_valid((*state).impos) then ptr_free,(*state).impos
        (*state).impos=ptr_new((*state).mrow)

        p3d_rss_plot,state,/imagebar,/imagesmap,/specplot, $
            imagesingleelement=oldsingleelement,oldelement=oldelement, $
            arrow=(*state).enteredspatmap

        ;; Updating widgets:
        widget_control,(*state).wid.tmor,set_value=strtrim((*state).mrow,2L)

        sensitive=tmp gt min((*state).pos.id)?1L:0L
        widget_control,(*state).wid.bmom,sensitive=sensitive
        widget_control,(*state).wid.bmrm,sensitive=sensitive
        sensitive=tmp lt max((*state).pos.id)?1L:0L
        widget_control,(*state).wid.bmop,sensitive=sensitive
        widget_control,(*state).wid.bmrp,sensitive=sensitive

      endif
    end ;; case: 'spatialmapselectid'

    'spatmapdecrementincrement': begin

      ok=0L
      case index of
        0L: begin ;; decrement id
          tmp=(*state).mfid
          while ~ok do idx=where((*state).pos.id eq --tmp,ok)
        end
        1L: begin ;; increment id
          tmp=(*state).mfid
          while ~ok do idx=where((*state).pos.id eq ++tmp,ok)
        end
        2L: begin ;; decrement row
          tmp=(*state).mrow
          while ~ok do idx=where((*state).pos.row eq --tmp,ok)
        end
        3L: begin ;; increment row
          tmp=(*state).mrow
          while ~ok do idx=where((*state).pos.row eq ++tmp,ok)
        end
      endcase

      oldelement=(*state).mfid
      olssingleelement=*(*state).impos
      (*state).mfid=(*state).pos.id[idx]
      (*state).mrow=(*state).pos.row[idx]

      if ptr_valid((*state).impos) then ptr_free,(*state).impos
      (*state).impos=ptr_new((*state).mrow)

      p3d_rss_plot,state,/imagebar,/imagesmap,/specplot, $
          imagesingleelement=oldsingleelement,oldelement=oldelement, $
          arrow=(*state).enteredspatmap and ~(*state).addmode

      ;; Updating widgets:
      sensitive=(*state).mfid gt min((*state).pos.id)?1L:0L
      widget_control,(*state).wid.bmom,sensitive=sensitive
      widget_control,(*state).wid.bmrm,sensitive=sensitive
      sensitive=(*state).mfid lt max((*state).pos.id)?1L:0L
      widget_control,(*state).wid.bmop,sensitive=sensitive
      widget_control,(*state).wid.bmrp,sensitive=sensitive
      widget_control,(*state).wid.tmod,set_value=strtrim((*state).mfid,2L)
      widget_control,(*state).wid.tmor,set_value=strtrim((*state).mrow,2L)

    end ;; case: 'spatmapdecrementincrement'

    'spatmapselectrow': begin
      widget_control,event.id,get_value=val
      
      defvalue=(*state).mrow
      defstr='The entered value must be of integer type.'
      on_ioerror,format_error
      tmp=long(val[0L])
      on_ioerror,NULL

      idx=where((*state).pos.row eq tmp,count)
      if ~count then begin
        ;; Resetting the text widget:
        widget_control,(*state).wid.tmor,set_value=strtrim((*state).mrow,2L)

        ;; Updating the status widget:
        tmpstr='There is no spatial element in the spectrum image at row '+ $
               strtrim(tmp,2L)+'.'
        widget_control,(*state).wid.lsta,set_value=tmpstr
      endif else if tmp ne (*state).mrow then begin

        oldelement=(*state).mfid
        olssingleelement=*(*state).impos
        (*state).mfid=(*state).pos.id[idx]
        (*state).mrow=tmp

        if ptr_valid((*state).impos) then ptr_free,(*state).impos
        (*state).impos=ptr_new((*state).mrow)

        p3d_rss_plot,state,/imagebar,/imagesmap,/specplot, $
            imagesingleelement=oldsingleelement,oldelement=oldelement, $
            arrow=(*state).enteredspatmap

        ;; Updating widgets:
        widget_control,(*state).wid.tmod,set_value=strtrim((*state).mfid,2L)
        sensitive=(*state).mfid gt min((*state).pos.id)?1L:0L
        widget_control,(*state).wid.bmom,sensitive=sensitive
        widget_control,(*state).wid.bmrm,sensitive=sensitive
        sensitive=(*state).mfid lt max((*state).pos.id)?1L:0L
        widget_control,(*state).wid.bmop,sensitive=sensitive
        widget_control,(*state).wid.bmrp,sensitive=sensitive

      endif
    end ;; case: 'spatmapselectrow'

    ;;================================================
    ;; Changing the orientation in the spatial map image:

    'orientation': begin

      (*state).orientation=index
      p3d_rss_plot,state,/imagesmap

    end ;; case: 'orientation'

    ;;================================================
    ;; Spectrum image; color range droplist:

    'imagecolordroplist': begin

      ;; Determining if the event came from the droplist or the menu buttons:
      if tmpstrname eq 'widget_button' then begin
        val=strtrim(index,2L)+'%'
        case index of
          100: choice=1L
          99.5: choice=2L
          99: choice=3L
          98: choice=4L
          97: choice=5L
          95: choice=6L
          90: choice=7L
          80: choice=8L
          70: choice=9L
        endcase

        ;; Updating the droplist widget:
        widget_control,(*state).wid.dcom,set_droplist_select=choice
      endif else begin
        ;; Extracting the percentage (droplist):
        widget_control,event.id,get_value=val
        val=val[event.index]
        choice=event.index
      endelse

      case choice of
        0: begin
          ;; Values are not changed:
          (*state).cmanual=1L
        end
        1: begin
          minval=min((*state).image,max=maxval)
          (*state).cmanual=0L
        end
        else: begin
          ;; Calculating the clipping range (histogram):
          p3d_misc_dataclip,(*state).image,range,percentage=val, $
              topwid=event.top,verbose=(*state).verbose,error=error, $
              debug=(*state).debug
          if error ne 0 then return

          minval=range[0L] & maxval=range[1L]
          (*state).cmanual=0L
        end
      endcase

      (*state).colmin =minval
      (*state).colmax =maxval
      (*state).mcolmin=minval
      (*state).mcolmax=maxval

      ;; Drawing the image anew:
      p3d_rss_plot,state,/imageplot,/imagebar,/imagesmap

      ;; Updating the color value widgets:
      widget_control,(*state).wid.tcoi,set_value=strtrim((*state).colmin,2L)
      widget_control,(*state).wid.tcoa,set_value=strtrim((*state).colmax,2L)

    end ;; case: 'imagecolordroplist'

    'imagecolorminmax': begin
      widget_control,event.id,get_value=val
      
      defvalue=index?(*state).colmax:(*state).colmin
      defstr='The entered value must be of decimal type.'
      on_ioerror,format_error
      tmp=float(val)
      on_ioerror,NULL

      if tmp ne (*state).colmin and tmp ne (*state).colmax then begin
        case index of
          0: begin
            (*state).colmin=tmp
            (*state).mcolmin=tmp
          end
          1: begin
            (*state).colmax=tmp
            (*state).mcolmax=tmp
          end
        endcase

        p3d_rss_plot,state,/imageplot,/imagebar,/imagesmap

        (*state).cmanual=1L ;; Making manual color value settings

        ;; Updating widgets:
        widget_control,(*state).wid.dcom,set_droplist_select=0L
      endif
    end ;; case: 'imagecolorminmax'

    ;;================================================
    ;; Show/hide error bars button:

    'showerrors': begin
      (*state).showerrors=~(*state).showerrors

      tmp=(*state).showerrors?'Hide errors':'Show errors'
      widget_control,(*state).wid.bser,set_value=tmp

      p3d_rss_plot,state,/specplot
    end ;; case: 'showerrors'

    ;;================================================
    ;; Sky subtraction, using the selected spectra:

    'skysubtract': begin

      if (*state).subtracted then begin
        ;; Restoring the un-subtracted spectrum image:
        if ptr_valid((*state).rimage) then begin
          (*state).image=*(*state).rimage & ptr_free,(*state).rimage
          if (*state).serr then $
             (*state).dimage=*(*state).rdimage & ptr_free,(*state).rdimage
        endif else begin
          errmsg='Cannot restore the spectrum image, a backup was never' + $
            ' stored.'
          widget_control,(*state).wid.lsta,set_value=errmsg
        endelse
        
        ;; Recalculating the 'added' spectrum:
        if ptr_valid((*state).addspec) then begin
          tmp=where((*state).pos.marked,count)
          if count ne 0L then begin
            tmp=(*state).image[*,tmp]
            *(*state).addspec=total(tmp,2L,/double)

            if (*state).serr then begin
              tmp=(*state).dimage[*,tmp]
              *(*state).addspee=total(tmp^2,2L,/double)
            endif
          endif ;; count ne 0L
        endif ;; ptr_valid((*state).addspec)

        uname='Click to subtract the currently marked spectrum from the sp' + $
          'ectral image.'
        value='Subtract'
        malue='Subtract the current spectrum from the image.'
        (*state).subtracted=0L
      endif else begin

        ;; Subtracting the current spectrum from the image:
        if (*state).addmode then begin
          tmp=where((*state).pos.marked,count)
          subspec=*(*state).addspec/count
          if (*state).serr then dsubspec=*(*state).addspee/count
        endif else begin
          subspec=*(*state).curspec
          if (*state).serr then dsubspec=(*(*state).curspee)^2
        endelse
        rubspec=rebin(subspec,(*state).nbins,(*state).nspec)

        if (*state).serr then $
           drubspec=rebin(dsubspec,(*state).nbins,(*state).nspec)

        (*state).rimage=ptr_new((*state).image) ;; storing the image
        (*state).image-=rubspec

        if (*state).serr then begin
          (*state).rdimage=ptr_new((*state).dimage) ;; storing the image
          (*state).dimage=sqrt((*state).dimage^2+drubspec)
        endif

        ;; Recalculating the 'added' spectrum:
        if ptr_valid((*state).addspec) then begin
          tmp=where((*state).pos.marked,count)
          if count ne 0L then begin
            tmp=(*state).image[*,tmp]
            *(*state).addspec=total(tmp,2L,/double)

            if (*state).serr then begin
              tmp=(*state).dimage[*,tmp]
              *(*state).addspee=total(tmp^2,2L,/double)
            endif
          endif ;; count ne 0L
        endif ;; ptr_valid((*state).addspec)

        uname='Click to restore the spectrum image, before subtraction.'
        value='Restore '
        malue='Restore the entry spectrum image'

        (*state).subtracted=1L
      endelse 

      ;; Setting the colors:
      choice=widget_info((*state).wid.dcom,/droplist_select)
      case choice of
        0: cmanual=1L                ;; Values are not changed:
        1: begin
          minval=min((*state).image,max=maxval)
          cmanual=0L
        end
        else: begin
          ;; Calculating the clipping range (histogram):
          widget_control,(*state).wid.dcom,get_value=val
          val=val[choice]
          p3d_misc_dataclip,(*state).image,range,percentage=val, $
              topwid=event.top,verbose=(*state).verbose,error=error, $
              debug=(*state).debug
          if error ne 0 then return

          minval=range[0L] & maxval=range[1L]
          cmanual=0L
        end
      endcase

      if ~cmanual then begin
        (*state).colmin =minval
        (*state).colmax =maxval
        (*state).mcolmin=minval
        (*state).mcolmax=maxval

        ;; Updating the color value widgets:
        widget_control,(*state).wid.tcoi,set_value=strtrim((*state).colmin,2L)
        widget_control,(*state).wid.tcoa,set_value=strtrim((*state).colmax,2L)
      endif

      ;; Drawing the image anew:
      p3d_rss_plot,state,/imageplot,/imagebar,/imagesmap,/specplot

      ;; Updating the spectrum y-range widgets:
      widget_control,(*state).wid.tpyl,set_value=strtrim((*state).specymin,2L)
      widget_control,(*state).wid.tpyh,set_value=strtrim((*state).specymax,2L)

      ;; Unsensitizing the sky subtraction buttons:
      if widget_info((*state).wid.bsks,/valid_id) then $
        widget_control,(*state).wid.bsks,set_uname=uname,set_value=value
      widget_control,(*state).wid.bska,set_uname=uname,set_value=malue


    end ;; case: 'skysubtract:

    ;;================================================
    ;; Save the marked spatial maps:

    'savespatialmaps': begin
      idx=where((*state).mwaves[0L,*] ne 0L,count)
      if count ne 0L then begin

        ;; Only saving if some wavelengths have been stored on the
        ;; spatial maps row:
        filename=dialog_pickfile(path=(*state).datapath,/write, $
            file='p3d_spatial_maps.fits',filter='*.fits', $
            /overwrite_prompt,dialog_parent=event.top,get_path=path, $
            title='p3d: save spatial maps')
        if filename ne '' then begin
          (*state).datapath=path

          data=(*state).msave[*,idx]

          ;; Creating a new fits-file:
          fxhmake,tmphdr,data
          for i=1L,count do begin
            tmp=(*state).mwaves[0L,idx[i-1L]]
            fxaddpar,tmphdr,'WAVE'+strtrim(i,2L)+'_0',tmp
            tmp=(*state).mwaves[1L,idx[i-1L]]
            fxaddpar,tmphdr,'WAVE'+strtrim(i,2L)+'_1',tmp
          endfor
          fxwrite,filename,tmphdr,data
        endif ;; filename ne ''

      endif else begin
        tmpstr='No spatial map has been "saved", which is why no d' + $
               'ata can be saved.'
        widget_control,(*state).wid.lsta,set_value=tmpstr
      endelse
    end ;; case: 'savespatialmaps'

    ;;================================================
    ;; Save the currently shown spectrum:

    'savespectrum': begin
      filename=dialog_pickfile(path=(*state).datapath,/write, $
          file='p3d_spectrum.fits',filter='*.fits',/overwrite_prompt, $
          dialog_parent=event.top,get_path=path, $
          title='p3d: save current spectrum')
      if filename ne '' then begin
        (*state).datapath=path

        if (*state).serr then begin
          tmp=strpos(filename,'.',/reverse_search)
          if tmp eq -1L then begin
            filenamee=filename+'_err'
          endif else begin
            fsfx=strmid(filename,tmp)
            filenamee=strmid(filename,0L,tmp)+'_err'+fsfx
          endelse
        endif ;; (*state).serr

        fxhmake,tmphdr,*(*state).spec

        if n_elements(*(*state).hdr) eq 1L and $
           (*(*state).hdr)[0L] eq '' then begin
          inputhdr=*(*state).hdr
        endif else begin
          inputhdr=*(*state).hdr
          fxaddpar,inputhdr,'BITPIX',fxpar(tmphdr,'BITPIX')
          fxaddpar,hdr,'BZERO',0
        endelse

        ;; Setting various header keywords:
        fxaddpar,inputhdr,'NAXIS',1L
        fxaddpar,inputhdr,'NAXIS1',n_elements(*(*state).spec)
        for k=2L,3L do begin
          if fxpar(inputhdr,'NAXIS'+strtrim(k,2L)) ne 0L then begin
            fxaddpar,inputhdr,'NAXIS'+strtrim(k,2L),1L
          endif
        endfor
        fxaddpar,inputhdr,'CRVAL1',(*state).wav0bn
        fxaddpar,inputhdr,'CDELT1',(*state).wavdsp
        fxaddpar,inputhdr,'CRPIX1',1L
        fxaddpar,inputhdr,'CTYPE1','AWAV'

        ;; Adding some information on how many spatial elements were
        ;; used to create the shown spectrum:
        comment=' The spectrum was divided by this number.'
        fxaddpar,inputhdr,'NCOMBINE',long((*state).meannumspec),comment

        ;; Saving the header in the state-structure:
        if ptr_valid((*state).hdr) then ptr_free,(*state).hdr
        (*state).hdr=ptr_new(inputhdr)

        ;; Writing the data:
        fxwrite,filename,inputhdr,*(*state).spec
        if (*state).serr then begin
	  spee=*(*state).spee
	  if (*state).addmode then spee=sqrt(spee)
	  fxwrite,filenamee,inputhdr,spee
	endif

        ;; Writing extended data with information on which spaxel(s) is/are
        ;; saved:

        if (*state).addmode then begin
          mask=(*state).pos.marked
        endif else begin
          ;; Only a single spatial element is saved:
          mask=lonarr((*state).nspec)
          idx=where((*state).pos.id eq (*state).mfid)
          mask[idx]=1L
        endelse

        fxhmake,tmphdr,mask
        fxaddpar,tmphdr,'XTENSION','IMAGE'
        writefits,filename,mask,tmphdr,/append

        fxhmake,tmphdr,*(*state).marr
        fxaddpar,tmphdr,'XTENSION','IMAGE'

        lower=(*state).marrbin[0L]
        upper=(*state).marrbin[1L]
        lbin=(*state).wavarr[lower]- $
             0.5d0*((*state).wavarr[lower+1L]-(*state).wavarr[lower])
        ubin=(*state).wavarr[upper]+ $
             0.5d0*((*state).wavarr[upper]-(*state).wavarr[upper-1L])

        fxaddpar,tmphdr,'WAVE_0',lbin
        fxaddpar,tmphdr,'WAVE_1',ubin
        writefits,filename,*(*state).marr,tmphdr,/append

        ;; Updating the status widget:
        tmpstr='Wrote the currently shown spectrum to "'+filename+'".'
        widget_control,(*state).wid.lsta,set_value=tmpstr

      endif ;; filename ne ''

    end ;; case: 'savespectrum'

    'colortable': begin
      xloadct,bottom=(*state).cbottom,group=event.top, $
          ncolors=!d.table_size-(*state).cbottom,updatecbdata=state, $
          /use_current,updatecallback='p3d_rss_xloadct'

       ;; Only allowing one instance of xloadct to be run:
       widget_control,(*state).wid.bcol,sensitive=0L
    end ;; case: 'colortable'

    'colortablesel': begin

      p3d_misc_colortable,long(index),bottom=(*state).cbottom,colors=colors, $
          topwid=event.top,error=error,verbose=(*state).verbose, $
          debug=(*state).debug
      if error ne 0 then return

      (*state).colortable=long(index)
      (*state).colors=colors

      p3d_rss_plot,state,/imageplot,/imagesmap,/specplot,topwid=event.top
      
    end ;; case: 'colortable'

    'viewfitsheader': begin
      xdisplayfile,(*state).fitsheadertitle,done_button='Exit',/grow, $
          text=*(*state).hdr,group=event.top, $
          title=p3d_misc_pathify((*state).fitsheadertitle,/dpath)
    end ;; case: 'viewfitsheader'

    ;;================================================
    ;; Character size:

    'charsize': begin
      widget_control,event.id,get_value=val
      val=val[event.index]
      (*state).charsize=float(val)

      p3d_rss_plot,state,/imageplot,/imagesmap,/specplot,topwid=event.top
    end ;; case: 'charsize'

    ;;================================================
    ;; Exit:

    'exit': begin
      widget_control,event.top,/destroy
      return
    end ;; case: 'exit'

    else: begin
      mark=strmid(uval,0L,16L) & slen=strlen(uval)
      case mark of
        ;; The row of spatial maps:
        'spatmapimagesave': begin
          ;; Finding out which widget generated the event:
          imageindex=long(strmid(uval,16L,slen-16L))

          ;; Only accept a pressed left mouse button in case it is the
          ;; draw widget that generated the event:
          if tmpstrname eq 'widget_draw' then $
             if event.press ne 1b then return

          p3d_rss_plot,state,xvar=(*state).mcol,imagerowspatmap=imageindex

;         I have not implemented the following as it seems pointless
;         to store the color values:
;         mcols[imageindex,*]=[colmin,colmax]

          ;; Indicating the wavelength range on the widget button:
          printwaves=strtrim(round((*state).mwaves[0L,imageindex]),2L)
          if round((*state).mwaves[1L,imageindex]) ne $
             round((*state).mwaves[0L,imageindex]) then $
             printwaves+='-'+strtrim(round((*state).mwaves[1L,imageindex]),2L)

          if widget_info((*state).wid.bsad[imageindex],/valid_id) then begin
            widget_control,(*state).wid.bsad[imageindex],set_value=printwaves
          endif else begin
            ;; If the 

            tmp_=' ['+(*state).wavunit+'].'
            uname=widget_info((*state).wid.dsad[imageindex],/uname)
            idx=strpos(uname,';',/reverse_search)
            uname=strmid(uname,0L,idx)+'; the wavelength range used is: '+ $
                  printwaves+tmp_

            widget_control,(*state).wid.dsad[imageindex],set_uname=uname
          endelse

        end ;; case: 'spatmapimagesave'
      endcase ;; 'mark'
    end ;; else

  endcase ;; uval

  return

format_error:
  widget_control,event.id,set_value=strtrim(defvalue,2L)
  widget_control,(*state).wid.lsta,set_value=defstr
  on_ioerror,NULL

  return
END ;;; procedure: p3d_rss_event


PRO p3d_rss_cleanup,id
  compile_opt hidden,IDL2

  ;; Get the state information:
  widget_control,id,get_uvalue=state

  if (*state).verbose ge 3 then print,'p3d_rss: cleanup.'

  if ptr_valid((*state).marr)    then ptr_free,(*state).marr
  if ptr_valid((*state).dmarr)   then ptr_free,(*state).dmarr
  if ptr_valid((*state).spec)    then ptr_free,(*state).spec
  if ptr_valid((*state).spee)    then ptr_free,(*state).spee
  if ptr_valid((*state).curspec) then ptr_free,(*state).curspec
  if ptr_valid((*state).curspee) then ptr_free,(*state).curspee
  if ptr_valid((*state).addspec) then ptr_free,(*state).addspec
  if ptr_valid((*state).addspee) then ptr_free,(*state).addspee
  if ptr_valid((*state).rimage)  then ptr_free,(*state).rimage
  if ptr_valid((*state).rdimage) then ptr_free,(*state).rdimage
  if ptr_valid((*state).impos)   then ptr_free,(*state).impos
  if ptr_valid((*state).hdr)     then ptr_free,(*state).hdr
;  if obj_valid((*state).icolowid) then obj_destroy,(*state).icolowid
;  if obj_valid((*state).ocolbobj) then obj_destroy,(*state).ocolbobj
  if ptr_valid(state)            then ptr_free,  state

  return
END ;;; procedure: p3d_rss_cleanup


PRO p3d_rss,image__,posfile,dimage=dimage__,hdr=hdr__,colorcut=colorcut__, $
        datapath=datapath,specrange=specrange__,orientation=orientation, $
        colortable=colortable,bottom=bottom,cindex=cindex,cindv=cindv, $
        title=title,fitsheadertitle=fitsheadertitle,tracking_events=track, $
        daxis=daxis,group_leader=group_leader, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_rss: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)
  if ~n_elements(group_leader) then group_leader=0L
  topwid=widget_info(group_leader,/valid_id)?group_leader:0L

  if keyword_set(help) then begin
    doc_library,'p3d_rss'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if ~n_elements(daxis) then daxis=1L
  sd=size(daxis)
  if sd[sd[0L]+2L] ne 1L or $
    (sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif

  s=size(image__)
  if s[0L] ne 2L or (s[s[0l]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='IMAGE must be set; to a two-dimensional array of floating poin' + $
           't type.'
    goto,error_handler
  endif
  image=float(image__)
  if daxis eq 2L then image=transpose(image)
  s=size(image)

  serr=0L & dimage=0L
  se=size(dimage__)
  if se[se[0L]+2L] ne 0L then begin
    if se[0L] ne 2L or $
      (se[se[0l]+1L] ge 6L and se[se[0L]+1L] le 11L) then begin
      errmsg='DIMAGE must, if set, be a two-dimensional array of decimal type.'
      goto,error_handler
    endif
    dimage=float(dimage__)
    if daxis eq 2L then dimage=transpose(dimage)
    se=size(dimage)
    if se[1L] ne s[1L] or se[se[0L]+2L] ne s[s[0L]+2L] then begin
      errmsg='DIMAGE must, if set, have the same number of elements as IMAGE.'
      goto,error_handler
    endif
    serr=1L
  endif ;; se[se[0L]+2L] ne 0L

  if ~n_elements(track) then track=1L
  sb=size(track)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0l]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='TRACK must be set to an integer scalar; 0||1.'
    goto,error_handler
  endif
  if track ne 0L and track ne 1L then begin
    errmsg='TRACK must be set to an integer scalar; 0||1, not "'+ $
           strtrim(track,2L)+'".'
    goto,error_handler
  endif

  if ~n_elements(orientation) then orientation=1L
  sb=size(orientation)
  if sb[sb[0L]+2L] ne 1L or $
    (sb[sb[0l]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
    errmsg='ORIENTATION must be set to an integer scalar; 0<=value<=3.'
    goto,error_handler
  endif
  if orientation lt 0L or orientation gt 3L then begin
    errmsg='ORIENTATION must be set to an integer scalar; 0<=value<=3, not' + $
           ' "'+strtrim(orientation,2L)+'".'
    goto,error_handler
  endif

  if ~keyword_set(spnull) then firstspec=0

  if ~n_elements(datapath) then cd,cur=datapath
  if ~file_test(datapath,/directory) then begin
    errmsg='DATAPATH, if specified, must point to an existing directory.'
    goto,error_handler
  endif

  tmp='p3d Spectrum Viewer'
  title=~n_elements(title) or size(title,/type) ne 7L?tmp:tmp+': '+title

  if ~n_elements(fitsheadertitle) then fitsheadertitle='Fits Header'
  sb=size(fitsheadertitle)
  if sb[sb[0L]+2L] ne 1L or sb[sb[0l]+1L] ne 7L then begin
    errmsg='FITSHEADERTITLE must, if set, be a scalar string.'
    goto,error_handler
  endif

  ;; Creating a shortened title for the widget base container, if possible:
  title=p3d_misc_pathify(title,/dpath)

  ;;========================================------------------------------
  ;; Setting up colortable-related variables:

  if ~n_elements(cindv) or ~n_elements(bottom) or $
     ~n_elements(cindex) then begin
    ;; Colors are only re-defined if the necessary arrays are not defined:
    p3d_misc_colortable,colortable,bottom=bottom,index=cindex,indv=cindv, $
        colors=colors,/define,verbose=verbose,error=error,debug=debug
    if error ne 0 then return
  endif else begin
    ;; Reading the colors:
    tvlct,red,green,blue,/get
    colors={red:red,green:green,blue:blue}
  endelse

  ;;========================================------------------------------
  ;; Loading the fiber position table:

  p3d_misc_read_postable,postable=posfile,rownum=rownum,id=id, $
      xpos=xpos,ypos=ypos,lens_size=lens_size,shape=shape, $
      nvertices=nvertices,/science,topwid=topwid,logunit=logunit, $
      verbose=verbose,debug=debug,error=error
  if error ne 0 then return

  ;; Extra settings for circular elements:
  if shape then begin
    if ~n_elements(nvertices) then nvertices=20L
    sb=size(nvertices)
    if sb[sb[0L]+2L] ne 1L or $
      (sb[sb[0l]+1L] ge 4L and sb[sb[0L]+1L] le 11L) then begin
      errmsg='NVERTICES must be set to a scalar integer.'
      goto,error_handler
    endif
  endif

  xposmin=min(xpos,max=xposmax)
  yposmin=min(ypos,max=yposmax)

  if shape then begin
    minoffset=lens_size[0L]*0.5 & maxoffset=minoffset
  endif else begin
    minoffset=0.0 & maxoffset=1.0
  endelse

  xrange=[xposmin-minoffset,xposmax+maxoffset]
  yrange=[yposmin-minoffset,yposmax+maxoffset]

  ;; Setting up a fiber position-related structure:
  pos={file:posfile,row:rownum,id:id,x:xpos,y:ypos,lsize:lens_size, $
       shape:shape,nvertices:nvertices,xrange:xrange,yrange:yrange, $
       marked:bytarr(n_elements(xpos))}

  ;;========================================------------------------------
  ;; Checking for the existence of an image header:

  nspec=s[2L] & nbins=s[1L]
  if ~n_elements(hdr__) then begin
    hdr=''
    wav0bn=0d0
    wavdsp=1d0
  endif else begin
    dstr=strtrim(daxis,2L)
    wav0bn=fxpar(hdr__,'CRVAL'+dstr)
    wavdsp=fxpar(hdr__,'CDELT'+dstr)
    if ~wavdsp then wavdsp=1.0
    hdr=hdr__
  endelse

  ;; Setting up the wavelength array:
  wavarr=dindgen(s[1L])*wavdsp+wav0bn
  wavlow=min(wavarr,max=wavhig)

  mcol=s[1L]/2L

  plotlow=0L & plothig=nbins-1L & plotdlt=nbins/25d0
  plotcen=0.5*(plothig+plotlow)

  wavunit=wav0bn le 0d0?'bin':''

  ;;========================================------------------------------
  ;; Setting screen size-dependent values:

  screensize=get_screen_size()
;screensize=[1024,768L] ;; for testing

  monoxrow=1L
  if screensize[1L] le 600L then begin
    mxsize=210L & mysize=210L
    rxsize=80L & rysize=80L
    ymax=180L
    symax=300L
    monoxrow=0L
  endif else if screensize[1L] le 800L then begin
    mxsize=250L & mysize=250L
    rxsize=90L & rysize=90L
    symax=300L
    ymax=250L
  endif else begin
    mxsize=250L & mysize=250L
    rxsize=110L & rysize=110L
    symax=400L
    ymax=400L
  endelse

  ;; Calculating the maximum allowed horizontal size:
  xofwid=10L & xoflo=10L & xofhi=xoflo+s[1L]
  imwidth=s[1L]+2L+2L*xofwid
  xmax=screensize[0L]- $ ;; # of x-pixels
    mxsize- $            ;; the size of the spatial map
    5L- $                ;; spacing of bdispwid
    10L-2L- $            ;; row indicator space
    10L                  ;; saving 10 pixels on the sides

  ;;==============================--------------------
  ;; Selecting the center-most spatial element [mxsize/2,mysize/2]:

  p3d_rss_find_closest_element,mxsize/2d0,mysize/2d0,mxsize,mysize, $
      mfid,mrow,fiberpos=pos,orientation=orientation

  ;; Sampling the current spectrum to plot:
  curspec=image[*,mrow]
  curspee=serr?dimage[*,mrow]:0L
  addspec=dblarr(s[1L])
  addspee=serr?addspec:0L
  spec=curspec
  spee=serr?curspee:0L

  charsize=1.0
  angstr=string("305b) ;; "

  ;;========================================------------------------------
  ;; Setting up a widget hierarchy:

  nmsave=10L
  wid={bbas:0L,bsks:0L,bska:0L,bcol:0L,bbil:0L,bbir:0L,bbwl:0L,bbwr:0L, $
       bprr:0L,bmar:0L,bsad:lonarr(nmsave),bara:lonarr(2),bsim:0L,bsip:0L, $
       bspr:0L,bpld:0L,bplu:0L,bser:0L,bmom:0L,bmop:0L,bsva:lonarr(3L), $
       bmrm:0L,bmrp:0L,bmod:0L,bdco:0L,bdcc:lonarr(4), $
       dcom:0L,dimd:0L,dspm:0L,dsad:lonarr(nmsave),dplo:0L,dspc:0L, $
       lsta:0L,lmnm:0L, $
       tbic:0L,tbib:0L,tmod:0L,tmor:0L,tprl:0L,tprh:0L, $
       tpyl:0L,tpyh:0L,tpsh:0L,tcoi:0L,tcoa:0L,tzom:0L, $
       isma:0L,imdi:0L,iplo:0L,isad:lonarr(nmsave)}

  blfile=filepath(subdir=['resource','bitmaps'],'shift_left.bmp')
  brfile=filepath(subdir=['resource','bitmaps'],'shift_right.bmp')
  bufile=filepath(subdir=['resource','bitmaps'],'shift_up.bmp')
  bdfile=filepath(subdir=['resource','bitmaps'],'shift_down.bmp')

  wid.bbas=widget_base(title=title,/column,mbar=mbbarwid, $
      /base_align_center,group_leader=topwid,space=2L,xpad=0L,ypad=2L, $
      sensitive=0L,tlb_frame_attr=1L,resource_name='p3d')

  ;; Setting up a menu bar:
  mfilewid=widget_button(mbbarwid,value='File',/menu,uname='With the opti' + $
      'ons of this menu you can create new data files and exit this tool.', $
      tracking_events=track)

  ;;===========
  ;; File menu:

  bfitswid=widget_button(mfilewid,value='View the fits header of the data', $
      uvalue='viewfitsheader',uname='Click to view the fits header of the ' + $
      'viewed data.',tracking_events=track)
  bsamiwid=widget_button(mfilewid,/separator,value='Save spatial maps', $
      uvalue='savespatialmaps',uname='Click to save the spatial maps ' + $
      'to a fits-file.',tracking_events=track,accelerator='CTRL+W')
  bsaspwid=widget_button(mfilewid,value='Save the current spectrum', $
      uvalue='savespectrum',uname='Click to save the currently shown spect' + $
      'rum to a fits-file.',tracking_events=track,accelerator='CTRL+S')

  ;; Color tables:
  bcolmwid=widget_button(mfilewid,/menu,/separator,value='Select colortable',$
      uvalue='colortablemenu',uname='Click to select a colortable', $
      tracking_events=track)

  loadct,get_names=names
  wid.bcol=widget_button(bcolmwid,value='xloadct',uvalue='colortable', $
      uname='Click to select a colortable using xloadct', $
      tracking_events=track)

  names=['Sauron',names]
  nnames=n_elements(names)
  nbuttons=nnames/12 & mbuttons=nnames mod 12
  bcolwwid=lonarr(nnames)
  imax=nbuttons+(mbuttons gt 0L?1L:0L)
  iimax=lonarr(imax)+11L
  iimax[imax-1L]=mbuttons-1L

  k=0L
  for i=0L,imax-1L do begin
    bcolqwid=widget_button(bcolmwid,/menu,separator=~i,value='colortables '+ $
        strtrim(i+1L,2L)+'/'+strtrim(imax,2L),uvalue='colortablemenu', $
        uname='Click to choose a colortable in this group', $
        tracking_events=track)

    for j=0L,iimax[i] do begin
      bcolwwid[k]=widget_button(bcolqwid,value=names[k], $
          uvalue='colortablesel_'+strtrim(k-1L,2L),uname='Click to select '+ $
          'the '+names[k]+' colortable',tracking_events=track)
      k++
    endfor
  endfor

  bexitwid=widget_button(mfilewid,/separator,value='Exit',uvalue='exit', $
      uname=' ',tracking_events=track,accelerator='Ctrl+Q')

  ;;===========
  ;; Image view menu:

  mviewwid=widget_button(mbbarwid,value='Image view',/menu, $
      uname='With the options of this menu you can change aspects related ' + $
      'to the spectrum images.',tracking_events=track)

  values=['100%','99.5%','99%','98%','97%','95%','90%','80%','70%']
  bsicowid=lonarr(n_elements(values))
  for i=0L,n_elements(values)-1L do begin
    tmp=strmid(values[i],0L,strpos(values[i],'%'))
    str='Color map: use '+values[i]+' of the pixel values'
    bsicowid[i]=widget_button(mviewwid,value=str, $
        uvalue='imagecolordroplist_'+tmp,uname='Click to use '+values[i]+ $
        ' of the pixel value range when calculating lower and upper limits' + $
        ' of the color map.',tracking_events=track, $
        accelerator='F'+strtrim(i+1L,2L))
  endfor

  wid.bsim=widget_button(mviewwid,value='Shown wavelength bin: next lower', $
      uvalue='imagebinincrementdecrement_0',uname='Press this button to de' + $
      'crement the wavelength bin number by one.', $
      tracking_events=track,accelerator='CTRL+G',/separator)
  wid.bsip=widget_button(mviewwid,value='Shown wavelength bin: next higher', $
      uvalue='imagebinincrementdecrement_1',uname='Press this button to in' + $
      'crement the wavelength bin number by one.', $
      tracking_events=track,accelerator='CTRL+H')

  wid.bmom=widget_button(mviewwid,value='Spatial element ID: next lower', $
      uvalue='spatmapdecrementincrement_0',uname='Click to decrement the ' + $
      'selected element by ID; with 1.',tracking_events=track, $
      sensitive=mfid gt min(pos.id),accelerator='Ctrl+P',/separator)
  wid.bmop=widget_button(mviewwid,value='Spatial element ID: next higher', $
      uvalue='spatmapdecrementincrement_1',uname='Click to increment the ' + $
      'selected element by ID; with 1.',tracking_events=track, $
      sensitive=mfid lt max(pos.id),accelerator='Ctrl+N')

  wid.bmrm=widget_button(mviewwid,value='Spatial element image row: next' + $
      ' lower',uvalue='spatmapdecrementincrement_2',uname='Click to decre' + $
      'ment the selected element image row; with 1.',tracking_events=track, $
      sensitive=mfid gt min(pos.id),accelerator='Ctrl+O')
  wid.bmrp=widget_button(mviewwid,value='Spatial element image row: next' + $
      ' higher',uvalue='spatmapdecrementincrement_3',uname='Click to incr' + $
      'ement the selected element image row; with 1.',tracking_events=track, $
      sensitive=mfid lt max(pos.id),accelerator='Ctrl+B')

  ;;===========
  ;; Spatial maps image menu:

  mviewwid=widget_button(mbbarwid,value='Spatial map',/menu, $
      uname='With the options of this menu you can change aspects related ' + $
      'to the spatial map.',tracking_events=track)

  values=['a single spectrum','a summed spectrum','an averaged spectrum']
  tmp='; requires the wanted elements to be marked.'
  valxtra=['a single spectrum','a summed spectrum'+tmp, $
           'an averaged spectrum'+tmp]
  for i=0L,n_elements(values)-1L do begin
    tstr=~i?'* ':'  '
    str=tstr+'Spatial elements, use '+values[i]
    tmp=strtrim(i,2L)
    wid.bsva[i]=widget_button(mviewwid,value=str, $
        uvalue='addmode_'+tmp,uname='Choose to look at '+valxtra[i], $
        tracking_events=track,accelerator='Ctrl+F'+strtrim(i+1L,2L))
  endfor

  values=['all','none']

  wid.bara[0L]=widget_button(mviewwid,value='Mark all spatial elements', $
      uvalue='spatmapmarkallnone_0',uname='Click to mark all spatial eleme' + $
      'nts.',tracking_events=track,accelerator='Ctrl+T',/separator, $
      sensitive=0L)

  wid.bara[1L]=widget_button(mviewwid,value='Unmark all spatial elements', $
      uvalue='spatmapmarkallnone_1',uname='Click to unmark all spatial elem'+ $
      'ents.',tracking_events=track,accelerator='Ctrl+Y',sensitive=0L)

  wid.bska=widget_button(mviewwid,value='Subtract the current spectrum fro' + $
      'm the image.',uvalue='skysubtract',uname='Click to subtract the cur' + $
      'rently marked spectrum from the spectrum image.',tracking_events=track,$
      sensitive=1L,/separator,accelerator='Ctrl+D')

  ;;===========
  ;; Spectrum view menu:

  mviewwid=widget_button(mbbarwid,value='Spectrum view',/menu, $
      uname='With the options of this menu you can change aspects related ' + $
      'to the spectrum plot.',tracking_events=track)

  values=['100%','99.5%','99%','98%','97%','95%','90%','80%','70%']
  bspcowid=lonarr(n_elements(values))
  for i=0L,n_elements(values)-1L do begin
    tmp=strmid(values[i],0L,strpos(values[i],'%'))
    str='Y-range: use '+values[i]+' of the bins'
    bspcowid[i]=widget_button(mviewwid,value=str, $
        uvalue='spectrumyrangedroplist_'+tmp,uname='Click to use '+values[i]+ $
        ' of the pixel value range when calculating the y-range in the spe' + $
        'ctral plot.',tracking_events=track, $
        accelerator='SHIFT+F'+strtrim(i+1L,2L))
  endfor

  str=strtrim(plotdlt*wavdsp,2L)
  tmp='[Shift down] Press this button to shift the shown wavelength range by '
  unamedown=tmp+'-'+str+' ['+wavunit+'].'
  unameupp =tmp+str+' ['+wavunit+'].'
  bspslwid=widget_button(mviewwid,value='Shift wavelength range - lower', $
      /separator,uvalue='wavesupdown_0',uname=unamedown, $
      tracking_events=track,accelerator='CTRL+C')
  bspsrwid=widget_button(mviewwid,value='Shift wavelength range - higher', $
      uvalue='wavesupdown_1',uname=unameupp,tracking_events=track, $
      accelerator='CTRL+V')
  bspinwid=widget_button(mviewwid,value='Zoom wavelength range - zoom in', $
      uvalue='wavezoominout_0',uname='[Zoom in] Click to make the waveleng' + $
      'th range smaller.',tracking_events=track,accelerator='CTRL+Z')
  bspouwid=widget_button(mviewwid,value='Zoom wavelength range - zoom out', $
      uvalue='wavezoominout_1',uname='[Zoom out] Click to make the wavelen' + $
      'gth range larger.',tracking_events=track,accelerator='CTRL+X')

  wid.bspr=widget_button(mviewwid,value='Reset wavelength range', $
      uvalue='wavereset',uname='Press this button to use the minimu' + $
      'm-maximum wavelength range that is available in the data.', $
      tracking_events=track,sensitive=0L,accelerator='CTRL+A')

  ;;=============
  ;; Display row:

  bdispwid=widget_base(wid.bbas,/row,/base_align_center, $
      space=5L,xpad=0L,ypad=0L)

  bdisawid=widget_base(bdispwid,/column,/base_align_center, $
      /align_top,space=2L,xpad=0L,ypad=0L)

  bbtmpwid=widget_base(bdisawid,/row,/base_align_top, $
               space=0L,xpad=0L,ypad=0L)
  dimdxwid=0L
  if imwidth gt xmax or s[2L] gt ymax then begin 
    tmptmwid=widget_base(bbtmpwid,space=0L,xpad=0L,ypad=0L) ;; req. for y-off.
;   dimdxwid=widget_draw(tmptmwid,xsize=hlpwid,ysize=s[2L],retain=2L, $
;       uname='This widget indicates which image row(s) is/are currently s' + $
;       'elected.',tracking_events=track,yoffset=2L)

    wid.dimd=widget_draw(bbtmpwid,xsize=imwidth,ysize=s[2L],/scroll, $
        uvalue='spectrumimage',/button_events,/motion_events, $
        /keyboard_events,ignore_accelerators=0, $
        retain=2L,x_scroll_size=imwidth<xmax, y_scroll_size=s[2L]<ymax, $
        uname='This image shows '+strtrim(s[2L],2L)+' spectra that are st' + $
        'acked in the vertical direction. Click in the image to show the s' + $
        'patial image of another wavelength.',tracking_events=track)
  endif else begin
    wid.dimd=widget_draw(bbtmpwid,xsize=imwidth,ysize=s[2L], $
        uvalue='spectrumimage',/button_events,retain=2L, $
        /motion_events,/keyboard_events,ignore_accelerators=0, $
        uname='This image shows '+strtrim(s[2L],2L)+' spectra that are st' + $
        'acked in the vertical direction. Click in the image to show the s' + $
        'patial image of another wavelength.',tracking_events=track)
;    dimdxwid=widget_draw(bbtmpwid,xsize=hlpwid,ysize=s[2L],retain=2L, $
;       uname='This widget indicates which image row(s) is/are currently s' + $
;       'elected.',tracking_events=track)
  endelse

  bdipawid=widget_base(bdisawid,/row,/base_align_center, $
                       space=30L,xpad=0L,ypad=0L)

  ;; First group of widgets:
  bcommwid=widget_base(bdipawid,/row,/base_align_center, $
                       space=0L,xpad=0L,ypad=0L)
  values=['user','100%','99.5%','99%','98%','97%','95%','90%','80%','70%']
  wid.dcom=widget_droplist(bcommwid,value=values,uvalue='imagecolordroplist', $
      uname='Select the range of values which are used when determining t' + $
      'he minimum and maximum values to plot.',tracking_events=track)
  wid.tcoi=widget_text(bcommwid,value=' ',/editable,xsize=7L, $
      uvalue='imagecolorminmax_0',uname='This is the minimum pixel value t' + $
      'hat is used in the scaling of the color map of the spectrum image.', $
      tracking_events=track,resource_name='field',/kbrd_focus_events)
  wid.tcoa=widget_text(bcommwid,value=' ',/editable,xsize=7L, $
      uvalue='imagecolorminmax_1',uname='This is the maximum pixel value t' + $
      'hat is used in the scaling of the color map of the spectrum image.', $
      tracking_events=track,resource_name='field',/kbrd_focus_events)

  ;; Second group of widgets:
  bcobawid=widget_base(bdipawid,/row,/base_align_center, $
                       space=0L,xpad=0L,ypad=0L)

; lcol1wid=widget_label(bcobawid,value='Select bin:',/align_left)
  wid.bbil=widget_button(bcobawid,value=blfile,tracking_events=track, $
      uvalue='imagebinincrementdecrement_0',uname='Press this button to de' + $
      'crement the wavelength bin number by one.', $
      /bitmap,resource_name='field')
  wid.tbic=widget_text(bcobawid,value=strtrim(s[1L]/2L,2L),xsize=6, $
      /editable,uvalue='imagebinnumber',uname='This value indicates which ' + $
      '(central) wavelength bin is shown in the spatial map to the' + $
      ' right.',tracking_events=track,resource_name='field',/kbrd_focus_events)
  wid.bbir=widget_button(bcobawid,value=brfile,tracking_events=track, $
      uvalue='imagebinincrementdecrement_1',uname='Press this button to in' + $
      'crement the wavelength bin number by one.', $
      /bitmap,resource_name='field')

  ;; Third group of widgets:
  bbandwid=widget_base(bdipawid,/row,/base_align_center, $
      space=0L,xpad=0L,ypad=0L)

  lbal1wid=widget_label(bbandwid,value='Bandwidth:')
  wid.bbwl=widget_button(bbandwid,value=blfile,tracking_events=track, $
      uvalue='spatialmapbandwidthchange_0',sensitive=0L, $
      uname='Press this button to decrement the bandwidth number by one.', $
      /bitmap,resource_name='field')

  wid.tbib=widget_text(bbandwid,value='0',uvalue='bandwidth',/editable, $
      xsize=3,uname='This value indicates the bin half-width when showing ' + $
      'the spatial map to the right.',/kbrd_focus_events, $
      tracking_events=track,resource_name='field')
  wid.bbwr=widget_button(bbandwid,value=brfile,tracking_events=track, $
      uvalue='spatialmapbandwidthchange_1', $
      uname='Press this button to increment the bandwidth number by one.', $
      /bitmap,resource_name='field')

;; Colorbar test:
;  dcolowid=widget_draw(bbandwid,graphics_level=2L,xsize=400L,ysize=25L,retain=0L)

  ;;=====================
  ;; Spatial map:

  bspmawid=widget_base(bdispwid,/column,/base_align_center, $
      /align_top,space=0L,xpad=0L,ypad=0L)

  wid.dspm=widget_draw(bspmawid,xsize=mxsize,ysize=mysize,retain=2L, $
      /button_events,/motion_events,/keyboard_events,uvalue='spatmap', $
      uname='This spatial map shows a spatial image of the data ' + $
      'for the selected wavelength (span); right-click to change the orien' + $
      'tation.',tracking_events=track,ignore_accelerators=0)

  ;; Setting up a context menu for this draw widget:
  wid.bdco=widget_base(wid.dspm,/context_menu)
  wid.bdcc[0L]=widget_button(wid.bdco,value='Set: North [right], East [up]', $
      uvalue='orientation_0',uname='Click this button to set the orientati' + $
      'on: Right-North, Up-East',tracking_events=track)
  wid.bdcc[1L]=widget_button(wid.bdco,value='Set: North [up]   , East [left]',$
      uvalue='orientation_1',uname='Click this button to set the orientati' + $
      'on: Up-North, Left-East',tracking_events=track)
  wid.bdcc[2L]=widget_button(wid.bdco,value='Set: North [left] , East [down]',$
      uvalue='orientation_2',uname='Click this button to set the orientati' + $
      'on: Left-North, Down-East',tracking_events=track)
  wid.bdcc[3L]=widget_button(wid.bdco,value='Set: North [down] , East [rig' + $
      'ht]',uvalue='orientation_3',uname='Click this button to set the ori' + $
      'entation: Down-North, Right-East',tracking_events=track)


  bmoidwid=widget_base(bspmawid,/row,/base_align_center, $
      space=0L,xpad=0L,ypad=0L)

  if monoxrow then begin
    values=['Single','Sum','Average']
    wid.bmod=widget_droplist(bmoidwid,value=values,uvalue='addmode', $
        uname='Choose which spectrum to look at: a single, a summed, or an' + $
        ' averaged spectrum; the latter two requires marking the wanted el' + $
        'ements.',tracking_events=track)

    values=['all','none']
    wid.bmar=widget_droplist(bmoidwid,value=values,tracking_events=track, $
        uvalue='spatmapmarkallnone',uname='Choose to mark all or none of t' + $
        'he spatial elements',sensitive=0L)

    bsky=widget_button(bmoidwid,value='Subtract',uvalue='skysubtract', $
        uname='Click to subtract the currently marked spectrum from the sp' + $
        'ectral image.',tracking_events=track,sensitive=1L)
  endif else begin
    ;; The skipped functions must be added as accelerators:

    wid.bmod=0L
    wid.bmar=0L
    wid.bsks=0L
  endelse


  bbselwid=widget_base(bspmawid,/row,space=5L,xpad=0L,ypad=0L, $
      /base_align_center)

  wid.tmod=widget_text(bbselwid,value=strtrim(mfid,2L),xsize=5,/editable, $
      uvalue='spatialmapselectid',uname='This integer indicates the id of ' + $
      'the currently selected spatial element; click to edit.', $
      tracking_events=track,resource_name='field',/kbrd_focus_events)
  wid.tmor=widget_text(bbselwid,value=strtrim(mrow,2L), $
      uvalue='spatmapselectrow',tracking_events=track,resource_name='field', $
      xsize=5,/editable,uname='This integer indicates the row in the image' + $
      ' of the currently selected spatial element; click to edit.', $
      /kbrd_focus_events)
  wid.lmnm=widget_label(bbselwid,value='    0',uvalue='mononum',sensitive=0L, $
      xsize=5L*!d.x_ch_size,uname='This integer indicates the number of sp' + $
      'atial elements that have been summed up to create the shown spectrum.',$
      tracking_events=track,/sunken_frame)

  ;;==========================
  ;; Spatial maps row:

  bmosawid=widget_base(wid.bbas,/row,/base_align_center, $
      space=5L,xpad=0L,ypad=0L)

  msave=fltarr(nspec,nmsave)
  mwaves=fltarr(2,nmsave)

  tmpstr=monoxrow?'.':'; no image-slice has been saved yet.'
  for i=0L,nmsave-1L do begin
    bsavewid=widget_base(bmosawid,/column,/base_align_center, $
        space=0L,xpad=0L,ypad=0L)
    wid.dsad[i]=widget_draw(bsavewid,xsize=rxsize,ysize=rysize,retain=2L, $
        uname='This is spatial map number '+strtrim(i+1L,2L)+'/'+ $
        strtrim(nmsave,2L)+tmpstr,tracking_events=track, $
        uvalue='spatmapimagesave'+strtrim(i,2L),button_events=~monoxrow)
    if monoxrow then $
       wid.bsad[i]=widget_button(bsavewid,value=' ',xsize=rxsize,ysize=20, $
         uvalue='spatmapimagesave'+strtrim(i,2L),resource_name='field', $
	 uname='Click to save the current spatial map in this field; the w' + $
         'avelength range will be indicated.',tracking_events=track)
  endfor

  ;;=============
  ;; Status line:

  wid.lsta=widget_label(wid.bbas,/sunken_frame,value='Welcome to the p3d s' + $
      'pectral viewer.',uname='This line shows information on what is goin' + $
      'g on in this tool.',tracking_events=track,resource_name='field')

  ;;==============
  ;; Spectrum plot:

  ;; The x-size should be defined by the user:
  wid.dplo=widget_draw(wid.bbas,xsize=(s[1L]<xmax)+mxsize,ysize=symax, $
      /motion_events,/keyboard_events,ignore_accelerators=0,retain=2L, $
      uvalue='spectrumplot',uname='This area shows the single, or summed, ' + $
      'spectrum that is selected in the spatial map.', $
      tracking_events=track)

  ;; widget row:
  bplrawid=widget_base(wid.bbas,/row,/base_align_center, $
      space=15L,xpad=0L,ypad=0L)

  values=string([0.8,0.9,1.0,1.1,1.2,2.0],format='(f3.1)')
  bcharwid=widget_droplist(bplrawid,value=values,uvalue='charsize', $
      title='Charsize:')

  wid.bser=widget_button(bplrawid,value='Show errors',uvalue='showerrors', $
      uname='Click this button to show or hide error bars.',sensitive=serr, $
      tracking_events=track)

  batmpwid=widget_base(bplrawid,/row,/base_align_center, $
                       space=0L,xpad=0L,ypad=0L)
  values=['user','100%','99.5%','99%','98%','97%','95%','90%','80%','70%']
  wid.dspc=widget_droplist(batmpwid,value=values,tracking_events=track, $
      uvalue='spectrumyrangedroplist',uname='Select the range of values of' + $
      ' the spectrum, which are used when determining the y-range to plot.')
  wid.tpyl=widget_text(batmpwid,value=' ',/editable,/kbrd_focus_events, $
      xsize=10L,uvalue='spectrumyrange_0',tracking_events=track, $
      uname='This is the lower value of the y-range.',resource_name='field')
  lpll0wid=widget_label(batmpwid,value='-')
  wid.tpyh=widget_text(batmpwid,value=' ',/editable,/kbrd_focus_events, $
      xsize=10L,uvalue='spectrumyrange_1',tracking_events=track, $
      uname='This is the upper value of the y-range.',resource_name='field')

  batmpwid=widget_base(bplrawid,/row,space=0L,xpad=0L,ypad=0L)
  lpll1wid=widget_label(batmpwid,value='x:')
  wid.tprl=widget_text(batmpwid,value=strtrim(wavlow,2L),/editable,xsize=10L, $
      uvalue='wavelowhigh_0',uname='This is the lower value of the shown w' + $
      'avelength range ['+wavunit+'].',tracking_events=track, $     
      resource_name='field',/kbrd_focus_events)
  lpll0wid=widget_label(batmpwid,value='-')
  wid.tprh=widget_text(batmpwid,value=strtrim(wavhig,2L),/editable,xsize=10L, $
      uvalue='wavelowhigh_1',uname='This is the upper value of the shown w' + $
      'avelength range ['+wavunit+'].',tracking_events=track, $
      resource_name='field',/kbrd_focus_events)
  wid.bprr=widget_button(batmpwid,value='Reset',uvalue='wavereset', $
      uname='Press this button to use the minimum-maximum wavelength range' + $
      ' that is available in the data.',tracking_events=track,sensitive=0L)

  batmpwid=widget_base(bplrawid,/row,space=0L,xpad=0L,ypad=0L)
  lpll3wid=widget_label(batmpwid,value='x-shift:')
  str=strtrim(plotdlt*wavdsp,2L)
  wid.bpld=widget_button(batmpwid,value=blfile,uvalue='wavesupdown_0', $
      uname='Press this button to shift the shown wavelength range by -'+ $
      str+' ['+wavunit+'].',tracking_events=track,/bitmap, $
      resource_name='field')
  wid.tpsh=widget_text(batmpwid,value=strtrim(str,2L),/editable, $
      uvalue='wavesval',xsize=5,uname='This value is used when shifting th' + $
      'e wavelength range; given in integer multiples of the bin width ' + $
      strtrim(wavdsp,2L)+' '+wavunit+'.',tracking_events=track, $
      resource_name='field',/kbrd_focus_events)
  wid.bplu=widget_button(batmpwid,value=brfile,uvalue='wavesupdown_1', $
      uname='Press this button to shift the shown wavelength range by '+ $
      str+' ['+wavunit+'].',tracking_events=track,/bitmap, $
      resource_name='field')

  batmpwid=widget_base(bplrawid,/row,space=0L,xpad=0L,ypad=0L)
  lpll4wid=widget_label(batmpwid,value='x-zoom:')
  bzooiwid=widget_button(batmpwid,value=bdfile,uvalue='wavezoominout_0', $
      uname='[Zoom in] Click to make the wavelength range smaller.', $
      tracking_events=track,/bitmap,resource_name='field')
  zoomfac=90.0 & tmpstr=string(zoomfac,format='(f5.1)')
  wid.tzom=widget_text(batmpwid,value=tmpstr,/editable,xsize=5L, $
      uvalue='wavezoomfactor',uname='When clicking the zoom buttons the w' + $
      'avelength range is zoomed in or out by this percentage of the curr' + $
      'ent wavelength range.',tracking_events=track,resource_name='field', $
      /kbrd_focus_events)
  bzooowid=widget_button(batmpwid,value=bufile,uvalue='wavezoominout_1', $
      uname='[Zoom out] Click to make the wavelength range larger.', $
      tracking_events=track,/bitmap,resource_name='field')

  ;;======================
  ;; Realizing the widget:

  widget_control,wid.bbas,/realize
  widget_control,wid.bbas,update=0L
  widget_control,wid.bbas,tlb_get_size=basesize
;  widget_control,wid.dplo,xsize=basesize[0L]-10L

  ;; Retrieving draw widget ids:
  widget_control,wid.dimd,get_value=tmp & wid.imdi=tmp
  widget_control,wid.dspm,get_value=tmp & wid.isma=tmp
  widget_control,wid.dplo,get_value=tmp & wid.iplo=tmp
  for i=0L,nmsave-1L do begin
    widget_control,wid.dsad[i],get_value=tmp
    wid.isad[i]=tmp
  endfor

  widget_control,bcharwid,set_droplist_select=2L

  ;; Calculating the new width of the spectrum image:
  wbase=widget_info(wid.bbas,/geometry)

  wmbar=widget_info(mbbarwid,/geometry)
  wdisp=widget_info(bdispwid,/geometry)
  wdisa=widget_info(bdisawid,/geometry)
  wdipa=widget_info(bdipawid,/geometry)
  wmono=widget_info(bspmawid,/geometry)
  wstat=widget_info(wid.lsta,/geometry)
  wmosa=widget_info(bmosawid,/geometry)
  wbbtm=widget_info(bbtmpwid,/geometry)
  wdimi=widget_info(wid.dimd,/geometry) ;; the vertical image row bar
;  if widget_info(dimdxwid,/valid_id) then $
;     wdimd=widget_info(dimdxwid,/geometry) $ ;; the vertical image row bar
;  else wdimd={xsize:0L}
  wplot=widget_info(wid.dplo,/geometry)
  wplra=widget_info(bplrawid,/geometry)


;;; Colorbar testing:
;  widget_control,dcolowid,get_value=icolowid
;  ocolbobj=obj_new('IDLgrView',viewplane_rect=[0.,0.,400,25])
;  oModel=obj_new('IDLgrModel')
;
;  tvlct,red,green,blue,/get
;  oPalette=obj_new('IDLgrPalette',red,green,blue)
;  oColorbar=obj_new('IDLgrColorbar',palette=oPalette,dimensions=[400,10], $
;                   show_axis=1L)
;
;  oModel->Add,oColorbar
;  ocolbobj->Add,oModel
;  icolowid->Draw,ocolbobj

  ;;==============================--------------------
  ;; The x-size:

  ;; The total number of pixels that is used:
  totxsize=screensize[0L]-15L-2L*(wbase.margin+wbase.xpad)

  ;; The available number of pixels for the image draw widget:
  available_x0=totxsize - $
               wdisp.space-2L*wdisp.xpad - $ ;; the im.display row base widget
               wmono.xsize-                $ ;; the width of the spatial map
                           2L*wdisa.xpad;- $ ;; the image display column
;              wbbtm.space-2L*wbbtm.xpad - $ ;; the image display inner row
;              wdimd.xsize                   ;; the width of the image row bar

  ;; Resizing the spectrum image draw widget:
  wdimw=widget_info(wid.dimd,/geometry)
  scrw=wdimw.scr_xsize-wdimw.xsize ;; The width of the scroll bar
  widget_control,wid.dimd,xsize=(available_x0<wdimw.draw_xsize)-scrw

  ;; Calculating the new width of the spectrum plot (comparing with
  ;; the width of the spectrum image row):
  wdisp=widget_info(bdispwid,/geometry)
  available_x1=totxsize
  widget_control,wid.dplo,xsize=available_x1<wdisp.xsize

  ;;==============================--------------------
  ;; The y-size:

  ;; The following works for as long as the base widget is not offset
  ;; in the y-direction by default:
  totysize=screensize[1L]-wbase.yoffset-2L*(wbase.margin+wbase.ypad)- $
           4L*wbase.space - 30L

  available_y1=totysize    - $  ;; the avaliable number of pixles in y
               wmbar.ysize - $  ;; the vertical size of the menu bar
               wstat.ysize - $  ;; the vertical size of the status line
               wmosa.ysize - $  ;; the vertical size of the spatial map-bar
               wplra.ysize - $  ;; the vertical size of the plot control bar
               wdisp.ysize      ;; the vertical size of the spectrum image row

  widget_control,wid.dplo,ysize=available_y1<symax

  ;;==============================--------------------
  ;; Setting the horizontal size of the status label widget:

  wbase=widget_info(wid.bbas,/geometry)
  tmpval=long(wbase.xsize)
  widget_control,wid.lsta,xsize=tmpval

  widget_control,wid.bbas,update=1L

  ;;======================
  ;; Setting the minimum and maximum values of the spatial map:

  ok=0L
  if n_elements(colorcut__) ne 0L then begin
    ok=1L
    if n_elements(colorcut__) ne 2L then begin
      ok=0L
    endif else begin
      colorcut=float(colorcut__)
      colmin=min(colorcut,max=colmax)
    endelse
  endif

  ;; Calculating the color clipping range (histogram):
  if ~ok then begin
    widget_control,wid.dcom,set_droplist_select=4L ;; Setting default to 98%
    p3d_misc_dataclip,image,range,percentage=98,verbose=verbose,error=error, $
                      debug=debug
    if error ne 0 then return

    colmin=range[0L]
    colmax=range[1L]
  endif else begin
    widget_control,wid.dcom,set_droplist_select=0L ;; Setting default to 'user'
  endelse

  cmanual=ok?1L:0L ;; Are colors user-defined or changed 'automatically'?

  mcolmin=colmin
  mcolmax=colmax

  ;; Updating the color value widgets:
  widget_control,wid.tcoi,set_value=strtrim(colmin,2L)
  widget_control,wid.tcoa,set_value=strtrim(colmax,2L)

  ;;======================
  ;; Setting the minimum and maximum values of the spectrum plot:

  ok=0L
  if n_elements(specrange__) ne 0L then begin
    ok=1L
    if n_elements(specrange__) ne 2L then begin
      ok=0L
    endif else begin
      specrange=float(specrange__)
      specymin=min(specrange,max=specymax)
    endelse
  endif

  if ~ok then begin
    widget_control,wid.dspc,set_droplist_select=4L ;; Setting default to 98%
    p3d_misc_dataclip,spec,range,percentage=98,verbose=verbose,error=error, $
                      debug=debug
    if error ne 0 then return

    specymin=range[0L]
    specymax=range[1L]
  endif else begin
    widget_control,wid.dspc,set_droplist_select=0L ;; Setting default to 'user'
  endelse

  smanual=ok?1L:0L ;; Are colors user-defined or changed 'automatically'?

  ;; Updating the plot range widgets:
  widget_control,wid.tpyl,set_value=strtrim(specymin,2L)
  widget_control,wid.tpyh,set_value=strtrim(specymax,2L)

  ;; Saving all settings in a state variable:
  state={image:image,dimage:dimage,si:size(image),hdr:ptr_new(hdr), $
         nspec:nspec,nbins:nbins,pos:pos,colmin:colmin,colmax:colmax, $
         marr:ptr_new(),marrbin:[0L,0L],mxsize:mxsize,mysize:mysize, $
         rxsize:rxsize,rysize:rysize,orientation:orientation, $
         mcolmin:mcolmin,mcolmax:mcolmax,mcol:mcol,mrow:mrow,mfid:mfid, $
         curspec:ptr_new(curspec),addspec:ptr_new(addspec),spec:ptr_new(), $
         curspee:ptr_new(curspee),addspee:ptr_new(addspee),spee:ptr_new(), $
         dmarr:ptr_new(),nmsave:nmsave,msave:msave,mwaves:mwaves, $
         meanspec:0L,meannumspec:1d0,wavarr:wavarr,wavunit:wavunit, $
         addmode:0,wavlow:wavlow,wavhig:wavhig,wav0bn:wav0bn,wavdsp:wavdsp, $
         plotlow:plotlow,plothig:plothig,plotcen:plotcen,plotdlt:plotdlt, $
         specymin:specymin,specymax:specymax,bandwidth:0L,impos:ptr_new(mrow),$
         xoflo:xoflo,xofhi:xofhi,xofwid:xofwid, $
         rimage:ptr_new(),rdimage:ptr_new(),subtracted:0L,serr:serr, $
         showerrors:0L, $
         ;; widget ids:
         wid:wid,zoomfac:zoomfac,cbottom:bottom,colors:colors, $
         colortable:colortable, $
         cindex:cindex,cindv:cindv,cmanual:cmanual,smanual:smanual, $
         xsave:-1d0,angstr:angstr,charsize:charsize,enteredspatmap:0L, $
         buttonmode:0b,pressed:0L,mpressed:0b,savx:!x,savy:!y,savp:!p, $
         datapath:datapath,fitsheadertitle:fitsheadertitle, $
         top:wid.bbas,debug:debug,verbose:verbose}
  state.spec=state.curspec
  if state.serr then state.spee=state.curspee

  ;; Storing state information:
  state=ptr_new(state,/no_copy)
  widget_control,wid.bbas,set_uvalue=state


  ;;========================================------------------------------
  ;; Making first draws in the image windows:

  p3d_rss_plot,state,/imageplot,/imagesmap,/specplot,topwid=wid.bbas

  ;;========================================------------------------------
  ;; Managing the widget events:

  xmanager,'p3d_rss',wid.bbas,/no_block,cleanup='p3d_rss_cleanup'

  widget_control,wid.bbas,/sensitive

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_rss
